/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Alan W Black                           */
/*                      Date   :  April 1996                             */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Basic intonation utilities common between different                   */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"
#include "intonation.h"
#include "modules.h"
#include "lexicon.h"

EST_Stream_Item *add_target(EST_Utterance &u, float pos,float val)
{
    EST_Stream_Item item;
    
    item.init("Target");
    EST_String sval = ftoString(val);
    item.set_name(sval);
    item.set_end(pos);

    u.stream("Target").append(item);
    return u.stream("Target").tail();

}

EST_Stream_Item *add_IntEvent(EST_Utterance &u,const EST_String &label)
{
    EST_Stream_Item item;
    
    item.init("IntEvent");
    item.set_name(label);

    u.stream("IntEvent").append(item);
    return u.stream("IntEvent").tail();
}

void add_IntEvent_to_syl(EST_Utterance &u,EST_Stream_Item &syl,const EST_String &label)
{
    // Add an IntEvent to this syllable

    link(*add_IntEvent(u,label),syl);
}

EST_Val ff_position_type(EST_Utterance &u,EST_Stream_Item &s)
{
    /* Position of syllable in this word: initial, mod, final */
    int iaddr = ffeature(u,s,"Word.Syllable.addr");
    int faddr = ffeature(u,s,"Word.Syllable:last.addr");

    if ((s.addr() == iaddr) && (s.addr() == faddr))
	return EST_Val("single");
    else if (s.addr() == iaddr)
	return EST_Val("initial");
    else if (s.addr() == faddr)
	return EST_Val("final");
    else 
	return EST_Val("mid");
}

EST_Val ff_lex_stress(EST_Utterance &u,EST_Stream_Item &s)
{
    // Part of an experiment.  This word in the lexion and find out 
    // what the stress markings are in there for this syllable 
    EST_String word = ffeature(u,s,"Word.name");
    LISP entry = lex_lookup_word(word,NIL);
    LISP lex_syls = car(cdr(cdr(entry)));
    int num_syls = ffeature(u,s,"Word.Syllable:num");
    int word_pos = ffeature(u,s,"pos_in_word");

    if (num_syls == siod_llength(lex_syls))
    {
	/* Lets brashly assume that these are therefore the same */
	/* syllably structure  */
	return EST_Val(get_c_string(car(cdr(siod_nth(word_pos,lex_syls)))));
    }
    /* Hmm what should be do here ? */
    else if (word_pos < siod_llength(lex_syls))
    {   
	cwarn << "Syllable mismatch for " << word << endl;
	return EST_Val(get_c_string(car(cdr(siod_nth(word_pos,lex_syls)))));
    }
    else
    {
	cwarn << "Failed syllable mismatch for " << word << endl;
	return EST_Val("0");
    }
}

EST_Val ff_syl_break(EST_Utterance &u,EST_Stream_Item &s)
{
    // 0 internal syl end, 1 word end, 4 phrase end (ToBI 3 and 4)
    EST_Relation *words = s.link("Word");
    EST_Relation *syls;

    if (words->head() == 0)
	return EST_Val(0); // no word so assumes its standalone
    else
    {
	syls = u.ritem("Word",(*words)(words->head())).link("Syllable");
	if ((syls->head() != 0) &&
	    (s.addr() == u.ritem("Syllable",(*syls)(syls->tail())).addr()))
	{
	    EST_String ss = ff_word_break(u,u.ritem("Word",(*words)(words->head()))).string();
	    const char *sc= ss;
	    if ((sc[0] == '1') || (sc[0] =='2'))
		return EST_Val(1);
	    else if (sc[0] == '3')
		return EST_Val(3);
	    else
		return EST_Val(4);
	}
	else
	    return EST_Val(0);
    }
}

EST_Val ff_syl_accented(EST_Utterance &u,EST_Stream_Item &s)
{
    // t if syllable is accented or not 
    (void)u; // unused parameter

    if (s.link("IntEvent")->head() != 0)
	return EST_Val(1);
    else
	return EST_Val(0);
}

static EST_Val ff_tobi_accent(EST_Utterance &u,EST_Stream_Item &s)
{
    // First tobi accent related to syllable
    EST_Relation *i_rels;
    EST_Stream_Item ievent;
    EST_TBI *p;

    i_rels = (s.link("IntEvent"));

    for (p=i_rels->head(); p != 0; p=next(p))
    {
	ievent = u.ritem("IntEvent",(*i_rels)(p));
	if (ievent.name().contains("*"))
	    return EST_Val(ievent.name());
    }

    return EST_Val("NONE");

}

static EST_Val ff_tobi_endtone(EST_Utterance &u,EST_Stream_Item &s)
{
    // First tobi endtone (phrase accent or boundary tone)
    EST_Relation *i_rels;
    EST_Stream_Item ievent;
    EST_TBI *p;

    i_rels = (s.link("IntEvent"));

    for (p=i_rels->head(); p != 0; p=next(p))
    {
	ievent = u.ritem("IntEvent",(*i_rels)(p));
	if (ievent.name().contains("%") ||
	    ievent.name().contains("-"))
	    return EST_Val(ievent.name());
    }

    return EST_Val("NONE");

}

EST_Val ff_syl_accent(EST_Utterance &u,EST_Stream_Item &s)
{
    // (first) accent or NONE on given syllable
    EST_Relation *i_rels;
    EST_Stream_Item IntEvent;

    i_rels = (s.link("IntEvent"));

    if (i_rels->head() == 0)
	return EST_Val("NONE");
    else if (i_rels->length() > 0)
    {
	IntEvent = u.ritem("IntEvent", (*i_rels)(i_rels->head()));
	return EST_Val(IntEvent.name());
    }
    else
	return EST_Val("multi");
}

static EST_Val ff_syl_numphones(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of phones in syllable
    (void)u; // unused parameter
    EST_Relation *segs;

    segs = s.link("Segment");

    return EST_Val(segs->length());
}

static EST_Val ff_syl_onsetsize(EST_Utterance &u,EST_Stream_Item &s)
{
    // number of segments in the onset
    EST_Relation *segs = s.link("Segment");
    int size;
    EST_TBI *p;

    for (p=segs->head(),size=0; p != 0; p=next(p),size++)
    {
	if (ph_is_vowel(u.ritem("Segment",(*segs)(p)).name()))
	    return EST_Val(size);
    }
    
    return EST_Val(size);

}

static EST_Val ff_syl_vowel(EST_Utterance &u,EST_Stream_Item &s)
{
    // the vowel in the syllable
    EST_Relation *segs = s.link("Segment");
    EST_TBI *p;

    for (p=segs->head(); p != 0; p=next(p))
    {
	const EST_Stream_Item &seg = u.ritem("Segment",(*segs)(p));
	if (ph_is_vowel(seg.name()))
	    return EST_Val(seg.name());
    }

    // no vowel 
    return EST_Val("novowel");

}

static EST_Val ff_syl_codasize(EST_Utterance &u,EST_Stream_Item &s)
{
    // number of segments in the coda
    EST_Relation *segs = s.link("Segment");
    int size;
    EST_TBI *p;

    for (p=segs->tail(),size=0; p != 0; p=prev(p),size++)
    {
	if (ph_is_vowel(u.ritem("Segment",(*segs)(p)).name()))
	    return EST_Val(size);
    }
    
    return EST_Val(size);

}

static EST_Val ff_syl_pc_unvox(EST_Utterance &u,EST_Stream_Item &s)
{
    // Returns percentage of syllable from start to first voiced phone
    EST_Relation *segs = s.link("Segment");
    float dur = s.dur();
    float unvox = 0;
    EST_TBI *p;

    for (p=segs->head(); p != 0; p=next(p))
    {
	const EST_Stream_Item &seg = u.ritem("Segment",(*segs)(p));
	if ((ph_is_vowel(seg.name())) ||
	    (ph_is_voiced(seg.name())))
	    break;
	unvox += seg.dur();
    }
    
    return EST_Val((int)(unvox/dur*100));
}

static EST_Val ff_syl_vowel_start(EST_Utterance &u,EST_Stream_Item &s)
{
    // Returns start time of vowel in syllable (or start of syllable)
    EST_Relation *segs = s.link("Segment");
    EST_TBI *p;

    for (p=segs->head(); p != 0; p=next(p))
    {
	const EST_Stream_Item &seg = u.ritem("Segment",(*segs)(p));
	if (ph_is_vowel(seg.name()))
	    return EST_Val((float)seg.start());
    }
    // There isn't a vowel, so just take start of syl
    return EST_Val((float)s.start());
}

static EST_Val ff_seg_onsetcoda(EST_Utterance &u,EST_Stream_Item &s)
{
    // onset if seg in onset, coda otherwise (vowel is in coda)
    EST_Relation *syls = s.link("Syllable");
    EST_Relation *segs;
    EST_TBI *p;

    if (syls->head() == 0)
	return EST_Val("onset");
    else
    {
	segs = u.ritem("Syllable", (*syls)(syls->head())).link("Segment");
	for (p=segs->head(); p != 0; p=next(p))
	{
	    if (ph_is_vowel(u.ritem("Segment",(*segs)(p)).name()))
		return EST_Val("coda");
	    if (u.ritem("Segment",(*segs)(p)).addr() == s.addr())
		return EST_Val("onset");
	}
	cwarn << "Segment not in its own syllable\n" << endl;;
	return EST_Val("onset");
    }
}

static EST_Val ff_seg_pos_in_syl(EST_Utterance &u,EST_Stream_Item &s)
{
    // position of segment in syllable
    EST_Relation *syls = s.link("Syllable");
    EST_Relation *segs;
    EST_TBI *p;
    int addr = s.addr(), pos=0;

    if (syls->head() == 0)
	return EST_Val("0");
    else
    {
	segs = u.ritem("Syllable", (*syls)(syls->head())).link("Segment");
	for (pos=0,p=segs->head(); p != 0; p=next(p),pos++)
	{
	    if (u.ritem("Segment",(*segs)(p)).addr() == addr)
		return EST_Val(pos);
	}
	cwarn << "Segment not in its own syllable" << endl;;
	return EST_Val(pos);
    }
}

static EST_Val ff_seg_syl_initial(EST_Utterance &u,EST_Stream_Item &s)
{
    // 1 if seg is syllable initial, 0 otherwise.
    EST_Relation *syls = s.link("Syllable");
    EST_Relation *segs;
    int addr = s.addr();

    if (syls->head() == 0)
	return EST_Val("1");
    else
    {
	segs = u.ritem("Syllable", (*syls)(syls->head())).link("Segment");
	if (u.ritem("Segment",(*segs)(segs->head())).addr() == addr)
	    return EST_Val("1");
	else
	    return EST_Val("0");
    }
}

static EST_Val ff_seg_syl_final(EST_Utterance &u,EST_Stream_Item &s)
{
    // 1 if seg is syllable initial, 0 otherwise.
    EST_Relation *syls = s.link("Syllable");
    EST_Relation *segs;
    int addr = s.addr();

    if (syls->head() == 0)
	return EST_Val("1");
    else
    {
	segs = u.ritem("Syllable", (*syls)(syls->tail())).link("Segment");
	if (u.ritem("Segment",(*segs)(segs->tail())).addr() == addr)
	    return EST_Val("1");
	else
	    return EST_Val("0");
    }
}

static EST_Val ff_syl_pos_in_word(EST_Utterance &u,EST_Stream_Item &s)
{
    // position of syllable in word
    EST_Relation *words = s.link("Word");
    EST_Relation *syls;
    EST_TBI *p;
    int addr = s.addr(), pos=0;

    if (words->head() == 0)
	return EST_Val("0");
    else
    {
	syls = u.ritem("Word", (*words)(words->head())).link("Syllable");
	for (pos=0,p=syls->head(); p != 0; p=next(p),pos++)
	{
	    if (u.ritem("Syllable",(*syls)(p)).addr() == addr)
		return EST_Val(pos);
	}
	cwarn << "Syllable not in its own word" << endl;;
	return EST_Val(pos);
    }
}

static EST_Val ff_syl_midpitch(EST_Utterance &u,EST_Stream_Item &s)
{
    // pitch of mid vowel in syllable
    EST_Relation *segs = s.link("Segment");
    EST_TBI *p;

    for (p=segs->head(); p != 0; p=next(p))
    {
	if (ph_is_vowel(u.ritem("Segment",(*segs)(p)).name()))
	    return ffeature(u,u.ritem("Segment",(*segs)(p)),"Target.name");
    }

    // must be a silence or a syllabic consonant

    return EST_Val(0.0);

}

static EST_Val ff_syl_startpitch(EST_Utterance &u,EST_Stream_Item &s)
{
    // pitch at start of syllable 
    // average of first segment and previous segment target (if exists)
    EST_Relation *segs = s.link("Segment");
    EST_Stream_Item *seg;
    EST_TBI *p;
    float target;

    if (segs->head() == 0)
	return EST_Val(0.0);
    else
    {
	for (p=segs->head(); p!=0; p=next(p))
	    if ((int)ffeature(u,u.ritem("Segment",(*segs)(p)),"Target.name")
			      != -1)
		break;
	if (p == 0) return EST_Val(-1.0);
	seg = &u.ritem("Segment",(*segs)(p));
	if (prev(seg) == 0)
	    return ffeature(u,*seg,"Target.name");
	else
	{
	    target = ((float)ffeature(u,*seg,"Target.name") +
		(float)ffeature(u,*(prev(seg)),"Target.name"))/2.0;
	    return EST_Val(target);
	}
    }
}

static EST_Val ff_syl_endpitch(EST_Utterance &u,EST_Stream_Item &s)
{
    // pitch at end of syllable 
    // average of kast segment and next segment target (if exists)
    EST_Relation *segs = s.link("Segment");
    EST_Stream_Item *seg;
    float target;
    EST_TBI *p;

    if (segs->tail() == 0)
	return EST_Val(0.0);
    else
    {
	for (p=segs->tail(); p!=0; p=prev(p))
	    if ((int)ffeature(u,u.ritem("Segment",(*segs)(p)),"Target.name")
			      != -1)
		break;
	if (p==0) return EST_Val(-1.0);
	seg = &u.ritem("Segment",(*segs)(p));
	if (next(seg) == 0)
	    return ffeature(u,*seg,"Target.name");
	else
	{
	    target = ((float)ffeature(u,*seg,"Target.name") +
		(float)ffeature(u,*(next(seg)),"Target.name"))/2.0;
	    return EST_Val(target);
	}
    }
}

static EST_Val ff_seg_pitch(EST_Utterance &u,EST_Stream_Item &s)
{
    // Return interpolated pitch at mid-point of s
    EST_Stream_Item *t,*lastt;
    float spoint,deltaf0,deltatime;

    for (lastt=t=u.stream("Target").head(); next(t) != 0; t=next(t))
    {
	if (s.mid() <= t->end())
	    break;
	lastt=t;
    }

    if (lastt == 0)
	return EST_Val((float)0.0);

    deltaf0 = atof(t->name())-atof(lastt->name());
    deltatime = t->end() - lastt->end();
    if (deltatime <= 0)
	spoint = atof(lastt->name());
    else
	spoint = atof(lastt->name()) + 
	    (deltaf0*((s.mid()-lastt->end())/deltatime));

    if (spoint > 35)
	return EST_Val(spoint);
    else
	return EST_Val((float)0.0);
}

static EST_Val ff_syl_in(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of syllables to since last phrase break 
    int count;
    EST_Stream_Item *p;
    int sb;

    for (count=0,p=prev(&s); p != 0; p=prev(p),count++)
    {
	sb = ff_syl_break(u,*p);
	if ((sb != 1) && (sb != 0))
	    return EST_Val(count);
    }
    return EST_Val(count);
}

static EST_Val ff_syl_out(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of syllables since last phrase break 
    int count;
    EST_Stream_Item *p;
    int sb;

    for (count=0,p=&s; p != 0; p=next(p),count++)
    {
	sb = ff_syl_break(u,*p);
	if ((sb != 1) && (sb != 0))
	    return EST_Val(count);
    }
    return EST_Val(count);
}

static EST_Val ff_ssyl_in(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of stressed syllables since last phrase break 
    int count;
    EST_Stream_Item *p;
    int sb;

    for (count=0,p=prev(&s); p != 0; p=prev(p))
    {
	sb = ff_syl_break(u,*p);
	if ((sb != 1) && (sb != 0))
	    return EST_Val(count);
	if (p->feature("stress") == "1")
	    count ++;
    }
    return EST_Val(count);
}

static EST_Val ff_ssyl_out(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of stressed syllables to next phrase break 
    int count;
    EST_Stream_Item *p;
    int sb;

    for (count=0,p=&s; p != 0; p=next(p))
    {
	sb = ff_syl_break(u,*p);
	if ((sb != 1) && (sb != 0))
	    return EST_Val(count);
	if (p->feature("stress") == "1")
	    count ++;
    }
    return EST_Val(count);
}

static EST_Val ff_asyl_in(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of accented syllables since last phrase break 
    int count;
    EST_Stream_Item *p;
    int sb;

    for (count=0,p=prev(&s); p != 0; p=prev(p))
    {
	sb = ff_syl_break(u,*p);
	if ((sb != 1) && (sb != 0))
	    return EST_Val(count);
	if (ff_syl_accented(u,*p) == 1)
	    count ++;
    }
    return EST_Val(count);
}

static EST_Val ff_asyl_out(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of accented syllables to next phrase break 
    int count;
    EST_Stream_Item *p;
    int sb;

    for (count=0,p=&s; p != 0; p=next(p))
    {
	sb = ff_syl_break(u,*p);
	if ((sb != 1) && (sb != 0))
	    return EST_Val(count);
	if (ff_syl_accented(u,*p) == 1)
	    count ++;
    }
    return EST_Val(count);
}
	    
static EST_Val ff_last_accent(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of syllables since last accented syllable
    int count;
    EST_Stream_Item *p;

    for (count=0,p=prev(&s); p != 0; p=prev(p),count++)
	if (ff_syl_accented(u,*p) == 1)
	    return EST_Val(count);

    return EST_Val(count);
}

static EST_Val ff_next_accent(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of syllables to next accented syllable
    int count;
    EST_Stream_Item *p;

    for (count=0,p=next(&s); p != 0; p=next(p),count++)
	if (ff_syl_accented(u,*p) == 1)
	    return EST_Val(count);

    return EST_Val(count);
}

static EST_Val ff_sub_phrases(EST_Utterance &u,EST_Stream_Item &s)
{
    // Number of non-major phrase breaks since last major phrase break
    int count;
    EST_Stream_Item *p;
    int sb;

    for (count=0,p=prev(&s); p != 0; p=prev(p))
    {
	sb = ff_syl_break(u,*p);
	if (sb == 4)
	    return EST_Val(count);
	if ((sb == 3) || (sb == 2))
	    count ++;
    }

    return EST_Val(count);
}


LISP FT_Intonation_Default_Utt(LISP args);
LISP FT_Int_Targets_Default_Utt(LISP args);
LISP FT_Intonation_Simple_Utt(LISP args);
LISP FT_Int_Targets_Simple_Utt(LISP args);
LISP FT_Intonation_Tree_Utt(LISP args);
LISP FT_Int_Targets_LR_Utt(LISP args);
LISP FT_Int_Targets_General_Utt(LISP utt);

void festival_Intonation_init(void)
{

    festival_def_utt_module("Intonation_Default",FT_Intonation_Default_Utt,
    "(Intonation_Default UTT)\n\
  this method is such a bad intonation module that it does nothing at all.\n\
  This utterance module is called when the Parameter Int_Method is not\n\
  set or  set to Default.  This module is called through the Intonation\n\
  module. [see Default intonation]");
    festival_def_utt_module("Int_Targets_Default",FT_Int_Targets_Default_Utt,
    "(Int_Targets_Default UTT)\n\
  This module creates two Targets causing a simple downward continuous\n\
  F0 through the whole utterance.  The code is in an appropriate named file\n\
  called duffint.  This module is called when the Parameter\n\
  Int_Method is not set or set to Default.  This module is called through\n\
  the Int_Targets module.  Optional parameters for a start value (default\n\
  130) and end value (default 110) may be set in the variable\n\
  diffint_params.  This can be used to generate a monotone intonation\n\
  with a setting like (set! duffint_params '((start 100) (end 100))).\n\
  [see Default intonation]");
    festival_def_utt_module("Intonation_Simple",FT_Intonation_Simple_Utt,
    "(Intonation_Simple)\n\
  Assign accents to each content word, creating an IntEvent stream. This \n\
  utterance module is called when the Parameter Int_Method is set to \n\
  Simple.  This module is called through the Intonation module.\n\
  [see Simple intonation]");
    festival_def_utt_module("Int_Targets_Simple",FT_Int_Targets_Simple_Utt,
    "(Int_Targets_Simple UTT)\n\
  Naively add targets for hat shaped accents for each accent in the \n\
  IntEvent stream.  This module is called when the Parameter Int_Method is\n\
  set to Simple.  This module is called through the Int_Targets module.\n\
  [see Simple intonation]");
    festival_def_utt_module("Int_Targets_General",FT_Int_Targets_General_Utt,
    "(Int_Targets_General UTT)\n\
  Add targets based on the functions defined in int_general_params.  This\n\
  method allows quite detailed control over the general of targets per\n\
  syllable, see manual for details and examples.  This module is called\n\
  when the Parameter Int_Method is set to General.  This module is called\n\
  through the Int_Targets module. [see General intonation]");
    festival_def_utt_module("Intonation_Tree",FT_Intonation_Tree_Utt,
    "(Intonation_Tree UTT)\n\
  Use the CART trees in int_tone_cart_tree and int_accent_cart_tree to\n\
  create an IntEvent stream of tones and accents related to syllables.\n\
  This module is called through the Intonation module and is selected\n\
  when the Parameter Int_Method is ToBI. [see Tree intonation]");
    festival_def_utt_module("Int_Targets_LR",FT_Int_Targets_LR_Utt,
    "(Int_Targets_LR UTT)\n\
  Predict Target F0 points using linear regression from factors such as\n\
  accent, tone, stress, position in phrase etc.  This utterance module is\n\
  called through the module Int_Targets when the Parameter Int_Method is\n\
  set to ToBI, even though this technique is not restricted to the ToBI\n\
  labelling system. [see Tree intonation]");

    // Intonation feature functions 
    festival_def_ff("accented","Syllable",ff_syl_accented,
    "Syllable.accented\n\
  Returns 1 if syllable is accented, 0 otherwise.  A syllable is\n\
  accented if there is at least one IntEvent related to it.");
    festival_def_ff("syl_accent","Syllable",ff_syl_accent,
    "Syllable.syl_accent\n\
  Returns the name of the accent related to the syllable.  NONE is returned\n\
  if there are no accents, and multi is returned if there is more than one.");
    festival_def_ff("tobi_accent","Syllable",ff_tobi_accent,
    "Syllable.tobi_accent\n\
  Returns the ToBI accent related to syllable.  ToBI accents are\n\
  those which contain a *.  NONE is returned if there are none.  If\n\
  there is more than one ToBI accent related to this syllable the\n\
  first one is returned.");
    festival_def_ff("tobi_endtone","Syllable",ff_tobi_endtone,
    "Syllable.tobi_endtone\n\
  Returns the ToBI endtone related to syllable.  ToBI end tones are\n\
  those IntEvent labels which contain a % or a - (i.e. end tones or\n\
  phrase accents).  NONE is returned if there are none.  If\n\
  there is more than one ToBI end tone related to this syllable the\n\
  first one is returned.");
    festival_def_ff("syl_onsetsize","Syllable",ff_syl_onsetsize,
    "Syllable.syl_onsetsize\n\
  Returns the number of segments before the vowel in this syllable.  If\n\
  there is no vowel in the syllable this will return the total number\n\
  of segments in the syllable.");
    festival_def_ff("syl_vowel","Syllable",ff_syl_vowel,
    "Syllable.syl_vowel\n\
  Returns the name of the vowel within this syllable.  Note this is not\n\
  the general form you probably want.  You can't refer to ph_* features \n\
  of this.  Returns \"novowel\" is no vowel can be found.");
    festival_def_ff("syl_codasize","Syllable",ff_syl_codasize,
    "Syllable.syl_codasize\n\
  Returns the number of segments after the vowel in this syllable.  If\n\
  there is no vowel in the syllable this will return the total number\n\
  of segments in the syllable." );
    festival_def_ff("seg_onsetcoda","Segment",ff_seg_onsetcoda,
    "Segment.onsetcoda\n\
  Returns onset if this segment is before the vowel in the syllable it\n\
  is contained within.  Returns coda if it is the vowel or after.  If\n\
  the segment is not in a syllable it returns onset.");
    festival_def_ff("syl_numphones","Syllable",ff_syl_numphones,
    "Syllable.syl_numphones\n\
  Returns number of phones in syllable.");
    festival_def_ff("syl_pc_unvox","Syllable",ff_syl_pc_unvox,
    "Syllable.syl_pc_unvox\n\
  Percentage of total duration of unvoiced segments from\n\
  start of syllable. (i.e. percentage to start of first voiced segment)");
    festival_def_ff("syl_vowel_start","Syllable",ff_syl_vowel_start,
    "Syllable.syl_vowel_start\n\
  Start position of vowel in syllable.  If there is no vowel the start\n\
  position of the syllable is returned.");
    festival_def_ff("syl_in","Syllable",ff_syl_in,
    "Syllable.syl_in\n\
  Returns number of syllables since last phrase break.  This is 0 if\n\
  this syllable is phrase initial.");
    festival_def_ff("syl_out","Syllable",ff_syl_out,
    "Syllable.syl_out\n\
  Returns number of syllables to next phrase break.  This is 0 if\n\
  this syllable is phrase final.");
    festival_def_ff("ssyl_in","Syllable",ff_ssyl_in,
    "Syllable.ssyl_in\n\
  Returns number of stressed syllables since last phrase break, not\n\
  including this one.");
    festival_def_ff("ssyl_out","Syllable",ff_ssyl_out,
    "Syllable.ssyl_out\n\
  Returns number of stressed syllables to next phrase break, not including\n\
  this one.");
    festival_def_ff("asyl_in","Syllable",ff_asyl_in,
    "Syllable.asyl_in\n\
  Returns number of accented syllables since last phrase break, not\n\
  including this one.  Accentedness is as defined by the syl_accented\n\
  feature.");
    festival_def_ff("asyl_out","Syllable",ff_asyl_out,
    "Syllable.asyl_in\n\
  Returns number of accented syllables to the next phrase break, not\n\
  including this one.  Accentedness is as defined by the syl_accented\n\
  feature.");
    festival_def_ff("last_accent","Syllable",ff_last_accent,
    "Syllable.last_accent\n\
  Returns the number of syllables since last accented syllable.");
    festival_def_ff("next_accent","Syllable",ff_next_accent,
    "Syllable.next_accent\n\
  Returns the number of syllables to the next accented syllable.");
    festival_def_ff("sub_phrases","Syllable",ff_sub_phrases,
    "Syllable.sub_phrases\n\
  Returns the number of non-major phrase breaks since last major\n\
  phrase break.  Major phrase breaks are 4, as returned by syl_break,\n\
  minor phrase breaks are 2 and 3.");
    festival_def_ff("start_f0","Syllable",ff_syl_startpitch,
    "Syllable.start_f0\n\
  The pitch value at the start of this syllable.  This is found from\n\
  the average of the target related to first segment related to this \n\
  syllable and the previous segment (if it exists).");
    festival_def_ff("pitch","Segment",ff_seg_pitch,
    "Segment.pitch\n\
  The mid-pitch of this segment found by interpolating the previous and\n\
  next target points.  Interpolation is linear.");
    festival_def_ff("mid_f0","Syllable",ff_syl_midpitch,
    "Syllable.mid_f0\n\
  The pitch value at the middle of the vowel in this syllable.  In no\n\
  vowel is found 0.0 is returned.");
    festival_def_ff("end_f0","Syllable",ff_syl_endpitch,
    "Syllable.end_f0\n\
  The pitch value at the end of this syllable.  This is found from\n\
  the average of the last target related to thsi syllable and the next\n\
  target.");
    festival_def_ff("syl_break","Syllable",ff_syl_break,
    "Syllable.syl_break\n\
  The break level after this syllable.  Word internal is syllables\n\
  return 0, non phrase final words return 1.  Final syllables in \n\
  phrase final words return the name of the phrase they are related to.\n\
  Note the occasional \"-\" that may appear of phrase names is removed\n\
  so that this feature function returns a number in the range 0,1,2,3,4.");
    festival_def_ff("pos_in_syl","Segment",ff_seg_pos_in_syl,
    "Segment.pos_in_syl\n\
  The position of this segment in the syllable it is related to.  The index\n\
  counts from 0.  If this segment is not related to a syllable this \n\
  returns 0.");
    festival_def_ff("syl_initial","Segment",ff_seg_syl_initial,
    "Segment.syl_initial\n\
  Returns 1 if this segment is the first segment in the syllable it\n\
  is related to, or if it is not related to any syllable.");
    festival_def_ff("syl_final","Segment",ff_seg_syl_final,
    "Segment.syl_final\n\
  Returns 1 if this segment is the last segment in the syllable it\n\
  is related to, or if it is not related to any syllable.");
    festival_def_ff("pos_in_word","Syllable",ff_syl_pos_in_word,
    "Syllable.pos_in_word\n\
  The position of this syllable in the word it is related to.  The index\n\
  counts from 0.  If this syllable is not related to a word then 0 is\n\
  returned.");
    festival_def_ff("position_type","Syllable",ff_position_type,
    "Syllable.position_type\n\
  The type of syllable with respect to the word it it related to.  This\n\
  may be any of: single for single syllable words, initial for word\n\
  initial syllables in a poly-syllabic word, final for word final\n\
  syllables in poly-syllabic words, and mid for syllables within \n\
  poly-syllabic words.");
    festival_def_ff("lex_stress","Syllable",ff_lex_stress,
    "Syllable.lex_stress\n\
  An experimental feature, *not* simply lexical stress, use \"stress\"\n\
  for that.  This feature function looks up the related word in the\n\
  lexicon and returns the lexical stress of the corresponding syllable\n\
  in the lexical entry.");
}
    
