/*
 *  Sarien AGI :: Copyright (C) 1999 Dark Fiber 
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sarien.h"
#include "agi.h"
#include "lzw.h"
#include "words.h"
#include "objects.h"
#include "picture.h"
#include "view.h"
#include "logic.h"
#include "sound.h"
#include "gfx.h"
#include "console.h"

UINT16	agi_v3_init (void);
UINT16	agi_v3_deinit (void);
UINT16	agi_v3_detect_game (UINT8 *);
UINT16	agi_v3_load_resource (UINT8, UINT16);
UINT16	agi_v3_unload_resource (UINT8, UINT16);
UINT16	agi_v3_load_dir (AGI_DIR *, FILE *, UINT32, UINT32);
UINT8	*agi_v3_load_vol_res (AGI_DIR *);


AGI_LOADER agi_v3 = {
	3,
	0,
	agi_v3_init,
	agi_v3_deinit,
	agi_v3_detect_game,
	agi_v3_load_resource,
	agi_v3_unload_resource
};


UINT16 agi_v3_detect_game(UINT8 *gn)
{
	UINT16	ec=err_Unk;
	char	x[256];
	UINT8	*xname;
	short	l;

	_D (("(\"%s\")", gn));
	gdir=(UINT8*)strdup((char*)gn);

	strcpy(x, (char*)"*vol.0");
	fixpath(GAMEDIR, (UINT8*)x);
	if(__file_exists(path)==0)
	{
		xname=__file_name(path);
		/* remove the DIR from xname */
		l=strlen((char*)xname);
		if(l>=5)
			l-=5;
		xname[l]=0;
		gname=xname;

		loader->int_version=0x3149;		/* setup for 3.002.149 */
		ec=v3id_game();
	}
	else
		ec=err_InvalidAGIFile;

	return ec;
}



UINT16 agi_v3_init(void)
{
	UINT16	ec=err_OK;

	struct	AGI3VOL
	{
		UINT32	sddr;
		UINT32	len;
	} agi_vol3[4];

	FILE *fp;
	UINT16 i;
	UINT16	xd[4];

	fixpath(GAMEDIR, (UINT8*)DIR);
	/*printf("fileopen : [%s]\n", path);*/
	fp=fopen((char*)path, "rb");
	if(fp!=NULL)
	{
		/* build offset table for v3 directory format */
		fread(&xd, 1, 8, fp);
		fseek(fp, 0x0L, SEEK_END);

		for(i=0; i<4; i++)
			agi_vol3[i].sddr=xd[i];

		agi_vol3[0].len=agi_vol3[1].sddr-agi_vol3[0].sddr;
		agi_vol3[1].len=agi_vol3[2].sddr-agi_vol3[1].sddr;
		agi_vol3[2].len=agi_vol3[3].sddr-agi_vol3[2].sddr;
		agi_vol3[3].len=ftell(fp)-agi_vol3[3].sddr;
		if(agi_vol3[3].len>256*3)
			agi_vol3[3].len=256*3;

		fseek(fp, 0x0L, SEEK_SET);

		/* read in directory files */
  		ec=agi_v3_load_dir(dir_logic, fp, agi_vol3[0].sddr, agi_vol3[0].len);
  		if(ec==err_OK)
	  		ec=agi_v3_load_dir(dir_pic, fp, agi_vol3[1].sddr, agi_vol3[1].len);
  		if(ec==err_OK)
	  		ec=agi_v3_load_dir(dir_view, fp, agi_vol3[2].sddr, agi_vol3[2].len);
  		if(ec==err_OK)
	  		ec=agi_v3_load_dir(dir_sound, fp, agi_vol3[3].sddr, agi_vol3[3].len);
	}
	else
	{
		printf("Failed to open \"%s\"\n", path);
		ec=err_BadFileOpen;
	}

	return ec;
}

UINT16 agi_v3_deinit(void)
{
	UINT16	ec=err_OK;

	/* unload words */
	/*agi_v3_unload_words();*/

	/* unload objects */
	/*agi_v3_unload_objects();*/

	return ec;
}


UINT16	agi_v3_load_dir(AGI_DIR *agid, FILE *fp, UINT32 offs, UINT32 len)
{
	UINT8	*mem;
	UINT16	ec=err_OK;
	UINT16	count;

	fseek(fp, offs, SEEK_SET);
	mem=(UINT8*)malloc(len+32);
	if(mem!=NULL)
	{
		fread(mem, 1, len, fp);

		/* set all directory resources to gone */
		for(count=0; count<MAX_DIRS; count++)
		{
			agid[count].volume=0xFF;
			agid[count].offset=_EMPTY;
		}

		/* build directory entries */
		for(count=0; count<len; count+=3)
		{
			agid[count/3].volume=hilo_getbyte(mem+count)>>4;
			agid[count/3].offset=(hilo_getpword(mem+count)&_EMPTY);
		}

		free(mem);
	}
	else
		ec=err_NotEnoughMemory;

	return ec;
}

UINT16	agi_v3_unload_resource(UINT8 restype, UINT16 resnum)
{
	UINT16	ec=err_OK;

	switch(restype)
	{
		case rLOGIC:
			if((dir_logic[resnum].flags&RES_LOADED)==RES_LOADED)
			{
				if(logics[resnum].data!=NULL)
					free(logics[resnum].data);
				logics[resnum].data=NULL;

				if(logics[resnum].num_texts!=0)
					free(logics[resnum].texts);
				logics[resnum].texts=0;

				dir_logic[resnum].flags&=~RES_LOADED;	/* strip loaded flag */
			}

			/* if cached, we end up here */
			logics[resnum].sIP=2;
			logics[resnum].cIP=2;
			break;

		case rPICTURE:
		    /* remove visual buffer & priority buffer if they exist */

			if((dir_pic[resnum].flags&RES_LOADED)==RES_LOADED)
			    if((dir_pic[resnum].flags&0x80)==0)
					unload_picture(resnum);

			/* if data is loaded & not cached, remove the raw data */
			/* if cached, raw data stays in memory */
			if((dir_pic[resnum].flags&RES_LOADED)==RES_LOADED)
			{
				if(pictures[resnum].rdata != NULL)
					free (pictures[resnum].rdata);
				pictures[resnum].rdata = NULL;
				dir_pic[resnum].flags &= ~RES_LOADED;
			}
			break;

		case rVIEW:
			if((dir_view[resnum].flags&RES_LOADED)==RES_LOADED)
			{
				/* unload view removes ALL data for a view */
				unload_view(resnum);

				if(views[resnum].rdata!=NULL)
					free(views[resnum].rdata);
				views[resnum].rdata=NULL;

				dir_view[resnum].flags&=~RES_LOADED;
			}
			else
				/* if cached, we end up here, just remove the loaded flag */
				dir_view[resnum].flags&=~RES_LOADED;
			break;

		case rSOUND:
			if((dir_view[resnum].flags&RES_LOADED)==RES_LOADED)
			{
				unload_sound(resnum);

				if(sounds[resnum].rdata!=NULL)
					free(sounds[resnum].rdata);
				sounds[resnum].rdata=NULL;
			}
			dir_sound[resnum].flags&=~RES_LOADED;
			break;

	}
	return ec;
}


/*
 * This function does noting but load a raw resource into memory,
 * if further decoding is required, it must be done by another
 * routine.
 *
 * NULL is returned if unsucsessfull.
 */

UINT8* agi_v3_load_vol_res (AGI_DIR *agid)
{
	UINT8	x[256];
	UINT8	*data;
	FILE	*fp;
	UINT8	*comp_buffer;

	_D (("(%p)", agid));
	sprintf((char*)x, "vol.%i", agid->volume);
	fixpath(GAMEDIR, x);
	data=NULL;

	if (agid->offset != _EMPTY && (fp = fopen((char*)path, "rb")) != NULL)
	{
		fseek (fp, agid->offset, SEEK_SET);
		fread (&x, 1, 7, fp);

		if (hilo_getword(x) != 0x1234) {
			/* FIXME */
			gfx->deinit_video_mode();
			printf("ACK! BAD RESOURCE!!!\n");
			exit(0);
		}
	
		agid->len = lohi_getword (x + 3);	/* uncompressed size */
		agid->clen = lohi_getword (x + 5);	/* compressed len */

		comp_buffer=(UINT8*)calloc(1, agid->clen+32);
		fread(comp_buffer, 1, agid->clen, fp);

		if (x[2] & 0x80 || agid->len == agid->clen) {
			/* do not decompress */
			data=comp_buffer;

#if 0
			/* CM: added to avoid problems in
			 *     convert_v2_v3_pic() when clen > len
			 *     e.g. Sierra demo 4, first picture
			 *     (Tue Mar 16 13:13:43 EST 1999)
			 */
			agid->len = agid->clen;

			/* Now removed to fix Gold Rush! in demo4 */
#endif
		} else {
			/* it is compressed */
			data = (UINT8*)calloc (1, agid->len+32);
			LZW_expand (comp_buffer, data, agid->len);
			free (comp_buffer);
			agid->flags |= RES_COMPRESSED;
		}
		
		fclose(fp);
	}
	else
	{
		/* we have a bad volume resource */
		/* set that resource to NA */
		agid->offset=_EMPTY;
	}

	return	data;
}


/*

Loads a resource into memory, a raw resource is loaded in
with above routine, then further decoded here.

*/
UINT16	agi_v3_load_resource(UINT8 restype, UINT16 resnum)
{
	UINT16	ec=err_OK;
	UINT8	*data=NULL;

	if(resnum>MAX_DIRS)
		return err_BadResource;

	switch(restype)
	{
	case rLOGIC:
		/* load resource into memory, decrypt messages at the end
		 * and build the message list */
		/* if logic is in memory) */
		if((dir_logic[resnum].flags&RES_LOADED)!=RES_LOADED)
		{
			/* if logic is already in memory, unload it */
			loader->unload_resource(rLOGIC, resnum);

			/* load raw resource into data */
			data=agi_v3_load_vol_res(&dir_logic[resnum]);
			logics[resnum].data=data;

			/* uncompressed logic files need to be decrypted */
			if(data!=NULL)
			{
				/* resloaded flag gets set by decode logic */
				/* needed to build string table */
				ec=decode_logic(resnum);

				logics[resnum].sIP=2;
			}
			else
				ec=err_BadResource;

			/*logics[resnum].sIP=2;*/	/* saved IP = 2 */
			/*logics[resnum].cIP=2;*/	/* current IP = 2 */

			logics[resnum].cIP=logics[resnum].sIP;
       		}

		/* if logic was cached, we get here */
		/* reset code pointers incase it was cached */

		/* we start at 2 to skip the size of the logic */
		/*logics[resnum].sIP=2;*/	/* saved IP = 2 */
		/*logics[resnum].cIP=2;*/	/* current IP = 2 */
		logics[resnum].cIP=logics[resnum].sIP;
		break;

	case rPICTURE:
		/* if picture is currently NOT loaded *OR* cacheing is off,
		 * unload the resource (caching==off) and reload it
		 */
		if ((dir_pic[resnum].flags & RES_LOADED) != RES_LOADED)
		{
			loader->unload_resource (rPICTURE, resnum);
			data = agi_v3_load_vol_res (&dir_pic[resnum]);

			if (data != NULL)
			{
				data = convert_v2_v3_pic (data,
					dir_pic[resnum].len);
				pictures[resnum].rdata = data;
				dir_pic[resnum].flags |= RES_LOADED;
			}
			else
				ec=err_BadResource;
		}
		break;

	case rSOUND:
		if((dir_sound[resnum].flags&RES_LOADED)==RES_LOADED)
			break;

		data=agi_v3_load_vol_res(&dir_sound[resnum]);
		if(data!=NULL)
		{
			sounds[resnum].rdata=data;
			dir_sound[resnum].flags|=RES_LOADED;
			decode_sound(resnum);
		}
		else
			ec=err_BadResource;
		break;

	case rVIEW:
		/* Load a VIEW resource into memory...
		   Since VIEWS alter the view table ALL the time
		   can we cache the view? or must we reload it all
		   the time?
		*/
		/* load a raw view from a VOL file into data */
		if((dir_view[resnum].flags&RES_LOADED)!=RES_LOADED)
		{
			loader->unload_resource(rVIEW, resnum);
			data=agi_v3_load_vol_res(&dir_view[resnum]);
			if(data!=NULL)
			{
				views[resnum].rdata=data;
				dir_view[resnum].flags|=RES_LOADED;
				ec=decode_view(resnum);
			}
			else
				ec=err_BadResource;
		}
		break;

	default:
		ec=err_BadResource;
		break;
	}

	return ec;
}

