/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */
package com.kvisco.xsl;

import java.util.Hashtable;

import org.w3c.dom.*;

/**
 * Represents an MultiplicativeExpr
 * @author Keith Visco (kvisco@ziplink.net)
**/
public class MultiplicativeExpr implements Expr {

    public static final short MULTIPLY      = 0;
    public static final short DIVIDE        = 1;
    public static final short MODULUS       = 2;
    public static final short QUOTIENT      = 3;
    
    private Expr leftExpr = null;
    private Expr rightExpr = null;
    
    private short op = MULTIPLY;
    
    private static Hashtable ops = null;
    static {
        ops = new Hashtable(4);
        ops.put(Names.MULTIPLY_OP,  new Short(MULTIPLY));
        ops.put(Names.DIV_OPNAME,   new Short(DIVIDE));
        ops.put(Names.MOD_OPNAME,   new Short(MODULUS));
        ops.put(Names.QUO_OPNAME,   new Short(QUOTIENT));
    }
    
      //---------------/
     //- Constructor -/
    //---------------/
    
    /**
     * Creates a new MultiplicativeExpr using the default operator
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this MultiplicativeExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this MultiplicativeExpr
     * <BR><B>Note:</B> the default operator is MultiplicativeExpr.MULITPLY
    **/
    public MultiplicativeExpr(Expr leftSideExpr, Expr rightSideExpr) {
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
    } //-- MultiplicativeExpr 
    
    /**
     * Creates a new MultiplicativeExpr
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this MultiplicativeExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this MultiplicativeExpr
     * @param additiveOp the additive operator for this MultiplicativeExpr
     * @exception InvalidExprException when the additive operator is 
     * invalid 
    **/
    public MultiplicativeExpr
        (Expr leftSideExpr, Expr rightSideExpr, short operator) 
        throws InvalidExprException
    {
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
        if ((operator < 0) || (operator > ops.size()))
            throw new InvalidExprException("invalid operator for multiplicative expression");
        this.op = operator;
    } //-- MultiplicativeExpr 
    
    /**
     * Creates a new MultiplicativeExpr
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this MultiplicativeExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this MultiplicativeExpr
     * @param additiveOp the additive operator for this MultiplicativeExpr
     * @exception InvalidExprException when the additive operator is 
     * invalid 
    **/
    public MultiplicativeExpr
        (Expr leftSideExpr, Expr rightSideExpr, String operator) 
        throws InvalidExprException
    {
        this.op = -1;
        if (operator != null) {
            Short sval = (Short)ops.get(operator);
            if (sval != null) this.op = sval.shortValue();
        }
        if ((this.op < 0) || (this.op > ops.size()))
            throw new InvalidExprException("invalid operator for multiplicative expression");
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
    } //-- MultiplicativeExpr
        
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Returns the type of Expr this Expr represents
     * @return the type of Expr this Expr represents
    **/
    public short getExprType() {
        return Expr.NUMBER;
    } //-- getExprType
    
    /**
     * Evaluates this Expr using the given context Node and ProcessorState
     * @param context the current context Node
     * @param ps the ProcessorState that contains the current processing 
     * environment
     * @return the ExprResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        double value = 0;
        
        if ((leftExpr == null) || (rightExpr == null))
            return new NumberResult(Double.NaN);

        double lValue = leftExpr.evaluate(context, ps).numberValue();
        double rValue = rightExpr.evaluate(context, ps).numberValue();
        
        switch(op) {
            
            case MULTIPLY:
                value = lValue * rValue;
                break;
            case DIVIDE:
                value = lValue / rValue;
                break;
            case MODULUS:
                value = (lValue % rValue);
                break;
            case QUOTIENT:
                value = Math.floor(lValue / rValue);
                break;
            default:
                break;
        }
    
        return new NumberResult(value);
    } //-- evaluate
    
    public static boolean isMultiplicativeOperator(String operator) {
        if (operator == null) return false;
        return (ops.get(operator) != null);
    } //-- isMulitplicativeOperator
    
    /**
     * Returns the String representation of this EqualityExpr
     * @return the String representation of this EqualityExpr
    **/
    public String toString() {
        
        
        StringBuffer sb = new StringBuffer();
        if (leftExpr != null)
            sb.append(leftExpr.toString());
        else
            sb.append("null");
        sb.append(" ");
        
        switch(op) {
            
            case MULTIPLY:
                sb.append(Names.MULTIPLY_OP);
                break;
            case DIVIDE:
                sb.append(Names.DIV_OPNAME);
                break;
            case MODULUS:
                sb.append(Names.MOD_OPNAME);
                break;
            case QUOTIENT:
                sb.append(Names.QUO_OPNAME);
                break;
            default:
                break;
        }
        
        sb.append(" ");
        if (rightExpr != null)
            sb.append(rightExpr.toString());
        else sb.append("null");
        
        return sb.toString();
    } //-- toString
    
    /* */
} //-- MultiplicativeExpr