########################################################################
#
#	File:			EmUtils.pm
#
#	Purpose:		High-level utilities for scripting the Palm OS
#					Emulator
#
#	Description:	This file contains the following useful utilities:
#
#					Wait
#						Wait for a signal from the Emulator.
#
#					TapPen
#						Tap the pen at the given x,y location.
#
#					TapButton
#						Tap the pen on the button with the given name.
#
########################################################################

package EmUtils;

use EmFunctions;	# HostSignalWait, HostSignalResume, EvtEnqueuePenPoint, etc.

use Exporter ();
@ISA = qw(Exporter);

@EXPORT = qw(
	Wait
	TapPen
	TapButton
);


########################################################################
#
#	FUNCTION:		Wait
#
#	DESCRIPTION:	Wait for a signal from the Palm OS Emulator.
#
#	PARAMETERS:		timeout in milliseconds.
#
#	RETURNED:		List containing the HostSignalResume error code
#					and the number of the signalled event.
#
########################################################################

sub Wait
{
	my $timeout = $_[0];
	if (not defined $timeout)
	{
		$timeout = 0x7fffffff;
	}

	my ($err, $signal) = HostSignalWait ($timeout);
	die "Didn't hear back from Poser, stopped" if ($err != 0);

	HostSignalResume ();

	($err, $signal);
}


########################################################################
#
#	FUNCTION:		TapPen
#
#	DESCRIPTION:	Simulate a tap at the given location, then wait for
#					the next null event.
#
#	PARAMETERS:		x, y coordinates.
#
#	RETURNED:		Nothing.
#
########################################################################

sub TapPen
{
	EvtEnqueuePenPoint ({x => 256 - $_[0], y => 256 - $_[1]});
	EvtEnqueuePenPoint ({x => -1, y => -1});

	Wait(5000);
}


########################################################################
#
#	FUNCTION:		TapButton
#
#	DESCRIPTION:	Simulate a tap on the named button, then wait for
#					the next null event.
#
#	PARAMETERS:		Name of the button to tap on.
#
#	RETURNED:		Nothing.
#
########################################################################

sub TapButton
{
	my ($button_name) = @_;

	my ($form) = FrmGetActiveForm();
	my ($num_objects) = FrmGetNumberOfObjects($form);

	for $ii (0..$num_objects - 1)
	{
		my ($object_type) = FrmGetObjectType($form, $ii);

		if ($object_type == frmControlObj)
		{
			my ($obj_ptr) = FrmGetObjectPtr ($form, $ii);
			my ($address, $label) = CtlGetLabel($obj_ptr);

			if ($label eq $button_name)
			{
				my (%bounds) = FrmGetObjectBounds($form, $ii);

				TapPen(	$bounds{left} + $bounds{width} / 2,
						$bounds{top} + $bounds{height} / 2);

				last;	# break out of the for loop.
			}
		}
	}
}
