/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _ERRORHANDLING_H_
#define _ERRORHANDLING_H_

#include <list>
#include <string>

class SessionFile;
struct SystemCallContext;


// ParamList contains a sequence of key/value pairs.  The even-numbered
// strings are the keys, and the odd-numbered strings are the values.
// The key/value pairs are used in string template substitution.  When
// substitution needs to take place, the keys and values are pulled
// out of the list one pair at a time.  The template string is then
// traversed, with each key being replaced by its value.
//
// Keys are typically of the form "%text" (with the "%" showing up in
// both the key text and the string template text), but could actually
// be in any form that is unique by convention.
//
// To create a parameter list, do something like the following:
//
//		ParamList	paramList;
//		paramList.push_back (string ("%foo"));
//		paramList.push_back (string ("bar"));
//		... repeat as necessary ...

typedef list<string>	ParamList;


// This is our "exception type". It is the type that we throw when
// we throw an exception. Compatible with PowerPlant's ExceptionCode.

typedef long	ErrCode;


// This is our "string resource ID" type.

typedef int		StrCode;


// Platform-independent errors numbers that can be thrown from
// platform-independent code.  These numbers ideally should not
// collide with platform-specific error codes. Mac error codes
// are 16-bit negative numbers, so take the range 0...-32768.
// Windows error codes are generally in the range 0...<big positive number>.
// Palm OS error codes are 16-bit positive numbers, so are
// in the range 0...32767.  Thus, Mac and Windows errors don't
// collide. Palm OS error codes collide with Windows, so we
// shift them down into the range 0xFFFBxxxx, where xxxx is the
// Palm OS error code. Finally, we have our own Emulator-specific
// error codes. Those we'll keep in 0xFFFDxxxx.

const ErrCode	kError_NoError							= 0;
const ErrCode	kError_EmuErrBase						= 0xFFFE0000;	// arbitrary value
const ErrCode	kError_EmuErrRange						= 0x00010000;	// arbitrary value

const ErrCode	kError_OutOfMemory						= kError_EmuErrBase + 1;
const ErrCode	kError_BadROM							= kError_EmuErrBase + 2;
const ErrCode	kError_WrongROMForType					= kError_EmuErrBase + 3;
const ErrCode	kError_UnsupportedROM					= kError_EmuErrBase + 4;

const ErrCode	kError_CantDownloadROM_BadBaudRate		= kError_EmuErrBase + 10;
const ErrCode	kError_CantDownloadROM_SerialPortBusy	= kError_EmuErrBase + 11;
const ErrCode	kError_CantDownloadROM_Generic			= kError_EmuErrBase + 12;
const ErrCode	kError_UnimplementedTrap				= kError_EmuErrBase + 13;
const ErrCode	kError_CommOpen							= kError_EmuErrBase + 14;
const ErrCode	kError_CommNotOpen						= kError_EmuErrBase + 15;
const ErrCode	kError_CantOpenPort						= kError_EmuErrBase + 16;
const ErrCode	kError_InvalidPort						= kError_EmuErrBase + 17;
const ErrCode	kError_CantFindSerialPort				= kError_EmuErrBase + 18;

const ErrCode	kError_OnlySameType						= kError_EmuErrBase + 20;
const ErrCode	kError_OnlyOnePSF						= kError_EmuErrBase + 21;
const ErrCode	kError_OnlyOneROM						= kError_EmuErrBase + 22;
const ErrCode	kError_UnknownType						= kError_EmuErrBase + 23;


const ErrCode	kError_PalmOSErrBase	= 0xFFFC0000;	// arbitrary value
const ErrCode	kError_PalmOSErrRange	= (1 << (sizeof (Err) * 8));


inline ErrCode ConvertFromPalmError (Err err)
	{ return (ErrCode) (err + kError_PalmOSErrBase); }

inline Err ConvertToPalmError (ErrCode err)
	{ return (Err) (err - kError_PalmOSErrBase); }


inline Bool IsPalmError (ErrCode err)
	{ return err >= kError_PalmOSErrBase && err < kError_PalmOSErrBase + kError_PalmOSErrRange; }

inline Bool IsEmuError (ErrCode err)
	{ return err >= kError_EmuErrBase && err < kError_EmuErrBase + kError_EmuErrRange; }

inline Bool IsPlatformError (ErrCode err)
	{ return !IsPalmError (err) && !IsEmuError (err); }

// Values passed to ReportInvalidPC.
enum
{
	kUnmappedAddress,
	kNotInCodeSegment,
	kOddAddress
};

class Errors
{
	public:
		static void				Initialize			(void);
		static void				Reset				(void);
		static void				Save				(SessionFile&);
		static void				Load				(SessionFile&);
		static void				Dispose				(void);


		// These three functions check for the indicated error condition.  If
		// there is an error, they display a "Could not foo because bar." message
		// in a dialog with an OK button.  If an optional recovery string is
		// provided, the message is "Could no foo because bar. Do this." message.
		// If "throwAfter" is true, Errors::Scram is called after the dialog
		// is dismissed.
		//
		// All of these functions bottleneck through DoDialog.
		//
		// "operation" and "recovery" are "kPart_Foo" values.
		// "error" is Mac or Windows error number, or a kError_Foo value.
		// "err" is a Palm OS error number.
		// "p" is a pointer to be tested.

		static void				ReportIfError		(StrCode operation, ErrCode error, StrCode recovery = 0, Bool throwAfter = true);
		static void				ReportIfPalmError	(StrCode operation, Err err, StrCode recovery = 0, Bool throwAfter = true);
		static void				ReportIfNULL		(StrCode operation, void* p, StrCode recovery = 0, Bool throwAfter = true);


		// Report that the indicated error condition occured.  Messages are generally
		// of the form "<Application> <version> just tried to <foo>. <More explanatory
		// text>. Report this to the program author."

		enum EAccessType
		{
			kOKAccess,
			kRegisterAccess,

			kLowMemAccess,
			kGlobalVarAccess,
			kScreenAccess,
			kMemMgrAccess,
			kUnlockedChunkAccess,
			kLowStackAccess,
			kUninitializedStackAccess,
			kFreeChunkAccess,
			kUninitializedChunkAccess,

			kStorageHeapAccess,
			kUnknownAccess,

			kStackOverflowWarning,
			kStackOverflowFatal,

			kLongMemSemaphore,

			kSizelessFormObject,
			kOffscreenFormObject,

			kA5Access
		};

		static int				ReportPreventedAccess		(uaecptr address,
															 long size,
															 Bool forRead,
															 EAccessType);

		static int				ReportUnimplementedTrap		(uae_u16 trapWord, int resourceBase);
		static int				ReportUnhandledException	(uae_s32 exceptionNumber);
		static int				ReportSANEUsage				(void);
		static int				ReportCorruptedHeap			(int, uaecptr);
		static int				ReportStackOverflow			(Bool fatal);
		static int				ReportLongMemSemaphore		(unsigned long elapsedMSecs);
		static int				ReportSizelessFormObject	(Word id, const RectangleType&);
		static int				ReportOffscreenFormObject	(Word id, const RectangleType&);
		static int				ReportA5Access				(uaecptr iAddress,
															 long size,
															 Bool forRead);
		static int				ReportInvalidSystemCall		(const SystemCallContext&);
		static int				ReportWroteToROM			(uaecptr address,
															 long size,
															 Bool forRead);

		static int				ReportInvalidPC				(uaecptr address, int reason);


		// Display the message generated by ErrDisplayFileLineMsg or SysFatalAlert.

		static int				SysFatalAlert				(const char* appMsg);


		// Add a user-supplied substitution rule.  Parameters are deleted after they
		// are used (by making a call to ReplaceParameters), so they need to be
		// re-established after every call to that function.

		static void				SetParameter		(const string& key, const string& value);
		static void				SetParameter		(const string& key, const unsigned char* value);
		static Bool				SetErrorParameters	(StrCode operation, ErrCode error, StrCode recovery);
		static void				SetStandardParameters	(void);


		// Displays a dialog box with the given message and according to
		// the given flags.  Returns which button was clicked.

		static int				DoDialog			(StrCode messageID, int flags);
		static int				DoDialog			(const char* title, const char* msg, int flags);


		// Creates and returns a string based on the string template and
		// parameter list.

		static string			ReplaceParameters	(StrCode templateID);
		static string			ReplaceParameters	(StrCode templateID, ParamList& params);
		static string			ReplaceParameters	(const string&, ParamList& params);


		// Convert error numbers into string IDs that describe the error.

		static int				GetIDForError			(ErrCode error);
		static int				GetIDForRecovery		(ErrCode error);

		// Get the name and version of the current application.

		static void				GetAppName				(string& appNameUC,
														 string& appNameLC);
		static void				GetAppVersion			(string& appVersion);

		// Return whether or not the current application has already been warned
		// about a particular error.  If not, this function also sets the bit
		// saying that this warning has been issued. Note that not all warnings
		// are "once only".  AlreadyWarned may always return true for some
		// nasty kinds of error types.

		static Bool				AlreadyWarned			(EAccessType kind);

		// If an error condition is detected, throws an error number.
		// No message is reported.

		static void				Throw				(ErrCode error);
		static void				ThrowIfError		(ErrCode error);
		static void				ThrowIfPalmError	(Err error);
		static void				ThrowIfNULL			(void*);

		// Call this function as a "silent failure" bottleneck.
		// An exception will be thrown that will unwind the stack
		// to some top-level, but nothing will be reported.

		static void				Scram				(void);

		static void				ClearWarningFlags	(void);

		// Flags for DoDialog.

		enum
		{
			// Use one of these. kErrorAlert is the default.

			kNoteAlert			= 0x0001,
			kCautionAlert		= 0x0002,
			kErrorAlert			= 0x0004,
			kAlertMask			= 0x000F,

			// Use one of the following three groups.

			// If you use this group, use one or both flags.
			// If you use one flag, the other button will be hidden.

			kOK					= 0x0010,
			kCancel				= 0x0020,
			kOKCancel			= kOK | kCancel,
			kOKCancelMask		= 0x0030,

			// If you use this group, use both flags.

			kYes				= 0x0040,
			kNo					= 0x0080,
			kYesNo				= kYes | kNo,
			kYesNoMask			= 0x00C0,

			// If you use this group, use one or more flags.
			// The other buttons will appear, but disabled.

			kContinue			= 0x0100,
			kDebug				= 0x0200,
			kReset				= 0x0400,
			kNextGremlin		= 0x0800,
			kContinueDebug		= kContinue	| kDebug,
			kContinueReset		= kContinue			 | kReset,
			kDebugReset			=			  kDebug | kReset,
			kContinueDebugReset	= kContinue | kDebug | kReset,
			kContinueDebugResetMask = 0x0F00
		};
};


#endif /* _ERRORHANDLING_H_ */

