/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _PLATFORM_FILES_H_
#define _PLATFORM_FILES_H_

#include "ErrorHandling.h"		// ErrCode
#include "StreamHandle.h"		// StreamHandle


class FileReference
{
	public:
								FileReference	(void);
								FileReference	(const FileReference&);
								FileReference	(const char*);
								FileReference	(const string&);
								FileReference	(const unsigned char*);
								~FileReference	(void);

		FileReference&			operator=		(const FileReference&);

		Bool					IsSpecified		(void) const;
		Bool					Exists			(void) const;

		Bool					IsPRC			(void) const;
		Bool					IsPDB			(void) const;
		Bool					IsPQA			(void) const;
		Bool					IsPSF			(void) const;
		Bool					IsROM			(void) const;

		string					GetFileName		(void) const;

		bool					operator==		(const FileReference&) const;
		bool					operator!=		(const FileReference&) const;
		bool					operator<		(const FileReference&) const;
		bool					operator>		(const FileReference&) const;

		bool					FromPrefString	(const string&);
		string					ToPrefString	(void) const;

		FILE*					OpenAsFILE		(const char* mode) const;

		// I'm kind of opposed to making GetFilePath public.  It breaks the
		// enacapsulation of FileReference.  However, there are some places
		// where getting to this information is handy (such as in the Windows
		// code that opens the File dialogs in the same directory as a
		// previously opened file).  Until I can figure a better way for
		// handling that scenario, I'll leave GetFilePath public.
	private:
	public:
		string					GetFilePath		(void) const;

#if defined (__MACOS__)
	public:
								FileReference	(const FSSpec&);
								FileReference	(AliasHandle);

		AliasHandle				GetAlias		(void) const;
		FSSpec					GetFSSpec		(void) const;

	private:
		void					UpdateAlias		(void);
		void					UpdateSpec		(void);

		Bool					EqualAlias		(AliasHandle, AliasHandle) const;
		Bool					EqualFSSpec		(const FSSpec&, const FSSpec&) const;

		Bool					IsType			(OSType type, const char* suffix) const;

		AliasHandle				fFileAlias;
		FSSpec					fFileSpec;
#endif

#if defined (_WINDOWS) || defined (UNIX)

	private:
		string					fFilePath;
#endif
};


enum
{
	kCreateNew,			// Creates a new file; fails if it already exists.
	kCreateAlways,		// Always creates a new file, destroying any existing file.
	kOpenExisting,		// Opens the file it it exists, fails otherwise.
	kOpenAlways,		// Opens the file if it exists, creates it otherwise.
	kTruncateExisting,	// Opens and clears existing file, fails otherwise.

	kOpenTypeMask = 0x0F,

	kOpenRead	= 16,
	kOpenWrite	= 32,
	kOpenReadWrite = kOpenRead | kOpenWrite
};

class FileHandle : public StreamHandle
{
	public:
							FileHandle		(const FileReference&,
											 long openMode,
											 uae_u32 creator = 0,
											 uae_u32 fileType = 0);
		virtual				~FileHandle		(void);

		FileReference		GetReference	(void);
		Bool				IsOpen			(void);

		void				SetPos			(long offset, SeekMode);
		uae_s32				GetPos			(void);

		void				SetLength		(uae_s32);
		uae_s32				GetLength		(void);

		void				Read			(uae_s32 length, void* buffer);
		void				Write			(uae_s32 length, const void* buffer);

	protected:
		void				Open			(const FileReference&,
											 long openMode,
											 uae_u32 creator,
											 uae_u32 fileType);

#if defined (__MACOS__)
		void				CreateNew		(const FSSpec&,
											 long openMode,
											 uae_u32 creator,
											 uae_u32 fileType);
		void				CreateAlways	(const FSSpec&,
											 long openMode,
											 uae_u32 creator,
											 uae_u32 fileType);
		void				OpenExisting	(const FSSpec&,
											 long openMode,
											 uae_u32 creator,
											 uae_u32 fileType);
		void				OpenAlways		(const FSSpec&,
											 long openMode,
											 uae_u32 creator,
											 uae_u32 fileType);
		void				TruncateExisting(const FSSpec&,
											 long openMode,
											 uae_u32 creator,
											 uae_u32 fileType);
		char				GetPermission	(long openMode);
#endif

		void				Close			(void);

		void				Throw			(ErrCode);
		void				SetFileNameParameter	(void);

	private:
			// Protect the copy constructor so that we don't
			// accidentally make copies (what does it mean to
			// copy a reference to an open file?).

							FileHandle		(const FileHandle&);

		FileReference		fFileRef;

#if defined (__MACOS__)
		short				fRefNum;
#endif

#if defined (_WINDOWS)
		void*				fHandle;	// == HANDLE
#endif

#if defined (UNIX)
		FILE *				fStream;
#endif
};

#endif /* _PLATFORM_FILES_H_ */

