// command.C
//
// This program is free software. See the file COPYING for details.
// Author: Mattias Engdegrd, 1997-1999

#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <qlabel.h>
#include <qlayout.h>
#include <qaccel.h>

#include "command.h"
#include "qps.h"
#include "proc.h"
#include "uidstr.h"
#include "svec.C"
#include "dialogs.h"

void Command::call(Procinfo *p)
{
    QString s;
    int len = cmdline.length();
    for(int i = 0;;) {
	int v = cmdline.find('%', i);
	if(v < 0) {
	    s.append(cmdline.right(len - i));
	    break;
	} else {
	    s.append(cmdline.mid(i, v - i));
	    if(++v >= len)
		break;
	    QString subst;
#if QT_VERSION >= 200
	    char c = cmdline[v].latin1();
#else
	    char c = cmdline[v];
#endif	    
	    switch(c) {
	    case 'p':
		subst.setNum(p->pid);
		break;
	    case 'c':
		subst = p->comm;
		break;
	    case 'C':
		subst = p->cmdline;
		break;
	    case 'u':
		subst = Uidstr::userName(p->uid);
		break;
	    case '%':
		subst = "%";
		break;
	    }
	    s.append(subst);
	    i = v + 1;
	}
    }
    int ret = system(s);
    if(ret) {
	QString msg;
	msg = "The command:\n\n";
	msg.append(s);
	if(ret == -1) {
	    msg.append("\n\nfailed with the error:\n\n");
	    const char *e = (errno == EAGAIN)
		            ? "Too many processes" : strerror(errno);
	    msg.append(e ? e : "Unknown error");
	} else if(ret & 0xff) {
	    msg.append("\n\nwas terminated by signal ");
	    msg.append(QString().setNum(ret & 0xff));
	    msg.append(".");
	} else if(ret == 0x7f00) {
	    msg.append("\n\ncould not be executed because it was not "
		       "found,\nor you did not have execute permission.");
	} else {
	    msg.append("\n\nexited with status ");
	    msg.append(QString().setNum(ret >> 8));
	    msg.append(".");
	}
	MessageDialog::message("Command Failed",
			       msg, "OK", MessageDialog::warningIcon());
    }
}

static void fixWidget(QWidget *w)
{
    w->setFixedSize(w->sizeHint());
}

CommandDialog::CommandDialog()
{
    setCaption("Edit Commands");
    QVBoxLayout *tl = new QVBoxLayout(this, 10);

    lb = new QListBox(this);
    lb->setMinimumSize(150, 150);
    
    tl->addWidget(lb);

    QHBoxLayout *hl = new QHBoxLayout;
    tl->addLayout(hl);

    add = new QPushButton("Add...", this);
    fixWidget(add);
    hl->addWidget(add);

    edit = new QPushButton("Edit...", this);
    fixWidget(edit);
    hl->addWidget(edit);

    del = new QPushButton("Delete", this);
    fixWidget(del);
    hl->addWidget(del);

    connect(lb, SIGNAL(highlighted(int)), SLOT(set_buttons(int)));
    connect(lb, SIGNAL(selected(int)), SLOT(edit_command(int)));
    connect(add, SIGNAL(clicked()), SLOT(add_new()));
    connect(del, SIGNAL(clicked()), SLOT(del_current()));
    connect(edit, SIGNAL(clicked()), SLOT(edit_current()));

    QAccel *acc = new QAccel(this);
    acc->connectItem(acc->insertItem(ALT + Key_W), this, SLOT(hide()));
    acc->connectItem(acc->insertItem(Key_Escape), this, SLOT(hide()));

    for(int i = 0; i < Qps::commands->size(); i++)
	lb->insertItem((*Qps::commands)[i]->name);

    set_buttons(0);
    tl->freeze();
}

void CommandDialog::set_buttons(int)
{
    bool sel = (lb->currentItem() >= 0);
    del->setEnabled(sel);
    edit->setEnabled(sel);
}

void CommandDialog::edit_current()
{
    edit_command(lb->currentItem());
}

static int find_command(QString s)
{
    for(int i = 0; i < Qps::commands->size(); i++)
	if(s == (*Qps::commands)[i]->name)
	    return i;
    return -1;
}

void CommandDialog::edit_command(int index)
{
    Command *c = (*Qps::commands)[find_command(lb->text(index))];
    
    CommandEditDialog ced(this, FALSE, c->name, c->cmdline);
    if(ced.exec()) {
	c->name = ced.name->text();
	c->cmdline = ced.cmdline->text();
	lb->changeItem(c->name, index);
	emit command_change();
    }
}

void CommandDialog::add_new()
{
    CommandEditDialog ced(this, TRUE, "", "");
    if(ced.exec()) {
	Qps::commands->add(new Command(ced.name->text(), ced.cmdline->text()));
	lb->insertItem(ced.name->text());
	emit command_change();
    }
}

void CommandDialog::del_current()
{
    int i = find_command(lb->text(lb->currentItem()));
    if(i >= 0) {
	delete (*Qps::commands)[i];
	Qps::commands->remove(i);
	lb->removeItem(lb->currentItem());
	set_buttons(0);
	emit command_change();
    }
}

CommandEditDialog::CommandEditDialog(QWidget *parent, bool add,
				     QString initial_name,
				     QString initial_command)
    : QDialog(parent, 0, TRUE),
      newcmd(add)
{
    setCaption(add ? "Add Command" : "Edit Command");
    QVBoxLayout *tl = new QVBoxLayout(this, 10);
    QHBoxLayout *h1 = new QHBoxLayout;
    tl->addLayout(h1);

    
    QLabel *l1 = new QLabel("Description:", this);
    fixWidget(l1);
    h1->addWidget(l1);
    name = new QLineEdit(this);
    name->setFixedHeight(name->sizeHint().height());
    name->setMinimumWidth(150);
    h1->addWidget(name);
    name->setText(initial_name);
    name->selectAll();
    name->setFocus();

    QLabel *l2 = new QLabel("Command Line:", this);
    l2->setFixedHeight(l2->sizeHint().height());
    l2->setAlignment(AlignVCenter | AlignLeft);
    tl->addWidget(l2);
    cmdline = new QLineEdit(this);
    cmdline->setFixedHeight(cmdline->sizeHint().height());
    cmdline->setMinimumWidth(250);
    tl->addWidget(cmdline);
    cmdline->setText(initial_command);

    QLabel *l3 = new QLabel("Substitutions:\n"
			    "%p\tPID\n%c\tCOMM\n%C\tCMDLINE\n%u\tUSER\n"
			    "%%\tliteral '%'", this);
    l3->setAlignment(AlignVCenter | AlignLeft | ExpandTabs);
    QFont f(font());
    f.setBold(FALSE);
    l3->setFont(f);
    fixWidget(l3);
    tl->addWidget(l3);
    QHBoxLayout *h2 = new QHBoxLayout;
    tl->addLayout(h2);
    
    QPushButton *ok = new QPushButton("OK", this);
    fixWidget(ok);
    h2->addWidget(ok);
    connect(ok, SIGNAL(clicked()), SLOT(accept()));
    ok->setDefault(TRUE);

    h2->addStretch(1);

    QPushButton *cancel = new QPushButton("Cancel", this);
    fixWidget(cancel);
    h2->addWidget(cancel);
    connect(cancel, SIGNAL(clicked()), SLOT(reject()));
    QAccel *acc = new QAccel(this);
    acc->connectItem(acc->insertItem(Key_Escape), this, SLOT(reject()));

    tl->freeze();
}

void CommandEditDialog::done(int ret)
{
    if(ret == Accepted) {
	QString n = name->text(), c = cmdline->text();
	n = n.stripWhiteSpace();
	c = c.stripWhiteSpace();
	if(n.isEmpty() || c.isEmpty()) {
	    MessageDialog::message("Bad Command",
				   "The description and command cannot "
				   "be empty.", "OK",
				   MessageDialog::warningIcon());
	    return;
	}
	name->setText(n);
	cmdline->setText(c);
	if(newcmd && find_command(n) >= 0) {
	    QString s = "The command '";
	    s.append(n);
	    s.append("' already exists.\nPlease use another name.");
	    MessageDialog::message("Command Exists", s, "OK",
				   MessageDialog::warningIcon());
	    return;
	}
    }
    QDialog::done(ret);
}

