#include <std.h>

#include <func/apt.h>

#include <list/colors.h>

#include <interface/coreui.h>
#include <interface/formatting.h>

#include <apt-pkg/acquire-item.h>
#include <apt-pkg/acquire-worker.h>
#include <apt-pkg/strutl.h>

#include <signal.h>

#include "acquire_status.h"
#include "acquire.h"

// AcquireStatus::AcquireStatus - Constructor                           /*{{{*/
AcquireStatus::AcquireStatus(pkgAcquireInterface * fetchObj)
{
	Master = fetchObj;
	WasAborted = false;
};

// AcquireStatus::Start - Downloading has started                       /*{{{*/
void AcquireStatus::Start()
{
//      MorePulses = true;
	pkgAcquireStatus::Start();
	ID = 1;
};

// AcquireStatus::IMSHit - Called when an item got a HIT response       /*{{{*/
void AcquireStatus::IMSHit(pkgAcquire::ItemDesc & Itm)
{
	Master->PrintOutputF("Hit %s", Itm.Description.c_str());

	if (Itm.Owner->FileSize != 0)
		Master->PrintOutputF(" [%sb]", SizeToStr(Itm.Owner->FileSize).c_str());

	Master->PrintOutputF("\n");
	Master->output->Refresh();

	Update = true;
};

// AcquireStatus::Fetch - An item has started to download               /*{{{*/
/* This prints out the short description and the expected size */
void AcquireStatus::Fetch(pkgAcquire::ItemDesc & Itm)
{
	Update = true;

	if (Itm.Owner->Complete == true)
		return;

	Itm.Owner->ID = ID++;

	Master->PrintOutputF("Get:%d %s", Itm.Owner->ID, Itm.Description.c_str());

	if (Itm.Owner->FileSize != 0)
		Master->PrintOutputF(" [%sb]", SizeToStr(Itm.Owner->FileSize).c_str());

	Master->PrintOutputF("\n");
	Master->output->Refresh();
};

// AcquireStatus::Done - Completed a download                           /*{{{*/
void AcquireStatus::Done(pkgAcquire::ItemDesc & Itm)
{
	Update = true;
};

// AcquireStatus::Fail - Called when an item fails to download          /*{{{*/
/* We print out the error text  */
void AcquireStatus::Fail(pkgAcquire::ItemDesc & Itm)
{
	if (Itm.Owner->Status == pkgAcquire::Item::StatDone)
	{
		Master->PrintOutputF("Ign %s\n", Itm.Description.c_str());
	}
	else
	{
		Master->PrintOutputF("Err %s\n", Itm.Description.c_str());
		Master->PrintOutputF("  `-> %s\n", Itm.Owner->ErrorText.c_str());
	}

	Master->output->Refresh();

	Update = true;
};

// AcquireStatus::Abort - User selected to stop the transfer           /*{{{*/
/* Sets a flag so that Pulse will return false and the fetcher will
   exit with pkgAcquire::Cancelled */
void AcquireStatus::Abort()
{
	WasAborted = true;
}

// AcquireStatus::Stop - Finished downloading                           /*{{{*/
/* This prints out the bytes downloaded and the overall average line
   speed */
void AcquireStatus::Stop()
{
	pkgAcquireStatus::Stop();

	Master->DestroyWindows();
	endwin();

	if (FetchedBytes != 0)
	{
		printf("\nFetched %sb in %s (%sb/s)\n\n", SizeToStr(FetchedBytes).c_str(), TimeToStr(ElapsedTime).c_str(), SizeToStr(CurrentCPS).c_str());
		fflush(stdout);
		sleep(2);
	}
}

// AcquireStatus::Pulse - Regular event pulse                           /*{{{*/
/* This draws the current progress. Each line has an overall percent
   meter and a per active item status meter along with an overall 
   bandwidth and ETA indicator. */
bool AcquireStatus::Pulse(pkgAcquire * Owner)
{
	pkgAcquireStatus::Pulse(Owner);

	if (Master->extra != NULL)
		Master->Game->pulse();

	typedef struct
	{
		char *Status;
		unsigned long ID;
		char *ShortDesc;
		unsigned long CurrentSize;
		unsigned long TotalSize;
		unsigned long Percentage;
	} pInd;

	pInd Indicators[3];

	bool Shown = false;

	Master->itemized->Erase();
	Master->total->Erase();

	int J = 0, Z = 0;
	for (pkgAcquire::Worker * I = Owner->WorkersBegin(); I != 0; I = Owner->WorkerStep(I), Z++)
	{
		// There is no item running
		if ((I->CurrentItem == 0) && I->Status.empty())
		{
			continue;
		}

		if (I->CurrentItem == 0)
		{
			Indicators[J].Status = (char *) I->Status.c_str();
			Indicators[J].ID = 0;
			Indicators[J].ShortDesc = 0;
			Indicators[J].CurrentSize = I->CurrentSize;
		}
		else
		{
			Indicators[J].Status = NULL;
			Indicators[J].ID = I->CurrentItem->Owner->ID;
			Indicators[J].ShortDesc = (char *) I->CurrentItem->ShortDesc.c_str();
			Indicators[J].CurrentSize = I->CurrentSize;
		}

		Indicators[J].TotalSize = I->TotalSize;
		Indicators[J].Percentage = 100;
		if (Indicators[J].TotalSize != 0)
			Indicators[J].Percentage = long (double (Indicators[J].CurrentSize * 100.0) / double (Indicators[J].TotalSize));

		J++;

		Shown = true;

		if (J == 2)
			break;
	}

	// Show something..
	if (Shown == false)
	{
		mvwprintw(Master->total->body(), 1, 3, "Working ...");
		Master->total->Refresh();
		return true;
	}

	/* Put in the ETA and cps meter, block off signals to prevent strangeness
	   during resizing  -  Uhhhh */
/* 
   sigset_t Sigs,OldSigs;
   sigemptyset(&Sigs);
   sigaddset(&Sigs,SIGWINCH);
   sigprocmask(SIG_BLOCK,&Sigs,&OldSigs);
   sigprocmask(SIG_UNBLOCK,&OldSigs,0);
*/

	char buf[COLS];

	// Draw all worker progress bars
	for (int K = 0; K < J; K++)
	{
		wattrset(Master->itemized->body(), Pair(COLOR_BLACK, COLOR_WHITE));

		for (int H = 2; H < Master->itemized->cols() - 2; H++)
			mvwaddch(Master->itemized->body(), K + 1, H, ' ');

		if (Indicators[K].Status != 0)
		{
			mvwprintw(Master->itemized->body(), K + 1, 2, " %s", Indicators[K].Status);
			Master->itemized->Refresh();
		}
		else
		{
			if (Indicators[K].TotalSize != 0)
			{
				double barlen = (double (Indicators[K].CurrentSize) / double (Indicators[K].TotalSize)) *double (Master->itemized->cols() - 4);

				char buf[COLS];

				sprintf(buf, " %s, %s/%s, %02d%%%% ",
					Indicators[K].ShortDesc, SizeToStr(Indicators[K].CurrentSize).c_str(), SizeToStr(Indicators[K].TotalSize).c_str(), (int) Indicators[K].Percentage);

				wmove(Master->itemized->body(), K + 1, 2);

				if (barlen < strlen(buf))
				{
					char redbuf[COLS];
					int I;

					for (I = 0; I < barlen; I++)
						redbuf[I] = buf[I];

					redbuf[I] = 0;

					wattrset(Master->itemized->body(), Pair(COLOR_WHITE, COLOR_RED));
					wprintw(Master->itemized->body(), redbuf);
					wattrset(Master->itemized->body(), Pair(COLOR_BLACK, COLOR_WHITE));
					wprintw(Master->itemized->body(), buf + I);
				}
				else
				{
					wattrset(Master->itemized->body(), Pair(COLOR_WHITE, COLOR_RED));
					wprintw(Master->itemized->body(), buf);

					for (int H = 0; H < (barlen - strlen(buf)); H++)
						waddch(Master->itemized->body(), ' ');
				}

				wattrset(Master->itemized->body(), Pair(COLOR_BLACK, COLOR_WHITE));
			}
		}
	}

	// Draw the total progress bar

	double barlen = (double (CurrentBytes + CurrentItems) / double (TotalBytes + TotalItems)) *(double (Master->itemized->cols() - 4));

	unsigned Total_Percentage = 0;

	if ((TotalBytes + TotalItems) != 0)
		Total_Percentage = unsigned ((double (CurrentBytes + CurrentItems) / double (TotalBytes + TotalItems)) *(double (100.0)));

	if (CurrentCPS != 0)
	{
		unsigned long ETA = (unsigned long) ((TotalBytes - CurrentBytes) / CurrentCPS);
		sprintf(buf, " total, %sB/s, %s, %02d%%%% ", SizeToStr(CurrentCPS).c_str(), TimeToStr(ETA).c_str(), Total_Percentage);
	}
	else
	{
		sprintf(buf, " %02d%%%% ", Total_Percentage);
	}

	wattrset(Master->total->body(), Pair(COLOR_BLACK, COLOR_WHITE));

	for (int x = 2; x < Master->total->cols() - 2; x++)
		mvwaddch(Master->total->body(), 1, x, ' ');

	wmove(Master->total->body(), 1, 2);

	if (barlen < strlen(buf))
	{
		char redbuf[COLS];
		int I;

		for (I = 0; I < barlen; I++)
			redbuf[I] = buf[I];

		redbuf[I] = 0;

		wattrset(Master->total->body(), Pair(COLOR_WHITE, COLOR_RED));
		wprintw(Master->total->body(), redbuf);
		wattrset(Master->total->body(), Pair(COLOR_BLACK, COLOR_WHITE));
		wprintw(Master->total->body(), buf + I);
	}
	else
	{
		wattrset(Master->total->body(), Pair(COLOR_WHITE, COLOR_RED));
		wprintw(Master->total->body(), buf);

		for (int H = 0; H < (barlen - strlen(buf)); H++)
			waddch(Master->total->body(), ' ');
	}

	Master->itemized->Refresh();
	Master->total->Refresh();

	Update = false;

	return !WasAborted;
}

// AcquireStatus::MediaChange - Media need to be swapped                /*{{{*/
/* Prompt for a media swap */
bool AcquireStatus::MediaChange(string Media, string Drive)
{
	Master->PrintOutputF("Media Change: Please insert the disc labeled '%s' in the drive '%s' and press enter", Media.c_str(), Drive.c_str());

	Master->output->Refresh();

	char C = 0;
	while (C != '\n' && C != '\r')
		read(STDIN_FILENO, &C, 1);

	Update = true;
	return true;
}
