/*
    ext2_resize.c -- ext2 resizer
    Copyright (C) 1998,99 Lennert Buytenhek <lbuijten@cs.leidenuniv.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include "ext2.h"

static int ext2_add_group(struct ext2_fs *fs, blk_t groupsize)
{
	blk_t admin;
	int   group;
	blk_t groupstart;
	blk_t newgdblocks;
	int   sparse;

	fprintf(stderr, "ext2_add_group\n");

	if (fs->sb.s_blocks_count != fs->sb.s_first_data_block + fs->numgroups * fs->sb.s_blocks_per_group)
	{
		fprintf(stderr, "ext2_add_group: last (existing) group isn't complete!\n");
		return 0;
	}

	group = fs->numgroups;
	sparse = ext2_is_group_sparse(fs, group);
	groupstart = fs->sb.s_first_data_block + group * fs->sb.s_blocks_per_group;

	admin = fs->adminblocks;
	if (!sparse)
		admin -= fs->gdblocks + 1;

	if (groupsize < fs->adminblocks || groupsize > fs->sb.s_blocks_per_group)
	{
		fprintf(stderr, "ext2_add_group: groups of %i blocks are impossible!\n", groupsize);
		return 0;
	}

	newgdblocks = howmany((fs->numgroups + 1) * sizeof(struct ext2_group_desc), fs->blocksize);
	if (newgdblocks != fs->gdblocks)
	{
		int i;

		for (i=0;i<fs->numgroups;i++)
			if (ext2_is_group_sparse(fs, i))
			{
				blk_t start;

				start = fs->sb.s_first_data_block + i * fs->sb.s_blocks_per_group;
				ext2_set_block_state(fs, start + fs->gdblocks + 1, 1, 1);
			}

		fs->gdblocks++;
		fs->adminblocks++;
		if (sparse)
			admin++;
	}

	fs->numgroups++;

	fs->sb.s_inodes_count += fs->sb.s_inodes_per_group;
	fs->sb.s_blocks_count += groupsize;
	fs->sb.s_free_blocks_count += groupsize - admin;
	fs->sb.s_free_inodes_count += fs->sb.s_inodes_per_group;

	{
		blk_t off;
		blk_t sparseoff;

		off = groupstart;
		sparseoff = off + fs->itoffset - 2;

		if (sparse)
		{
			fs->gd[group].bg_block_bitmap = sparseoff;
			fs->gd[group].bg_inode_bitmap = sparseoff + 1;
		}
		else
		{
			fs->gd[group].bg_block_bitmap = off;
			fs->gd[group].bg_inode_bitmap = off + 1;
		}

		fs->gd[group].bg_inode_table = sparseoff + 2;			/* Hey, I don't know _why_ either */
	}

	fs->gd[group].bg_free_blocks_count = groupsize - admin;
	fs->gd[group].bg_free_inodes_count = fs->sb.s_inodes_per_group;
	fs->gd[group].bg_used_dirs_count   = 0;

	{
		struct ext2_buffer_head *bh;
		int i;

		bh = ext2_bcreate(fs, fs->gd[group].bg_block_bitmap);

		if (sparse)
		{
			bh->data[0] |= _bitmap[0];
			for (i=1;i<=fs->gdblocks;i++)
				bh->data[i>>3] |= _bitmap[i&7];
		}

		i = fs->gd[group].bg_block_bitmap - groupstart;
		bh->data[i>>3] |= _bitmap[i&7];

		i = fs->gd[group].bg_inode_bitmap - groupstart;
		bh->data[i>>3] |= _bitmap[i&7];

		for (i=0;i<fs->inodeblocks;i++)
		{
			blk_t j;

			j = fs->gd[group].bg_inode_table - groupstart + i;
			bh->data[j>>3] |= _bitmap[j&7];
		}

		for (i=groupsize;i<fs->sb.s_blocks_per_group;i++)
			bh->data[i>>3] |= _bitmap[i&7];

		ext2_brelse(bh, 0);         /* this is a block bitmap */
	}

	ext2_zero_blocks(fs, fs->gd[group].bg_inode_bitmap, 1);
	ext2_zero_blocks(fs, fs->gd[group].bg_inode_table, fs->inodeblocks);

	fs->metadirty = 1;
	ext2_sync(fs);

	return 1;
}

static int ext2_del_group(struct ext2_fs *fs)
{
	blk_t admin;
	int   group;
	blk_t groupsize;
	blk_t newgdblocks;
	int   sparse;

	fprintf(stderr, "ext2_del_group\n");

	group = fs->numgroups - 1;
	sparse = ext2_is_group_sparse(fs, group);

	admin = fs->adminblocks;
	if (!sparse)
		admin -= fs->gdblocks + 1;

	groupsize = fs->sb.s_blocks_count - fs->sb.s_first_data_block - group * fs->sb.s_blocks_per_group;

	if (fs->sb.s_free_blocks_count < groupsize - admin)
	{
		fprintf(stderr, "ext2_del_group: filesystem is too occupied to remove a group!\n");
		return 0;
	}

	if (fs->sb.s_free_inodes_count < fs->sb.s_inodes_per_group)
	{
		fprintf(stderr, "ext2_del_group: filesystem has too many allocated inodes to remove a group!\n");
		return 0;
	}

	if (fs->gd[group].bg_free_inodes_count != fs->sb.s_inodes_per_group)
	{
		fprintf(stderr, "ext2_del_group: this should not happen anymore!\n");
		return 0;
	}

	newgdblocks = howmany((fs->numgroups - 1) * sizeof(struct ext2_group_desc), fs->blocksize);
	if (newgdblocks != fs->gdblocks)
	{
		int i;

		for (i=0;i<fs->numgroups;i++)
			if (ext2_is_group_sparse(fs, i))
			{
				blk_t start;

				start = fs->sb.s_first_data_block + i * fs->sb.s_blocks_per_group;
				ext2_set_block_state(fs, start + fs->gdblocks, 0, 1);
			}

		fs->gdblocks--;
		fs->adminblocks--;
		if (sparse)
			admin--;
	}

	if (fs->gd[group].bg_free_blocks_count != groupsize - admin)
	{
		blk_t i;
		blk_t num;
		blk_t offset;

		offset = fs->sb.s_first_data_block + group * fs->sb.s_blocks_per_group;
		num = fs->sb.s_blocks_per_group;

		for (i=0;i<num;i++)
			if (ext2_is_data_block(fs, offset+i) && ext2_get_block_state(fs, offset+i))
			{
				fprintf(stderr, "error: block relocator should have relocated %i\n", offset+i);
				return 0;
			}
	}

	fs->numgroups--;

	fs->sb.s_inodes_count -= fs->sb.s_inodes_per_group;
	fs->sb.s_blocks_count -= groupsize;
	fs->sb.s_free_blocks_count -= groupsize - admin;
	fs->sb.s_free_inodes_count -= fs->sb.s_inodes_per_group;

	fs->metadirty = 1;
	ext2_sync(fs);

	return 1;
}

static int ext2_grow_group(struct ext2_fs *fs, blk_t newsize)
{
	int   group;
	blk_t groupoff;
	blk_t gblocks;
	blk_t i;

	fprintf(stderr, "ext2_grow_group\n");

	group = fs->numgroups - 1;
	groupoff = group * fs->sb.s_blocks_per_group + fs->sb.s_first_data_block;
	gblocks = fs->sb.s_blocks_count - groupoff;

	if (newsize < gblocks)
	{
		fprintf(stderr, "ext2_grow_group: called to shrink group!\n");
		return 0;
	}

	if (gblocks == newsize)
	{
		fprintf(stderr, "ext2_grow_group: nothing to do!\n");
		return 0;
	}

	for (i=gblocks;i<newsize;i++)
		ext2_set_block_state(fs, groupoff + i, 0, 1);

	fs->sb.s_blocks_count += newsize - gblocks;

	fs->metadirty = 1;
	ext2_sync(fs);

	return 1;
}

static int ext2_shrink_group(struct ext2_fs *fs, blk_t newsize)
{
	blk_t admin;
	int   group;
	blk_t groupoff;
	blk_t gblocks;
	int   i;

	fprintf(stderr, "ext2_shrink_group\n");

	group = fs->numgroups - 1;

	admin = fs->adminblocks;
	if (!ext2_is_group_sparse(fs, group))
		admin -= fs->gdblocks + 1;

	groupoff = group * fs->sb.s_blocks_per_group + fs->sb.s_first_data_block;
	gblocks = fs->sb.s_blocks_count - groupoff;

	if (newsize < admin)
	{
		fprintf(stderr, "ext2_shrink_group: cant shrink a group to %i blocks\n", newsize);
		return 0;
	}

	if (newsize > gblocks)
	{
		fprintf(stderr, "ext2_shrink_group: called to grow group!\n");
		return 0;
	}

	if (gblocks == newsize)
	{
		fprintf(stderr, "ext2_shrink_group: nothing to do!\n");
		return 0;
	}

	for (i=newsize;i<gblocks;i++)
	{
		if (ext2_get_block_state(fs, groupoff + i))
		{
			fprintf(stderr, "error: block relocator should have relocated %i\n", groupoff + i);
			return 0;
		}

		ext2_set_block_state(fs, groupoff + i, 1, 0);
	}

	i = gblocks - newsize;
	fs->metadirty = 1;
	fs->sb.s_blocks_count -= i;
	fs->sb.s_free_blocks_count -= i;
	fs->gd[group].bg_free_blocks_count -= i;

	fs->metadirty = 1;
	ext2_sync(fs);

	return 1;
}






static int ext2_grow_fs(struct ext2_fs *fs, blk_t newsize)
{
	blk_t diff;
	blk_t sizelast;

	fprintf(stderr, "ext2_grow_fs\n");

	if (!ext2_block_relocate(fs, newsize))
		return 0;

	if (!ext2_metadata_push(fs, newsize))
		return 0;

	diff = newsize - fs->sb.s_blocks_count;
	sizelast = fs->sb.s_blocks_count - fs->sb.s_first_data_block - (fs->numgroups-1)*fs->sb.s_blocks_per_group;

	if (sizelast != fs->sb.s_blocks_per_group)
	{
		blk_t growto;

		growto = sizelast + diff;
		if (growto > fs->sb.s_blocks_per_group)
			growto = fs->sb.s_blocks_per_group;

		if (!ext2_grow_group(fs, growto))
			return 0;

		diff -= growto - sizelast;
	}

	while (diff)
	{
		sizelast = min(diff, fs->sb.s_blocks_per_group);
		if (!ext2_add_group(fs, sizelast))
			return 0;

		diff -= sizelast;
	}

	return 1;
}

static int ext2_shrink_fs(struct ext2_fs *fs, blk_t newsize)
{
	blk_t diff;
	int newgroups;
	blk_t sizelast;

	fprintf(stderr, "ext2_shrink_fs\n");

	newgroups = howmany(newsize - fs->sb.s_first_data_block, fs->sb.s_blocks_per_group);
	if (fs->sb.s_blocks_count - fs->sb.s_free_blocks_count > newsize)
	{
		fprintf(stderr, "Your filesystem is too occupied to resize it to %i\n", newsize);
		fprintf(stderr, "blocks. Sorry.\n");
		return 0;
	}

	if (fs->sb.s_inodes_count - fs->sb.s_free_inodes_count > newgroups * fs->sb.s_inodes_per_group)
	{
		fprintf(stderr, "Your filesystem has too much occupied inodes to resize it\n");
		fprintf(stderr, "to %i blocks. Sorry.\n", newsize);
		return 0;
	}

	if (!ext2_inode_relocate(fs, newgroups))
		return 0;

	if (!ext2_block_relocate(fs, newsize))
		return 0;

	diff = fs->sb.s_blocks_count - newsize;

	while (diff > 0)
	{
		sizelast = fs->sb.s_blocks_count - fs->sb.s_first_data_block -
			(fs->numgroups - 1) * fs->sb.s_blocks_per_group;

		if (diff < sizelast)
		{
			if (!ext2_shrink_group(fs, sizelast - diff))
				return 0;

			diff = 0;
		}
		else
		{
			if (!ext2_del_group(fs))
				return 0;

			diff -= sizelast;
		}
	}

	return 1;
}


static int ext2_determine_itoffset(struct ext2_fs *fs)
{
	int i;

	fs->itoffset = fs->gd[0].bg_inode_table - fs->sb.s_first_data_block;

	for (i=0;i<fs->numgroups;i++)
	{
		blk_t start;
		blk_t bb;
		blk_t ib;
		blk_t it;

		start = fs->sb.s_first_data_block + (i * fs->sb.s_blocks_per_group);
		it = start + fs->itoffset;

		if (ext2_is_group_sparse(fs, i))
		{
			ib = it - 1;
			bb = it - 2;
		}
		else
		{
			bb = start;
			ib = start + 1;
		}

		if (fs->gd[i].bg_block_bitmap != bb ||
		    fs->gd[i].bg_inode_bitmap != ib ||
		    fs->gd[i].bg_inode_table != it)
		{
			fprintf(stderr, "This ext2 filesystem has a rather strange layout!\n");
			fprintf(stderr, "Please run dumpe2fs over this filesystem and it to\n");
			fprintf(stderr, "<lbuijten@cs.leidenuniv.nl>. I won't resize it, sorry.\n");
			return 0;
		}
	}

	return 1;
}


int ext2_resize_fs(struct ext2_fs *fs, blk_t newsize)
{
	blk_t container;
	blk_t residue;
	int status;

	fprintf(stderr, "ext2_resize_fs\n");

	if (!ext2_determine_itoffset(fs))
		return 0;

	residue = (newsize - fs->sb.s_first_data_block) % fs->sb.s_blocks_per_group;
	if (residue && residue <= fs->adminblocks)
	{
		fprintf(stderr, "ext2_resize_fs: %i is an impossible size for an ext2 fs! rounding down to %i\n",
			newsize, newsize-residue);
		newsize -= residue;
	}

	if (newsize == fs->sb.s_blocks_count)
		return 1;

	container = fs->devhandle->ops->get_size(fs->devhandle->cookie);
	if (newsize > container)
	{
		fprintf(stderr, "ext2_resize_fs: your container (i.e. the file or partition\n");
		fprintf(stderr, "                the filesystem) is in is not big enough for\n");
		fprintf(stderr, "                a filesystem of %i blocks (it's only\n", newsize);
		fprintf(stderr, "                %i blocks)\n", container);
		return 0;
	}

	if ((fs->relocator_pool = (unsigned char *)malloc(ext2_relocator_pool_size << 10)) == NULL)
	{
		fprintf(stderr, "ext2_resize: error allocating relocator pool!\n");
		return 0;
	}
	fs->relocator_pool_end = fs->relocator_pool + (ext2_relocator_pool_size << 10);

	if (newsize < fs->sb.s_blocks_count)
		status = ext2_shrink_fs(fs, newsize);
	else
		status = ext2_grow_fs(fs, newsize);

	free(fs->relocator_pool);
	fs->relocator_pool = NULL;
	fs->relocator_pool_end = NULL;

	return status;
}
