#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dnsconf.h"
#include "internal.h"

/*
	Replace or add a record in this DOMAIN definition.
	new_rec has been created with new. It will be managed (even deleted)
	by this function.
*/
PUBLIC int ZONE::set (FQHOST &fq, RECORD *new_rec)
{
	/* #Specification: record file / multiple origin
		dnsconf can update with multiple $ORIGIN statement.
		It will find the closest origin.
	*/
	ORIGIN *match_ori = NULL;
	int minlevel = 100;
	for (int i=0; i<origins.getnb(); i++){
		ORIGIN *ori = origins.getitem(i);
		char hostpart[200];
		int level = fq.is_member(ori->origin.get(),hostpart);
		if (level > 0 && level < minlevel){
			match_ori = ori;
			new_rec->sethostpart(hostpart);
			minlevel = level;
		}
	}
	if (match_ori != NULL){
		match_ori->tbrec.add (new_rec);
		updatesoa();
	}
	return 0;
}
/*
	Erase a record in this DOMAIN definition.
	new_rec has been created with new. It is used to find the record
	to delete. It is also deleted by this function.
*/
PUBLIC int ZONE::unset (RECORD *new_rec)
{
	bool found = false;
	for (int i=0; i<origins.getnb() && !found; i++){
		ORIGIN *ori = origins.getitem(i);
		for (int o=0; o<ori->tbrec.getnb(); o++){
			RECORD *rec = ori->tbrec.getitem(o);
			if (rec->cmp(new_rec) == 0){
				ori->tbrec.remove_del(rec);
				o--;
				found = true;
			}
		}
	}
	if (found) updatesoa();
	delete new_rec;
	return 0;
}

/*
	Erase all records in this DOMAIN definition using the "left" element.
*/
PUBLIC int ZONE::unset_left (const char *left)
{
	bool found = false;
	for (int i=0; i<origins.getnb() && !found; i++){
		ORIGIN *ori = origins.getitem(i);
		for (int o=0; o<ori->tbrec.getnb(); o++){
			RECORD *rec = ori->tbrec.getitem(o);
			if (rec->cmp_left(left) == 0){
				ori->tbrec.remove_del(rec);
				o--;
				found = true;
			}
		}
	}
	if (found) updatesoa();
	return 0;
}


/*
	Locate all record of a type using the left value as the key.
*/
PUBLIC int ZONE::locate_left (
	FQHOST &fq,
	RECORD_TYPE rtype,
	RECORDS &recs)
{
	recs.neverdelete();
	for (int i=0; i<origins.getnb(); i++){
		ORIGIN *ori = origins.getitem(i);
		char hostpart[200];
		int level = fq.is_member(ori->origin.get(),hostpart);
		if (level > 0){
			for (int o=0; o<ori->tbrec.getnb(); o++){
				RECORD *rec = ori->tbrec.getitem(o);
				if (rec->is (rtype)
					&& rec->cmp_left (hostpart)==0){
					recs.add (rec);
				}
			}
		}
	}
	return recs.getnb();
}
/*
	Locate all IP address from reverse mapping
*/
PUBLIC int ZONE::locate_rev (
	const char *fullname,
	IP_ADDRS &tba)
{
	for (int i=0; i<origins.getnb(); i++){
		ORIGIN *ori = origins.getitem(i);
		const char *dom = ori->origin.get();
		for (int o=0; o<ori->tbrec.getnb(); o++){
			RECORD *rec = ori->tbrec.getitem(o);
			if (rec->is (RTYPE_PTR)
				&& rec->cmp_right (fullname)==0){
				RECORD_IN_PTR *pt = (RECORD_IN_PTR*)rec;
				char revip[100];
				sprintf (revip,"%s.%s",pt->addr.get(),dom);
				IP_ADDR *ra  = new IP_ADDR(revip);
				ra->reverse();
				tba.add(ra);
			}
		}
	}
	return tba.getnb();
}

/*
	Locate all record of a type using the right value as the key.
*/
PUBLIC int ZONE::locate_right (
	const char *value,
	RECORD_TYPE rtype,
	SSTRINGS &recs)
{
	int ret = 0;
	for (int i=0; i<origins.getnb(); i++){
		ORIGIN *ori = origins.getitem(i);
		const char *suffix = ori->origin.get();
		for (int o=0; o<ori->tbrec.getnb(); o++){
			RECORD *rec = ori->tbrec.getitem(o);
			if (rec->is (rtype)
				&& rec->cmp_right (value)==0){
				const char *name = rec->get_left();
				if (name != NULL){
					SSTRING *s = new SSTRING(name);
					int len = strlen(name);
					if (len > 0 && name[len-1] != '.'){
						if (strcmp(name,"@")==0){
							s->setfrom (suffix);
						}else{
							s->append (".");
							s->append (suffix);
						}
					}
					recs.add (s);
					ret++;
				}
			}
		}
	}
	return ret;
}

/*
	Locate all record of a type using the right value as the key.
*/
PUBLIC int PRIMARYS::locate_right (
	const char *value,
	RECORD_TYPE rtype,
	SSTRINGS &recs)
{
	int ret = 0;
	int n = getnb();
	for (int i=0; i<n; i++){
		ret += getitem(i)->locate_right(value,rtype,recs);
	}
	return ret;
}

/*
	Locate all host which point to a given IP.
*/
PUBLIC int DNS::locate_ip (
	const char *ip,
	SSTRINGS &recs)
{
	return primarys.locate_right (ip,RTYPE_A,recs);
}

/*
	Locate a domain in the DNS
	Return NULL if not found.
*/
PUBLIC PRIMARY *DNS::locate_domain(const char *domain)
{
	PRIMARY *ret = NULL;
	for (int d=0; d<primarys.getnb(); d++){
		PRIMARY *p = primarys.getitem(d);
		if (p->domain.icmp(domain)==0){
			ret = p;
			break;
		}
	}
	return ret;
}

