# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: datatab.tcl,v 2.2 1999/08/28 10:48:16 jfontain Exp $}

class dataTable {                       ;# module data view in the form of a table, that can be sorted using a column as a reference

    set ::dataTable::(list) {}
    set ::dataTable::(scrollbarBorderWidth) [expr {$widget::(default,ScrollbarBorderWidth)==0?0:1}]
    set ::dataTable::(scrollbarWidth) [expr {2*$widget::(default,ScrollbarWidth)/3}]

    proc dataTable {this parentPath args} composite {
        [new scroll table $parentPath\
            -scrollbarwidth $(scrollbarWidth) -scrollbarelementborderwidth $(scrollbarBorderWidth)\
            -width $global::viewerWidth -height $global::viewerHeight\
        ]
        $args
    } {
        set path $composite::($composite::($this,base),scrolled,path)
        # only allow interactive colun resizing
        # use arrow cursor instead of default insertion cursor, meaningless since cell editing is disabled
        $path configure -font $font::(mediumNormal) -rows 1 -state disabled -titlerows 1 -roworigin -1 -colstretchmode unset\
            -variable dataTable::${this}data -resizeborders col -cursor {} -bordercursor sb_h_double_arrow -highlightthickness 0\
            -takefocus 0
        $path tag configure select -background white

        # remove all class bindings for we do not use any and they would cause interferences
        bindtags $path [list $path [winfo toplevel $path] all]

        # allow border resizing with first button. does not interfere with drag bindings since command does nothing unless mouse
        # click occured in a column border, which cannot be the case when dragging (see drag validation procedure in this class)
        bind $path <ButtonPress-1> "dataTable::buttonPress $this %x %y"
        bind $path <Button1-Motion> "if \$dataTable::($this,columnBorderHit) {%W border dragto %x %y}"

        set ($this,sortOrder) increasing
        set ($this,tablePath) $path

        lappend ::dataTable::(list) $this

        composite::complete $this
        setupDataView $this                                              ;# wait till -data and eventually -view options are defined
    }

    proc ~dataTable {this} {
        variable ${this}data

        if {[string length $composite::($this,-data)]>0} {
            setTrace $this 0                                                                                         ;# remove trace
        }
        catch {unset ${this}data}                                                                      ;# eventually free table data
        if {[info exists ($this,arrow)]} {
            eval delete $($this,arrow) $($this,tips)
        }
        if {[info exists ($this,drag)]} {
            delete $($this,drag)
        }
        if {[info exists ($this,selector)]} {                                       ;# selector may not exist if dragging disallowed
            delete $($this,selector)
        }
        ldelete ::dataTable::(list) $this
    }

    proc options {this} {
        return [list\
            [list -columnwidths columnWidths ColumnWidths {} {}]\
            [list -data data Data {} {}]\
            [list -draggable draggable Draggable 0 0]\
            [list -titlefont titleFont TitleFont $font::(mediumBold) $font::(mediumBold)]\
            [list -view view View {} {}]\
        ]
    }

    # list of column widths (as specified in the tkTable manual), applied to existing columns, can be empty.
    proc set-columnwidths {this value} {
        updateColumnWidths $this
    }

    proc set-titlefont {this value} {
        if {[string length $composite::($this,-data)]==0} return                                         ;# nothing to do if no data

        set path $($this,tablePath)
        for {set column 0} {$column<[llength $($this,dataColumns)]} {incr column} {
            $path.$column.label configure -font $value
        }
    }

    proc set-data {this value} {                                                ;# value must be a fully qualified module data array
        if {$composite::($this,complete)} {
            error {option -data cannot be set dynamically}
        }
        if {![array exists $value]} {
            error "\"$value\" argument is not an existing array"
        }
    }

    proc set-draggable {this value} {
        if {$composite::($this,complete)} {
            error {option -draggable cannot be set dynamically}
        }
        if {!$value} return                                                                                           ;# no dragging
        set path $($this,tablePath)
        set ($this,drag) [new dragSite -path $path -validcommand "dataTable::validateDrag $this"]
        dragSite::provide $($this,drag) DATACELLS "dataTable::dragData $this"

        set ($this,selector) [new tableSelector -selectcommand "dataTable::setCellsState $this"]
        bind $path <ButtonRelease-1> "dataTable::buttonRelease $this %x %y"
        bind $path <Control-ButtonRelease-1> "dataTable::toggleSelection $this %x %y"
        bind $path <Shift-ButtonRelease-1> "dataTable::extendSelection $this %x %y"
    }

    # override default view defined in -data option with visibleColumns and sort members
    proc set-view {this value} {                                            ;# value must be a fully qualified module sub data array
        if {$composite::($this,complete)} {
            error {option -view cannot be set dynamically}
        }
        if {![array exists $value]} {
            error "\"$value\" argument is not an existing array"
        }
    }

    proc buttonRelease {this x y} {
        if {!$($this,columnBorderHit)} {                                   ;# if column was resized, do not interfere with selection
            set number [expr {[$($this,tablePath) cget -rows]-1}]                                   ;# calculate number of data rows
            if {$number==0} return
            scan [$($this,tablePath) index @$x,$y] %d,%d row column
            if {$row<0} return                                                                           ;# title row, nothing to do
            if {[info exists ($this,selector)]} {
                selector::select $($this,selector) $row,$column
            }
        }
        unset ($this,columnBorderHit)
    }

    proc columnSort {this column} {                        ;# sort table rows using the column that the user selected as a reference
        if {$column==$($this,sortColumn)} {                                                                  ;# sort the same column
            if {[string equal $($this,sortOrder) increasing]} {                                                ;# but toggle sorting
                set ($this,sortOrder) decreasing
            } else {
                set ($this,sortOrder) increasing
            }
        } else {                                                                ;# sort for the first time or for a different column
            set ($this,sortColumn) $column
            set ($this,sortOrder) increasing
        }
        # deselect all cells since reordering rows renders selection meaningless
        if {[info exists ($this,selector)]} {
            selector::clear $($this,selector)
        }
        update $this                                                                                     ;# update table immediately
    }

    proc update {this args} {                                   ;# update display using module data. ignore eventual trace arguments
        variable ${this}data
        upvar #0 $composite::($this,-data) data                                              ;# data must be visible at global level

        set path $($this,tablePath)

        set cursor [$path cget -cursor]                                                                               ;# save cursor
        $path configure -cursor watch                                                                  ;# show user that we are busy
        ::update idletasks

        set sortColumn $($this,sortColumn)
        set lists {}
        if {[regexp {^integer|real$} $data($sortColumn,type)]} {                                                     ;# numeric type
            foreach name [array names data *,$sortColumn] {
                scan $name %u dataRow
                if {[catch {expr {double($data($dataRow,$sortColumn))}}]} {                   ;# handle empty values or ? characters
                    lappend lists [list $dataRow 0]                                                          ;# assume 0 for sorting
                } else {
                    lappend lists [list $dataRow $data($dataRow,$sortColumn)]
                }
            }
        } else {
            foreach name [array names data *,$sortColumn] {
                scan $name %u dataRow
                lappend lists [list $dataRow $data($dataRow,$sortColumn)]
            }
        }
        # sort data rows according to sort column (column numbering is identical for table data and source data)
        set lists [lsort -$($this,sortOrder) -$data($sortColumn,type) -index 1 $lists]

        if {[info exists ($this,selector)]} {
            set selector $($this,selector)
        }
        set changed 0                                                        ;# keep track of whether any rows were added or removed
        set row 0
        set rows {}
        foreach pair $lists {
            set dataRow [lindex $pair 0]
            if {![info exists ${this}data($row,dataRow)]} {                                                       ;# gather new rows
                lappend rows $row
            }
            set ${this}data($row,dataRow) $dataRow                                        ;# keep track of table / data rows mapping
            set column 0
            foreach dataColumn $($this,dataColumns) {
                set ${this}data($row,$column) $data($dataRow,$dataColumn)
                incr column
            }
            incr row
        }
        $path configure -rows [expr {$row+1}]                                           ;# fit to data (take into account title row)

        set columns [llength $($this,dataColumns)]

        if {[llength $rows]>0} {                                                                      ;# one or more rows were added
            set changed 1
            set cells {}
            foreach new $rows {
                for {set column 0} {$column<$columns} {incr column} {
                    lappend cells $new,$column
                }
            }
            if {[info exists selector]} {
                selector::add $selector $cells                                           ;# make selector aware of new cells at once
            }
        }

        set rows {}
        while {[info exists ${this}data($row,dataRow)]} {                                                     ;# gather removed rows
            lappend rows $row
            incr row
        }
        if {[llength $rows]>0} {                                                                    ;# one or more rows were removed
            set changed 1
            set cells {}
            foreach old $rows {
                unset ${this}data($old,dataRow)
                for {set column 0} {$column<$columns} {incr column} {
                    lappend cells $old,$column
                    unset ${this}data($old,$column)
                }
            }
            if {[info exists selector]} {
                selector::remove $selector $cells                                    ;# make selector aware of removed cells at once
            }
        }

        if {$changed&&[info exists selector]} {
            selector::clear $selector                  ;# deselect all cells since new or deleted rows renders selection meaningless
        }

        $path configure -cursor $cursor                                                                            ;# restore cursor
        ::update idletasks
    }

    proc dragData {this format} {
        variable ${this}data

        set data $composite::($this,-data)
        set list {}
        foreach cell [selector::selected $($this,selector)] {
            scan $cell %d,%d row column
            # data cell format is array(row,column)
            lappend list ${data}([set ${this}data($row,dataRow)],[set ${this}data($column,dataColumn)])
        }
        return $list
    }

    proc validateDrag {this x y} {
        if {$($this,columnBorderHit)} {
            return 0                                                                              ;# resizing a column: prevent drag
        }
        if {[$($this,tablePath) cget -rows]<=1} {
            return 1                                                  ;# allow dragging of empty table (with eventually 1 title row)
        }
        # allow dragging only from a selected cell
        return [expr {[lsearch -exact [selector::selected $($this,selector)] [$($this,tablePath) index @$x,$y]]>=0}]
    }

    proc setCellsState {this cells select} {
        if {$select} {
            eval $($this,tablePath) tag cell select $cells
        } else {
            eval $($this,tablePath) tag cell {{}} $cells
        }
    }

    proc toggleSelection {this x y} {
        set cell [$($this,tablePath) index @$x,$y]
        scan $cell %d row
        if {$row<0} return                                                                         ;# prevent selection on title row
        selector::toggle $($this,selector) $cell
    }

    proc extendSelection {this x y} {
        set cell [$($this,tablePath) index @$x,$y]
        scan $cell %d row
        if {$row<0} return                                                                         ;# prevent selection on title row
        selector::extend $($this,selector) $cell
    }

    proc updateSortingArrow {this column} {
        set path $widget::($($this,arrow),path)

        set label $($this,tablePath).$column.label                 ;# copy title label bindings for contextual help and mouse action
        foreach event {<Enter> <Leave> <ButtonRelease-1>} {
            bind $path $event [bind $label $event]
        }
        if {[string equal $($this,sortOrder) increasing]} {
            widget::configure $($this,arrow) -direction down
        } else {
            widget::configure $($this,arrow) -direction up
        }
        # place arrow in sorted column title frame on the right side of label
        place $path -in $($this,tablePath).$column -anchor e -relx 1 -rely 0.5 -relheight 1
    }

    proc createTitles {this} {
        upvar #0 $composite::($this,-data) data                                              ;# data must be visible at global level

        set path $($this,tablePath)
        set font $composite::($this,-titlefont)
        set column 0
        foreach dataColumn $($this,dataColumns) {
            # create table title labels in separate windows
            set frame [frame $path.$column]                       ;# use a frame as a container for label and eventual sorting arrow
            # force default arrow cursor as column resizing cursor sticks when moving across columns
            set label [label $path.$column.label -font $font -text $data($dataColumn,label) -cursor top_left_arrow]
            place $label -relwidth 1 -relheight 1         ;# use placer so that sorting arrow can eventually be displayed over label
            $path window configure -1,$column -window $frame -padx 2 -pady 2 -sticky nsew
            bind $label <ButtonRelease-1> "dataTable::columnSort $this $dataColumn; dataTable::updateSortingArrow $this $column"
            # setup context sensitive help on titles using help strings from module data
            bind $label <Enter> "lifoLabel::push $global::messenger [list $data($dataColumn,message)]"
            bind $label <Leave> "lifoLabel::pop $global::messenger"
            lappend ($this,tips) [new widgetTip -path $label -text {click to toggle sort}]
            incr column
        }
        $path configure -cols $column                                                              ;# fit table to number of columns
        updateColumnWidths $this

        set arrow [new arrowButton $path -state disabled -borderwidth 0 -highlightthickness 0 -width 12]
        widget::configure $arrow -disabledforeground [widget::cget $arrow -foreground]                   ;# make arrow fully visible
        # force default arrow cursor as column resizing cursor sticks when moving across columns
        $widget::($arrow,path) configure -cursor top_left_arrow
        lappend ($this,tips) [new widgetTip -path $widget::($arrow,path) -text {click to toggle sort}]
        set ($this,arrow) $arrow
    }

    proc buttonPress {this x y} {
        foreach {row column} [$($this,tablePath) border mark $x $y] {}
        # do not allow resizing with rightmost column edge
        set ($this,columnBorderHit)\
            [expr {[info exists column]&&([string length $column]>0)&&($column<([$($this,tablePath) cget -cols]-1))}]
    }

    proc setupDataView {this} {
        if {[string length $composite::($this,-data)]==0} return                                         ;# nothing to do if no data

        variable ${this}data
        if {[string length $composite::($this,-view)]>0} {
            upvar #0 $composite::($this,-view) data                                          ;# data must be visible at global level
        } else {
            upvar #0 $composite::($this,-data) data                                          ;# data must be visible at global level
        }
        if {[catch {set columns $data(visibleColumns)}]} {               ;# if not user defined visibility, make all columns visible
            set columns {}                                                                                  ;# gather column indices
            foreach name [array names data *,label] {
                if {[scan $name %u column]>0} {
                    lappend columns $column
                }
            }
        }
        set ($this,dataColumns) [lsort -integer $columns]                                                ;# then sort and store them
        set ($this,sortColumn) [lindex $data(sort) 0]
        if {[lsearch -exact $columns $($this,sortColumn)]<0} {
            error "sort column $($this,sortColumn) is not visible"
        }
        set ($this,sortOrder) [lindex $data(sort) 1]
        set column 0
        foreach dataColumn $($this,dataColumns) {                                              ;# store table / data columns mapping
            set ${this}data($column,dataColumn) $dataColumn
            if {$dataColumn==$($this,sortColumn)} {
                set sortColumnIndex $column
            }
            incr column
        }
        createTitles $this
        updateSortingArrow $this $sortColumnIndex
        setupColumnsAnchoring $this
        setTrace $this 1
    }

    proc updateColumnWidths {this} {                                                        ;# best apply widths to existing columns
        set path $($this,tablePath)
        set index 0
        foreach width $composite::($this,-columnwidths) {
            $path width $index $width
            if {[incr index]>=[$path cget -cols]} return                                                          ;# no more columns
        }
    }

    proc initializationConfiguration {this} {
        set path $($this,tablePath)
        for {set column 0} {$column<[$path cget -cols]} {incr column} {
            lappend widths [$path width $column]
        }
        return [list -columnwidths $widths]
    }

    proc setTrace {this on} {
        if {$on} {
            set command variable
        } else {
            set command vdelete
        }
        trace $command $composite::($this,-data)(updates) w "dataTable::update $this"                          ;# track data updates
    }

    proc setupColumnsAnchoring {this} {                                              ;# handle eventual data anchor column attribute
        upvar #0 $composite::($this,-data) data

        set column -1
        set path $($this,tablePath)
        foreach dataColumn $($this,dataColumns) {
            incr column
            if {[catch {set anchor $data($dataColumn,anchor)}]} continue      ;# no anchor for this column, default (center) is used
            if {![regexp {^center|left|right$} $anchor]} {
                error "bad anchor value \"$anchor\": must be center, left or right"
            }
            if {[string equal $anchor center]} continue                       ;# nothing to do as center is column anchoring default
            if {![$path tag exists $anchor]} {        ;# create anchor tag as needed and use valid anchor values (see above) as name
                array set convert {left w right e}
                $path tag configure $anchor -anchor $convert($anchor)
            }
            $path tag col $anchor $column
        }
    }

}
