/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel2.h"

void ATL_drefsymv
(
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const double               ALPHA,
   const double               * A,
   const int                  LDA,
   const double               * X,
   const int                  INCX,
   const double               BETA,
   double                     * Y,
   const int                  INCY
)
{
/*
 * Purpose
 * =======
 *
 * ATL_drefsymv performs the matrix-vector operation
 *
 *    y := alpha * A * x + beta * y,
 *
 * where alpha and beta are scalars, x and y are n-element vectors and A
 * is an n by n symmetric matrix.
 *
 * Arguments
 * =========
 *
 * UPLO    (input)                       const enum ATLAS_UPLO
 *         On entry, UPLO  specifies whether the upper or lower triangu-
 *         lar part of the array A is to be referenced as follows:
 *
 *             UPLO = AtlasUpper   Only the upper triangular part of A
 *                                 is to be referenced.
 *
 *             UPLO = AtlasLower   Only the lower triangular part of A
 *                                 is to be referenced.
 *
 *         Unchanged on exit.
 *
 * N       (input)                       const int
 *         On entry, N specifies the order of the matrix A. N must be at
 *         least zero. Unchanged on exit.
 *
 * ALPHA   (input)                       const double
 *         On entry, ALPHA specifies the scalar alpha.   When  ALPHA  is
 *         supplied as zero then  A and X  need not be set on input. Un-
 *         changed on exit.
 *
 * A       (input)                       const double *
 *         On entry,  A  points  to an array of size equal to or greater
 *         than   LDA * n * sizeof(   double  ).   Before   entry   with
 *         UPLO = AtlasUpper, the leading  n by n  upper triangular part
 *         of the array  A must contain the upper triangular part of the
 *         symmetric matrix  and  the strictly  lower triangular part of
 *         A is not referenced. Before entry with UPLO = AtlasLower, the
 *         leading  n by n  lower  triangular part of the array  A  must
 *         contain the lower triangular part of the symmetric matrix and
 *         the  strictly upper triangular part of  A  is not referenced.
 *         Unchanged on exit.
 *
 * LDA     (input)                       const int
 *         On entry, LDA  specifies the leading dimension of A as decla-
 *         red  in  the  calling  (sub) program.  LDA  must be  at least
 *         MAX( 1, n ). Unchanged on exit.
 *
 * X       (input)                       const double *
 *         On entry,  X  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCX ) ) * sizeof(   double  ),
 *         that contains the vector x. Unchanged on exit.
 *
 * INCX    (input)                       const int
 *         On entry, INCX specifies the increment for the elements of X.
 *         INCX must not be zero. Unchanged on exit.
 *
 * BETA    (input)                       const double
 *         On entry,  BETA  specifies the scalar  beta.   When  BETA  is
 *         supplied as zero then Y  need not be set on input.  Unchanged
 *         on exit.
 *
 * Y       (input/output)                double *
 *         On entry,  Y  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCY ) ) * sizeof(   double  ),
 *         that contains the vector y.  Before entry with BETA non-zero,
 *         the incremented array  Y  must contain the vector y. On exit,
 *         Y is overwritten by the updated vector y.
 *
 * INCY    (input)                       const int
 *         On entry, INCY specifies the increment for the elements of Y.
 *         INCY must not be zero. Unchanged on exit.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   int                        i, iaij, ix, iy, j, jaj, jx, jy, ldap1 = LDA + 1;
   register double            t0, t1;
/* ..
 * .. Executable Statements ..
 *
 */
   if( ( N == 0 ) || ( ( ALPHA == ATL_dZERO ) &&
                       ( BETA  == ATL_dONE  ) ) ) return;

   if( BETA != ATL_dONE )
   {
      if( BETA == ATL_dZERO )
      { for( i = 0, iy = 0; i < N; i++, iy += INCY ) Y[iy]  = ATL_dZERO; }
      else
      { for( i = 0, iy = 0; i < N; i++, iy += INCY ) Y[iy] *= BETA;      }
   }
   if( ALPHA == ATL_dZERO ) return;

   if( UPLO == AtlasUpper )
   {
      for( j = 0,      jaj = 0,    jx  = 0,    jy  = 0;
           j < N; j++, jaj += LDA, jx += INCX, jy += INCY )
      {
         t0 = ALPHA * X[jx];
         t1 = ATL_dZERO;
         for( i = 0,      iaij  = jaj, ix  = 0,    iy  = 0;
              i < j; i++, iaij += 1,   ix += INCX, iy += INCY )
         {
            Y[iy] += A[iaij] * t0;
            t1    += A[iaij] * X[ix];
         }
         Y[jy] += A[iaij] * t0 + ALPHA * t1;
      }
   }
   else
   {
      for( j = 0,      jaj  = 0,     jx  = 0,    jy  = 0;
           j < N; j++, jaj += ldap1, jx += INCX, jy += INCY )
      {
         t0 = ALPHA * X[jx];
         t1 = ATL_dZERO;
         Y[jy] += t0 * A[jaj];
         for( i = j+1,    iaij  = 1+jaj, ix  = jx + INCX, iy  = jy + INCY;
              i < N; i++, iaij += 1,     ix += INCX,      iy += INCY )
         {
            Y[iy] += A[iaij] * t0;
            t1    += A[iaij] * X[ix];
         }
         Y[jy] += ALPHA * t1;
      }
   }
/*
 * End of ATL_drefsymv
 */
}
