(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                system.ml4                                *)
(****************************************************************************)

(* camlp4o ../../config/coq_config.cmo ../../../tools/camlp4/ocamlpp.cmo *)

(* This file contains the functions which are dependent of the system.
   It is pre-processed with camlp4, using the "ifdef" syntaxe defined in
   the file ../../../tools/camlp4/ocamlpp.ml.

   The file ../../config/coq_config.ml (which is created at configuration time)
   contains the list of the ``#define'' as a list of capitalized strings
   [defined]. *)

open Std
open More_util
open Unix
open Pp
open Char

let getcwd = 
  Unix.getcwd

let safe_getenv_def var def =
  try Sys.getenv var
  with Not_found ->
    warning ("Environnement variable "^var^" not found: using '"^def^"' .");
    def

let safe_getenv n = safe_getenv_def n ("$"^n)
let safe_getlogin () =
  ifdef Unix then
    try getlogin()
    with _ -> (getpwuid(getuid())).pw_name
  else
    failwith "getlogin() not available !"


let home = (safe_getenv_def "HOME" ".");;

(* slash base [d1;d2;...;dn] construit le fichier/repertoire
   base/d1/d2/.../dn *)
let slash = List.fold_left Filename.concat;;



(* TODO: rendre glob un peu plus portable ! *)
let glob fname =
    let get_revexp_env n =
        List.rev(explode_chars(safe_getenv n)) in
    let get_revexp_pwdir n =
        List.rev(explode_chars(getpwnam n).pw_dir) in
    let rec aux sofar =
      parser
        [< ''/';
           rslt= (parser
                    [< ''~' ;
                       uname= (parser
                                 [< n= p_atom >] -> n
                               | [< >] -> safe_getlogin()) ;
                       s >] -> aux (get_revexp_pwdir uname) s
                   | [< s >] -> aux ('/'::sofar) s) >] -> rslt
      | [< ''$' ; a= p_atom ; s >] ->
          aux ((get_revexp_env a)@sofar) s
      | [< 'c ; s >] -> aux (c::sofar) s
      | [< >] -> List.rev sofar in

    let fname = if fname <> "" & fname.[0] = '~' then ("/"^fname) else fname in
      implode_chars (aux [] (Stream.of_string fname))



let make_suffix name suffix=
  if Filename.check_suffix name suffix then name else (name ^ suffix)


let open_trapping_failure open_fun name suffix =
  try open_fun (glob (make_suffix name suffix))
  with _ -> error ("Can't open " ^ name)


let file_readable_p na =
  try access (glob na) [R_OK];true
  with Unix_error (_, _, _) -> false


let open_in_maybe_compressed s =
(* this is used to intern the state. So the file is not text, we open it in 
   binary mode *) 
  try open_in_bin s
  with e -> (if file_readable_p (s^".z") then
               open_process_in ("gzcat "^s^".z")
             else raise e)
;;

(* Load path. *)
let lOADPATH = ref ([]:string list)

(*For AddPath*)
let add_path dir =
  lOADPATH := add_set dir !lOADPATH

let del_path dir =
  if not (List.mem dir !lOADPATH) then
    error (dir ^ " not on loadpath")
  else
    lOADPATH := rmv_set dir !lOADPATH

(*Gives the list of all the directories under dir*)
let alldir dir=
  let ini=Unix.getcwd()
  and tmp=Filename.temp_file "coq" "rec"
  and lst=ref []
  in
    Unix.chdir dir;
    (let bse=Unix.getcwd()
     in
       let _ = Sys.command ("find "^bse^" -type d >> "^tmp) in
       let inf=open_in tmp
       in
         try
           (while true do
              lst:=(!lst)@[input_line inf]
            done;
            [])
         with
            End_of_file ->
              close_in inf;
              Sys.remove tmp;
              Unix.chdir ini;
              !lst);;

(*For RecAddPath*)
let radd_path dir=
  List.iter add_path (alldir dir);;

let search_paths () = !lOADPATH

let search_in_path path filename =
  let rec search = function
      dir :: rem ->
	let f = glob (Filename.concat dir filename) in
	if file_readable_p f then f else search rem
    | [] ->
	raise Not_found
  in
    search path

(*For LoadPath or MLPath*)
let is_in_path lpath filename =
  try
    let _ = search_in_path lpath filename in true
  with
    Not_found -> false


let where_in_path lpath filename =
  search_in_path lpath filename


let round_char=ref (chr 45)

let round_reset()=
  round_char:=(chr 45);
  print_char(!round_char)

let round_step()=
  (if !round_char=(chr 45) then
     round_char:=(chr 92)
   else
     if !round_char=(chr 92) then
       round_char:=(chr 124)
     else
       if !round_char=(chr 124) then
         round_char:=(chr 47)
       else
         if !round_char=(chr 47) then
           round_char:=(chr 45));
  print_char(chr 8);
  print_char(!round_char);
  Pervasives.flush Pervasives.stdout

let round_end()=
  print_char(chr 8)

(*Loads all the .vo files in the LOADPATH*)
let all_vo_in_path () =
  let wrg=ref 0
  in
    let res=
      List.flatten (List.map (fun dir ->
        try
          (let dh=Unix.opendir dir
           and st=ref []
           and stg=ref ""
           in
             round_step();
             try
               (while true do
                  stg:=Unix.readdir dh;
                  if (Filename.check_suffix !stg ".vo") then
                    st:=(!st)@[Filename.chop_suffix !stg ".vo"];
                  round_step ()
                done;
                [])
             with
                End_of_file -> Unix.closedir dh; !st)
        with
           Unix.Unix_error (_,"opendir",_) ->
             print_char(chr 8);
             (if !wrg=0 then
               (print_char(chr 32);
                Pervasives.flush Pervasives.stdout;
                print_string
                  ("\n>>>>>>>Warning: there is no directory \""^dir^"\"\n"))
              else
                print_string
                  (">>>>>>>Warning: there is no directory \""^dir^"\"\n"));
             Pervasives.flush Pervasives.stdout;
             incr wrg;
             []) (List.rev (search_paths ())))
    in
      if !wrg>0 then
        (print_string
           (">>>>>>>Total: "^(string_of_int (!wrg))^" warning(s)\n");
         Pervasives.flush Pervasives.stdout);
      res;;

(* Rendu portable entre Unix/Mac/PC par emploi judicieux
   du module filename. -XL. *)

let find_file_in_path path name =
  let globname = glob name in
  if not (Filename.is_relative globname) then globname
  else 
    try search_in_path path name
    with
      Not_found ->
	errorlabstrm "System.find_file_in_path"
          (hOV 0 [< 'sTR"Can't find file" ; 'sPC ; 'sTR name ; 'sPC ;
                    'sTR"on loadpath" >])


let (extern_intern :
      	(int * string) -> ((string -> 'a -> unit) * (string -> 'a)))
  = function (magic,suffix) ->

  let extern_state name val_0 = 
    try
      let (expname,channel) =
(* we open the file in binary mode since it contains values and not text
   this is important in the windows environement. HL *)
        open_trapping_failure (fun n -> n,open_out_bin n) name suffix in
      try
        output_binary_int channel magic;
        output_value channel val_0;
        close_out channel
      with e -> 
        ((try Sys.remove expname
          with _ -> pPNL [< 'sTR"Warning: " ; 'sTR"Could not remove file " ;
                            'sTR expname ; 'sTR" which is corrupted !!" >]);
         raise e)
    with Sys_error s -> error ("System error: " ^ s)

  and intern_state name = 
    try
      let fname = find_file_in_path !lOADPATH (make_suffix name suffix) in
      let channel = open_in_maybe_compressed fname in
      if input_binary_int channel <> magic then
         error (fname^" not compiled with the current version of Coq");
      let v = input_value(channel) in
      close_in channel; 
      v
    with Sys_error s -> error("System error: " ^ s)

  in (extern_state,intern_state)


ifdef Unix then
  type process_times = Unix.process_times 
else 
  type process_times = int

let process_time () = 
  ifdef Unix then
    match Unix.times () with
    {Unix.tms_utime = ut;Unix.tms_stime = st} -> (ut,st)
  else (0.0,0.0)

let timestamp () = ifdef Unix then (time(),times()) else (0.0,0)

let fmt_time_difference (startreal,start) (stopreal,stop) =
  ifdef Unix then 
    [< 'rEAL(stopreal -. startreal); 'sTR" secs ";
       'sTR"(";
       'rEAL((-.) stop.tms_utime start.tms_utime); 'sTR"u";
       'sTR",";
       'rEAL((-.) stop.tms_stime start.tms_stime); 'sTR"s";
       'sTR")" >]
  else
    [< >]  


(* $Id: system.ml4,v 1.23 1999/07/19 10:49:48 herbelin Exp $ *)
