(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                astterm.ml                                *)
(****************************************************************************)

open Std;;
open Pp;;
open Names;;
open CoqAst;;
open Ast;;
open Vectops;;
open Generic;;
open Term;;
open Environ;;
open Termenv;;
open Impuniv;;
open Himsg;;
open Multcase_astterm;; 
open Evd;;

(****************************************************************)
(* Arguments normally implicit in the "Implicit Arguments mode" *)
(* but explicitely given                                        *)

let dest_explicit_arg = function
    (DOP1(XTRA ("!", [Num(_,j)]), t)) -> (j,t)
  | _ -> raise Not_found

let is_explicit_arg = function
    (DOP1(XTRA ("!", [Num(_,j)]), t)) -> true
  | _ -> false

let explicitize_appl l args = 
 let rec aux n l args acc =
 match (l,args) with 
  (i::l',a::args') -> 
   if i=n 
   then try let j,a' = dest_explicit_arg a in
            if j > i
            then aux (n+1) l' args (mkExistential::acc)
            else if j = i
                 then aux (n+1) l' args' (a'::acc)
                 else error "Bad explicitation number"
        with Not_found -> aux (n+1) l' args (mkExistential::acc)
   else if is_explicit_arg a then error "Bad explicitation number"
   else aux (n+1) l args' (a::acc)
 | ([],_) -> (List.rev acc)@args
 | (_,[]) -> (List.rev acc)
in aux 1 l args []
;;

let search_ref_with_impl k env id =
  try match k with
      CCI -> Machops.search_reference1 env id
    | FW  -> Machops.search_freference1 env id
    | OBJ -> anomaly "search_ref_cci_fw"
  with UserError _ -> (search_synconst k id,[])

let absolutize_var_with_impl k sigma env id =
  if (is_existential_id id) then
    match k with
 	CCI -> Machops.lookup_exist sigma env (evar_of_id k id),[]
      | FW  -> error "existentials in fterms not implemented"
      | OBJ -> anomaly "absolutize_var"
  else
  if (mem_glob id env) then
    ((VAR id), (try Vartab.implicits_of_var k id
                 with _ -> []))
  else
  try search_ref_with_impl k env id
  with Not_found ->  error_var_not_found "get_typ_and_impargs" id

let absolutize k sigma env constr = 
 let rec absrec env constr = match constr with
    VAR id -> fst (absolutize_var_with_impl k sigma env id)
  | Rel _ as t -> t
  | DOP1(XTRA ("!", [Num _]), _) -> error "Bad explicitation number"
  | DOPN(AppL,cl) -> (* Detect implicit args occurrences *)
     let cl1 = Array.to_list cl in
     let f = List.hd cl1 in
     let args = List.tl cl1 in
       begin match f with 
           VAR id ->
             let (c, impargs) = absolutize_var_with_impl k sigma env id in
             let newargs = explicitize_appl impargs args in
               mkAppList c (List.map (absrec env) newargs)
         | DOPN((Const _ | MutInd _ | MutConstruct _) as op, _) ->
	     let id = id_of_global op in
             let (c, impargs) = search_ref_with_impl k env id in
             let newargs = explicitize_appl impargs args in
               mkAppList c (List.map (absrec env) newargs)
         | (DOP1(XTRA ("!",[]), t)) ->
             mkAppList (absrec env t) (List.map (absrec env) args)
         | _ -> mkAppL (Array.map (absrec env) cl)
       end

  (* Pattern branches have a special absolutization *)
  | DOP1(XTRA("LAMEQN",_),_) as eqn  -> absolutize_eqn (absrec env) k env eqn

  | DOP0 _ as t -> t
  | DOP1(oper,c) -> DOP1(oper,absrec env c)
  | DOP2(oper,c1,c2) -> DOP2(oper,absrec env c1,absrec env c2)
  | DOPN(oper,cl) -> DOPN(oper,Array.map (absrec env) cl)
  | DOPL(oper,cl) -> DOPL(oper,List.map (absrec env) cl)
  | DLAM(na,c)    -> DLAM(na,absrec (add_rel (na,()) env) c)
  | DLAMV(na,cl)  -> DLAMV(na,Array.map (absrec (add_rel (na,()) env)) cl)

 in absrec env constr
;;

(* Fonctions exportes *)
let absolutize_cci sigma env constr = absolutize CCI sigma env constr
let absolutize_fw  sigma env constr = absolutize FW  sigma env constr

let dbize_sp = function
    Path(loc,sl,s) ->
      (try section_path sl s
       with Invalid_argument _ | Failure _ ->
         anomaly_loc(loc,"Astterm.dbize_sp",
                     [< 'sTR"malformed section-path" >]))
  | ast -> anomaly_loc(Ast.loc ast,"Astterm.dbize_sp",
                     [< 'sTR"not a section-path" >]);;

let rec iterated_binder oper ty = function
    DLAM(na,body) ->
    DOP2(oper,ty,DLAM(na,iterated_binder oper (lift 1 ty) body))
  | body -> body
;;

let dbize_op loc opn pl cl =
    match (opn,pl,cl) with
        ("META",[Num(_,n)], []) -> DOP0(Meta n)
      | ("XTRA",(Str(_,"MLCASE"))::tlpl,_) ->
          DOPN(XTRA("MLCASE",tlpl),Array.of_list cl)
      | ("XTRA",(Str(_,s))::tlpl,_) when (multcase_kw s) ->
          DOPN(XTRA(s,tlpl),Array.of_list cl)
      | ("XTRA",(Str(_,s))::tlpl,[]) -> DOP0(XTRA(s,tlpl))
      | ("XTRA",(Str(_,s))::tlpl,[c]) -> DOP1(XTRA(s,tlpl),c)
      | ("XTRA",(Str(_,s))::tlpl,[c1;c2]) -> DOP2(XTRA(s,tlpl),c1,c2)
      | ("XTRA",(Str(_,s))::tlpl,_) -> DOPN(XTRA(s,tlpl), Array.of_list cl)

      | ("PROP", [Id(_,s)], []) ->
          (try DOP0(Sort(Prop (contents_of_str s)))
           with Invalid_argument s -> anomaly_loc (loc,"Astterm.dbize_op",
                                                   [< 'sTR s >]))
      | ("TYPE", [], []) -> DOP0(Sort(Type(dummy_univ)))
      | ("IMPLICIT", [], []) -> DOP0(Implicit)

      | ("CAST", [], [c1;c2]) -> DOP2(Cast,c1,c2)
      | ("PROD", [], [c1;(DLAM _ as c2)]) -> DOP2(Prod,c1,c2)
      | ("LAMBDA", [], [c1;(DLAM _ as c2)]) -> DOP2(Lambda,c1,c2)
      | ("PRODLIST", [], [c1;(DLAM _ as c2)]) -> iterated_binder Prod c1 c2
      | ("LAMBDALIST", [], [c1;(DLAM _ as c2)]) -> iterated_binder Lambda c1 c2
      | ("APPLIST", [], _) -> DOPN(AppL,Array.of_list cl)
      | ("CONST", [sp], _) -> DOPN(Const (dbize_sp sp),Array.of_list cl)
      | ("ABST", [sp], _) ->
          (try global_abst (dbize_sp sp) (Array.of_list cl)
           with
               Failure _ | Invalid_argument _ ->
                 anomaly_loc(loc,"Astterm.dbize_op",
                             [< 'sTR"malformed abstraction" >])
             | Not_found -> user_err_loc(loc,"Astterm.dbize_op",
                                         [< 'sTR"Unbound abstraction" >]))
      | ("MUTIND", [sp;Num(_,tyi)], _) ->
          DOPN(MutInd (dbize_sp sp, tyi),Array.of_list cl)
            
      | ("MUTCASE", [], p::c::l) -> mkMutCase None p c l

      | ("MUTCONSTRUCT", [sp;Num(_,tyi);Num(_,n)], _) ->
          DOPN(MutConstruct ((dbize_sp sp, tyi),n),Array.of_list cl)

      | ("SQUASH",[],[_]) -> user_err_loc
            (loc,"Astterm.dbize_op",
             [< 'sTR "Unrecognizable braces expression." >])

      | (op,_,_) -> anomaly_loc
            (loc,"Astterm.dbize_op",
             [< 'sTR "Unrecognizable constr operator: "; 'sTR op >])
;;



let split_params =
 let rec sprec acc = function
     (Id _ as p)::l -> sprec (p::acc) l
   | (Str _ as p)::l -> sprec (p::acc) l
   | (Num _ as p)::l -> sprec (p::acc) l
   | (Path _ as p)::l -> sprec (p::acc) l
   | l -> (List.rev acc,l)
 in sprec []
    
;;

let dbize =
 let rec dbrec env = function
    Nvar(loc,s) ->
      let id = id_of_string s in
        (try
           match lookup_id id env with
               RELNAME(n,_) -> Rel n
             | _ -> VAR id
         with Not_found -> VAR id)

   | Slam(_,ona,Node(_,"V$",l)) ->
       let na =
         (match ona with Some s -> Name (id_of_string s) | _ -> Anonymous)
       in DLAMV(na,Array.of_list (List.map (dbrec (add_rel (na,()) env)) l))

   | Slam(_,ona,t) ->
       let na =
         (match ona with Some s -> Name (id_of_string s) | _ -> Anonymous)
       in DLAM(na, dbrec (add_rel (na,()) env) t)

   | Node(_,"FIX", (Nvar (_,iddef))::ldecl) ->
       let (rlf,lf,ln,lA,lt) =
         let rec decrec lf ln lA lt = function
             [] -> (lf,List.rev lf, List.rev ln, List.rev lA, List.rev lt)
           | Node(_,"NUMFDECL", [Nvar(_,fi); Num(_,ni); astA; astT])::rest ->
               decrec ((id_of_string fi)::lf) (ni-1::ln)
                 ((dbrec env astA)::lA) (astT::lt) rest
           | Node(_,"FDECL", [Nvar(_,fi); Node(_,"BINDERS",bl); astA; astT])
             ::rest ->
               let destruct_binder = function
		   Node(_,"BINDER",c::idl) ->
		     List.map (fun id -> (id_of_string (nvar_of_ast id),c)) idl
		 | _ -> anomaly "BINDER is expected" in
	       let binders = List.flatten (List.map destruct_binder bl) in
	       let mkLambdaC(x,a,b) =
		 ope("LAMBDA",[a;slam(Some (string_of_id x),b)]) in
	       let mkLambdaCit =
		 List.fold_right (fun (x,a) b -> mkLambdaC(x,a,b)) in
	       let mkProdC (x,a,b) =
		 ope("PROD",[a;slam(Some (string_of_id x),b)]) in
	       let mkProdCit =
		 List.fold_right (fun (x,a) b -> mkProdC(x,a,b)) in
               let ni = List.length binders - 1 in
               let arity = dbrec env (mkProdCit binders astA) in
               decrec ((id_of_string fi)::lf) (ni::ln)
                 (arity::lA) ((mkLambdaCit binders astT)::lt) rest
           | _ -> anomaly "FDECL is expected"
         in decrec [] [] [] [] ldecl 
       in let n =
	    try (index (id_of_string iddef) lf) -1
            with Failure _ -> error_fixname_unbound "dbize (FIX)" false iddef
       in let new_env = List.fold_left
                          (fun env fid -> add_rel (Name fid,()) env) env lf 
       in let defs = Array.of_list (List.map (dbrec new_env) lt)
       in let c = List.fold_left (fun c fi -> DLAM(Name fi,c))
                    (DLAMV(Name (List.hd rlf),defs)) (List.tl rlf)
       in DOPN(Fix(Array.of_list ln,n),Array.append (Array.of_list lA) [|c|])
            
   | Node(_,"COFIX", (Nvar(_,iddef))::ldecl) ->
       let (rlf,lf,lA,lt) = 
         let rec decrec lf lA lt = function
             [] -> (lf,List.rev lf, List.rev lA, List.rev lt)
           | Node(_,"CFDECL", [Nvar(_,fi); astA; astT])::rest -> 
               decrec ((id_of_string fi)::lf)
                 ((dbrec env astA)::lA) (astT::lt) rest
           | _ -> anomaly "CFDECL is expected"
         in decrec [] [] [] ldecl 
       in let n =
            try (index (id_of_string iddef) lf) -1
            with Failure _ -> error_fixname_unbound "dbize (COFIX)" true iddef
       in let new_env = List.fold_left
                          (fun env fid -> add_rel (Name fid,()) env) env lf 
       in let defs = Array.of_list (List.map (dbrec new_env) lt)
       in let c = List.fold_left (fun c fi -> DLAM(Name fi,c))
                    (DLAMV(Name (List.hd rlf),defs)) (List.tl rlf)
       in DOPN(CoFix(n),Array.append (Array.of_list lA) [|c|])

 | Node(loc,opn,tl) ->
     let (pl,tl) = split_params tl in
     dbize_op loc opn pl (List.map (dbrec env) tl)

 | _ -> anomaly "dbize: unexpected ast"

 in dbrec
;;



let dbize_kind k sigma env com =
  let c =
    try dbize env com
    with e ->
      wrap_error
 	(Ast.loc com, "dbize_kind",
         [< 'sTR"During conversion from explicit-names to" ; 'sPC ;
           'sTR"debruijn-indices" >], e,
         [< 'sTR"Perhaps the input is malformed" >]) in

  let c =
    try absolutize k sigma env c
    with e -> 
      wrap_error
 	(Ast.loc com, "Astterm.dbize_kind",
         [< 'sTR"During the relocation of global references," >], e,
         [< 'sTR"Perhaps the input is malformed" >])
  in c
;;

let dbize_cci sigma env com = dbize_kind CCI sigma env com
let dbize_fw  sigma env com = dbize_kind FW sigma env com

(* constr_of_com takes an environment of typing assumptions,
 * and translates a command to a constr.
 *)

let raw_constr_of_com sigma env com =
  let c = dbize_cci sigma (unitize_env env) com in
    try Sosub.soexecute c
    with Failure _|UserError _ -> error_sosub_execute CCI com;;

let raw_fconstr_of_com sigma env com =
  let c = dbize_fw sigma (unitize_env env) com in
    try Sosub.soexecute c
    with Failure _|UserError _ -> error_sosub_execute FW com;;

let raw_constr_of_compattern sigma env com =
  let c = dbize_cci sigma (unitize_env env) com in
    try Sosub.try_soexecute c
    with Failure _|UserError _ -> error_sosub_execute CCI com;;



(* Globalization of AST quotations *)

let mk_glob sigma env id =
  try Termast.bdize_no_casts false env (absolutize_cci sigma env (VAR id))
  with UserError _ -> raise Not_found
;;

let ast_adjust_consts sigma = (* locations are kept *)
 let rec dbrec env = function
    Nvar(loc,s) as ast ->
      let id = id_of_string s in
        if (Ast.isMeta s) or (is_existential_id id) then ast
        else
          if List.mem id (ids_of_env env) then ast
	  else
            (try Ast.set_loc loc (mk_glob sigma env id)
             with (UserError _ | Not_found) -> 
               try let _ = search_synconst CCI id in ast
               with Not_found ->
                 warning ("Could not globalize "^s);
                 ast)

   | Slam(loc,None,t) -> Slam(loc,None,dbrec (add_rel (Anonymous,()) env) t)

   | Slam(loc,Some na,t) ->
       let env' = add_rel (Name (id_of_string na),()) env in
         Slam(loc,Some na,dbrec env' t)
   | Node(loc,opn,tl) -> Node(loc,opn,List.map (dbrec env) tl)
   | x -> x
 in dbrec
;;


let globalize_command ast =
  let (sigma,sign) = Termenv.initial_sigma_sign() in
    ast_adjust_consts sigma (gLOB sign) ast
;;


(* Avoid globalizing in non command ast for tactics *)
let rec glob_ast sigma env = function 
  | Node(loc,"COMMAND",[c]) ->
      Node(loc,"COMMAND",[ast_adjust_consts sigma env c])
  | Node(loc,"COMMANDLIST",l) -> 
      Node(loc,"COMMANDLIST", List.map (ast_adjust_consts sigma env) l)
  | Slam(loc,None,t) ->
      Slam(loc,None,glob_ast sigma (add_rel (Anonymous,()) env) t)
  | Slam(loc,Some na,t) ->
      let env' = add_rel (Name (id_of_string na),()) env in
        Slam(loc,Some na, glob_ast sigma env' t)
  | Node(loc,opn,tl) -> Node(loc,opn,List.map (glob_ast sigma env) tl)
  | x -> x
;;

let globalize_ast ast =
  let (sigma,sign) = Termenv.initial_sigma_sign() in
    glob_ast sigma (gLOB sign) ast
;;

(* Installation of the AST quotations. "command" is used by default. *)
open Pcoq;;
define_quotation true "command" (map_entry globalize_command Command.command);;
define_quotation false "tactic" (map_entry globalize_ast Tactic.tactic);;
define_quotation false "vernac" (map_entry globalize_ast Vernac.vernac);;


(* $Id: astterm.ml,v 1.21 1999/09/08 12:39:45 herbelin Exp $ *)
