(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA                        ENS-CNRS                *)
(*              Rocquencourt                        Lyon                    *)
(*                                                                          *)
(*                                Coq V5.10                                 *)
(*                              Nov 25th 1994                               *)
(*                                                                          *)
(****************************************************************************)
(*                           multcase_astterm.ml                            *)
(****************************************************************************)

open Std;;
open Pp;;
open More_util;;
open Names;;
open Vectops;;
open Generic;;
open Term;;
open CoqAst;;
open Impuniv;;


(* This file contains functions dealing with multiple_case ast.
 * Functions are linked to absolutize via absrec (that is the closure
 * of  absolutize).  This file is also linked to the system through termast.ml
 *)

(* === Interface with termast.ml === *)

(* Xtra Asts that should always be translated as a DOPN *)
let multiple_case_keyw =
  ["MULTCASE"; "AS"; "EQN"; "TOMATCH"];;
let  multcase_kw s = List.mem s  multiple_case_keyw;;

(* == Error messages == *)
          
let mal_formed_mssg () =
  [<'sTR "malformed macro of multiple case" >];;

(* == Warning messages == *)

let warning_uppercase uplid =
  let vars =
    prlist_with_sep pr_spc (fun v -> [< 'sTR(Ast.id_of_ast v) >]) uplid in
  pP [<'sTR "Warning: the variable(s) "; vars;
       'sTR " start(s) with upper case in a pattern"; 'cUT >]
;;

(* determines if id starts with uppercase *)
let is_uppercase_var v =
 match (Ast.id_of_ast v).[0] with
    'A'..'Z' -> true
 | _  -> false
;;

let is_underscore id = id = id_of_string "_";;

(* absolutize_eqn replaces pattern variables that are global. This
 * means that we are going to remove the pattern abstractions that
 * correspond to constructors.
 * Warning: Names that are global but do not correspond to
 * constructors are considered non-known_global. E.g. if nat appears
 * in a pattern is considered as a variable name.  known_global should
 * be refined in order to consider known_global names only those that
 * correspond to constructors of inductive types.
 *)
let known_global k env id =
  let srch =
    match k with
      CCI -> Machops.search_reference
    | FW -> Machops.search_freference
    | _ -> anomaly "known_global" in
  try hd_is_constructor (srch env id) 
  with  (Not_found | UserError _) -> 
    (try  hd_is_constructor (Environ.search_synconst k id) 
    with Not_found -> false)
;;

let rec abs_eqn k env avoid = function
    (v::ids, DLAM(na,t)) ->
      let id = id_of_string (Ast.id_of_ast v) in
      if known_global k env id
      then abs_eqn k env avoid (ids, subst1 (VAR id) t)
      else
	let id' = if is_underscore id then id else next_ident_away id avoid in
      	let v' = Ast.ide (string_of_id id') in
	let (nids,nt) = abs_eqn k env (id'::avoid) (ids,t) in
	(v'::nids, DLAM(na,nt))
  | ([],t) -> ([],t)
  | _ -> assert false
;;

let rec absolutize_eqn absrec k env = function
    DOP1(XTRA("LAMEQN",ids),lam_eqn) ->
      let gids = ids_of_sign (get_globals env) in
      let (nids,neqn) =	abs_eqn k env gids (ids, lam_eqn) in
      let uplid = filter is_uppercase_var nids in
      let _ = if uplid <> [] then warning_uppercase uplid in
      DOP1(XTRA("LAMEQN",nids), absrec neqn)
  | _ -> anomalylabstrm "absolutize_eqn" (mal_formed_mssg())
;;


(* destruct_multcase mc_constr = 
  <predicate, terms_to_match, 
         [| [u1,patt_11;..patt_1n];...;[u1,patt_11..patt_1n]|]>
*)
(* Pb: in dest_eqn, ids can clash with ids of initial_sign ? *)
let destruct_multcase   =
  let rec dest_eqn = function
      ([], DOPN(XTRA("EQN",[]),pl)) -> Array.to_list pl
    | (id::ids, DLAM(_,t)) ->
 	dest_eqn (ids, subst1 (VAR (id_of_string id)) t)
    | _ -> anomalylabstrm "destruct_eqn" (mal_formed_mssg()) in
  let destruct_eqn  = 
    Array.map
      (function
	  DOP1(XTRA("LAMEQN",pl),lam_eqn) ->
	    dest_eqn (List.map Ast.id_of_ast pl, lam_eqn)
       	| _ -> anomalylabstrm "destruct_eqn" (mal_formed_mssg())) in
  let destruct_tomatch  = function
      DOPN(XTRA("TOMATCH",pl),v) -> v
    | _  -> anomalylabstrm "destruct_tomatch" (mal_formed_mssg())
  in function 
      (DOPN(XTRA("MULTCASE",pl),v)) ->
        v.(0), 
        destruct_tomatch v.(1), 
        destruct_eqn (Array.sub v 2 ((Array.length v)- 2))
    | _  -> anomalylabstrm "destruct_multcase" (mal_formed_mssg())
;;

(* $Id: multcase_astterm.ml,v 1.10 1999/09/03 17:50:39 barras Exp $ *)
