------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ D B U G                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.39 $                             --
--                                                                          --
--          Copyright (C) 1996-1999 Free Software Foundation, Inc.          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

--  Expand routines for generation of special declarations used by the
--  debugger. In accordance with the Dwarf 2.2 specification, certain
--  type names are encoded to provide information to the debugger.

with Sinfo; use Sinfo;
with Types; use Types;
with Uintp; use Uintp;

package Exp_Dbug is

   --------------------------------------
   -- Basic Encoding of External Names --
   --------------------------------------

   --  In the absence of pragma Interface, entities declared at the global
   --  level in library-level packages have names obtained by folding all
   --  letters to lower case, and replacing periods with two underscores.

   --  For library-level procedures, the name is folded to all lower case
   --  letters, and the characters _ada_ are prepended (the primary reason
   --  for this is to avoid confusion with similarly named C procedures,
   --  and in particular with the C procedure Main.

   --  In the case of a set of overloaded subprograms in the same
   --  package, the names are serialized by adding one of the two suffixes:

   --    $n    (dollar sign)
   --    __nn  (two underscores)

   --  where nn is a serial number (1 for the first overloaded function,
   --  2 for the second, etc.). The former suffix is used when a dollar
   --  sign is a valid symbol on the target machine and the latter is
   --  used when it is not.

   --  Operator names are encoded using the following encodings:

   --    Oabs       abs
   --    Oand       and
   --    Omod       mod
   --    Onot       not
   --    Oor        or
   --    Orem       rem
   --    Oxor       xor
   --    Oeq        =
   --    One        /=
   --    Olt        <
   --    Ole        <=
   --    Ogt        >
   --    Oge        >=
   --    Oadd       +
   --    Osubtract  -
   --    Oconcat    &
   --    Omultiply  *
   --    Odivide    /
   --    Oexpon     **

   --  These names are prefixed by the normal full qualification, and
   --  suffixed by the overloading identification. So for example, the
   --  second operator "=" defined in package Extra.Messages would
   --  have the name:

   --    extra__messages__Oeq__2

   --------------------------------
   -- Handling of Numeric Values --
   --------------------------------

   --  All numeric values here are encoded as strings of decimal digits.
   --  Only integer values need to be encoded. A negative value is encoded
   --  as the corresponding positive value followed by a lower case m for
   --  minus to indicate that the value is negative (e.g. 2m for -2).

   ----------------------------
   -- Note on Implicit Types --
   ----------------------------

   --  The compiler creates implicit type names in many situations where
   --  a type is present semantically, but no specific name is present.
   --  For example:

   --     S : Integer range M .. N;

   --  Here the subtype of S is not integer, but rather an anonymous subtype
   --  of Integer. Where possible, the compiler generates names for such
   --  anonymous types that are related to the type from which the subtype
   --  is obtained as follows:

   --     T name suffix

   --  where name is the name from which the subtype is obtained, using lower
   --  case letters and underscores, and suffix starts with an upper case
   --  letter. For example, the name for the above declaration of S might be:

   --     TintegerS4b

   --  If the debugger is asked to give the type of an entity and the type
   --  has the form T name suffix, it is probably appropriate to just use
   --  "name" in the response since this is what is meaningful to the
   --  programmer.

   -------------------------
   -- Type Name Encodings --
   -------------------------

   --  In the following typ is the name of the type as normally encoded by
   --  the debugger rules, i.e. a non-qualified name, all in lower case,
   --  with standard encoding of upper half and wide characters

      ------------------------
      -- Encapsulated Types --
      ------------------------

      --  In some cases, the compiler encapsulates a type by wrapping it in
      --  a structure. For example, this is used when a size or alignment
      --  specification requires a larger type. Consider:

      --    type y is mod 2 ** 64;
      --    for y'size use 256;

      --  In this case the compile generates a structure type y___PAD, which
      --  has a single field whose name is F. This single field is 64 bits
      --  long and contains the actual value.

      --  A similar encapsulation is done for some packed array types,
      --  in which case the structure type is y__LJM and the field name
      --  is OBJECT.

      --  When the debugger sees an object of a type whose name has a
      --  suffix not otherwise mentioned in this specification, the type
      --  is a record containing a single field, and the name of that field
      --  is all upper-case letters, it should look inside to get the value
      --  of the field, and neither the outer structure name, nor the
      --  field name should appear when the value is printed.

      -----------------------
      -- Fixed-Point Types --
      -----------------------

      --   Fixed-point types are encoded using a suffix that indicates the
      --   delta and small values. The actual type itself is a normal
      --   integer type.

      --     typ___XF_nn_dd
      --     typ___XF_nn_dd_nn_dd

      --   The first form is used when small = delta. The value of delta (and
      --   small) is given by the rational nn/dd, where nn and dd are decimal
      --   integers.
      --
      --   The second form is used if the small value is different from the
      --   delta. In this case, the first nn/dd rational value is for delta,
      --   and the second value is for small.

      ------------------------------
      -- VAX Floating-Point Types --
      ------------------------------

      --   Vax floating-point types are represented at run time as integer
      --   types, which are treated specially by the code generator. Their
      --   type names are encoded with the following suffix:

      --     typ___XFF
      --     typ___XFD
      --     typ___XFG

      --   representing the Vax F Float, D Float, and G Float types. The
      --   debugger must treat these specially. In particular, printing
      --   these values can be achieved using the debug procedures that
      --   are provided in package System.Vax_Float_Operations:

      --     procedure Debug_Output_D (Arg : D);
      --     procedure Debug_Output_F (Arg : F);
      --     procedure Debug_Output_G (Arg : G);

      --   These three procedures take a Vax floating-point argument, and
      --   output a corresponding decimal representation to standard output
      --   with no terminating line return.

      --------------------
      -- Discrete Types --
      --------------------

      --   Discrete types are coded with a suffix indicating the range in
      --   the case where one or both of the bounds are discriminants or
      --   variable.

      --   Note: at the current time, we also encode static bounds if they
      --   do not match the natural machine type bounds, but this may be
      --   removed in the future, since it is redundant for most debug
      --   formats ???

      --     typ___XD
      --     typ___XDL_lowerbound
      --     typ___XDU_upperbound
      --     typ___XDLU_lowerbound__upperbound

      --   If a discrete type is a natural machine type (i.e. its bounds
      --   correspond in a natural manner to its size), then it is left
      --   unencoded. The above encoding forms are used when there is a
      --   constrained range that does not correspond to the size or that
      --   has discriminant references or other non-static bounds.

      --   The first form is used if both bounds are dynamic, in which case
      --   two constant objects are present whose names are typ___L and
      --   typ___U in the same scope as typ, and the values of these constants
      --   indicate the bounds. As far as the debugger is concerned, these
      --   are simply variables that can be accessed like any other variables.
      --   In the enumeration case, these values correspond to the Enum_Rep
      --   values for the lower and upper bounds.

      --   The second form is used if the upper bound is dynamic, but the
      --   lower bound is either constant or depends on a discriminant of
      --   the record with which the type is associated. The upper bound
      --   is stored in a constant object of name typ___U as previously
      --   described, but the lower bound is encoded directly into the
      --   name as either a decimal integer, or as the discriminant name.

      --   The third form is similarly used if the lower bound is dynamic,
      --   but the upper bound is static or a discriminant reference, in
      --   which case the lower bound is stored in a constant object of
      --   name typ___L, and the upper bound is encoded directly into the
      --   name as either a decimal integer, or as the discriminant name.

      --   The fourth form is used if both bounds are discriminant references
      --   or static values, with the encoding first for the lower bound,
      --   then for the upper bound, as previously described.

      ------------------
      -- Biased Types --
      ------------------

      --   Only discrete types can be biased, and the fact that they are
      --   biased is indicated by a suffix of the form:

      --     typ___XB_lowerbound__upperbound

      --   Here lowerbound and upperbound are decimal integers, with the
      --   usual (postfix "m") encoding for negative numbers. Biased
      --   types are only possible where the bounds are static, and the
      --   values are represented as unsigned offsets from the lower
      --   bound given. For example:

      --     type Q is range 10 .. 15;
      --     for Q'size use 3;

      --   The size clause will force values of type Q in memory to be
      --   stored in biased form (e.g. 11 will be represented by the
      --   bit pattern 001).

      ----------------------------------------------
      -- Record Types with Variable-Length Fields --
      ----------------------------------------------

      --  The debugging formats do not fully support these types, and indeed
      --  some formats simply generate no useful information at all for such
      --  types. In order to provide information for the debugger, gigi creates
      --  a parallel type in the same scope with one of the names

      --    type___XVE
      --    type___XVU

      --  The former name is used for a record and the latter for the union
      --  that is made for a variant record (see below) if that union has
      --  variable size. These encodings suffix any other encodings that
      --  might be suffixed to the type name.

      --  The idea here is to provide all the needed information to interpret
      --  objects of the original type in the form of a "fixed up" type, which
      --  is representable using the normal debugging information.

      --  There are three cases to be dealt with. First, some fields may have
      --  variable positions because they appear after variable-length fields.
      --  To deal with this, we encode *all* the field bit positions of the
      --  special ___XV type in a non-standard manner.

      --  The idea is to encode not the position, but rather information
      --  that allows computing the position of a field from the position
      --  of the previous field. The algorithm for stepping from one field
      --  to another is as follows:

      --    1. Take the current bit position in the record, i.e. the first
      --       unused bit after the previous field, or zero if this is the
      --       first field of the record.

      --    2. If an alignment is given (see below), then round the current
      --       bit position up, if needed, to the next multiple of that
      --       alignment.

      --    3. If a bit offset is given (see below), then add the bit offset
      --       to the current bit position.

      --  The bit offset is encoded as the position of the field. A value
      --  of zero means that step 3 can be skipped (or zero added).

      --  The alignment, if present, is encoded in the field name of the
      --  record, which has a suffix:

      --    fieldname___XVAnn

      --  where the nn after the XA indicates the alignment value in storage
      --  units. This encoding is present only if an alignment is present.

      --  Second, the variable-length fields themselves are represented by
      --  replacing the type by a special access type. The designated type
      --  of this access type is the original variable-length type, and the
      --  fact that this field has been transformed in this way is signalled
      --  by encoding the field name as:

      --    field___XVL

      --  where field is the original field name. If a field is both
      --  variable-length and also needs an alignment encoding, then the
      --  encodings are combined using:

      --    field___XVLnn

      --  Note: the reason that we change the type is so that the resulting
      --  type has no variable-length fields. At least some of the formats
      --  used for debugging information simply cannot tolerate variable-
      --  length fields, so the encoded information would get lost.

      --  Third, in the case of a variant record, the special union
      --  that contains the variants is replaced by a normal C union.
      --  In this case, the positions are all zero.

      --  As an example of this encoding, consider the declarations:

      --    type Q is array (1 .. V1) of Float;       -- alignment 4
      --    type R is array (1 .. V2) of Long_Float;  -- alignment 8

      --    type X is record
      --       A : Character;
      --       B : Float;
      --       C : String (1 .. V3);
      --       D : Float;
      --       E : Q;
      --       F : R;
      --       G : Float;
      --    end record;

      --  The encoded type looks like:

      --    type anonymousQ is access Q;
      --    type anonymousR is access R;

      --    type X___XVE is record
      --       A        : Character;               -- position contains 0
      --       B        : Float;                   -- position contains 24
      --       C___XVL  : access String (1 .. V3); -- position contains 0
      --       D___XVA4 : Float;                   -- position contains 0
      --       E___XVL4 : anonymousQ;              -- position contains 0
      --       F___XVL8 : anonymousR;              -- position contains 0
      --       G        : Float;                   -- position contains 0
      --    end record;

      --  Notes:

      --  1) The B field could also have been encoded by using a position
      --  of zero, and an alignment of 4, but in such a case, the coding by
      --  position is preferred (since it takes up less space). We have used
      --  the (illegal) notation access xxx as field types in the example
      --  above, but in actual practice.

      --  2) The E field does not actually need the alignment indication
      --  but this may not be detected in this case by the conversion
      --  routines.

      --  All discriminants always appear before any variable-length
      --  fields that depend on them. So they can be located independent
      --  of the variable-length field, using the standard procedure for
      --  computing positions described above.

      --  The size of the __XVE or ___XVU record or union is set to the
      --  alignment (in bytes) of the original object so that the debugger
      --  can calculate the size of the original type.

      ---------------------
      -- Record subtypes --
      ---------------------

      --  If type is a subtype of a record type, we produce a type that
      --  has the name of the subtype followed by "___XVS" and is a record
      --  consisting of one field, whose name is that of the record type
      --  of which this type is a subtype.

      --  The layout and fields of the subtype can be displayed by using
      --  the information in the type. If a list of fields is provided
      --  in the subtype, they will correspond correctly to the fields
      --  in the actual record, but the structure will not be the same.
      --  Both the subtype and the type may have corresponding ___XVE types.

      --  The size of the subtype should be obtained from subtype-specific
      --  information. A variable whose name is the name of the subtype
      --  followed by "___XVZ" will be created to hold the size (in bits)
      --  of the subtype if it is not a constant and cannot be derived from
      --  the list of fields in the subtype.

      -----------------
      -- Array Types --
      -----------------

      --  Since there is no way for the debugger to obtain the index subtypes
      --  for an array type, we produce a type that has the name of the
      --  array type followed by "___XA" and is a record whose field names
      --  are the names of the types for the bounds. The types of these
      --  fields is an integer type which is meaningless.

      --  To conserve space, we do not produce this type unless one of
      --  the index types is either an enumeration type, has a variable
      --  upper bound, has a lower bound different from the constant 1,
      --  is a biased type, or is wider than "sizetype".

      --  Given the full encoding of these types (see above description for
      --  the encoding of discrete types), this means that all necessary
      --  information for addressing arrays is available. In some
      --  debugging formats, some or all of the bounds information may
      --  be available redundantly, particularly in the fixed-point case,
      --  but this information can in any case be ignored by the debugger.

   function Get_Encoded_Type_Name (E : Entity_Id) return Boolean;
   --  Return True if type name needs to be encoded according to the above
   --  rules. In that case, the suffix for the encoded name, not including
   --  the initial three underscores is stored in Name_Buffer with the
   --  length of the name in Name_Len and an ASCII.NUL character stored
   --  following the name. If no encoding is required, then False is returned
   --  and the values in Name_Buffer and Name_Len are undefined.

   ---------------------------
   -- Packed Array Encoding --
   ---------------------------

   --  For every packed array, two types are created, and both appear in
   --  the debugging output.

   --    The original declared array type is a perfectly normal array type,
   --    and its index bounds indicate the original bounds of the array.

   --    The corresponding packed array type, which may be a modular type, or
   --    may be an array of bytes type (see Exp_Pakd for full details). This
   --    is the type that is actually used in the generated code and for
   --    debugging information for all objects of the packed type.

   --  The name of the corresponding packed array type is:

   --    ttt___XPnnn

   --  where
   --    ttt is the name of the original declared array
   --    nnn is the component size in bits (1-31)

   --  When the debugger sees that an object is of a type that is encoded
   --  in this manner, it can use the original type to determine the bounds,
   --  and the component size to determine the packing details.

   --  Packed arrays are represented in tightly packed form, with no extra
   --  bits between components. This is true even when the component size
   --  is not a factor of the storage unit size, so that as a result it is
   --  possible for components to cross storage unit boundaries.

   --  The layout in storage is identical, regardless of whether the
   --  implementation type is a modular type or an array-of-bytes type.
   --  See Exp_Pakd for details of how these implementation types are used,
   --  but for the purpose of the debugger, only the starting address of
   --  the object in memory is significant.

   --  The following example should show clearly how the packing works in
   --  the little-endian and big-endian cases:

   --     type B is range 0 .. 7;
   --     for B'Size use 3;

   --     type BA is array (0 .. 5) of B;
   --     pragma Pack (BA);

   --     BV : constant BA := (1,2,3,4,5,6);

   --  Little endian case

   --        BV'Address + 2   BV'Address + 1    BV'Address + 0
   --     +-----------------+-----------------+-----------------+
   --     | 0 0 0 0 0 0 1 1 | 0 1 0 1 1 0 0 0 | 1 1 0 1 0 0 0 1 |
   --     +-----------------+-----------------+-----------------+
   --       <---------> <-----> <---> <---> <-----> <---> <--->
   --       unused bits  BV(5)  BV(4) BV(3)  BV(2)  BV(1) BV(0)
   --
   --  Big endian case
   --
   --        BV'Address + 0  BV'Address + 1    BV'Address + 2
   --     +-----------------+-----------------+-----------------+
   --     | 0 0 1 0 1 0 0 1 | 1 1 0 0 1 0 1 1 | 1 0 0 0 0 0 0 0 |
   --     +-----------------+-----------------+-----------------+
   --       <---> <---> <-----> <---> <---> <-----> <--------->
   --       BV(0) BV(1)  BV(2)  BV(3) BV(4)  BV(5)  unused bits

   function Make_Packed_Array_Type_Name
     (Typ   : Entity_Id;
      Csize : Uint)
      return  Name_Id;
   --  This function is used in Exp_Pakd to create the name that is encoded
   --  as described above. The entity Typ provides the name ttt, and the
   --  value Csize is the component size that provides the nnn value.

   --------------------------------------
   -- Pointers to Unconstrained Arrays --
   --------------------------------------

   --  There are two kinds of pointers to arrays. The debugger can tell
   --  which format is in use by the form of the type of the pointer.

   --    Fat Pointers

   --      Fat pointers are represented as a struct with two fields. This
   --      struct has two distinguished field names:

   --        P_ARRAY is a pointer to the array type. The name of this
   --        type is the unconstrained type followed by "___XUA". This
   --        array will have bounds which are the discriminants, and
   --        hence are unparsable, but will give the number of
   --        subscripts and the component type.

   --        P_BOUNDS is a pointer to a struct, the name of  whose type is the
   --        unconstrained array name followed by "___XUB" and which has
   --        fields of the form

   --           LBn (n a decimal integer) lower bound of n'th dimension
   --           UBn (n a decimal integer) upper bound of n'th dimension

   --        The bounds may be any integral type. In the case of an
   --        enumeration type, Enum_Rep values are used.

   --      The debugging information will sometimes reference an anonymous
   --      fat pointer type. Such types are given the name xxx___XUP, where
   --      xxx is the name of the designated type. If the debugger is asked
   --      to output such a type name, the appropriate form is "access xxx".

   --    Thin Pointers

   --      Thin pointers are represented as a pointer to a structure
   --      with two fields. The name of the type of that structure is that
   --      of the unconstrained array followed by "___XUT".

   --      The field ARRAY contains the array value. This array field is
   --      typically a variable-length array, and consequently the entire
   --      record structure will be encoded as previously described,
   --      resulting in a type with suffix "___XUT__XVE".

   --      The field BOUNDS is a struct containing the bounds as above.

   --------------------------------------
   -- Tagged Types and Type Extensions --
   --------------------------------------

   --  A type C derived from a tagged type P has a field named "_parent"
   --  of type P that contains its inherited fields. The type of this
   --  field is usually P (encoded as usual if it has a dynamic size),
   --  but may be a more distant ancestor, if P is a null extension of
   --  that type.

   --  The type tag of a tagged type is a field named _tag, of type void*.
   --  If the type is derived from another tagged type, its _tag field is
   --  found in its _parent field.

   -----------------------------
   -- Variant Record Encoding --
   -----------------------------

   --  The variant part of a variant record is encoded as a single field
   --  in the enclosing record, whose name is:

   --     discrim___XVN

   --  where discrim is the unqualified name of the variant. This field name
   --  is built by gigi (not by code in this unit). In the case of an
   --  Unchecked_Union record, this discriminant will not appear in the
   --  record, and the debugger must proceed accordingly (basically it
   --  can treat this case as it would a C union).

   --  The type corresponding to this field has a name that is obtained
   --  by concatenting the type name with the above string and is similar
   --  to a C union, in which each member of the union corresponds to one
   --  variant. However, unlike a C union, the size of the type may be
   --  variable even if each of the components are fixed size, since it
   --  includes a computation of which variant is present. In that case,
   --  it will be encoded as above and a type with the suffix "__XVN__XVU"
   --  will be present.

   --  The name of the union member is encoded to indicate the choices, and
   --  is a string given by the following grammar:

   --    union_name ::= {choice} | others_choice
   --    choice ::= simple_choice | range_choice
   --    simple_choice ::= S number
   --    range_choice  ::= R number T number
   --    number ::= {decimal_digit} [m]
   --    others_choice ::= O (upper case letter O)

   --  The m in a number indicates a negative value. As an example of this
   --  encoding scheme, the choice 1 .. 4 | 7 | -10 would be represented by

   --    R1T4S7S10m

   --  In the case of enumeration values, the values used are the
   --  actual representation values in the case where an enumeration type
   --  has an enumeration representation spec (i.e. they are values that
   --  correspond to the use of the Enum_Rep attribute).

   --  The type of the inner record is given by the name of the union
   --  type (as above) concatenated with the above string. Since that
   --  type may itself be variable-sized, it may also be encoded as above
   --  with a new type with a further suffix of "___XVU".

   --  As an example, consider:

   --    type Var (Disc : Boolean := True) is record
   --       M : Integer;

   --       case Disc is
   --         when True =>
   --           R : Integer;
   --           S : Integer;

   --         when False =>
   --           T : Integer;
   --       end case;
   --    end record;

   --    V1 : Var;

   --  In this case, the type var is represented as a struct with three
   --  fields, the first two are "disc" and "m", representing the values
   --  of these record  components.

   --  The third field is a union of two types, with field names S1 and O.
   --  S1 is a struct with fields "r" and "s", and O is a struct with
   --  fields "t".

   procedure Get_Variant_Encoding (V : Node_Id);
   --  This procedure is called by Gigi with V being the variant node.
   --  The corresponding encoding string is returned in Name_Buffer with
   --  the length of the string in Name_Len, and an ASCII.NUL character
   --  stored following the name.

   ---------------------------------
   -- Subtypes of Variant Records --
   ---------------------------------

   --  A subtype of a variant record is represented by a type in which the
   --  union field from the base type is replaced by one of the possible
   --  values. For example, if we have:

   --    type Var (Disc : Boolean := True) is record
   --       M : Integer;

   --       case Disc is
   --         when True =>
   --           R : Integer;
   --           S : Integer;

   --         when False =>
   --           T : Integer;
   --       end case;

   --    end record;
   --    V1 : Var;
   --    V2 : Var (True);
   --    V3 : Var (False);

   --  Here V2 for example is represented with a subtype whose name is
   --  something like TvarS3b, which is a struct with three fields. The
   --  first two fields are "disc" and "m" as for the base type, and
   --  the third field is S1, which contains the fields "r" and "s".

   --  The debugger should simply ignore structs with names of the form
   --  corresponding to variants, and consider the fields inside as
   --  belonging to the containing record.

   -------------------------------------------
   -- Character literals in Character Types --
   -------------------------------------------

   --  Character types are enumeration types at least one of whose
   --  enumeration literals is a character literal. Enumeration literals
   --  are usually simply represented using their identifier names. In
   --  the case where an enumeration literal is a character literal, the
   --  name aencoded as described in the following paragraph.

   --  A name QUhh, where each 'h' is a lower-case hexadecimal digit,
   --  stands for a character whose Unicode encoding is hh, and
   --  QWhhhh likewise stands for a wide character whose encoding
   --  is hhhh. The representation values are encoded as for ordinary
   --  enumeration literals (and have no necessary relationship to the
   --  values encoded in the names).

   --  For example, given the type declaration

   --    type x is (A, 'C', B);

   --  the second enumeration literal would be named QU43 and the
   --  value assigned to it would be 1.

   -------------------
   -- Modular Types --
   -------------------

   --  A type declared

   --    type x is mod N;

   --  Is encoded as a subrange of an unsigned base type with lower bound
   --  0 and upper bound N. That is, there is no name encoding; we only use
   --  the standard encodings provided by the debugging format.  Thus,
   --  we give these types a non-standard interpretation: the standard
   --  interpretation of our encoding would not, in general, imply that
   --  arithmetic on type x was to be performed modulo N (especially not
   --  when N is not a power of 2).

   ---------------------
   -- Context Clauses --
   ---------------------

   --  The SGI Workshop debugger requires a very peculiar and nonstandard
   --  symbol name containing $ signs to be generated that records the
   --  use clauses that are used in a unit. GDB does not use this name,
   --  since it takes a different philsophy of universal use visibility,
   --  with manual resolution of any ambiguities.

   --  The routines and data in this section are used to prepare this
   --  specialized name, whose exact contents are described below. Gigi
   --  will output this encoded name only in the SGI case (indeed, not
   --  only is it useless on other targets, but hazardous, given the use
   --  of the non-standard character $ rejected by many assemblers.

   --  "Use" clauses are encoded as follows:

   --    _LSS__ prefix for clauses in a subprogram spec
   --    _LSB__ prefix for clauses in a subprogram body
   --    _LPS__ prefix for clauses in a package spec
   --    _LPB__ prefix for clauses in a package body

   --  Following the prefix is the fully qualified filename, followed by
   --  '$' separated names of fully qualified units in the "use" clause.
   --  If a unit appears in both the spec and the body "use" clause, it
   --  will appear once in the _L[SP]S__ encoding and twice in the _L[SP]B__
   --  encoding. The encoding appears as a global symbol in the object file.

   procedure Save_Unitname_And_Use_List
     (Main_Unit_Node : Node_Id;
      Main_Kind      : Node_Kind);
   --  Creates a string containing the current compilation unit name
   --  and a dollar sign delimited list of packages named in a Use_Package
   --  clause for the compilation unit. Needed for the SGI debugger.

   --  The following variables are used for communication between the front
   --  end and the debugging output routines in Gigi.

   type Char_Ptr is access all Character;
   pragma Convention (C, Char_Ptr);
   --  Character pointers accessed from C

   Spec_Context_List, Body_Context_List : Char_Ptr;
   --  List of 'with' clauses for spec and body, respectively

   Spec_Filename, Body_Filename : Char_Ptr;
   --  Filenames for the spec and body, respectively

end Exp_Dbug;
