#
# TCL Library for tkCVS
#

#
# $Id: logcanvas.tcl,v 1.13.2.23 1999/10/28 05:04:14 dorothyr Exp $
# 
# Contains procedures used for the log canvas for tkCVS.
#

set const(boxx) 80
set const(boxy) 30
set const(spacex) 160
set const(spacey) 16
set const(textheight) 12
set cvscanv ""

proc new_logcanvas {localfile filelog} {
  #
  # Creates a new log canvas.  filelog must be the output of a cvs
  # log or rlog command.  If localfile is not "no file" then it is
  # the file name in the local directory that this applies to.
  #
  global cvs
  global revdate
  global revwho
  global revcomment
  global revbranches
  global revlist
  global cvscanv
  global tags
  global const

  #puts "new_logcanvas \{$localfile $filelog\}"
  unset cvscanv
  # Height and width to draw boxes
  set cvscanv(boxx) $const(boxx)
  set cvscanv(boxy) $const(boxy)
  # Gaps between boxes
  set cvscanv(gapx) [expr $cvscanv(boxx) + $const(spacex)]
  set cvscanv(gapy) [expr $cvscanv(boxy) + $const(spacey)]
  # Indent at top left of canvas
  set cvscanv(indx) 20
  set cvscanv(indy) 20
  # Static type variables used while drawing on the canvas.
  set cvscanv(xhigh) [expr $cvscanv(boxx) + $cvscanv(indx)]
  set cvscanv(yhigh) [expr $cvscanv(boxy) + $cvscanv(indy)]
  set cvscanv(xlow) 0
  set cvscanv(ylow) 0
  set cvscanv(maxy) $cvscanv(indy)

  if [info exists revlist] { unset revlist }
  if [info exists revdate] { unset revdate }
  if [info exists revwho] { unset revwho }
  if [info exists revbranches] { unset revbranches }
  if [info exists revcomment] { unset revcomment }
  if [info exists tags] { unset tags }

  static {canvasnum 0}

  # Make the canvas

  incr canvasnum
  set logcanvas ".logcanvas$canvasnum"
  toplevel $logcanvas

  frame $logcanvas.up -relief groove -border 2
  frame $logcanvas.up.left
  frame $logcanvas.up.right
  frame $logcanvas.up1 -relief groove -border 2
  frame $logcanvas.up1.left
  frame $logcanvas.up1.right
  frame $logcanvas.up2 -relief groove -border 2
  frame $logcanvas.up2.left
  frame $logcanvas.up2.right
  frame $logcanvas.down -relief groove -border 2

  pack $logcanvas.up -side top -fill x
  pack $logcanvas.up.left -side left -fill both
  pack $logcanvas.up.right -side left -fill both -expand 1
  pack $logcanvas.up1 -side top -fill x
  pack $logcanvas.up1.left -side left -fill both
  pack $logcanvas.up1.right -side left -fill both -expand 1
  pack $logcanvas.up2 -side top -fill x
  pack $logcanvas.up2.left -side left -fill both
  pack $logcanvas.up2.right -side left -fill both -expand 1
  pack $logcanvas.down -side bottom -fill x

  label $logcanvas.lfname -anchor w -text "RCS File Name"
  entry $logcanvas.tfname
  # This is a hidden entry that stores the local file name.
  entry $logcanvas.tlocalfile
  $logcanvas.tlocalfile delete 0 end
  $logcanvas.tlocalfile insert end $localfile

  label $logcanvas.lvers1 -anchor w -text "Revision A"
  label $logcanvas.lwho1 -anchor w -text "Committed by"
  label $logcanvas.ldate1 -anchor w -text "Date"
  label $logcanvas.lcomment1 -anchor w -text "Log"

  entry $logcanvas.tvers1 -relief sunken
  label $logcanvas.twho1 -anchor w -text "--"
  label $logcanvas.tdate1 -anchor w -text "--"
  text  $logcanvas.tcomment1 -height 5 -width 75

  label $logcanvas.lvers2 -anchor w -text "Revision B"
  label $logcanvas.lwho2 -anchor w -text "Committed by"
  label $logcanvas.ldate2 -anchor w -text "Date"
  label $logcanvas.lcomment2 -anchor w -text "Log"

  entry $logcanvas.tvers2 -relief sunken
  label $logcanvas.twho2 -anchor w -text "--"
  label $logcanvas.tdate2 -anchor w -text "--"
  text  $logcanvas.tcomment2 -height 5 -width 75

  pack $logcanvas.lfname \
    -in $logcanvas.up.left \
    -side top -fill x -pady 2
  pack $logcanvas.tfname \
    -in $logcanvas.up.right \
    -side top -fill x -pady 0
  pack $logcanvas.lvers1 $logcanvas.lwho1 \
      $logcanvas.ldate1 $logcanvas.lcomment1 \
    -in $logcanvas.up1.left \
    -side top -fill x -pady 0
  pack $logcanvas.tvers1 $logcanvas.twho1 \
      $logcanvas.tdate1 $logcanvas.tcomment1 \
    -in $logcanvas.up1.right \
    -side top -fill x -pady 0
  pack $logcanvas.lvers2 $logcanvas.lwho2 \
      $logcanvas.ldate2 $logcanvas.lcomment2 \
    -in $logcanvas.up2.left \
    -side top -fill x -pady 0
  pack $logcanvas.tvers2 $logcanvas.twho2 \
      $logcanvas.tdate2 $logcanvas.tcomment2 \
    -in $logcanvas.up2.right \
    -side top -fill x -pady 0

  canvas $logcanvas.canvas -relief sunken -border 2 \
    -yscrollcommand "$logcanvas.yscroll set" \
    -xscrollcommand "$logcanvas.xscroll set"
  scrollbar $logcanvas.xscroll -relief sunken -orient horizontal \
    -command "$logcanvas.canvas xview"
  scrollbar $logcanvas.yscroll -relief sunken \
    -command "$logcanvas.canvas yview"

  #
  # Create buttons
  #
  button $logcanvas.help -text "Help" \
    -command log_browser
  button $logcanvas.view -text "View" \
    -command "logcanvas_view $logcanvas"
  button $logcanvas.diff -text "Diff" \
    -command "logcanvas_diff $logcanvas"
  pack $logcanvas.help $logcanvas.view $logcanvas.diff \
    -in $logcanvas.down -side left \
    -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1
  if {$localfile != "no file"} {
    button $logcanvas.join -text "Merge Branch to Head" \
      -command "logcanvas_join $localfile $logcanvas"
    button $logcanvas.delta -text "Merge Changes to Head" \
      -command "logcanvas_delta $localfile $logcanvas"
    pack $logcanvas.join $logcanvas.delta \
      -in $logcanvas.down -side left \
      -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1
  }
  button $logcanvas.viewtags -text "View Tags" \
    -command "nop"
  button $logcanvas.quit -text "Quit" \
    -command "destroy $logcanvas"
  pack $logcanvas.viewtags $logcanvas.quit \
    -in $logcanvas.down -side left \
    -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1

  #
  # Put the canvas on to the display.
  #
  pack $logcanvas.xscroll -side bottom -fill x -padx 2 -pady 2
  pack $logcanvas.yscroll -side right -fill y -padx 2 -pady 2
  pack $logcanvas.canvas -fill both -expand 1

  $logcanvas.canvas delete all

  #
  # Window manager stuff.
  #
  wm title $logcanvas "CVS Log Browser"
  wm minsize $logcanvas 1 1

  # Collect the history from the RCS log
  parse_cvslog $logcanvas $filelog

  # Sort it into order - this makes drawing the tree much easier
  set revlist [lsort -command sortrevs [array names revdate]]
  #puts $revlist

  # Now draw the revision tree
  set n 0
  foreach rev $revlist {
    set revprev [lindex $revlist [expr $n-1]]
    logcanvas_draw_box $logcanvas $rev $revprev
    incr n
  }
  # Find stray tags that aren't attached to revisions
  stray_tags $logcanvas

  $logcanvas.canvas yview moveto 0
  return $logcanvas
}

proc stray_tags { w } {
  global revdate
  global revlist
  global tags
  global cvscanv
  global const

  set sortags [lsort -command sortrevs [array names tags]]
  foreach t $sortags {
    set foundowner 0
    if {! [info exists revdate($t)]} {
      # If its revision doesn't exist, it may be a sticky tag
      #puts "$t    $tags($t) has no revision"
      set pospart [split $t "."]
      set poslen [llength $pospart]
      set nextolast [expr [llength $pospart] - 2]
      set num_nextolast [lindex $pospart $nextolast]
      if {$num_nextolast != 0} {
         # Forget it, it's just an orphaned tag
         #puts " not interested in $t because $num_nextolast is not 0"
         continue
      }
      set revb [expr [lindex $pospart [expr [llength $pospart] - 1]]]
      set pospart [lreplace $pospart $nextolast $nextolast $revb]
      set pospart [lreplace $pospart end end 1]
      set pospart [join $pospart "."]
      #puts " mangled brachrev $pospart"
      if {! [info exists revdate($pospart)]} {
        # A branch for this sticky doesn't seem to exist
        set pospart [split $t "."]
        set trunk [join [lrange $pospart 0 [expr $nextolast - 1]] "."]
        set stub "$trunk.$revb"
        #puts " => $tags($t) looks like a branch bud coming off $trunk"
        if { [info exists cvscanv(posx$trunk)] && \
            [info exists cvscanv(posx$trunk)] } {
          set xbegin [expr $cvscanv(posx$trunk) + $cvscanv(boxx)]
          set ybegin [expr $cvscanv(posy$trunk) + $cvscanv(boxy)]
        set xend [expr $xbegin + 45] 
        set yend [expr $ybegin + 0]
          if [info exists lasttrunk] {
            if {$trunk == $lasttrunk } {
              if [info exists score] {
                incr score
              } else {
                set score 1
              }
              incr yend [expr $const(textheight) * $score]
            } else {
              set score 0
            }
          }
          $w.canvas create line \
            $xbegin $ybegin $xend $yend \
            -fill red
          $w.canvas create oval \
            $xend [expr $yend - 3] \
            [expr $xend + 6] [expr $yend + 3] \
            -outline red
          $w.canvas create text \
            [expr $xend + 10] $yend \
            -text "$tags($t)" \
            -font -*-Helvetica-Bold-R-Normal-*-11-* \
            -anchor w -fill red
          set lasttrunk $trunk
        }
      }
    }
  }
}

proc sortrevs {a b} {
  #
  # Proc for lsort -command, to sort revision numbers
  # Return -1 if a<b, 0 if a=b, and 1 if a>b

  set alist [split $a "."]
  set blist [split $b "."]
  set alength [llength $alist]
  set blength [llength $blist]
  set shorter [expr {($alength < $blength) ? $alength : $blength}]

  # As soon as a field is greater than its counterpart, return
  for {set i 0} {$i <= $shorter} {incr i} {
    set A($i) [lrange $alist $i $i]
    set B($i) [lrange $blist $i $i]
      if {$A($i) < $B($i)} { return -1}
      if {$A($i) > $B($i)} { return 1}
  }
  # Tie breaker needed
  if {$alength < $blength} {
    return -1
  } elseif {$alength > $blength} {
    return 1
  } else {
    return 0
  }
}

proc logcanvas_draw_box {logcanvas rev revprev} {
  #
  # Draws a box containing a revision of a file.
  #
  global cvscanv
  global const
  global tags
  global revwho
  global revbranches
  global revlist

  set parts [split $rev "."]
  set depth [llength $parts]
  set branchn [expr $depth - 2]
  set prevparts [split $revprev "."]
  set prevdepth [llength $prevparts]

  # Default - if theres no previous revision, put it at the beginning index
  set parent ""
  set y $cvscanv(indy)
  set x $cvscanv(indx)

  set leaf [lindex $parts [expr $depth - 1]]
  set branch [join [lrange $parts 0 $branchn] "."]
  set prevbranch [join [lrange $prevparts 0 [expr $prevdepth - 2]] "."]
  set branchroot [join [lrange $parts 0 [expr $depth - 3]] "."]
  set branchnum [lrange $parts $branchn $branchn]
  set prevroot [join [lrange $prevparts 0 [expr $prevdepth - 3]] "."]
  #puts "$rev\tleaf $leaf\tbranch $branch\tbranchnum $branchnum\troot $branchroot"

  # Else, find the parent and place this one after it
  if [info exists cvscanv(posy$revprev)] {
    if {$depth <= $prevdepth} {
      if {[string compare $branch $prevbranch] != 0} {
        # We're on a different branch now, so the parent isn't revprev.
        for {set int [expr $leaf - 1]} {$int >= 0} {incr int -1} {
          foreach r $revlist {
            #puts "comparing $r with $branch.$int"
            if {[string compare $r "$branch.$int"] == 0} {
              #puts "Found parent $r on same branch"
              set parent $r  
              set x $cvscanv(posx$parent)
              set y [expr $cvscanv(posy$parent) - $cvscanv(gapy)]
              break
            }
          }
          if {$parent != ""} {
            break
          }
        }
        if {$parent == ""} {
          # Check our list of branches and see if this matches one of them.
          foreach br [array names revbranches] {
            foreach b $revbranches($br) {
              if {[string compare $b $branch] == 0} {
                set parent $br
                #puts "  $b matched $branch, so parent is $br"
                set x [expr $cvscanv(posx$parent) + $cvscanv(gapx)]
                set y [expr $cvscanv(posy$parent) - $cvscanv(gapy)]
                set offset [check_xspace $x $y $rev]
                #puts "check_xspace returned $offset"
                set x [expr $x + ($offset * $cvscanv(gapx))]
              }
            }
          }
          # We're on an already-existing branch although it's not the
          # same as the immediately preceding revision.  Look for
          # a parent at the same branching level.
          if {$parent == ""} {
            set shortlist ""
            foreach r $revlist {
              set l [llength [split $r "."]]
              if {$l == $depth} {
                lappend shortlist $r
              }
            }
            #puts "trying to find a parent for $rev from $shortlist"
            foreach item $shortlist {
              if {[sortrevs $item $rev] < 0} {
                set parent $item
                set x $cvscanv(posx$parent)
                set y [expr $cvscanv(posy$parent) - $cvscanv(gapy)]
                #puts "  picked parent $parent from list"
                continue
              }
            }
          }
        }
        if {$parent == ""} {
          #puts "Didn't find a parent for $rev"
        }
      } else {
        # branch is the same as previous, so parent is $revprev
        set parent $revprev
        set x $cvscanv(posx$parent)
        set y [expr $cvscanv(posy$parent) - $cvscanv(gapy)]
        #puts "  parent $parent is the previous rev"
      }
    } else {
      # $depth > $prevdepth, meaning we have a new branch
      # Find parent by simply truncating rev number
      set parent [join [lrange $parts 0 [expr $depth - 3]] "."]
      #puts "  new branch, found parent $parent by truncation"
      # If it's on the main trunk, it doesn't need any special processing
      if {$depth == 2} {
        set x $cvscanv(posx$parent)
        set y [expr $cvscanv(posy$parent) - $cvscanv(gapy)]
      } else {
        # Else, maybe theres a branch already occupying this position
        # so we have to offset it
        set x [expr $cvscanv(posx$parent) + $cvscanv(gapx)]
        set y [expr $cvscanv(posy$parent) - $cvscanv(gapy)]
        set offset [check_xspace $x $y $rev]
        #puts "check_xspace returned $offset"
        set x [expr $x + ($offset * $cvscanv(gapx))]
      }
    }
  }
  #if {$parent != ""} {
    #puts " parent $parent"
  #}

  # If the parent has more than two tags, it needs more vertical space
  set extratags 0
  if [info exists tags($parent)] {
    set ntags [llength $tags($parent)]
    if {$ntags > 2} {
      set extratags [expr $ntags - 2]
    }
  }
  set y [expr $y - (12 * $extratags)]

  # draw the box and remember its position
  logcanvas_rectangle $logcanvas $rev $x $y $parent
  set cvscanv(posx$rev) $x
  set cvscanv(posy$rev) $y
}

proc check_xspace {x y rev} {
  #
  # See if we have to offset a branch to the right.
  # Return number of columns to offset by.
  #
  global cvscanv
  
  #puts "In check_xspace \{$x $y $rev\}"
  set offsetcolumns 0
  set stack ""
  # Collect all the x position data
  foreach xb [array names cvscanv] {
    if [string match "posx*" $xb] {
      lappend stack $xb
    }
  }
  
  # sometimes the offset algorithm is sensitive to the order that 
  # the versions are placed in the stack. Remove this sensitivity
  # with a sort - default ascii increasing is fine. Leif E.

  set stack [lsort $stack]

  set ymin 0
  set ymax -1000000
  # Find the things with the same x position and check their y positions
  foreach xb $stack {
    set d [string range $xb 4 end]
    if {$x == $cvscanv(posx$d)} {
      # We found something else with the same x position
      #puts "  ymin $ymin\tymax $ymax"
      #puts "Match on X positions: x $x\ty $y\tposx($d): $cvscanv(posx$d)\tposy($d): $cvscanv(posy$d)"

      if {$cvscanv(posy$d) < $ymin} {
        set oldymin $ymin
        set ymin $cvscanv(posy$d)
      }
      if {$cvscanv(posy$d) > $ymax} {
        set oldymax $ymax
        set ymax $cvscanv(posy$d)
      }
      # Check if the potential y pos will overwrite an existing box          
      if {$y >= $ymin && $y <= $ymax} {
        incr offsetcolumns
        set x [expr $x + $cvscanv(gapx)]
        if {[check_xspace $x $y $rev] == 0} {
	  continue
        } else {
          incr offsetcolumns
          set x [expr $x + $cvscanv(gapx)]
        }
      }
    }
  }
  return $offsetcolumns
}

proc logcanvas_rectangle {logcanvas rev x y revprev} {
  #
  # Breaks out some of the code from the logcanvas_draw_box procedure.
  #
  global cvscanv
  global cvscfg
  global const
  global tags
  global revwho
  global revdate
  global revcomment

  set parts [split $rev "."]
  set prevparts [split $revprev "."]

  $logcanvas.canvas create rectangle \
    $x $y \
    [expr $x + $cvscanv(boxx)] [expr $y + $cvscanv(boxy)] \
    -width 3 \
    -tags v$rev

  # draw connecting line
  if [info exists cvscanv(posx$revprev)] {
    if {[llength $parts] > [llength $prevparts]} {
      set xbegin [expr $cvscanv(posx$revprev) + $cvscanv(boxx)]
      set ybegin [expr $cvscanv(posy$revprev) + ($cvscanv(boxy)/2)]
    } else {
      set xbegin [expr $cvscanv(posx$revprev) + ($cvscanv(boxx)/2)]
      set ybegin $cvscanv(posy$revprev)
    }
    set xend [expr $x + ($cvscanv(boxx)/2)]
    set yend [expr $y + $cvscanv(boxy)]
    $logcanvas.canvas create line $xbegin $ybegin $xend $yend
  }

  # Make sure the scrolling region is big enough
  if {$cvscanv(xlow) > $x} {
    set cvscanv(xlow) $x
  }
  if {$cvscanv(ylow) > $y} {
    # sckick  Added code to compensate for more than two tags
    #  sckick Start of change
    #    set cvscanv(ylow) $y
    set extratags 0
    if [info exists tags($rev)] {
      set ntags [llength $tags($rev)]
      if {$ntags > 2} {
        set extratags [expr $ntags - 2]
      }
    }
    set cvscanv(ylow) [expr $y - (12 * $extratags)]
  }
  # sckick End of  change

  if {$cvscanv(xhigh) < [expr $x + $cvscanv(boxx) + $cvscanv(gapx)]} {
    set cvscanv(xhigh) [expr $x + $cvscanv(boxx)  + $cvscanv(gapx)+ 10]
  }
  if {$cvscanv(yhigh) < [expr $y + $cvscanv(boxy)]} {
    set cvscanv(yhigh) [expr $y + $cvscanv(boxy) + 10]
  }
  $logcanvas.canvas configure \
    -scrollregion "$cvscanv(xlow) $cvscanv(ylow) \
    $cvscanv(xhigh) $cvscanv(yhigh)"

  # Put the version number in the box
  $logcanvas.canvas create text \
    [expr $x + 4] [expr $y + 2] \
    -anchor nw -text $rev  \
    -tags v$rev
  $logcanvas.canvas create text \
    [expr $x + 4] [expr $y + 14] \
    -anchor nw -text $revwho($rev) \
    -tags v$rev

  if [info exists tags($rev)] {
    set n 0
    foreach item $tags($rev) {
      if [info exists cvscfg(tagcolour,$item)] {
        set tagcolour $cvscfg(tagcolour,$item)
      } else {
        set tagcolour black
      }
      $logcanvas.canvas create text \
        [expr $x + $cvscanv(boxx) + 2] [expr $y + $cvscanv(boxy) - $n] \
        -anchor sw -text $item \
        -fill $tagcolour \
        -tags v$rev
      incr n $const(textheight)
    }
  }
  #puts "About to add sticky tags for branches, point is $rev"
  # Mangle the rev number to become the branch number
  set pospart [split $rev "."]
  # do this only for the first revision on a branch
  if {[expr  [lindex $pospart end]] == 1} {
    set poslen [llength $pospart]
    set revb [expr [lindex $pospart [expr [llength $pospart] - 2]]]
    set pospart [lreplace $pospart end end $revb]
    set pospart [lreplace $pospart [expr $poslen - 2] [expr $poslen - 2]  0]
    # revision list is now the branch number, restore to a string
    set pospart [join $pospart "."]
    # arbitrary constant chosen to push sticky branch label under the box
    # there is almost certainly a better way (an array var somewhere ??)
    set n 15
    foreach item [array names tags] {
      if [string match $pospart $item] {
        set parts  [split $item "."] 
        # Rip last number off revision string
        set stub [join [lreplace [split $rev "."] end end] "."]
        # For each sticky tag for this branch, write it on the canvas
        foreach t $tags($item) {
          $logcanvas.canvas create text \
              $x \
              [expr $y + [expr $n + $cvscanv(boxy)]] \
              -anchor sw -text "($stub) $t" \
              -font -*-Helvetica-Bold-R-Normal-*-12-* \
              -fill blue \
              -tags v$rev
          incr n $const(textheight)
        }
      }
    }
  }
  
  # Bind to the tag.
  $logcanvas.canvas bind v$rev <ButtonPress-1> \
    "$logcanvas.tvers1 delete 0 end
    $logcanvas.tvers1 insert end $rev
    $logcanvas.twho1 configure -text $revwho($rev)
    $logcanvas.tdate1 configure -text $revdate($rev)
    $logcanvas.tcomment1 delete 1.0 end
    $logcanvas.tcomment1 insert end {$revcomment($rev)}"
  $logcanvas.canvas bind v$rev <ButtonPress-3> \
    "$logcanvas.tvers2 delete 0 end
    $logcanvas.tvers2 insert end $rev
    $logcanvas.twho2 configure -text $revwho($rev)
    $logcanvas.tdate2 configure -text $revdate($rev)
    $logcanvas.tcomment2 delete 1.0 end
    $logcanvas.tcomment2 insert end {$revcomment($rev)}"
}

proc logcanvas_view {logcanvas} {
  #
  # Views the selected version.
  #
  set ver1 [$logcanvas.tvers1 get]
  set localfile [$logcanvas.tlocalfile get]

  if {$localfile != "no file"} {
    #puts "cvs_view_r $ver1 $localfile"
    cvs_view_r $ver1 $localfile
  } else {
    set fname [$logcanvas.tfname get]
    #puts "rcs_fileview $fname $ver1"
    rcs_fileview $fname $ver1
  }
}

proc logcanvas_diff {logcanvas} {
  #
  # Diffs two versions.
  #
  set ver1 [$logcanvas.tvers1 get]
  set ver2 [$logcanvas.tvers2 get]
  set localfile [$logcanvas.tlocalfile get]
  #puts "In [pwd]"
  #puts "Trying to diff $localfile"
  if {$localfile != "no file"} {
    #puts "cvs_diff_r $ver1 $ver2 $localfile"
    cvs_diff_r $ver1 $ver2 $localfile
  } else {
    set fname [$logcanvas.tfname get]
    #puts "rcs_filediff $fname $ver1 $ver2"
    rcs_filediff $fname $ver1 $ver2
  }
}

proc logcanvas_join {localfile logcanvas} {
  #
  # Joins a branch version to the head version.
  #
  set ver1 [$logcanvas.tvers1 get]
  set versions [split $ver1 "."]
  set depth [llength $versions]
  if {$depth < 4} {
    cvsfail "Please select a branch version for this function!"
    return 1
  }

  cvs_join $localfile $ver1
}

proc logcanvas_delta {localfile logcanvas} {
  #
  # Merges changes in the delta between two versions to the head
  # version.
  #
  set ver1 [$logcanvas.tvers1 get]
  set ver2 [$logcanvas.tvers2 get]

  cvs_delta $localfile $ver1 $ver2
}

proc parse_cvslog {logcanvas filelog} {
  #
  # Splits the rcs file up and parses it using a simple state machine.
  #
  global revdate
  global revwho
  global revcomment
  global tags
  global revbranches

  set loglist [split $filelog "\n"]
  set logstate "rcsfile"
  foreach logline $loglist {
    switch -exact -- $logstate {
      "rcsfile" {
        # Look for the first text line which should give the file name.
        set fileline [split $logline]
        if {[lindex $fileline 0] == "RCS"} {
          $logcanvas.tfname delete 0 end
          $logcanvas.tfname insert end [lindex $fileline 2]
          set logstate "tags"
          set taglist ""
          continue
        }
      }
      "tags" {
        # Any line with a tab leader is a tag
        if { [string index $logline 0] == "\t" } {
          set taglist "$taglist$logline\n"
          set tagitems [split $logline ":"]
          set tagstring [string trim [lindex $tagitems 0]]
          set tagrevision [string trim [lindex $tagitems 1]]
          lappend tags($tagrevision) $tagstring
        } else {
          if {$logline == "description:"} {
            # No more tags after this point
            $logcanvas.viewtags configure \
              -command "view_output Tags \"$taglist\""
            set logstate "searching"
            continue
          }
          if {$logline == "----------------------------"} {
            # Oops, missed something.
            $logcanvas.viewtags configure \
              -command "view_output Tags \"$taglist\""
            set logstate "revision"
            continue
          }
        }
      }
      "searching" {
        # Look for the line that starts a revision message.
        if {$logline == "----------------------------"} {
          set logstate "revision"
          continue
        }
      }
      "revision" {
        # Look for a revision number line
        set revline [split $logline]
        set revnum [lindex $revline 1]
        set logstate "date"
      }
      "date" {
        # Look for a date line.  This also has the name of the author.
        set dateline [split $logline]
        set revdate($revnum) [lindex $dateline 1]

        set who  [lindex $dateline 5]
        set revwho($revnum) [string range $who 0 \
                     [expr [string length $who] - 2]]

        set revcomment($revnum) ""
        set revbranches($revnum) ""
        set logstate "logmessage"
      }
      "logmessage" {
        # See if there are branches off this revision
        if [string match "branches:*" $logline] {
          set br [split $logline]
          set branches [lrange $logline 1 [llength $logline]]
          foreach br $branches {
            lappend revbranches($revnum) [string trimright $br ";"]
          }
          continue
        }
        # Read the log message which follows the date line.
        if {$logline == "----------------------------"} {
          set logstate "revision"
          continue
        } elseif {$logline == \
"============================================================================="} {
          set logstate "terminated"
          continue
        }
        # Process a revision log line
        regsub -all "\"" $logline "'" newline
        set revcomment($revnum) "$revcomment($revnum)$newline\n"
      }
      "terminated" {
        # ignore any further lines
        continue
      }
    }
  }
}

