/* Implementation of TLLTag class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLLTag.m,v 1.2 1998/02/23 14:17:35 tiggr Exp $  */

#define TLLTAG_DECLARE_PRIVATE_METHODS
#import "tl/support.h"
#import "tl/TLLTag.h"
#import "tl/TLLSubroutine.h"
#import "tl/TLCons.h"
#import "tl/subr.h"
#import "tl/TLGC.h"
#import "tl/predicates.h"

/* Symbols.  */
TLSymbol *Qerror_conditions, *Qerror, *Qno_catch, *Qeof;
TLSymbol *Qsignal;

/* The list of tags and the tag to be garbage protected while it executes
   its unwind forms.  */
static TLLTag *tags, *current_tag;

/* Pointer to previous garbage protection function.  */
static id (*previous_garbage_protect) (void);

/* Prevent recursive panics.  */
static int aborting = 0;

/* Definition of subroutines in this file.  */
static struct tl_subdef subdefs[] =
{
  DEFSUB (nil, @"catch",		Fcatch,			1, ARGS_UNEVAL),
  DEFSUB (nil, @"throw",		Fthrow,			2, 2),
  DEFSUB (nil, @"unwind-protect",	Funwind_protect,	1, ARGS_UNEVAL),
  DEFSUB (nil, @"condition-case",	Fcondition_case,	2, ARGS_UNEVAL),
  DEFSUB (&Qsignal, @"signal",		Fsignal,		2, 2),
  DEFSUB (0, 0, 0, 0, 0)
};

id
tlltag_garbage_protect (void)
{
  MARK (tags);
  MARK (current_tag);
  if (previous_garbage_protect)
    return (previous_garbage_protect ());
  return (nil);
} /* tlltag_garbage_protect */

id
Fcatch (id in_args)
{
  id tag, body, retval;
  GCDECL1;

  GCPRO1 (in_args);

  DECONS (in_args, tag, body);
  retval = [TLLTag protect: body tag: EVAL (tag) unwind: nil];

  GCUNPRO;
  return (retval);
} /* Fcatch */

id
Fcondition_case (id in_args)
  /* (condition-case VAR FORM HANDLERS)  */
{
  id retval, var, form, handlers;
  GCDECL1;

  GCPRO1 (in_args);

  DECONS (in_args, var, form);
  DECONS (form, form, handlers);
  retval = [TLLTag protect: CONS (form, nil) symbol: var handlers: handlers];

  GCUNPRO;
  return (retval);
} /* Fcondition_case */

#if SUPPORT_DEBUG
void
dump_invocation_backtrace (void)
{
  tll_invocation_info *ii;
  int i, j;

  for (i = tll_invocation_num - 1, ii = tll_invocation_stack + i;
       i >= 0; i--, ii--)
    if (ii->receiver)
      switch (ii->argc)
	{
	case -2:
	  formac (V_stderr_, @"%d: [%# %# ...forwarding...]\n",
		  i, ii->receiver, ii->name);
	  break;

	case -1:
	  formac (V_stderr_, @"%d: [%# %# ...computing arguments...]\n",
		  i, ii->receiver, ii->name);
	  break;

	default:
	  formac (V_stderr_, @"%d: [%# %#", i, ii->receiver, ii->name);

	  if (ii->argc)
	    for (j = 0; j < ii->argc; j++)
	      formac (V_stderr_, @" %#", ii->argv[j]);
	  else if (ii->argl)
	    formac (V_stderr_, @" %#", ii->argl);

	  formac (V_stderr_, @"]\n");
	  break;
	}
    else
      switch (ii->argc)
	{
	case -2:
	  formac (V_stderr_, @"%d: (%# ...forwarding...)\n", i, ii->name);
	  break;

	case -1:
	  formac (V_stderr_, @"%d: (%# ...computing arguments...)\n",
		  i, ii->name);
	  break;

	default:
	  formac (V_stderr_, @"%d: (%#", i, ii->name);

	  if (ii->argc)
	    for (j = 0; j < ii->argc; j++)
	      formac (V_stderr_, @" %#", ii->argv[j]);
	  else if (ii->argl)
	    formac (V_stderr_, @" %#", ii->argl);

	  formac (V_stderr_, @")\n");
	  break;
	}
} /* dump_invocation_backtrace */
#endif

id
Fsignal (id symbol, id data)
  /* (signal error-symbol data) */
{
#if SUPPORT_DEBUG
  if (1)
    dump_invocation_backtrace ();
#endif

  if (!tags)
    {
      if (!aborting)
	{
	  aborting = 1;
	  formac (V_stderr_, @"no tags to signal %#", CONS (symbol, data));
	}
      [V_stderr_ flushOutput];
      abort ();
    }
  [tags raise: symbol context: data];
  return (nil);
} /* Fsignal */

id
Fthrow (id tag, id value)
  /* (throw tag value) */
{
  if (!tags)
    {
      if (!aborting)
	{
	  aborting = 1;
	  formac (V_stderr_, @"no tags to throw %#", CONS (tag, value));
	}
      abort ();
    }
  [tags throw: tag value: value];
  return (nil);
} /* Fthrow */

id
Funwind_protect (id in_args)
  /* (unwind-protect body-form unwind-form ...)  */
{
  id form, cleanup, retval;
  GCDECL1;

  GCPRO1 (in_args);

  DECONS (in_args, form, cleanup);
  retval = [TLLTag protect: CONS (form, nil) tag: nil unwind: cleanup];

  GCUNPRO;
  return (retval);
} /* Funwind_protect */

@implementation TLLTag

/******************** private methods ********************/

+initialize
{
  id s = [CO_TLSymbol class];
  id t;

  [TLLSubroutine registerSubroutines: subdefs];

  Qerror_conditions = [s symbolWithName: @"error-conditions"];
  Qerror = [s symbolWithName: @"error"];
  Qeof = [s symbolWithName: @"eof"];
  Qno_catch = [s symbolWithName: @"no-catch"];
  t = CONS (Qerror, nil);
  [Qerror put: Qerror_conditions : t];
  [Qno_catch put: Qerror_conditions : CONS (Qno_catch, t)];
  [Qsignal put: Qerror_conditions : CONS (Qsignal, t)];
  [Qeof put: Qerror_conditions : CONS (Qeof, nil)];

  /* Install the tags garbage protection.  */
  previous_garbage_protect = tl_garbage_protect;
  tl_garbage_protect = tlltag_garbage_protect;
  return (self);
} /* +initialize */

-(void) catch: arg_tag value: arg_value
{
  if (tl_quit_inhibit)
    {
      if (!aborting)
	{
	  aborting = 1;
	  formac (V_stderr_, @"inhibited to catch %#",
		  CONS (arg_tag, arg_value));
	}
      abort ();
    }

  tags = next;
  ASGN_IVAR (next, nil);
  current_tag = self;

  [CO_TLSymbol popVarValues: binding_level];
#if SUPPORT_DEBUG
  tll_invocation_pop (stack_level);
#endif

  if (unwind_forms)
    Fprogn (unwind_forms);
  current_tag = nil;

  if (tag == arg_tag)
    {
      ASGN_IVAR (value, arg_value);
      longjmp (catch, 1);
    }

  if (tags)
    [tags catch: arg_tag value: arg_value];

  if (!aborting)
    {
      aborting = 1;
      formac (V_stderr_, @"no tags to catch %#", CONS (arg_tag, arg_value));
    }
  abort ();
} /* -catch:value: */

-initWithBody: body_form tag: evaluated_tag unwind: cleanup_form handlers: list
{
  GCDECL1;

  GCPRO1 (body_form);

  binding_level = [CO_TLSymbol bindingLevel];
#if SUPPORT_DEBUG
  stack_level = tll_invocation_num;
#endif

  ASGN_IVAR (unwind_forms, cleanup_form);
  ASGN_IVAR (tag, evaluated_tag);
  ASGN_IVAR (handlers, list);

  ASGN_IVAR (next, tags);
  tags = self;

  if (!setjmp (catch))
    {
      value = Fprogn (body_form);
      tags = next;
      next = nil;

      current_tag = self;
      if (unwind_forms)
	Fprogn (unwind_forms);
      current_tag = nil;
    }

  GCUNPRO;
  return (value);
} /* -initWithBody:tag:unwind: */

-(void) raise: sym context: data
{
  GCDECL2;

  if (tl_quit_inhibit)
    {
      if (!aborting)
	{
	  aborting = 1;
	  formac (V_stderr_, @"inhibited to raise %#", CONS (sym, data));
	}
      abort ();
    }

  GCPRO2 (sym, data);

  tags = next;
  next = nil;

  [CO_TLSymbol popVarValues: binding_level];
#if SUPPORT_DEBUG
  tll_invocation_pop (stack_level);
#endif

  current_tag = self;
  if (unwind_forms)
    Fprogn (unwind_forms);

  if (handlers)
    {
      TLCons *conditions = [sym get: Qerror_conditions];
      TLCons *handler, *next_handler;
      id c, next_c, condition;

      for (handler = handlers; handler; handler = next_handler)
	{
	  DECONS (handler, handler, next_handler);
	  DECONS (handler, condition, handler);

	  if (condition == Qt);
	  else if (!CONSP (condition))
	    condition = [conditions memq: condition];
	  else
	    for (c = condition; c; c = next_c)
	      {
		DECONS (c, c, next_c);
		condition = [conditions memq: c];
		if (condition)
		  break;
	      }

	  if (condition)
	    {
	      if (tag != Qnil)
		[tag pushVarValue: CONS (sym, data)];
	      value = Fprogn (handler);
	      [CO_TLSymbol popVarValues: binding_level];
#if SUPPORT_DEBUG
	      tll_invocation_pop (stack_level);
#endif
	    }
	}

      current_tag = nil;
      longjmp (catch, 1);
    }

  current_tag = nil;
  GCUNPRO;

  if (tags)
    [tags raise: sym context: data];

  if (!aborting)
    {
      aborting = 1;
      formac (V_stderr_, @"no tags to catch %#", CONS (sym, data));
    }
  abort ();
} /* -raise:context: */

-(TLLTag *) tagCatching: sym
{
    return ((!handlers && tag == sym) ? self
	    : next ? [next tagCatching: sym] : nil);
} /* -tagCatching: */

-throw: arg_tag value: arg_value
{
  TLLTag *t = [tags tagCatching: arg_tag];

  if (!t)
    [self raise: Qno_catch context: CONS (arg_tag, CONS (arg_value, nil))];
  [self catch: arg_tag value: arg_value];
  return (nil);
} /* throw:value: */
	
/******************** `public' methods ********************/

+protect: body tag: the_tag unwind: cleanup_form
{
  return ([[self gcAlloc] initWithBody: body tag: the_tag
	   unwind: cleanup_form handlers: nil]);
} /* +tag */

+protect: body symbol: var handlers: the_handlers
{
  return ([[self gcAlloc] initWithBody: body
	   tag: var unwind: nil handlers: the_handlers]);
} /* +protect:symbol:handlers: */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (tag);
  MARK (unwind_forms);
  MARK (handlers);
  MARK (value);
  MARK (next);
} /* -gcReference */

@end
