/* Interface to TLString class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLString.h,v 1.4 1998/01/13 11:53:09 tiggr Exp $  */

#import <stdarg.h>
#import "config.h"
#import "TLObject.h"

/* How deerly do we want protocol declarations, instead of only
   definitions!  */
@class TLNumber, TLRegex, TLString, TLMutableString;
@protocol TLString

/* Compare the receiving string with O, returning <0 if it is alphabetically
   preceding O, >0 if it follows O, or 0 if it is equal.  */
-(int) compare: (id <TLString>) o;

/* Return a pointer to the (zero terminated) C style character string held
   by the receiving object.  There is no guarantee the retuened string does
   not _contain_ zero-valued bytes.  */
-(const char *) cString;

/* Get a real string to do some difficult things with, i.e. things not in
   the TLString protocol.  */
-(TLString *) string;

/* Return the receiving object's string length, as either an integer or a
   TLNumber.  This non-underscored length is needed, since we can not have a
   category deliver a replacement method under the GNU runtime.  Actually,
   it is not such a bad idea, since the `objectLength' can be covered by the
   `length' symbol as a forward or an overriding defun.  */
-(int) length;
-(TLNumber *) objectLength;

@end

/* Now include those header files needing the TLString declaration.  */
#import "TLStream.h"
#import "TLVector.h"
#import "TLRange.h"

/* The GNU runtime comes with a clean (i.e. real) NXConstantString.  For
   NeXT's, we must invent it ourselves.  */
#if GNU_RUNTIME
#import <objc/NXConstStr.h>
#endif

#if NEXT_RUNTIME
@interface NXConstantString: Object
{
  char *c_string;
  unsigned int len;
}

-(const char *) cString;
-(int) length;

@end
#endif

/* Option flags for searching.  */
enum
{
  TLSEARCH_NONE,
  TLSEARCH_BACKWARD,
  TLSEARCH_FOLD_CASE = 2,
  TLSEARCH_FULL_MATCH = 4,
  /*  TLSEARCH_REGEX = 8,
  TLSEARCH_REGEX_FAST = 16, */
};

/* Factory methods do not really fit well into protocols.  Therefore,
   declare a separate protocol for checking implementation and another for
   typing arguments.  When invoking, cast to id to avoid the compiler
   warning.

   This just shows another problem with Objective-C's definition.  */
@protocol TLStringCreationUsage

-(id <TLString>) stringWithCString: (const char *) s length: (int) l;

@end

@protocol TLStringCreationDeclaration

+(id <TLString>) stringWithCString: (const char *) s length: (int) l;

@end

/* Make clear that NXConstantString conforms to the basic reading TLString
   methods.  */
@interface NXConstantString (TLNXConstantString) <TLString>
@end

/* General TLString-used byte hashing function.  */
unsigned int tol_hash_string (const char *s, int len);

/* Short-hand string creation.  */
#define TLSTRING(X)  [TLString stringWithCString: X]

/* A read-only string.  */
@interface TLString: TLObject <TLString, TLStringCreationDeclaration, TLVector>
{
  char *c_string;
  int len;
}

/******************** string creation ********************/

+(TLString *) stringWithCString: (const char *) s;
+(TLString *) stringWithCStringNoCopy: (char *) s;
+(TLString *) stringWithCStringNoCopy: (char *) an_s length: (int) l;
+(TLString *) stringWithFormat: (id <TLString>) fmt, ...;
+(TLString *) stringWithFormat: (id <TLString>) fmt: (va_list) ap;
+(TLString *) stringWithFormatVector: (id <TLVector>) vector;
+(TLString *) stringWithString: (id <TLString>) s;
+(TLString *) stringWithStream: (id <TLInputStream>) stream;
+(TLString *) stringWithStream: (id <TLInputStream>) stream
 length: (int) length;

/* Return a string containing S with a length of at least L (but longer if S
   is longer) and padded left (-1), right (1) or both (0: centering).  */
+(TLString *) stringWithString: (id <TLString>) s withLength: (int) l
 byPadding: (int) pad;

/* Return a new string, created from S while eliminating `% hex hex'
   quoting.  */
+(id <TLString>) stringWithHTMLQuotedCString: (const char *) s length: (int) l;

/* Return a new string which is some manipulated version of the receiving
   string.  */
-(id <TLString>) stringWithUppercase;
-(id <TLString>) stringWithLowercase;
-(id <TLString>) stringWithCapitals;

/* Return a substring of this string.  The range is mangled to fit within
   the receiving string.  */
-(TLString *) stringWithRange: (id <TLRange>) r;
-(TLString *) stringWithStart: (int) start length: (int) length;

/******************** miscellaneous ********************/

-(unsigned int) hash;

/* Return a HTML quoted or unquoted version of this string.  */
-(TLMutableString *) htmlQuoted;
-(TLMutableString *) htmlUnquoted;

/******************** retrieving information ********************/

/* Return the character at index IDX if IDX > 0 or the character at index
   (LENGTH + IDX) if IDX < 0.  Return -1 for out-of-bounds.  */
-(int) characterAtIndex: (int) idx;

-(id <TLVector>) componentsSeparatedByString: (id <TLString>) s;
-(id <TLVector>) componentsSeparatedByPattern: (id <TLString>) s;

/* Return the index of the first character match from the string S.  */
-(int) indexOfCharacterMatchFrom: (id <TLString>) s;

/* Return the range of the string S in the receiving string.  Return NIL iff
   S does not occur in the string or the substring indicated by R.  */
-(id <TLRange>) rangeOfString: (id <TLString>) s;

-(id <TLRange>) rangeOfString: (id <TLString>) s
		      options: (unsigned int) opt;

-(id <TLRange>) rangeOfString: (id <TLString>) s
			range: (id <TLRange>) r;

-(id <TLRange>) rangeOfString: (id <TLString>) s
			range: (id <TLRange>) r
		      options: (unsigned int) opt;

/* Shorthands for some the above.  */
-(id <TLRange>) strcmp: (id <TLString>) s;
-(id <TLRange>) strcasecmp: (id <TLString>) s;
-(id <TLRange>) strstr: (id <TLString>) s;
-(id <TLRange>) strcasestr: (id <TLString>) s;
-(id <TLRange>) strrstr: (id <TLString>) s;
-(id <TLRange>) strcaserstr: (id <TLString>) s;

/******************** streams and printing ********************/

/* Like normal `print:quoted:', but iff QP and the extra parameter WQ,
   quotes are printed surrounding the output.  */
-(void) print: (id <TLOutputStream>) stream quoted: (BOOL) qp
 withQuotes: (BOOL) wq;

/* Return a stream on this string.  */
-(id <TLStream>) stream;

/* Attempt to write non-persistently the characters from range R to the
   STREAM .  Returns the number of characters written, or NIL if the write
   failed.  */
-(id <TLNumber>) writeRange: (id <TLRange>) r
		   toStream: (id <TLOutputStream>) stream;

/******************** filenames ********************/

-(id <TLString>) basename;
-(id <TLString>) expandedFilename;
-(id <TLString>) expandedFilenameRelativeTo: (id <TLString>) dir;
-(id) fileExistsP;
-(BOOL) _fileExistsP;

@end
