/* Interface to TLTrie class.
   This file is part of TL, Tiggr's Library.
   Written by Pieter J. Schoenmakers<tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLTrie.h,v 1.1 1998/01/08 16:13:43 tiggr Exp $  */

#import "TLObject.h"
#import "TLEnumerator.h"

@class TLString;

enum tltrie_options
{
  /* Handle keys reversed.  */
  TLTO_REVERSED = 1,

  /* (Valid only for lookups.)  Match the longest prefix, instead of
     requiring an exact match.  */
  TLTO_PREFIX = 2,

  /* Map upper case characters to lower case.  */
  TLTO_FOLD_CASE = 4,
};

@interface TLTrie: TLObject
{
  /* The object for (seen from here) empty prefix.  */
  id object;

  /* Range of the VALUES.  */
  int start, end;

  /* Array of next nodes (indexed from zero, that being the next node for
     the char START), in case START and END indicate a non-empty range
     (START == END).  Otherwise, it is the non-empty TLString suffix to
     match for the OBJECT.  The suffix has _not_ been reversed (for easy
     memcmp of suffixes); consistency is ensured by proper usage by the
     user.  */
  void **values;
}

/******************** creation ********************/

/* Return a new empty trie.  */
+(TLTrie *) trie;

/* Basic retrieve method.  */
-(id) objectForKey: (TLString *) k options: (enum tltrie_options) options;

/* return the object stored for the key K.  */
-(id) objectForKey: (TLString *) k;
-(id) objectForReversedKey: (TLString *) k;

/* Return the object stored for the longest prefix of K.  */
-(id) objectForPrefix: (TLString *) k;
-(id) objectForReversedPrefix: (TLString *) k;

/* Basic store method.  TLTO_PREFIX is ignored.  */
-(id) setObject: (id) o forKey: (TLString *) k
	options: (enum tltrie_options) options;

/* Insert the object O for the key K into the trie and return the previous
   values for that key.  */
-setObject: (id) o forKey: (TLString *) k;
-setObject: (id) o forReversedKey: (TLString *) k;

#ifdef TLTRIE_DECLARE_PRIVATE_METHODS

-(id) objectForChars: (const char *) s limit: (const char *) e
	     options: (enum tltrie_options) options;

/* TLTO_PREFIX is ignored.  */
-(void) pushSuffixWithOptions: (enum tltrie_options) options;

/* TLTO_PREFIX is ignored.  */
-(id) setObject: (id) o forChars: (const char *) s limit: (const char *) e
	options: (enum tltrie_options) options;

#endif

@end
