<copyright> Block -- TOM closures (for Tesla compiler)
    Written by <a href="mailto:tiggr@gerbil.org">Pieter J. Schoenmakers</a>

    Copyright &copy; 1999 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: Block.t,v 1.10 1999/10/05 19:35:17 tiggr Exp $</id>
    </copyright>

implementation class
Block: State, Conditions
{
  // We want institutionalized configuration controls: in a specific file
  // per unit, compile-time settings which the user can change before
  // doing a total recompile.  For instance:
  //	static boolean check_block_selectors;
  // could be changed into:
  //	const check_block_selectors = FALSE;
  // Sun Aug  1 13:06:43 1999, tiggr@gerbil.org
  static boolean check_block_selectors;
}

<doc> Initialize the static control variables (only
    {check_block_selectors} up to now).  </doc>
void
  load Array arguments
{
  check_block_selectors = TRUE;
}

end;

implementation instance
Block
{
  <doc> Pointer to the actual code (a C function).  </doc>
  pointer code;

  <doc> The selector of the {eval} method of this block, which includes the
      formal argument and return types.  </doc>
  selector arguments;

  <doc> If this block employs block variables, the {variables} points to a
      struct holding those variables.  </doc>
  pointer variables;

  <doc> Pointer to the local variables of the enclosing method that are
      referenced from this block.  This is not set when the block does not
      reference its environment; it is cleared when the environment is
      exited.  A block that uses its environment checks upon entry to its
      eval method that the environment is still available.  </doc>
  pointer environment;

  <doc> A description of the variables in {variables}.  </doc>
  pointer block_description;
}

<doc> Designated initializer.  </doc>
id (self)
  initWithCode pointer block_c_function
       trigger selector full_arguments
       context pointer context
     variables (pointer, pointer) (vars, desc)
{
  (code, arguments) = (block_c_function, full_arguments);
  (environment, variables, block_description) = (context, vars, desc);
}

<doc> Return {FALSE} if the arguments in the {formal} and {actual}
    selectors match, or match enough.  Raise a {program-condition} for a
    mismatch (and return {TRUE}).  The precondition dictates that the fast
    check should be done autonomously.  </doc>
protected boolean (mismatch)
  arguments_fail (selector, selector) (formal, actual)
pre
  formal != actual
{
<c>
  if (!trt_selector_args_match (formal->in, actual->in))
    {
      if (formal->in->num == 0 && actual->in->num == 1 &&
	  actual->in->args[0] == TRT_TE_VOID)
	mismatch = 0;
      else
	mismatch = 1;
    }

  if (!trt_selector_args_match (formal->out, actual->out))
    {
      if (formal->out->num == 1 && actual->out->num == 0)
	mismatch = 0;
      else 
	mismatch = 1;
    }
</c>

  if (mismatch)
    [[SelectorCondition for self class program-condition
			message [[MutableByteString new]
				  print ("actual ", actual,
					 " mismatches formal ", formal)]
			selector actual] raise];
}

<doc> Generic {eval} method.  Faster versions, which are specilised on
    their arguments, are below.  </doc>
extern dynamic
  eval dynamic arguments;

<doc> The first of many (similar) type-specific {eval} methods.  </doc>
void
  eval
{
  pointer fn = code;

  // This check is too strict, especially for a void actual return type.
  // Sat Aug 28 18:31:42 1999, tiggr@gerbil.org
  if (check_block_selectors && cmd != arguments)
    if ([self arguments_fail (arguments, cmd)])
      return;

  <c>
     ((void (*) (void *, void *)) fn) (self, cmd);
  </c>
}

int (result)
  eval int a1
{
  pointer fn = code;

  if (check_block_selectors && cmd != arguments)
    if ([self arguments_fail (arguments, cmd)])
      return;

  <c>
     result = ((int (*) (void *, void *, int)) fn) (self, cmd, a1);
  </c>
}

void
  eval int a1
{
  pointer fn = code;

  if (check_block_selectors && cmd != arguments)
    if ([self arguments_fail (arguments, cmd)])
      return;

  <c>
     ((void (*) (void *, void *, int)) fn) (self, cmd, a1);
  </c>
}

Any (result)
  eval All a1
{
  pointer fn = code;

  if (check_block_selectors && cmd != arguments)
    if ([self arguments_fail (arguments, cmd)])
      return;

  <c>
     result = ((void * (*) (void *, void *, void *)) fn) (self, cmd, a1);
  </c>
}

<doc><h3>Memory management</h3></doc>

<doc> Release the memory used by this block.  </doc>
void
  dealloc
{
  pointer p = variables;

  <c>
     if (p)
       xfree (p);
  </c>

  [super dealloc];
}

<doc> Mark the block variables if needed.  </doc>
extern void
  gc_mark_elements;

<doc> Be informed that the block is going out of scope, invalidating the
    {environment}.  </doc>
void
  invalidate
{
  environment = ({pointer p;});
}

end;
