<copyright> Everything about coding.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1996, 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: coding.t,v 1.25 1998/07/08 13:22:19 tiggr Exp $</id>
    </copyright>

/******************** State (Coding) ********************/

<doc> This extension of {State} defines the functionality for encoding and
    decoding objects.  To be able to encode an object, it must at least
    properly implement {encodeUsingCoder}.  Similarly, to be decodable, it
    must implement {initWithCoder}.

    The `unit of archiving' is a class, not an extension.  This means that
    if an extension adds state information which needs to be archived (or
    encoded onto a {too.PortCoder}), the extension must re-implement the
    coding methods. </doc>
implementation class
State extension Coding

<doc> Return the current version of the class {cls}.  This is the version
    that will be written when coding instances of this class or a subclass
    thereof.  The default version is 0.

    A version should only be returned if {self} is identical to the class
    containing the method definition, i.e. the method is not inherited.
    Otherwise, the two are unequal, and the version of a subclass is
    requested that does not implement this method, and hence should return
    version 0.  </doc>
int
  version
{
}

<doc> Return {YES}.  </doc>
boolean
  persistent-coding-p
{
  = YES;
}

end;

implementation instance
State extension Coding

<doc><h4>Encoding methods</h4></doc>

<doc> Return the class to be put in the coded stream as the class of this
    object.  The default implementation simply returns {isa}, which is the
    receiving object's class.  </doc>
class (State)
  classForCoder Encoder coder
{
  = isa;
}

<doc> Encode the receiving object to the target {coder}.  Every object
    should first invoke this method of all its direct superclasses before
    encoding its instance variables, but only if {hasBeenCodedFor} for the
    class implementing the method returns {FALSE}.  </doc>
void
  encodeUsingCoder Encoder coder
{
  if (![coder hasBeenCodedFor [State self]])
    [coder startEncoding self];
}

<doc> Return {NO}.  </doc>
boolean
  persistent-coding-p
{
}

<doc> Return the object to be encoded on the {StreamEncoder} {coder}
    (i.e. archived or wired) instead of the receiving object.  The default
    implementation simply returns {self}.  </doc>
Any (self)
  replacementForStreamCoder StreamEncoder coder
{
}

<doc><h4>Decoding methods</h4></doc>

<doc> Initialize the receiving object from the {coder}.  After verifying
    that this method implementation has not yet been invoked (using
    {hasBeenCodedFor}), this method should invoke the implementation of
    this method by the superclasses, followed the fields that were encoded
    by this class.  Decoding must be done in the same order as encoding.

    Note that this method returns {void}.  An object can change the actual
    object returned from decoding by implementing {awakeAfterUsingCoder}.
    </doc>
void
  initWithCoder Decoder coder
{
}

<doc> Return the object to be the object retrieved from decoding instead
    of the receiving object.  The default implementation returns {self}.

    Objects can use this method to return their administered counterpart,
    like {UniqueString} objects do.

    Note that if an object is referenced during its decoding (i.e. object
    A is referenced by an object B which is decoded because B is
    (indirectly) referenced by A), it must not return a different object
    from {awakeAfterUsingCoder}.  If it does, a {coding-condition} is
    raised.  </doc>
id (self)
  awakeAfterUsingCoder Decoder coder
{
}

end;

/******************** Coder ********************/

implementation class
Coder: State, Conditions

<doc> The version of the coding scheme used.  The current version is 0.
    </doc>
int
  version
{
  = 0;
}

end;

implementation instance
Coder

end;

/******************** BinaryCoder ********************/

<doc> The {BinaryCoder} classes {BinaryEncoder} and {BinaryDecoder} can
    archive dearchive a graph of objects in a binary form onto/from a
    stream.  The format is rather simple: Every item stored is preceded by
    a tag byte indicating what the next item is.  There are a few
    secondary tags to introduce classes, etc.

    Every instance or class written is internally numbered in the order
    the objects are written.  References to these objects are encoded in
    the number of bytes necessary for the number of currently known
    objects.  The secondary tags {2} and {4} switch to 2 and 4 byte
    reference encoding, respectively.

    The {nil} object is denoted by the {0} tag.

    Selectors are encoded as a tag {S}, followed by the assigned selector
    number (which is an {int}, starting at 1) and the corresponding
    {Selector} object.  Selectors already encoded are denoted by a tag {s}
    and the {int} selector number.  The invalid selector (the default
    value of {selector} typed variables, also available as {[Runtime
    nullSelector]}) is identified by the tag {s} followed by 0 as the
    selector number. </doc>
implementation class
BinaryCoder: Coder

end;

implementation instance
BinaryCoder
{
  <doc> The number of bytes issued for a reference.  This starts with 1 (a
      byte), and can become 2 (a char) or 4 (an int).  </doc>
  int reference_size;
}

id
  init
{
  reference_size = 1;
  = [super init];
}

end;

/******************** Encoder ********************/

implementation class
Encoder: Coder

end;

implementation instance
Encoder
{
  <doc> Keyed on the objects already encoded, the value is the identifier
      (which is an {IntNumber}) used for this object.  This dict only
      contains temporary objects, i.e. objects that can be forgotten about
      after each {encodeRoot}.  </doc>
  MutableEqDictionary tmp_objects_done;

  <doc> Similar, the non-temporary objects.  This includes class objects
      and {Selector} objects.  </doc>
  MutableEqDictionary perm_objects_done;

  <doc> The set of conditional objects that were skipped.  </doc>
  MutableEqSet objects_skipped;

  <doc> The classes which, for the current object, have already done their
      part in the coding.  </doc>
  MutableEqSet coded_classes;

  <doc> The most recently issued object identifier.  </doc>
  int last_object_id;
}

<doc> The main entry {Encoder} method: encode the {object} and the whole
    object graph implied by it.  This method is not reentrant.  </doc>
void
  encodeRoot State object
{
  // Why can't the functionality of these enclosing methods be implemented
  // by overriding the encodeRoot method?
  // Wed Jul  2 21:12:03 1997, tiggr@tricky.es.ele.tue.nl
  [self startEncodingRoot object];
  tmp_objects_done = [MutableEqDictionary new];
  [self encode object];
  tmp_objects_done = nil;
  [self finishEncodingRoot object];
}

<doc> Designated initializer.  </doc>
id
  init
{
  perm_objects_done = [MutableEqDictionary new];
  = [super init];
}

<doc> Return {NO} if the object currently being encoded on this coder has
    not yet been encoded for {the_class}.  Return {YES} otherwise.  While
    coding an object, only the first invocation for a certain {the_class}
    will return {YES}; subsequent invocations will return {NO}.  </doc>
boolean
  hasBeenCodedFor class (State) the_class
{
  if (!coded_classes[the_class])
    {
      [coded_classes add the_class];

      All v = perm_objects_done[the_class];
      if (!v)
	perm_objects_done[the_class]
	  = [IntNumber with [self identityForClass the_class]];
    }
  else
    = YES;
}

<doc><h4>Encoding methods</h4></doc>

<doc> Encode the {object}, unconditionally.  </doc>
void
  encode State object
{
  State subject = !object ? nil : [self replacementObjectFor object];

  if (!subject)
    [self encodeNilObject];
  else
    {
      Number v = perm_objects_done[subject];

      if (!v)
	v = tmp_objects_done[subject];

      if (v != nil)
	[self encodeReference [v intValue]];
      else
	{
	  /* If this object was previously skipped and must be encoded
             now, the output graph will be wrong and we can not continue.  */
	  if (objects_skipped != nil && objects_skipped[subject] != nil)
	    [[SelectorCondition for self class coding-condition
				message "skipped conditional object now encoded"
				selector cmd] raise];

	  MutableEqSet save_coded_classes = coded_classes;

	  coded_classes = [MutableEqSet new];

	  /* The `startEncoding' method is invoked by the implementation
             of `encodeUsingCoder' by the State class.  */
	  [subject encodeUsingCoder self];
	  [self finishEncoding subject];

	  coded_classes = save_coded_classes;
	}
    }
}

<doc> Encode the {object}, but only if it already is part of the output
    graph.  If this is not the case, {nil} is encoded, and if later on in
    the coding process the object previously encoded as nil is encountered
    (unconditionally), a {program-condition} will be raised to flag the
    inconsistency.  </doc>
void
  encodeConditionally State object
{
  State subject = !object ? nil : [self replacementObjectFor object];

  if (subject == nil || tmp_objects_done[subject] != nil
      || perm_objects_done[subject] != nil)
    [self encode object];
  else
    {
      if (!objects_skipped)
	objects_skipped = [MutableEqSet new];
      [objects_skipped add subject];
      [self encodeNilObject];
    }
}

<doc> Encode the {boolean} {v}.  </doc>
deferred void
  encode boolean v;

<doc> Encode the {byte} {v}.  </doc>
deferred void
  encode byte v;

<doc> Encode the {char} {v}.  </doc>
deferred void
  encode char v;

<doc> Encode the {int} {v}.  </doc>
deferred void
  encode int v;

<doc> Encode the {long} {v}.  </doc>
deferred void
  encode long v;

<doc> Encode the {float} {v}.  </doc>
deferred void
  encode float v;

<doc> Encode the {double} {v}.  </doc>
deferred void
  encode double v;

<doc> Encode the {selector} v.  </doc>
deferred void
  encode selector v;

<doc> Encode the bytes in the range {(start, length)} from the array {r}.
    </doc>
deferred void
  encodeBytes (int, int) (start, length)
	 from ByteArray r
pre
  start >= 0 && length >= -1;

<doc> Encode the {length} bytes of which the first one resides at the
    {address}.  </doc>
deferred void
  encodeBytes (pointer, int) (address, length);

<doc><h4>Internal methods</h4></doc>

<doc> Return the object to be encoded to this coder instead of the
    {object}.  This method is implemented by subclasses to retrieve the
    actual object from the {object} itself, for instance by asking for it
    {replacementForStreamCoder} or {replacementForPortCoder}.  </doc>
deferred protected State
  replacementObjectFor State object;

<doc> Encode the {nil} reference.  </doc>
deferred protected void
  encodeNilObject;

<doc> Encode a reference to the object known as {v}.  </doc>
deferred protected void
  encodeReference int v;

<doc> Return the identity to be used for the non-class {object}.  This
    returns the next value of {last_object_id}.  </doc>
protected int
  identityFor All object
{
  = ++last_object_id;
}

<doc> Return the identity to be used for the class object {a_class}.  This
    returns the next value of {last_object_id}.  </doc>
protected int
  identityForClass class (State) a_class
{
  = ++last_object_id;
}

class (State)
  startEncoding State object
{
  class (State) objclass = [object classForCoder self];

  // The following fragment is duplicated from `hasBeenCodedFor'.  Does it
  // deserve a seperate method?
  // Wed Jul  2 21:34:17 1997, tiggr@tricky.es.ele.tue.nl
  All v = perm_objects_done[objclass];

  if (!v)
    perm_objects_done[objclass]
      = [IntNumber with [self identityForClass objclass]];

  Number ident = [IntNumber with [self identityFor object]];

  if ([object coding-permanent-object-p])
    perm_objects_done[object] = ident;
  else
    tmp_objects_done[object] = ident;

  = objclass;
}

<doc> Invoked when the {object} has been encoded.  Default does nothing.
    </doc>
protected void
  finishEncoding All object
{
}

<doc> Invoked when coding starts with the root {object}.  Default does
    nothing.  </doc>
protected void
  startEncodingRoot All object
{
}

<doc> Invoked when coding the root {object} has finished.  Default does
    nothing.  </doc>
protected void
  finishEncodingRoot All object
{
}

end;

/******************** BinaryEncoder ********************/

implementation class
BinaryEncoder: BinaryCoder, Encoder

end;

implementation instance
BinaryEncoder
{
  <doc> The selector dictionary, from {Selector} to {IntNumber}.  </doc>
  MutableDictionary selectors;
}

<doc> Designated initializer.  </doc>
id
  init
{
  [super (BinaryCoder) init];
  = [super (Encoder) init];
}

class (State)
  startEncoding State object
{
  class (State) objclass = [super startEncoding object];
  IntNumber cv = perm_objects_done[objclass];
  IntNumber ov = tmp_objects_done[object];

  if (!ov)
    ov = perm_objects_done[object];

  [self writeByte '@'];
  [self writeReference [ov intValue]];
  [self writeReference [cv intValue]];
}

<doc> Invoked when the {object} has been encoded.  Emit a close paren.
    </doc>
protected void
  finishEncoding All object
{
  [self writeByte ')'];
}

protected void
  updateReferenceSize
{
  if (last_object_id == 256)
    {
      [self writeByte '2'];
      reference_size = 2;
    }
  else if (last_object_id == 65536)
    {
      [self writeByte '4'];
      reference_size = 4;
    }
}

protected int
  identityFor All object
{
  = [super identityFor object];
  [self updateReferenceSize];
}

<doc> Identify this class on the output {stream}, reporting its coding
    version.  </doc>
protected int
  identityForClass class (State) a_class
{
  String s = [a_class name], us = [[a_class unit] name];
  int i, v = [super identityForClass a_class];
  int n = [s length], un = [us length];

  [self updateReferenceSize];

  [self writeByte '#'];
  [self writeReference v];
  [self writeInt n + un + 1];
  for (i = 0; i < un; i++)
    [self writeByte us[i]];
  [self writeByte '.'];
  for (i = 0; i < n; i++)
    [self writeByte s[i]];

  [self writeInt [a_class version]];

  = v;
}

protected void
  encodeNilObject
{
  [self writeByte '0'];
}

protected void
  encodeReference int v
{
  [self writeByte 'r'];
  [self writeReference v];
}

void
  encode boolean v
{
  [self writeByte 'o'];
  [self writeByte byte (v ? 1 : 0)];
}

void
  encode byte v
{
  [self writeByte 'b'];
  [self writeByte v];
}

void
  encode char v
{
  [self writeByte 'c'];
  [self writeChar v];
}

void
  encode int v
{
  [self writeByte 'i'];
  [self writeInt v];
}

void
  encode long v
{
  [self writeByte 'l'];
  [self writeLong v];
}

void
  encode float v
{
  // How to write out this floating point value?  In a decimal
  // representation?  Or should the start of the stream identify the
  // native representation and should we output both the native
  // representation and the decimal one?  Should we only do so after
  // negotiation (in case of a PortCoder)?
  // Sun Dec 29 00:42:08 1996, tiggr@tricky.es.ele.tue.nl
  MutableByteString buffer = [MutableByteString withCapacity 20];

  [buffer print v];
  [self writeByte 'f'];
  [self encode buffer];
}

void
  encode double v
{
  // Should solve the print buffer stuff between this one, encode <float>,
  // the OutputStream one, etc.
  // Thu May  1 18:08:28 1997, tiggr@akebono.ics.ele.tue.nl
  MutableByteString buffer = [MutableByteString withCapacity 20];

  [buffer print v];
  [self writeByte 'd'];
  [self encode buffer];
}

void
  encode selector v
{
  IntNumber lo;
  Selector s;
  int l;

  if (!!v)
    {
      s = [Selector with v];
      if (!selectors)
	selectors = [MutableDictionary new];
      else
	lo = selectors[s];

      if (!lo)
	{
	  l = 1 + [selectors length];
	  lo = [IntNumber with l];
	  if (s != nil)
	    selectors[s] = lo;
	  [self writeByte 'S'];
	  [self writeInt l];
	  [self encode s];
	  return;
	}

      l = [lo intValue];
    }

  [self writeByte 's'];
  [self writeInt l];
}

void
  encodeBytes (pointer, int) (address, length)
{
  [self writeByte 'B'];
  [self writeInt length];
  [self writeBytes (address, length)];
}

void
  encodeBytes (int, int) (start, length)
	 from ByteArray r
{
  [self writeByte 'B'];

  (start, length) = [r adjustRange (start, length)];

  [self writeInt length];
  [self writeBytes (start, length) from r];
}

// The precondition isn't a real precondition, but more an invariant.
// Sat Dec 28 14:33:38 1996, tiggr@tricky.es.ele.tue.nl
protected void
  writeReference int r
pre
  reference_size == 1 || reference_size == 2 || reference_size == 4
{
  if (reference_size == 4)
    [self writeInt r];
  else if (reference_size == 2)
    [self writeChar char (r)];
  else
    [self writeByte byte (r)];
}

deferred protected void
  writeByte byte b;

deferred protected void
  writeBytes (int, int) (start, length)
	from ByteArray r;

deferred protected void
  writeBytes (pointer, int) (address, length);

protected void
  writeChar char c
{
  [self writeByte byte (c % 256)];
  [self writeByte byte (c / 256)];
}

protected void
  writeInt int i
{
  [self writeChar char (i % 0x10000)];
  [self writeChar char (i / 0x10000)];
}

protected void
  writeLong long l
{
  [self writeInt int (l % 0x100000000)];
  [self writeInt int (l / 0x100000000)];
}

end;

/******************** Decoder ********************/

<doc> The {Decoder} class defines the interface to all decoder classes, be
    it binary or textual, stream or port base.  </doc>
implementation class
Decoder: Coder

end;

implementation instance
Decoder
{
  <doc> Objects, indexed on their identity, as retrieved from this coder.
      </doc>
  MutableIntDictionary tmp_objects_done, perm_objects_done;

  // This should be a bitset, since the naming of the objects is
  // practically closed.
  // Wed Jul  2 21:41:48 1997, tiggr@tricky.es.ele.tue.nl
  <doc> The identity of the objects that have been referenced while being
      decoded.  </doc>
  IntegerRangeSet objects_referenced;

  <doc> Mapping from a class to the decoding version of that class.  </doc>
  MutableEqDictionary class_versions;

  <doc> The classes which, for the current object, have already done their
      part in the coding.  </doc>
  MutableEqSet coded_classes;
}

<doc> Designated initializer.  </doc>
id
  init
{
  perm_objects_done = [MutableIntDictionary new];
  objects_referenced = [IntegerRangeSet new];
  class_versions = [MutableEqDictionary new];

  = [super init];
}

<doc> This is the entry point for the user of this decoder.  The user
    invokes {decodeRoot} to retrieve an object, plus its underlying graph,
    from this decoder.  The object is returned.  </doc>
Any
  decodeRoot
{
  tmp_objects_done = [MutableIntDictionary new];
  = [self decode];
  tmp_objects_done = nil;
}

<doc> Return {NO} if the object currently being decoded on this coder has
    not yet been decoded for {the_class}.  Return {YES} otherwise.  While
    coding an object, only the first invocation for a certain {the_class}
    will return {YES}; subsequent invocations will return {NO}.  </doc>
boolean
  hasBeenCodedFor class (State) the_class
{
  if (!coded_classes[the_class])
    [coded_classes add the_class];
  else
    = YES;
}

<doc> Return the version of the class {cls} as encountered by this coder.
    The version can only be retrieved of classes already encountered
    curing the decoding process.  </doc>
int
  versionOfClass class (State) cls
pre
  class_versions[cls] != nil
{
  = [class_versions[cls] intValue];
}

<doc><h4>Decoding methods</h4></doc>

<doc> Retrieve an object from this decoder and return it.  </doc>
deferred Any
  decode;

deferred boolean
  decode;

deferred byte
  decode;

deferred char
  decode;

deferred int
  decode;

deferred long
  decode;

deferred float
  decode;

deferred double
  decode;

deferred selector
  decode;

<doc> Decode a sequence of bytes from the coder to newly allocated memory
    space.  Return the address and the length.  </doc>
deferred (pointer, int)
  decodeBytes;

<doc> Decode the {num} bytes from the coder to the {address}.  </doc>
deferred void
  decodeBytes int num
	   to pointer address;

<doc><h4>Private methods</h4></doc>

protected Any
  decodeObject class (State) cls
	    as int ref
{
  State o = [cls alloc];

  boolean permanent = [o coding-permanent-object-p];

  if (permanent)
    perm_objects_done[ref] = o;
  else
    tmp_objects_done[ref] = o;

  MutableEqSet saved_coded_classes = coded_classes;
  coded_classes = [MutableEqSet new];
  [o initWithCoder self];
  coded_classes = saved_coded_classes;

  [self finishDecoding o];

  State p = [o awakeAfterUsingCoder self];
  if (p != o)
    if ([objects_referenced member ref])
      [[SelectorCondition for self class coding-condition
			  message "referenced object changed address"
			  selector cmd] raise];
    else if (permanent)
      perm_objects_done[ref] = p;
    else
      tmp_objects_done[ref] = p;

  return Any (p);
}

<doc> Invoked by {decodeObject as}, after having invoked {initWithCoder},
    but before {awakeAfterUsingCoder}.  The default implementation does
    nothing.  </doc>
protected void
  finishDecoding All o
{
}

<doc> Return the object referenced as the number {i}.  </doc>
protected Any
  reference int i
{
  Any o;

  o = tmp_objects_done[i];
  if (!o)
    o = perm_objects_done[i];
  if (!o)
    [[SelectorCondition for self class coding-condition
			message "bad object reference" selector cmd] raise];
  else
    [objects_referenced add i];

  = o;
}

end;

/******************** BinaryDecoder ********************/

<doc> The {BinaryDecoder} is an abstract decoding class which can decode
    binary encoded objects.  It serves as the decoding engine for the
    {BinaryStreamDecoder} and {too.PortDecoder}.  </doc>
implementation class
BinaryDecoder: BinaryCoder, Decoder, C

end;

implementation instance
BinaryDecoder
{
  <doc> The selectors encountered so far, indexed on their identity.
      </doc>
  MutableArray selectors;
}

<doc> Designated initializer.  </doc>
id
  init
{
  selectors = [MutableObjectArray with nil];

  [super (BinaryCoder) init];

  = [super (Decoder) init];
}

<doc> Decode and return an object.  </doc>
Any
  decode
{
  = [self decode [self nextPrimary]];
}

<doc> Decode and return an object, announced by the tag {b}.  </doc>
Any
  decode byte b
{
  if (b == 'r')
    return [self readReference];
  else if (b == '0')
    return nil;
  else if (b != '@')
    [[SelectorCondition for self class coding-condition
      message [[MutableByteString new]
	       print ("expected object but encountered `", b, "'")]
      selector cmd] raise];

  State o, p;
  int oi = [self readReference];
  All ccls = [self readReference];

  = [self decodeObject class (State) (ccls) as oi];
}

boolean
  decode
{
  [self nextPrimary 'o'];
  = [self readByte] == byte (0) ? FALSE : TRUE;
}

byte
  decode
{
  [self nextPrimary 'b'];
  = [self readByte];
}

char
  decode
{
  [self nextPrimary 'c'];
  = [self readChar];
}

int
  decode
{
  [self nextPrimary 'i'];
  = [self readInt];
}

long
  decode
{
  [self nextPrimary 'l'];
  = [self readLong];
}

float
  decode
{
  [self nextPrimary 'f'];
  ByteString s = [self decode];

  = float ([s doubleValue]);
}

double
  decode
{
  [self nextPrimary 'd'];
  ByteString s = [self decode];

  = [s doubleValue];
}

selector (result)
  decode
{
  byte b = [self nextPrimary];
  Selector sel;

  if (b == 's')
    {
      int sid = [self readInt];
      if (sid != 0)
	sel = selectors[sid];
    }
  else if (b != 'S')
    [[SelectorCondition for self class coding-condition
      message [[MutableByteString new]
	       print ("expected selector but encountered `", b, "'")]
      selector cmd] raise];
  else
    {
      int sid = [self readInt];
      sel = [self decode];
      selectors[sid] = sel;
    }

  if (sel != nil)
    result = [sel selector];
}

(pointer, int)
  decodeBytes
{
  [self nextPrimary 'B'];

  int length = [self readInt];

  // The memory should be protected against a leak by a throw.
  // Mon Dec 30 14:01:57 1996, tiggr@tricky.es.ele.tue.nl
  pointer address = malloc (length);

  [self readBytes length to address];
  = (address, length);
}

void
  decodeBytes int length
	   to pointer address
{
  [self nextPrimary 'B'];

  int to_be = [self readInt];

  if (to_be != length)
    [[SelectorCondition for self class coding-condition
      message [[MutableByteString new]
	       print ("expected ", length, " instead of ", to_be)]
      selector cmd] raise];

  [self readBytes length to address];
}

<doc><h4>Internal methods</h4></doc>

<doc> Invoked when the object {o} has been decoded.  Read a close paren.
    </doc>
protected void
  finishDecoding All o
{
  byte b = [self nextPrimary];
  if (b != ')')
    [[Condition for o class coding-condition
		message [[MutableByteString new]
			  print ("more encoded than decoded (next='", b, "')")]]
      raise];
}

<doc> Return the next primary tag byte, handling secondary tags such as
    reference size changes and class declarations.

    If an unknown class is encountered, a {unknown-class-condition} is
    signaled.  A handler may return a replacement class to be used
    instead.  Failure to do so will later on result in a {nil-receiver}
    condition or a failed precondition.  </doc>
protected byte
  nextPrimary
{
  while (TRUE)
    {
      byte b = [self readByte];

      if (b == '2')
	reference_size = 2;
      else if (b == '4')
	reference_size = 4;
      else if (b == '#')
	{
	  /* Read a class definition.  */
	  int r = [self readReference];
	  int i, n = [self readInt];

	  /* Slow and low-level string reading, since we must be able to
             handle the ByteString class as well...  */
	  MutableByteString s = [MutableByteString withCapacity n];
	  for (i = 0; i < n; i++)
	    [s add [self readByte]];

	  /* Retrieve the class and administer its version.  */
	  class (State) cls = [Runtime classNamed s];
	  if (!cls)
	    cls = [[SelectorCondition for s class unknown-class-condition
				      message nil selector cmd] signal];
	  class_versions[cls] = [IntNumber with [self readInt]];

	  /* And note its reference number.  */
	  perm_objects_done[r] = cls;
	}
      else
	return b;
    }
}

<doc> Return the next primary tag byte, which must match {expected}.  If
    it doesn't, a {program-condition} is raised.  </doc>
protected byte
   nextPrimary byte expected
{
  byte b = [self nextPrimary];

  if (b != expected)
    [[SelectorCondition for self class coding-condition
      message [[MutableByteString new]
	       print ("expected `", expected, "' but encountered `", b, "'")]
      selector cmd]
     raise];

  = b;
}

<doc> Read an object reference from this decoder.  Depending on the
    {reference_size} this read 1, 2, or 4 bytes.  </doc>
protected int
  readReference
pre
  reference_size == 1 || reference_size == 2 || reference_size == 4
{
  if (reference_size == 4)
    = [self readInt];
  else if (reference_size == 2)
    = [self readChar];
  else
    = [self readByte];
}

<doc> Read an object reference from this decoder and return the object
    referenced.  This raises a {coding-condition} in case not a proper
    reference is encountered, or if the referenced object is unknown.
    </doc>
protected Any
  readReference
{
  = [self reference [self readReference]];
}

<doc> Return the next single {byte}.  </doc>
deferred protected byte
  readByte;

deferred protected void
  readBytes int num
	 to pointer address;

<doc> Return the next two bytes as a {char}.  </doc>
protected char
  readChar
{
  = [self readByte] + char (256 * [self readByte]);
}

<doc> Return the next four bytes as an {int}.  </doc>
protected int
  readInt
{
  = [self readChar] + 0x10000 * [self readChar];
}

<doc> Return the next 8 bytes as a {long}.  </doc>
protected long
  readLong
{
  = [self readInt] + 0x100000000 * [self readInt];
}

end;
