<copyright> Proxy class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: Proxy.t,v 1.11 1998/01/05 01:16:48 tiggr Exp $</id>
    </copyright>

/******************** State (Proxy) ********************/

<doc> This extension of {State} only provides the {isProxy} method, which
    allows one to discern between proxy and non-proxy objects.  </doc>
implementation class
State extension Proxy

end;

implementation instance
State extension Proxy

boolean
  isProxy
{
  = NO;
}

<doc> Return the object to be encoded by the {coder} instead of the
    receiving object.  This method is repeatedly invoked until an object
    returns {self}.  The default implementation retrieves a proxy from the
    {coder}'s {connection}.  </doc>
State
  replacementForPortCoder PortEncoder coder
{
  = State ([[coder connection] localProxyFor self]);
}

end;

/******************** Proxy ********************/

// Proxy should not inherit from State...
// Mon Jan  6 01:06:50 1997, tiggr@tricky.es.ele.tue.nl
implementation class
Proxy: State

end;

implementation instance
Proxy
{
  <doc> The {Connection} to which we belong.  </doc>
  Connection connection;

  <doc> Our identity with our {connection}.  </doc>
  int identity;
}

id
  initWithConnection Connection c
	    identity int ident
{
  (connection, identity) = (c, ident);
  = self;
}

Connection
  proxy_connection
{
  = connection;
}

int
  proxy_identity
{
  = identity;
}

<doc> Return {self}, since we know how to be sent over the wire.  </doc>
State
  replacementForPortCoder PortEncoder coder
{
  = self;
}

<doc> Have the {coder} encode us as a proxy; otherwise fail (which is the
    case when archiving instead of wiring).  </doc>
void
  encodeUsingCoder Encoder coder
{
  if (![coder encodeProxy self] && ![coder hasBeenCodedFor [Proxy self]])
    [self unimplemented cmd message: "encoding a proxy for archiving purposes"];
}

end;

/******************** LocalProxy ********************/

implementation class
LocalProxy: Proxy

end;

implementation instance
LocalProxy
{
  <doc> The object for which we stand.  </doc>
  public Any original;
}

<doc> Designated initializer.  </doc>
id
  initWithConnection Connection c
	    identity int i
		 for All object
{
  original = Any (object);

  = [super initWithConnection c identity i];
}

end;

/******************** RemoteProxy ********************/

implementation class
RemoteProxy: Proxy

end;

implementation instance
RemoteProxy
{
  <doc> Our {connection} is connected.  </doc>
  redeclare ConnectedConnection connection;
}

boolean
  isProxy
{
  = YES;
}

<doc> The low-level forwarding method.  This method is invoked for
    forwarding a invocation completing method and this is used by the
    {Proxy}.  </doc>
InvocationResult
  forwardSelector selector sel
	arguments pointer args
{
  = [connection forward [Invocation of sel to: self using args]];
}

<doc> Inform our {connection} from our death.  This messages the
    {Connection} class, since messaging objects from {dealloc} methods is
    not allowed.  We identify ourselves by our {identity} since passing
    around a dead object (which we are) is asking for trouble.  </doc>
void
  dealloc
{
  [Connection connection connection remoteProxyDead identity];
}

end;

/******************** NonProxy ********************/

<doc> Instances of (subclasses of) {NonProxy} are never proxies.  They
    always send a copy over the wire.  </doc>
implementation class
NonProxy: State

end;

implementation instance
NonProxy

<doc> Return {self} as we do not want to be proxied.  </doc>
id (self)
  replacementForPortCoder PortEncoder c
{
}

end;

/******************** Number (Proxy) ********************/

implementation class Number extension Proxy: NonProxy end;

implementation instance Number extension Proxy end;

/******************** Invocation (Proxy) ********************/

implementation class Invocation extension Proxy: NonProxy end;

implementation instance Invocation extension Proxy end;

/******************** InvocationResult (Proxy) ********************/

implementation class InvocationResult extension Proxy: NonProxy end;

implementation instance InvocationResult extension Proxy end;

/******************** Selector (Proxy) ********************/

implementation class Selector extension Proxy: NonProxy end;

implementation instance Selector extension Proxy end;

/******************** Collection (Proxy) ********************/

implementation class Collection extension Proxy: NonProxy end;

implementation instance Collection extension Proxy end;

/******************** MutableCollection (Proxy) ********************/

implementation class
MutableCollection extension Proxy: State

end;

implementation instance
MutableCollection extension Proxy

<doc> This is naughty: a {Collection}, through its inheritance of
    {NonProxy} returns {self} when asked its {replacementForPortCoder}.
    However, a {MutableCollection} must be proxied for maintaining the
    right semantics.  Hence, we redirect the method to our direct (though
    repeated) superclass, {State}.  </doc>
id
  replacementForPortCoder PortEncoder c
{
  = [super (State) replacementForPortCoder c];
}

end;
