(**
   A nice analog clock, to be placed everywhere you like.
**)

MODULE VOClock;

(*
    Demo for VisualOberon. A clock.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       E  := VOEvent,
       F  := VOFrame,
       G  := VOGUIObject,
       O  := VOObject,
       T  := VOText,

       C  := Calendar,
       SC := SysClock;

CONST
  callPeriod = 1;

  textSize   = 50;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 frame*      : LONGINT; (* the frame to use for the button *)
               END;


  Clock*     = POINTER TO ClockDesc;
  ClockDesc* = RECORD (G.GadgetDesc)
                 prefs      : Prefs;
                 frame      : F.Frame;
                 timeOut    : D.TimeOut;
                 text       : T.Text;
                 text1,
                 text2      : ARRAY textSize OF CHAR;
                 selected,
                 framed     : BOOLEAN;
               END;

VAR
  prefs* : Prefs;
  sinTab : ARRAY 16 OF LONGINT;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DOut;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(c : Clock);

  BEGIN
    c.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      c.SetBackgroundObject(p.background.Copy());
      c.backgroundObject.source:=c;
    END;
  END SetPrefs;

  PROCEDURE (c : Clock) Init*;

  BEGIN
    c.Init^;

    prefs.SetPrefs(c);

    c.selected:=FALSE;

    c.timeOut:=NIL;

    c.text:=NIL;
    c.text1:="";
    c.text2:="";

    c.frame:=NIL;
    c.framed:=TRUE;
  END Init;

  (**
    Tell if the clock should be framed. Defaults to TRUE.
  **)

  PROCEDURE (c : Clock) ShowFrame*(framed : BOOLEAN);

  BEGIN
    (* We cannot switch back to using no fraame if we already generated one *)
    ASSERT(c.frame=NIL);

    c.framed:=framed;
  END ShowFrame;

  (**
    Set the text to display in normal state and when clicked.
    Use placeholders defined in module Calendar. If you don't set
    a text, no space for text will be reserved. Length is the estimated
    length in characters. Note that VOClocck will use the width of a
    special character for calculation of the string length in pixel, so
    the real size may be different.

    NOTE
    Text must no exceed 50 characters. Currently text must be set
    on creation. But can then be altered on the fly.
  **)

  PROCEDURE (c : Clock) SetText*(text1,text2 : ARRAY OF CHAR);

  BEGIN
    COPY(text1,c.text1);
    COPY(text2,c.text2);
  END SetText;

  PROCEDURE (c : Clock) CalcSize*(display : D.Display);

  BEGIN

    NEW(c.frame);
    c.frame.Init;
    c.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    IF c.framed THEN
      c.frame.SetInternalFrame(c.prefs.frame);
    ELSE
      c.frame.SetInternalFrame(F.none);
    END;
    c.frame.CalcSize(display);

    c.width:=c.frame.leftBorder+4*display.spaceWidth+c.frame.rightBorder;
    c.height:=c.frame.topBorder+4*display.spaceHeight+c.frame.bottomBorder;


    IF (c.text1#"") OR (c.text2#"") THEN
      NEW(c.text);
      c.text.Init;
      c.text.SetFlags({G.horizontalFlex});
      c.CopyBackground(c.text);
      c.text.SetDefault(T.centered,{},D.tinyFont);
      (*c.text.SetText("");*)
      c.text.CalcSize(display);
      INC(c.height,c.text.height+display.spaceHeight);
    END;

    c.minWidth:=c.width;
    c.minHeight:=c.height;

    c.CalcSize^(display);
  END CalcSize;

  (**
    This method gets called when the window gets an event and looks for
    someone that processes it.

    If GetFocus return an object, that objets HandleEvent-method
    get called untill it gives away the focus.
  **)

  PROCEDURE (c : Clock) GetFocus*(event : E.Event):G.Object;

  BEGIN
    (* It makes no sense to get the focus if we are currently not visible *)
    IF ~c.visible OR c.disabled THEN
      RETURN NIL;
    END;

    (*
      When the left mousebutton gets pressed without any qualifier
      in the bounds of our button...
   *)

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & c.PointIsIn(event.x,event.y) & (event.button=E.button1) THEN
        (* We change our state to pressed and redisplay ourself *)
        c.selected:=TRUE;
        c.Redraw;

        (*
          Since we want the focus for waiting for buttonup we return
          a pointer to ourself.
        *)
        RETURN c;
      END;
    ELSE
    END;
    RETURN NIL;
  END GetFocus;

  (**
    Handles all events after GetFocus catched the focus for us.
  **)

  PROCEDURE (c : Clock) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    (*
      If the user releases the left mousebutton...
    *)
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        (* We get unselected again and must redisplay ourself *)
        c.selected:=FALSE;
        c.Redraw;
        RETURN TRUE;
      END;
    ELSE
    END;

    RETURN FALSE; (* No message we are interested in, get more msgs *)
  END HandleEvent;


  PROCEDURE sin(grad : LONGINT):LONGINT;

  BEGIN
    IF (grad>=0) & (grad<=90) THEN
      RETURN sinTab[grad DIV 6];
    ELSIF (grad>=91) & (grad<=179) THEN
      RETURN sin(-grad+180);
    ELSIF (grad>=180) & (grad<=269) THEN
      RETURN -sin(grad-180);
    ELSE (*grad>270*)
      RETURN -sin(360-grad);
    END;
  END sin;

  PROCEDURE cos(grad : LONGINT):LONGINT;

  BEGIN
    RETURN sin((grad+90) MOD 360);
  END cos;

  PROCEDURE (c : Clock) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    zx,zy,r,
    height   : LONGINT;
    text     : ARRAY textSize OF CHAR;
    sTime    : SC.DateTime;

  BEGIN
    c.Draw^(x,y,draw);

    c.frame.Resize(c.width,c.height);
    c.frame.Draw(c.x,c.y,draw);

    c.DrawBackground(c.x+c.frame.leftBorder,c.y+c.frame.topBorder,
                     c.width-c.frame.leftBorder-c.frame.rightBorder,
                     c.height-c.frame.topBorder-c.frame.bottomBorder);

    SC.GetClock(sTime);

    height:=c.height;
    IF c.text#NIL THEN
      DEC(height,c.text.height+c.display.spaceHeight);
    END;

    zx:=c.x+c.width DIV 2;
    zy:=c.y+height DIV 2;

    r:=G.MinLong(c.width-c.frame.leftBorder-c.frame.rightBorder-c.display.spaceWidth,
                 height-c.frame.topBorder-c.frame.bottomBorder-c.display.spaceHeight) DIV 2;

    draw.PushForeground(D.textColor);
    draw.DrawLine(zx,zy,
                  zx+(r * sin((LONG(sTime.hour) MOD 12)*30+sTime.minute DIV 2) DIV 1500),
                  zy-(r * cos((LONG(sTime.hour) MOD 12)*30+sTime.minute DIV 2) DIV 1500));
    draw.DrawLine(zx,zy,
                  zx+(r * sin(LONG(sTime.minute)*6) DIV 1000),
                  zy-(r * cos(LONG(sTime.minute)*6) DIV 1000));
    draw.PopForeground;

    draw.PushForeground(D.warnColor);
    draw.DrawLine(zx,zy,
                  zx+(r * sin(LONG(sTime.second)*6) DIV 1200),
                  zy-(r * cos(LONG(sTime.second)*6) DIV 1200));
    draw.PopForeground;

    IF c.text#NIL THEN
      c.text.Resize(c.width-c.display.spaceWidth-c.frame.leftBorder-c.frame.rightBorder,-1);
      IF c.selected THEN
        C.TimeToStr(sTime,c.text2,text);
      ELSE
        C.TimeToStr(sTime,c.text1,text);
      END;
      c.text.SetText(text);
      c.text.Draw(c.x+c.frame.leftBorder+c.display.spaceWidth DIV 2,
                  c.y+c.height-c.text.height-c.frame.bottomBorder-c.display.spaceHeight DIV 2,draw);
    END;

    IF c.timeOut=NIL THEN
      c.timeOut:=c.display.AddTimeOut(callPeriod,0,c);
    END;
  END Draw;

  PROCEDURE (c : Clock) Hide*;

  BEGIN
    IF c.timeOut#NIL THEN
      c.display.RemoveTimeOut(c.timeOut);
      c.timeOut:=NIL;
    END;
    IF c.visible THEN
      c.DrawHide;
      c.Hide^;
    END;
  END Hide;

  PROCEDURE (c : Clock) Receive*(msg : O.Message);

  BEGIN
    WITH msg:
      D.TimeOutMsg DO
        IF c.visible THEN
          c.timeOut:=NIL;
          c.Redraw;
        END;
    ELSE
    END;
  END Receive;

BEGIN
  sinTab[ 0]:= 000; (*  0 *)
  sinTab[ 1]:= 105; (*  6 *)
  sinTab[ 2]:= 208; (* 12 *)
  sinTab[ 3]:= 309; (* 18 *)
  sinTab[ 4]:= 407; (* 24 *)
  sinTab[ 5]:= 500; (* 30 *)
  sinTab[ 6]:= 588; (* 36 *)
  sinTab[ 7]:= 669; (* 42 *)
  sinTab[ 8]:= 743; (* 48 *)
  sinTab[ 9]:= 809; (* 54 *)
  sinTab[10]:= 866; (* 60 *)
  sinTab[11]:= 914; (* 66 *)
  sinTab[12]:= 951; (* 72 *)
  sinTab[13]:= 978; (* 78 *)
  sinTab[14]:= 995; (* 84 *)
  sinTab[15]:=1000; (* 90 *)

  NEW(prefs);
  prefs.Init;
END VOClock.