(**
  This module implements loading and displaying of xpm images using the
  widly spread xpmlib.
**)

MODULE VOXpmImage;

(*
    A class for loading and displaying xpm images.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT s   := SYSTEM,
       D   := VODisplay,
       G   := VOGUIObject,
       U   := VOUtil,

              C,
              Err,
       str := Strings,

              X11,
              xpm;

CONST
  plainMode  * = 0;
  buttonMode * = 1;

TYPE
  XpmImage*     = POINTER TO XpmImageDesc;
  XpmImageDesc* = RECORD (G.ImageDesc)
                    filename  : U.Text;
                    xpmImage  : xpm.XpmImage;
                    image,
                    shape,
                    selImage,
                    selShape  : X11.XImagePtr;
                    attrib,
                    selAttrib : xpm.XpmAttributes;

                    hAlign,
                    vAlign    : LONGINT;

                    mode      : LONGINT;
                  END;

  (**
    Initialized the object. Must be called.
  **)

  PROCEDURE (x : XpmImage) Init*;

  BEGIN
    x.Init^;

    x.filename:=NIL;
    x.image:=NIL;
    x.shape:=NIL;
    x.selImage:=NIL;
    x.selShape:=NIL;
    x.mode:=plainMode;
    x.hAlign:=G.alignCenter;
    x.vAlign:=G.alignCenter;
  END Init;

  (**
    Set the alignment of the image within the image area.
    See VOGUIObject for alignments supported.
  **)

  PROCEDURE (x : XpmImage) SetAlignment*(horiz, vert : LONGINT);

  BEGIN
    x.hAlign:=horiz;
    x.vAlign:=vert;
  END SetAlignment;

  (**
    Set the filename of the image. Must be set before use.
  **)

  PROCEDURE (x : XpmImage) SetFilename*(name : ARRAY OF CHAR);

  BEGIN
    NEW(x.filename,str.Length(name)+1);
    COPY(name,x.filename^);
  END SetFilename;

  (**
    Set the mode of the object.

    * plainMode
      Uses the fillColor when selecting.

    * buttonMode
      draw the image a little bit more left and down on select, so one gets
      a 3D feal when selecting-
  **)

  PROCEDURE (x : XpmImage) SetMode*(mode : LONGINT);

  BEGIN
    x.mode:=mode;
  END SetMode;

  PROCEDURE (x : XpmImage) CalcSize*(display : D.Display);

  VAR
    res,i   : LONGINT;
    symbols : xpm.ColorSymbolTable;
    window  : D.Window;


    PROCEDURE PrintError(err : LONGINT);

    VAR
      error  : C.charPtr1d;
      buffer : ARRAY 100 OF CHAR;
      x      : LONGINT;

    BEGIN
      error:=xpm.XpmGetErrorString(res);
      x:=0;
      WHILE error[x]#0X DO
        buffer[x]:=error[x];
        INC(x);
      END;
      buffer[x]:=0X;
      Err.String(buffer);
    END PrintError;
(*
    PROCEDURE PrintAttrib(VAR a : xpm.XpmAttributes);

    BEGIN
      Err.String("offset    : "); Err.LongInt(s.VAL(LONGINT,s.ADR(a.npixels))-s.VAL(LONGINT,s.ADR(a)),8); Err.Ln;
      Err.String("Size      : "); Err.LongInt(SIZE(xpm.XpmAttributes),8); Err.Ln;
      Err.String("closeness : "); Err.LongInt(a.closeness,8); Err.Ln;
      Err.String("colors    : "); Err.LongInt(a.ncolors,8); Err.Ln;
      Err.String("pixels    : "); Err.LongInt(a.npixels,8); Err.Ln;
      Err.String("alloced   : "); Err.LongInt(a.nalloc_pixels,8); Err.Ln;
    END PrintAttrib;
 *)
  BEGIN
    (*
      CalcSize may be called more than one but must not open the image twice,
      so we check f we already have been inited.
    *)

    IF ~(G.inited IN x.flags) THEN
      x.image:=NIL;
      x.shape:=NIL;
      x.selImage:=NIL;
      x.selShape:=NIL;

      s.NEW(symbols,(LEN(D.colorNames)+1)*SIZE(xpm.XpmColorSymbol));
      symbols[0].name:=NIL;
      symbols[0].value:=s.VAL(C.charPtr1d,s.ADR("none"));
      symbols[0].pixel:=display.GetX11Color(x.background);

      FOR i:=0 TO LEN(D.colorNames)-1 DO
        symbols[i+1].name:=s.VAL(C.charPtr1d,s.ADR(D.colorNames[i]));
        symbols[i+1].value:=NIL;
        symbols[i+1].pixel:=display.GetX11Color(i);
      END;

      res:=xpm.XpmReadFileToXpmImage(x.filename^,s.VAL(xpm.XpmImagePtr,s.ADR(x.xpmImage)),NIL);
      IF res=xpm.XpmSuccess THEN

        x.attrib.valuemask:={xpm.XpmCloseness,
                             xpm.XpmColorSymbols,
                             xpm.XpmRGBCloseness,
                             xpm.XpmExactColors,
                             xpm.XpmAllocCloseColors};
        x.attrib.closeness:=-1;
        x.attrib.red_closeness:=-1;
        x.attrib.blue_closeness:=-1;
        x.attrib.green_closeness:=-1;
        x.attrib.exactColors:=X11.False;
        x.attrib.alloc_close_colors:=X11.True;
        x.attrib.colorsymbols:=symbols;
        x.attrib.numsymbols:=LEN(D.colorNames)+1;

        res:=xpm.XpmCreateImageFromXpmImage(display.display,
                                            s.VAL(xpm.XpmImagePtr,s.ADR(x.xpmImage)),
                                            x.image,x.shape,
                                            s.VAL(xpm.XpmAttributesPtr,s.ADR(x.attrib)));
        IF res#xpm.XpmSuccess THEN
          Err.String("Error converting "); Err.String(x.filename^); Err.String(": "); PrintError(res); Err.Ln;
        END;
        (*PrintAttrib(x.attrib);*)

        IF (x.mode=plainMode) & ~(G.noHighlight IN x.flags) THEN
          symbols[1].pixel:=display.GetX11Color(D.fillColor);
        END;

        x.selAttrib.valuemask:={xpm.XpmCloseness,
                                xpm.XpmColorSymbols,
                                xpm.XpmRGBCloseness,
                                xpm.XpmExactColors,
                                xpm.XpmAllocCloseColors};
        x.selAttrib.closeness:=-1;
        x.selAttrib.red_closeness:=-1;
        x.selAttrib.blue_closeness:=-1;
        x.selAttrib.green_closeness:=-1;
        x.selAttrib.exactColors:=X11.False;
        x.selAttrib.alloc_close_colors:=X11.True;
        x.selAttrib.colorsymbols:=symbols;
        x.selAttrib.numsymbols:=LEN(D.colorNames)+1;

        res:=xpm.XpmCreateImageFromXpmImage(display.display,
                                            s.VAL(xpm.XpmImagePtr,s.ADR(x.xpmImage)),
                                            x.selImage,x.selShape,
                                            s.VAL(xpm.XpmAttributesPtr,s.ADR(x.selAttrib)));
        IF res#xpm.XpmSuccess THEN
          Err.String("Error converting "); Err.String(x.filename^); Err.String(": "); PrintError(res); Err.Ln;
        END;
        (*PrintAttrib(x.selAttrib);*)
        x.width:=x.xpmImage.width;
        x.height:=x.xpmImage.height;

        IF x.mode=buttonMode THEN
          INC(x.width);
          INC(x.height);
        END;
      ELSE
          Err.String("Error loading "); Err.String(x.filename^); Err.String(": "); PrintError(res); Err.Ln;
        x.width:=2*display.spaceWidth;
        x.height:=2*display.spaceHeight;
      END;

      x.minWidth:=x.width;
      x.minHeight:=x.height;

      (*
        We register ourself to our parent window.
        This asures that XpmImage.Free gets called when the window
        get destroyed. So we can free our ressources.
      *)

      window:=display.GetNewWindow();
      window.AddFreeList(x);
    END;

    x.CalcSize^(display);
  END CalcSize;

  PROCEDURE (i : XpmImage) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    a,b : LONGINT;

  BEGIN
    i.Draw^(x,y,draw);


    CASE i.hAlign OF
      G.alignLeft:
        a:=i.x;
    | G.alignCenter:
        a:=i.x+(i.width-i.xpmImage.width) DIV 2;
    | G.alignRight:
        a:=i.x+i.width-i.xpmImage.width;
    END;

    CASE i.vAlign OF
      G.alignTop:
        b:=i.y;
    | G.alignCenter:
        b:=i.y+(i.height-i.xpmImage.height) DIV 2;
    | G.alignRight:
        b:=i.y+i.height-i.xpmImage.height;
    END;

    IF (i.mode=buttonMode) & (D.selected IN draw.mode) THEN
      INC(a);
      INC(b);
    END;

    IF (D.selected IN draw.mode) & (i.mode=plainMode) & ~(G.noHighlight IN i.flags) THEN
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(i.x,i.y,i.width,i.height);
    ELSE
      draw.PushForeground(D.backgroundColor);
      draw.FillRectangle(i.x,i.y,i.width,i.height);
    END;
    draw.PopForeground;


    IF D.selected IN draw.mode THEN
      IF i.selImage#NIL THEN
        X11.XPutImage(i.display.display,draw.window,draw.gc,i.selImage,0,0,
                      a,b,i.xpmImage.width,i.xpmImage.height);
      END;
    ELSE
      IF i.image#NIL THEN
        X11.XPutImage(i.display.display,draw.window,draw.gc,i.image,0,0,
                      a,b,i.xpmImage.width,i.xpmImage.height);
      END;
    END;
  END Draw;

  PROCEDURE (x : XpmImage) Hide*;

  BEGIN
    IF x.visible THEN
      x.DrawHide;
      x.Hide^;
    END;
  END Hide;

  (**
    Must be called before image gets lost in the undergrounds of the
    garbarge collector or you possibly get memory leaks.
  **)

  PROCEDURE (x : XpmImage) Free*;

  BEGIN
    xpm.XpmFreeAttributes(s.VAL(xpm.XpmAttributesPtr,s.ADR(x.attrib)));
    xpm.XpmFreeAttributes(s.VAL(xpm.XpmAttributesPtr,s.ADR(x.selAttrib)));
    xpm.XpmFreeXpmImage(s.VAL(xpm.XpmImagePtr,s.ADR(x.xpmImage)));
    IF (x.image#NIL) & (x.image.f.destroy_image(x.image)=0) THEN END;
    x.image:=NIL;

    IF (x.selImage#NIL) & (x.selImage.f.destroy_image(x.selImage)=0) THEN END;
    x.selImage:=NIL;

    IF (x.shape#NIL) & (x.shape.f.destroy_image(x.shape)=0) THEN END;
    x.shape:=NIL;

    IF (x.selShape#NIL) & (x.selShape.f.destroy_image(x.selShape)=0) THEN END;
    x.selShape:=NIL;
  END Free;

END VOXpmImage.