/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPTich_macros.cc
|
| Revision    : 1.0
| Date        : 26/05/97
|
| Object      : None
|
| Description : Macro handler code file
|
| (c) Richard Kent 1997
|
| $Id$
|
****************************************************************************/

static char DPTich_macros_cc [] = "$Id$";

#include "DPTich_macros.h"

int localMacrosUsed = FALSE;
long numberMacros   = 0;
long currentMacro   = -1;
macroType macro [MAXMACROS];

#define ENDAIFFMACRO                                          \
{                                                             \
  delete [] reloadFilename;                                   \
  delete [] tempOutputFilename;                               \
  delete [] filename;                                         \
  return 0;                                                   \
}

#define REMOVETEMPFILES                                       \
{                                                             \
  if (macro [macroNumber].output == MACRODIALOGOUTPUT)        \
    if (remove (tempOutputFilename) == -1)                    \
      fl_show_alert ("Warning",                               \
      "Unable to remove temporary output file",               \
      tempOutputFilename,TRUE);                               \
  if (reloadUsed)                                             \
    if (remove (reloadFilename) == -1)                        \
      fl_show_alert ("Warning",                               \
      "Unable to remove temporary reload file",               \
      reloadFilename,TRUE);                                   \
  if (macro [macroNumber].processMode == MACROTEMPAIFFMODE)   \
    if (remove (filename) == -1)                              \
      fl_show_alert ("Warning",                               \
      "Unable to remove temporary save file",                 \
      filename,TRUE);                                         \
}

/*---------------------------------------------------------------------------
| FUNCTION loadMacros
---------------------------------------------------------------------------*/
int loadMacros ()
{
  char *directory;
  char *filename;
  FILE *file;
  int  macrosFound;
  
  macrosFound = FALSE;

  directory = getenv ("HOME");

  if (!localMacrosUsed && directory)
  {
    filename = new char [strlen (directory) + strlen (MACROSNAME) + 2];
    strcpy (filename,directory);
    if (strlen (filename) > 0 && filename [strlen (filename) - 1] != '/')
      strcat (filename,"/");
    strcat (filename,MACROSNAME);
    file = fopen (filename,"r");
    if (file)
      macrosFound = TRUE;
    else
      delete [] filename;
  }
  
  if (!macrosFound)
  {
    localMacrosUsed = TRUE;
    directory = ".";
    filename = new char [strlen (directory) + strlen (MACROSNAME) + 2];
    strcpy (filename,directory);
    if (strlen (filename) > 0 && filename [strlen (filename) - 1] != '/')
      strcat (filename,"/");
    strcat (filename,MACROSNAME);
    file = fopen (filename,"r");
    if (file)
      macrosFound = TRUE;
    else
      delete [] filename;
  }

  if (!macrosFound) return FALSE;
  
  // Read in macros

  const int  tempStringLen = 128;
  const char delimiters [] = " \t\n";
  const char textDelimiters [] = "\n";
  char tempString [tempStringLen];
  char *token;
  
  int i;
  
  i = numberMacros;

  while (i < MAXMACROS && readLine (file,tempString,tempStringLen))
  {
    if (!strlen (tempString)) continue;
    token = strtok (tempString,delimiters);
    if (!token) continue;
    if (!strcasecmp (token,"Name:"))
    {
      token = strtok (NULL,textDelimiters);
      if (token)
      {
        int firstNonSpace = findNonSpace (token);
        strncpy (macro [i].name,token + firstNonSpace,MACRONAMELENGTH);
      }
    }
    else if (!strcasecmp (token,"BufferMode:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"CurrentBuffer"))
          macro [i].bufferMode = MACROCURRENTBUFFER;
        else if (!strcasecmp (token,"AllBuffers"))
          macro [i].bufferMode = MACROALLBUFFERS;
      }
    }
    else if (!strcasecmp (token,"ProcessMode:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"Raw"))
          macro [i].processMode = MACRORAWMODE;
        else if (!strcasecmp (token,"Text"))
          macro [i].processMode = MACROTEXTMODE;
        else if (!strcasecmp (token,"TempAiff"))
          macro [i].processMode = MACROTEMPAIFFMODE;
        else if (!strcasecmp (token,"SaveLoadAiff"))
          macro [i].processMode = MACROSAVEAIFFMODE;
      }
    }
    else if (!strcasecmp (token,"Input:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"Buffer"))
          macro [i].input = MACROBUFFERINPUT;
        else if (!strcasecmp (token,"Range"))
          macro [i].input = MACRORANGEINPUT;
        else if (!strcasecmp (token,"EitherBufferOrRange"))
          macro [i].input = MACROEITHERINPUT;
        else if (!strcasecmp (token,"None"))
          macro [i].input = MACRONONEINPUT;
      }
    }
    else if (!strcasecmp (token,"Output:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"Buffer"))
          macro [i].output = MACROBUFFEROUTPUT;
        else if (!strcasecmp (token,"Dialog"))
          macro [i].output = MACRODIALOGOUTPUT;
        else if (!strcasecmp (token,"None"))
          macro [i].output = MACRONONEOUTPUT;
      }
    }
    else if (!strcasecmp (token,"DataFormat:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"Byte"))
          macro [i].dataFormat = MACROBYTEFORMAT;
        else if (!strcasecmp (token,"Short"))
          macro [i].dataFormat = MACROSHORTFORMAT;
        else if (!strcasecmp (token,"Long24"))
          macro [i].dataFormat = MACROLONG24FORMAT;
        else if (!strcasecmp (token,"Long32"))
          macro [i].dataFormat = MACROLONG32FORMAT;
        else if (!strcasecmp (token,"Float"))
          macro [i].dataFormat = MACROFLOATFORMAT;
        else if (!strcasecmp (token,"Double"))
          macro [i].dataFormat = MACRODOUBLEFORMAT;
      }
    }
    else if (!strcasecmp (token,"DataSigned:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"Signed"))
          macro [i].dataSigned = MACROSIGNED;
        else if (!strcasecmp (token,"Unsigned"))
          macro [i].dataSigned = MACROUNSIGNED;
      }
    }
    else if (!strcasecmp (token,"Command:"))
    {
      token = strtok (NULL,textDelimiters);
      if (token)
      {
        int firstNonSpace = findNonSpace (token);
        strncpy (macro [i].command,token + firstNonSpace,MACROCOMMANDLENGTH);
      }
    }
    else if (!strcasecmp (token,"EndMacro"))
    {
      fl_addto_browser (macroForm->browser,macro [i].name);
      fl_addto_browser (macroexecForm->browser,macro [i].name);
      numberMacros++;
      i++;
    }
  }
  
  fclose (file);
  delete [] filename;
  updateMacroForm ();
  macrosChanged = FALSE;
  return TRUE;
}  
  
/*---------------------------------------------------------------------------
| FUNCTION saveMacros
---------------------------------------------------------------------------*/
int saveMacros ()
{
  char *directory;
  char *filename;
  FILE *file;
  int  macrosFound;
  
  macrosFound = FALSE;

  directory = getenv ("HOME");

  if (!localMacrosUsed && directory)
  {
    filename = new char [strlen (directory) + strlen (MACROSNAME) + 2];
    strcpy (filename,directory);
    if (strlen (filename) > 0 && filename [strlen (filename) - 1] != '/')
      strcat (filename,"/");
    strcat (filename,MACROSNAME);
    file = fopen (filename,"w");
    if (file)
      macrosFound = TRUE;
    else
      delete [] filename;
  }
  
  if (!macrosFound)
  {
    localMacrosUsed = TRUE;
    directory = "";
    filename = new char [strlen (directory) + strlen (MACROSNAME) + 2];
    strcpy (filename,directory);
    if (strlen (filename) > 0 && filename [strlen (filename) - 1] != '/')
      strcat (filename,"/");
    strcat (filename,MACROSNAME);
    file = fopen (filename,"w");
    if (file)
      macrosFound = TRUE;
    else
      delete [] filename;
  }

  if (!macrosFound) return FALSE;
  
  // Write out macros

  int i;
  
  for (i=0; i<numberMacros; i++)
  {
    fprintf (file,"StartMacro\n");
    
    fprintf (file,"Name: %s\n",macro [i].name); 

    switch (macro [i].bufferMode)
    {
      case MACROCURRENTBUFFER :
        fprintf (file,"BufferMode: CurrentBuffer\n");
        break;
      case MACROALLBUFFERS :
        fprintf (file,"BufferMode: AllBuffers\n");
        break;
      default:
        break;
    }

    switch (macro [i].processMode)
    {
      case MACRORAWMODE :
        fprintf (file,"ProcessMode: Raw\n"); 
        break;
      case MACROTEXTMODE :
        fprintf (file,"ProcessMode: Text\n"); 
        break;
      case MACROTEMPAIFFMODE :
        fprintf (file,"ProcessMode: TempAiff\n"); 
        break;
      case MACROSAVEAIFFMODE :
        fprintf (file,"ProcessMode: SaveLoadAiff\n"); 
        break;
      default:
        break;
    }

    switch (macro [i].input)
    {
      case MACROBUFFERINPUT :
        fprintf (file,"Input: Buffer\n"); 
        break;
      case MACRORANGEINPUT :
        fprintf (file,"Input: Range\n"); 
        break;
      case MACROEITHERINPUT :
        fprintf (file,"Input: EitherBufferOrRange\n"); 
        break;
      case MACRONONEINPUT :
        fprintf (file,"Input: None\n"); 
        break;
      default :
        break;
    }

    switch (macro [i].output)
    {
      case MACROBUFFEROUTPUT :
        fprintf (file,"Output: Buffer\n"); 
        break;
      case MACRODIALOGOUTPUT :
        fprintf (file,"Output: Dialog\n"); 
        break;
      case MACRONONEOUTPUT :
        fprintf (file,"Output: None\n"); 
        break;
      default :
        break;
    }

    switch (macro [i].dataFormat)
    {
      case MACROBYTEFORMAT :
        fprintf (file,"DataFormat: Byte\n");
        break;
      case MACROSHORTFORMAT :
        fprintf (file,"DataFormat: Short\n");
        break;
      case MACROLONG24FORMAT :
        fprintf (file,"DataFormat: Long24\n");
        break;
      case MACROLONG32FORMAT :
        fprintf (file,"DataFormat: Long32\n");
        break;
      case MACROFLOATFORMAT :
        fprintf (file,"DataFormat: Float\n");
        break;
      case MACRODOUBLEFORMAT :
        fprintf (file,"DataFormat: Double\n");
        break;
      default :
        break;
    }

    switch (macro [i].dataSigned)
    {
      case MACROSIGNED :
        fprintf (file,"DataSigned: Signed\n");
        break;
      case MACROUNSIGNED :
        fprintf (file,"DataSigned: Unsigned\n");
        break;
      default :
        break;
    }

    fprintf (file,"Command: %s\n",macro [i].command);

    fprintf (file,"EndMacro\n");
  }

  fclose (file);
  delete [] filename;
  macrosChanged = FALSE;
  return TRUE;
}  
  
/*---------------------------------------------------------------------------
| FUNCTION defaultMacros
---------------------------------------------------------------------------*/
void defaultMacros (void)
{
  int i;
  numberMacros = 0;
  currentMacro = -1;
  for (i=0; i<MAXMACROS; i++)
  {
    strcpy (macro [i].name,"Untitled");
    macro [i].bufferMode  = MACROCURRENTBUFFER;
    macro [i].processMode = MACRORAWMODE;
    macro [i].input       = MACRONONEINPUT;
    macro [i].output      = MACRONONEOUTPUT;
    macro [i].dataFormat  = MACROBYTEFORMAT;
    macro [i].dataSigned  = MACROSIGNED;
    strcpy (macro [i].command,"");
  }
  fl_deselect_browser (macroForm->browser);
  fl_clear_browser (macroForm->browser);
  fl_clear_browser (macroexecForm->browser);
  macrosChanged = FALSE;
}

/*---------------------------------------------------------------------------
| CALLBACK macroName_cb
---------------------------------------------------------------------------*/
void macroName_cb (FL_OBJECT *ob,long data)
{
  int i;
  const char *name = fl_get_input (macroForm->name);

  if ((i = findMacro (name)) >= 0)
  {
    fl_select_browser_line (macroForm->browser,i + 1);
    fl_call_object_callback (macroForm->browser);
  }
  else
  {
    fl_deselect_browser (macroForm->browser);
    fl_call_object_callback (macroForm->storeButton);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK macroCommand_cb
---------------------------------------------------------------------------*/
void macroCommand_cb (FL_OBJECT *ob,long data)
{
  const char *name = fl_get_input (macroForm->name);
  if (findMacro (name) >= 0)
    fl_call_object_callback (macroForm->storeButton);
  else
    updateMacroForm ();
}

/*---------------------------------------------------------------------------
| CALLBACK macroButton_cb
---------------------------------------------------------------------------*/
void macroButton_cb (FL_OBJECT *ob,long data)
{
  const char *name = fl_get_input (macroForm->name);
  if (findMacro (name) >= 0)
    fl_call_object_callback (macroForm->storeButton);
  else
    updateMacroForm ();
}

/*---------------------------------------------------------------------------
| CALLBACK macroBrowser_cb
---------------------------------------------------------------------------*/
void macroBrowser_cb (FL_OBJECT *ob,long data)
{
  int current = fl_get_browser (macroForm->browser) - 1;
  if (current >= 0 && current < numberMacros)
  {
    currentMacro = current;
    updateMacroForm ();
  }
  else
  {
    fl_deselect_browser (macroForm->browser);
  }
}

/*---------------------------------------------------------------------------
| CALLBACK macroNewButton_cb
---------------------------------------------------------------------------*/
void macroNewButton_cb (FL_OBJECT *ob,long data)
{
  currentMacro = -1;
  fl_set_input (macroForm->name,"Untitled");
  fl_set_input (macroForm->command,"");
  fl_set_button (macroForm->currentBuffer,1);
  fl_set_button (macroForm->pipeRaw,1);    
  fl_set_button (macroForm->byteButton,1);
  fl_set_button (macroForm->signedButton,1);
  fl_set_button (macroForm->noneInput,1);
  fl_set_button (macroForm->noneOutput,1);
  fl_deselect_browser (macroForm->browser);
  updateMacroForm ();
}

/*---------------------------------------------------------------------------
| CALLBACK macroExecuteButton_cb
---------------------------------------------------------------------------*/
void macroExecuteButton_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  const char *name = fl_get_input (macroForm->name);
  char *errorMessage;
  
  updateBuffers ("Undo macro");
  waitCursorOn ();
  errorMessage = executeMacro (name);
  waitCursorOff ();
  if (errorMessage)
    fl_show_alert ("Warning","Cannot execute macro",errorMessage,TRUE);
  updateSample (sample);
}

/*---------------------------------------------------------------------------
| CALLBACK macroStoreButton_cb
---------------------------------------------------------------------------*/
void macroStoreButton_cb (FL_OBJECT *ob,long data)
{
  int i;
  const char *name = fl_get_input (macroForm->name);
  
  // Check for inconsistencies and sort them out
  if (fl_get_button (macroForm->aiffTemp))
  {
    if (fl_get_button (macroForm->noneInput))
      fl_set_button (macroForm->eitherInput,1);
  }
  else if (fl_get_button (macroForm->aiffSave))
  {
    if (!fl_get_button (macroForm->bufferInput))
      fl_set_button (macroForm->bufferInput,1);
  }
 
  if ((i = findMacro (name)) < 0)
  {
    if (numberMacros == MAXMACROS)
    {
      fl_show_alert ("Warning","Maximum number of macros stored",
        "Please remove a macro before storing a new one",TRUE);
      return;
    }
    i = numberMacros;
    numberMacros++;
    strncpy (macro [i].name,fl_get_input (macroForm->name),
      MACRONAMELENGTH);
    fl_addto_browser (macroForm->browser,macro [i].name);
    fl_addto_browser (macroexecForm->browser,macro [i].name);
  }
    
  if (fl_get_button (macroForm->currentBuffer))
    macro [i].bufferMode = MACROCURRENTBUFFER;
  else
    macro [i].bufferMode = MACROALLBUFFERS;

  if (fl_get_button (macroForm->pipeRaw))
    macro [i].processMode = MACRORAWMODE;
  else if (fl_get_button (macroForm->pipeText))
    macro [i].processMode = MACROTEXTMODE;
  else if (fl_get_button (macroForm->aiffTemp))
    macro [i].processMode = MACROTEMPAIFFMODE;
  else
    macro [i].processMode = MACROSAVEAIFFMODE;

  if (fl_get_button (macroForm->byteButton))
    macro [i].dataFormat = MACROBYTEFORMAT;
  else if (fl_get_button (macroForm->shortButton))
    macro [i].dataFormat = MACROSHORTFORMAT;
  else if (fl_get_button (macroForm->long24Button))
    macro [i].dataFormat = MACROLONG24FORMAT;
  else if (fl_get_button (macroForm->long32Button))
    macro [i].dataFormat = MACROLONG32FORMAT;
  else if (fl_get_button (macroForm->floatButton))
    macro [i].dataFormat = MACROFLOATFORMAT;
  else
    macro [i].dataFormat = MACRODOUBLEFORMAT;

  if (fl_get_button (macroForm->signedButton))
    macro [i].dataSigned = MACROSIGNED;
  else
    macro [i].dataSigned = MACROUNSIGNED;

  if (fl_get_button (macroForm->bufferInput))
    macro [i].input = MACROBUFFERINPUT;
  else if (fl_get_button (macroForm->rangeInput))
    macro [i].input = MACRORANGEINPUT;
  else if (fl_get_button (macroForm->eitherInput))
    macro [i].input = MACROEITHERINPUT;
  else
    macro [i].input = MACRONONEINPUT;

  if (fl_get_button (macroForm->bufferOutput))
    macro [i].output = MACROBUFFEROUTPUT;
  else if (fl_get_button (macroForm->dialogOutput))
    macro [i].output = MACRODIALOGOUTPUT;
  else
    macro [i].output = MACRONONEOUTPUT;

  strncpy (macro [i].command,fl_get_input (macroForm->command),
    MACROCOMMANDLENGTH);
    
  currentMacro = i;
  fl_select_browser_line (macroForm->browser,currentMacro + 1);
  macrosChanged = TRUE;
}

/*---------------------------------------------------------------------------
| CALLBACK macroDeleteButton_cb
---------------------------------------------------------------------------*/
void macroDeleteButton_cb (FL_OBJECT *ob,long data)
{
  int i;
  int j;
  const char *name = fl_get_input (macroForm->name);

  if ((i = findMacro (name)) >= 0)
  {
    for (j=i; j<(numberMacros-1); j++)
    {
      strcpy (macro [j].name,macro [j+1].name);
      macro [j].bufferMode  = macro [j+1].bufferMode;
      macro [j].processMode = macro [j+1].processMode;
      macro [j].input       = macro [j+1].input;
      macro [j].output      = macro [j+1].output;
      macro [j].dataFormat  = macro [j+1].dataFormat;
      macro [j].dataSigned  = macro [j+1].dataSigned;
      strcpy (macro [j].command,macro [j+1].command);
    }
    fl_deselect_browser (macroForm->browser);
    fl_delete_browser_line (macroForm->browser,i + 1);
    fl_delete_browser_line (macroexecForm->browser,i + 1);
    numberMacros--;
    macrosChanged = TRUE;
    currentMacro = -1;
    updateMacroForm ();
  }
}

/*---------------------------------------------------------------------------
| CALLBACK macroOK_cb
---------------------------------------------------------------------------*/
void macroOK_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (macroForm->macroForm);
}

/*---------------------------------------------------------------------------
| CALLBACK macrooutputOK_cb
---------------------------------------------------------------------------*/
void macrooutputOK_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (macrooutputForm->macrooutputForm);
}

/*---------------------------------------------------------------------------
| CALLBACK macroexecName_cb
---------------------------------------------------------------------------*/
void macroexecName_cb (FL_OBJECT *ob,long data)
{
}

/*---------------------------------------------------------------------------
| CALLBACK macroexecBrowserDbl_cb
---------------------------------------------------------------------------*/
void macroexecBrowserDbl_cb (FL_OBJECT *ob,long data)
{
  fl_call_object_callback (macroexecForm->browser);
  fl_call_object_callback (macroexecForm->OK);
}

/*---------------------------------------------------------------------------
| CALLBACK macroexecBrowser_cb
---------------------------------------------------------------------------*/
void macroexecBrowser_cb (FL_OBJECT *ob,long data)
{
  int i = fl_get_browser (macroexecForm->browser) - 1;
  if (i >= 0 && i < numberMacros)
    fl_set_input (macroexecForm->name,macro [i].name);
  else
    fl_set_input (macroexecForm->name,"");
}

/*---------------------------------------------------------------------------
| CALLBACK macroexecOK_cb
---------------------------------------------------------------------------*/
void macroexecOK_cb (FL_OBJECT *ob,long data)
{
  DPSample *sample = fl_get_sample (mainForm->sample);
  const char *name = fl_get_input (macroexecForm->name);
  char *errorMessage;
  
  updateBuffers ("Undo macro");
  waitCursorOn ();
  errorMessage = executeMacro (name);
  waitCursorOff ();
  if (errorMessage)
    fl_show_alert ("Warning","Cannot execute macro",errorMessage,TRUE);
  updateSample (sample);
}

/*---------------------------------------------------------------------------
| CALLBACK macroexecCancel_cb
---------------------------------------------------------------------------*/
void macroexecCancel_cb (FL_OBJECT *ob,long data)
{
  fl_hide_form (macroexecForm->macroexecForm);
}

/*---------------------------------------------------------------------------
| FUNCTION updateMacroForm
---------------------------------------------------------------------------*/
void updateMacroForm (void)
{
  // Check for inconsistencies and sort them out
  if (fl_get_button (macroForm->aiffTemp))
  {
    if (fl_get_button (macroForm->noneInput))
      fl_set_button (macroForm->eitherInput,1);
  }
  else if (fl_get_button (macroForm->aiffSave))
  {
    if (!fl_get_button (macroForm->bufferInput))
      fl_set_button (macroForm->bufferInput,1);
  }
 
  if (currentMacro >= numberMacros)
    currentMacro = -1;
    
  if (currentMacro < 0)
  {
    fl_deselect_browser (macroForm->browser);
  }
  else
  {
    int i = currentMacro;
    
    fl_set_input (macroForm->name,macro [i].name);
    fl_set_input (macroForm->command,macro [i].command);

    switch (macro [i].bufferMode)
    {
      case MACROCURRENTBUFFER :
        fl_set_button (macroForm->currentBuffer,1);
        break;
      case MACROALLBUFFERS :
        fl_set_button (macroForm->allBuffers,1);
        break;
      default :
        break;
    }
    
    switch (macro [i].processMode)
    {
      case MACRORAWMODE :
        fl_set_button (macroForm->pipeRaw,1);
        break;
      case MACROTEXTMODE :
        fl_set_button (macroForm->pipeText,1);
        break;
      case MACROTEMPAIFFMODE :
        fl_set_button (macroForm->aiffTemp,1);
        break;
      case MACROSAVEAIFFMODE :
        fl_set_button (macroForm->aiffSave,1);
        break;
      default :
        break;
    }
    
    switch (macro [i].dataFormat)
    {
      case MACROBYTEFORMAT :
        fl_set_button (macroForm->byteButton,1);
        break;
      case MACROSHORTFORMAT :
        fl_set_button (macroForm->shortButton,1);
        break;
      case MACROLONG24FORMAT :
        fl_set_button (macroForm->long24Button,1);
        break;
      case MACROLONG32FORMAT :
        fl_set_button (macroForm->long32Button,1);
        break;
      case MACROFLOATFORMAT :
        fl_set_button (macroForm->floatButton,1);
        break;
      case MACRODOUBLEFORMAT :
        fl_set_button (macroForm->doubleButton,1);
        break;
      default :
        break;
    }

    switch (macro [i].dataSigned)
    {
      case MACROSIGNED :
        fl_set_button (macroForm->signedButton,1);
        break;
      case MACROUNSIGNED :
        fl_set_button (macroForm->unsignedButton,1);
        break;
      default :
        break;
    }
    
    switch (macro [i].input)
    {
      case MACROBUFFERINPUT :
        fl_set_button (macroForm->bufferInput,1);
        break;
      case MACRORANGEINPUT :
        fl_set_button (macroForm->rangeInput,1);
        break;
      case MACROEITHERINPUT :
        fl_set_button (macroForm->eitherInput,1);
        break;
      case MACRONONEINPUT :
        fl_set_button (macroForm->noneInput,1);
        break;
      default :
        break;
    }

    switch (macro [i].output)
    {
      case MACROBUFFEROUTPUT :
        fl_set_button (macroForm->bufferOutput,1);
        break;
      case MACRODIALOGOUTPUT :
        fl_set_button (macroForm->dialogOutput,1);
        break;
      case MACRONONEOUTPUT :
        fl_set_button (macroForm->noneOutput,1);
        break;
      default :
        break;
    }
    
    fl_select_browser_line (macroForm->browser,currentMacro + 1);
  }
}

/*---------------------------------------------------------------------------
| FUNCTION findMacro
---------------------------------------------------------------------------*/
long findMacro (const char *name)
{
  int i=0;
  int found=FALSE;
  if (!name) return -1;
  
  while (i<numberMacros && !found)
  {
    if (!strcmp (macro [i].name,name))
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return -1;
  return i;
}

/*---------------------------------------------------------------------------
| FUNCTION executeMacro
---------------------------------------------------------------------------*/
char *executeMacro (const char *name)
{
  int i;
  if ((i = findMacro (name)) < 0)
    return "Unable to find macro with the given name";
  
  if (macro [i].bufferMode == MACROCURRENTBUFFER)
  {
    return executeSingleMacro (i,current);
  }
  else
  {
    char *errorMessage;
    for (int j=0; j<8; j++)
    {
      errorMessage  = executeSingleMacro (i,j);
      if (errorMessage) return errorMessage;
    }
  }
  
  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION executeSingleMacro
---------------------------------------------------------------------------*/
char *executeSingleMacro (int macroNumber,int bufferNumber)
{
  // Can assume macroNumber is valid macro
  DPSample *sample;
  
  if (bufferNumber < 0 || bufferNumber >= 8)
    return "Illegal buffer number";
  
  sample = buffer [bufferNumber];
  if (macro [macroNumber].processMode == MACROSAVEAIFFMODE &&
    !sample->getValid ())
    return 0; // No error
  if (macro [macroNumber].processMode == MACROTEMPAIFFMODE &&
    !sample->getValid ())
    return 0; // No error
  if ((macro [macroNumber].input != MACRONONEINPUT ||
    macro [macroNumber].output == MACROBUFFEROUTPUT) &&
    !sample->getValid ())
    return 0; // No error
  if (macro [macroNumber].input == MACRORANGEINPUT &&
    !sample->getRangeValid ())
    return 0; // No error
  
  switch (macro [macroNumber].processMode)
  {
    case MACRORAWMODE :
    case MACROTEXTMODE :
    {
      FILE *pipeFile;
      FILE *tempFile;
      const char *tempDir = fl_get_sample_tempdir (mainForm->sample);
      char *tempNameTemplate;
      char *mktempFileName;
      char *tempFileName;
      char *tempCommand;
      char *command = macro [macroNumber].command;

      long i;
      long j;
        
      signed char    tempSignedChar;
      unsigned char  tempUnsignedChar;
      signed short   tempSignedShort;
      unsigned short tempUnsignedShort;
      signed long    tempSignedLong;
      unsigned long  tempUnsignedLong;
      float          tempFloat;
      double         tempDouble;

      tempNameTemplate = new char [strlen (tempDir) + 8];
      strcpy (tempNameTemplate,tempDir);
      if (strlen (tempNameTemplate) > 0 &&
        tempNameTemplate [strlen (tempNameTemplate) - 1] != '/')
        strcat (tempNameTemplate,"/");
      strcat (tempNameTemplate,"XXXXXX");
      mktempFileName = mktemp (tempNameTemplate);
      if (!mktempFileName)
      {
        delete [] tempNameTemplate;
        return "Unable to create temporary filename";
      }

      tempFileName = new char [strlen (tempNameTemplate) + 1];
      strcpy (tempFileName,tempNameTemplate);
      delete [] tempNameTemplate;

      tempCommand = new char
        [strlen (command) + 3 + strlen (tempFileName)];
      strcpy (tempCommand,command);
      strcat (tempCommand," >");
      strcat (tempCommand,tempFileName);

      pipeFile = popen (tempCommand,"w");
      delete [] tempCommand;  

      if (!pipeFile)
      {
        delete [] tempFileName;
        return "Cannot open pipe file";
      }
      
      // Input to command
      
      if (macro [macroNumber].input != MACRONONEINPUT)
      {
        long inputStart;
        long inputEnd;
        long inputChannels;
        long inputSample;
        long finished;
                
        if (macro [macroNumber].input == MACROBUFFERINPUT ||
          (macro [macroNumber].input == MACROEITHERINPUT &&
          !sample->getRangeValid ()))
        {
          inputStart = 0;
          inputEnd   = sample->getFrames ();
        }
        else
        {
          inputStart = sample->getRangeStart ();
          inputEnd   = sample->getRangeEnd ();
        }
        
        inputChannels = sample->getChannels ();
        finished = FALSE;
        
        if (macro [macroNumber].processMode == MACRORAWMODE)
        {
          for (i=inputStart; i<inputEnd && !finished; i++)
          {
            for (j=0; j<inputChannels && !finished; j++)
            {
              inputSample = sample->getFrame24 (i,j);
              if (macro [macroNumber].dataSigned == MACROSIGNED)
              {
                switch (macro [macroNumber].dataFormat)
                {
                  case MACROBYTEFORMAT :
                    tempSignedChar = inputSample / 65536;
                    if (fwrite (&tempSignedChar,
                      sizeof (tempSignedChar),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACROSHORTFORMAT :
                    tempSignedShort = inputSample / 256;
                    if (fwrite (&tempSignedShort,
                      sizeof (tempSignedShort),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACROLONG24FORMAT :
                    tempSignedLong = inputSample;
                    if (fwrite (&tempSignedLong,
                      sizeof (tempSignedLong),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACROLONG32FORMAT :
                    tempSignedLong = inputSample * 256;
                    if (fwrite (&tempSignedLong,
                      sizeof (tempSignedLong),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACROFLOATFORMAT :
                    tempFloat = ((float) inputSample) / MAX23;
                    if (fwrite (&tempFloat,
                      sizeof (tempFloat),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACRODOUBLEFORMAT :
                    tempDouble = ((double) inputSample) / MAX23;
                    if (fwrite (&tempDouble,
                      sizeof (tempDouble),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  default :
                    break;
                }
              }
              else
              {
                switch (macro [macroNumber].dataFormat)
                {
                  case MACROBYTEFORMAT :
                    tempUnsignedChar = inputSample / 65536 + MAX7;
                    if (fwrite (&tempUnsignedChar,
                      sizeof (tempUnsignedChar),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACROSHORTFORMAT :
                    tempUnsignedShort = inputSample / 256 + MAX15;
                    if (fwrite (&tempUnsignedShort,
                      sizeof (tempUnsignedShort),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACROLONG24FORMAT :
                    tempUnsignedLong = inputSample + MAX23;
                    if (fwrite (&tempUnsignedLong,
                      sizeof (tempUnsignedLong),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACROLONG32FORMAT :
                    tempUnsignedLong = inputSample * 256 + MAX31;
                    if (fwrite (&tempUnsignedLong,
                      sizeof (tempUnsignedLong),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACROFLOATFORMAT :
                    tempFloat = ((float) inputSample) / MAX24 + 0.5;
                    if (fwrite (&tempFloat,
                      sizeof (tempFloat),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  case MACRODOUBLEFORMAT :
                    tempDouble = ((double) inputSample) / MAX24 + 0.5;
                    if (fwrite (&tempDouble,
                      sizeof (tempDouble),1,pipeFile) != 1)
                      finished = TRUE;
                    break;
                  default :
                    break;
                }
              }
            }
          }
        }
        else
        {
          for (i=inputStart; i<inputEnd && !finished; i++)
          {
            for (j=0; j<inputChannels && !finished; j++)
            {
              inputSample = sample->getFrame24 (i,j);
              if (macro [macroNumber].dataSigned == MACROSIGNED)
              {
                switch (macro [macroNumber].dataFormat)
                {
                  case MACROBYTEFORMAT :
                    tempSignedChar = inputSample / 65536;
                    if (fprintf (pipeFile,"%d\n",tempSignedChar) <= 0)
                      finished = TRUE;
                    break;
                  case MACROSHORTFORMAT :
                    tempSignedShort = inputSample / 256;
                    if (fprintf (pipeFile,"%hd\n",tempSignedShort) <= 0)
                      finished = TRUE;
                    break;
                  case MACROLONG24FORMAT :
                    tempSignedLong = inputSample;
                    if (fprintf (pipeFile,"%ld\n",tempSignedLong) <= 0)
                      finished = TRUE;
                    break;
                  case MACROLONG32FORMAT :
                    tempSignedLong = inputSample * 256;
                    if (fprintf (pipeFile,"%ld\n",tempSignedLong) <= 0)
                      finished = TRUE;
                    break;
                  case MACROFLOATFORMAT :
                    tempFloat = ((float) inputSample) / MAX23;
                    if (fprintf (pipeFile,"%.20f\n",tempFloat) <= 0)
                      finished = TRUE;
                    break;
                  case MACRODOUBLEFORMAT :
                    tempDouble = ((double) inputSample) / MAX23;
                    if (fprintf (pipeFile,"%.20lf\n",tempDouble) <= 0)
                      finished = TRUE;
                    break;
                  default :
                    break;
                }
              }
              else
              {
                switch (macro [macroNumber].dataFormat)
                {
                  case MACROBYTEFORMAT :
                    tempUnsignedChar = inputSample / 65536 + MAX7;
                    if (fprintf (pipeFile,"%u\n",tempUnsignedChar) <= 0)
                      finished = TRUE;
                    break;
                  case MACROSHORTFORMAT :
                    tempUnsignedShort = inputSample / 256 + MAX15;
                    if (fprintf (pipeFile,"%hu\n",tempUnsignedShort) <= 0)
                      finished = TRUE;
                    break;
                  case MACROLONG24FORMAT :
                    tempUnsignedLong = inputSample + MAX23;
                    if (fprintf (pipeFile,"%lu\n",tempUnsignedLong) <= 0)
                      finished = TRUE;
                    break;
                  case MACROLONG32FORMAT :
                    tempUnsignedLong = inputSample * 256 + MAX31;
                    if (fprintf (pipeFile,"%lu\n",tempUnsignedLong) <= 0)
                      finished = TRUE;
                    break;
                  case MACROFLOATFORMAT :
                    tempFloat = ((float) inputSample) / MAX24 + 0.5;
                    if (fprintf (pipeFile,"%.20f\n",tempFloat) <= 0)
                      finished = TRUE;
                    break;
                  case MACRODOUBLEFORMAT :
                    tempDouble = ((double) inputSample) / MAX24 + 0.5;
                    if (fprintf (pipeFile,"%.20lf\n",tempDouble) <= 0)
                      finished = TRUE;
                    break;
                  default :
                    break;
                }
              }
            }
          }
        }
      }
      
      pclose (pipeFile);
      
      // Output from command
      
      if (macro [macroNumber].output == MACRODIALOGOUTPUT)
      {
        if (!fl_load_browser (macrooutputForm->browser,tempFileName))
        {
          delete [] tempFileName;
          return "Cannot load temp file into browser";
        }
        fl_show_form (macrooutputForm->macrooutputForm,
          FL_PLACE_MOUSE | FL_FREE_SIZE,FL_FULLBORDER,
          "Command Output");
      }
      else if (macro [macroNumber].output == MACROBUFFEROUTPUT)
      {
        tempFile = fopen (tempFileName,"r");
        if (!tempFile)
        {
          delete [] tempFileName;
          return "Cannot open temp file";
        }

        // Read data back into temp sample
        
        DPSample tempSample;
        long tempSampleLength;
        long outputChannels;
        long outputSample;
        long finished;

        // Default to one second of current sample format
        if (!tempSample.fresh (sample->getRate (),sample->getWidth (),
          sample->getChannels (),(long)sample->getRate ()))
        {
          fclose (tempFile);
          delete [] tempFileName;
          return "Unable to create temp sample";
        }
        
        // Read samples in (set tempSampleLength to actual length)

        outputChannels = sample->getChannels ();
        finished = FALSE;
        i = 0;
        
        if (macro [macroNumber].processMode == MACRORAWMODE)
        {
          while (!finished)
          {
            for (j=0; j<outputChannels && !finished; j++)
            {
              if (macro [macroNumber].dataSigned == MACROSIGNED)
              {
                switch (macro [macroNumber].dataFormat)
                {
                  case MACROBYTEFORMAT :
                    if (fread (&tempSignedChar,
                      sizeof (tempSignedChar),1,tempFile) != 1)
                      finished = TRUE;
                    else
                      outputSample = tempSignedChar * 65536;
                    break;
                  case MACROSHORTFORMAT :
                    if (fread (&tempSignedShort,
                      sizeof (tempSignedShort),1,tempFile) != 1)
                      finished = TRUE;
                    else
                      outputSample = tempSignedShort * 256;
                    break;
                  case MACROLONG24FORMAT :
                    if (fread (&tempSignedLong,
                      sizeof (tempSignedLong),1,tempFile) != 1)
                      finished = TRUE;
                    else
                    {
                      tempSignedLong = limitSLValue
                        (tempSignedLong,-MAX23,MAX23_1);
                      outputSample = tempSignedLong;
                    }
                    break;
                  case MACROLONG32FORMAT :
                    if (fread (&tempSignedLong,
                      sizeof (tempSignedLong),1,tempFile) != 1)
                      finished = TRUE;
                    else
                      outputSample = tempSignedLong / 256;
                    break;
                  case MACROFLOATFORMAT :
                    if (fread (&tempFloat,
                      sizeof (tempFloat),1,tempFile) != 1)
                      finished = TRUE;
                    else
                    {
                      tempFloat = limitFValue (tempFloat,-1.0,1.0);
                      outputSample = limitSLValue (
                        (long)(tempFloat * MAX23),-MAX23,MAX23_1);
                    }
                    break;
                  case MACRODOUBLEFORMAT :
                    if (fread (&tempDouble,
                      sizeof (tempDouble),1,tempFile) != 1)
                      finished = TRUE;
                    else
                    {
                      tempDouble = limitDValue (tempDouble,-1.0,1.0);
                      outputSample = limitSLValue (
                        (long)(tempDouble * MAX23),-MAX23,MAX23_1);
                    }
                    break;
                  default :
                    break;
                }
              }
              else
              {
                switch (macro [macroNumber].dataFormat)
                {
                  case MACROBYTEFORMAT :
                    if (fread (&tempUnsignedChar,
                      sizeof (tempUnsignedChar),1,tempFile) != 1)
                      finished = TRUE;
                    else
                      outputSample = (tempUnsignedChar - MAX7) * 65536;
                    break;
                  case MACROSHORTFORMAT :
                    if (fread (&tempUnsignedShort,
                      sizeof (tempUnsignedShort),1,tempFile) != 1)
                      finished = TRUE;
                    else
                      outputSample = (tempUnsignedShort - MAX15) * 256;
                    break;
                  case MACROLONG24FORMAT :
                    if (fread (&tempUnsignedLong,
                      sizeof (tempUnsignedLong),1,tempFile) != 1)
                      finished = TRUE;
                    else
                    {
                      tempUnsignedLong = limitULValue
                        (tempUnsignedLong,0,MAX24_1);
                      outputSample = tempUnsignedLong - MAX23;
                    }
                    break;
                  case MACROLONG32FORMAT :
                    if (fread (&tempUnsignedLong,
                      sizeof (tempUnsignedLong),1,tempFile) != 1)
                      finished = TRUE;
                    else
                      outputSample = (tempUnsignedLong - MAX31) / 256;
                    break;
                  case MACROFLOATFORMAT :
                    if (fread (&tempFloat,
                      sizeof (tempFloat),1,tempFile) != 1)
                      finished = TRUE;
                    else
                    {
                      tempFloat = limitFValue (tempFloat,0.0,1.0);
                      outputSample = limitSLValue (
                        (long)((tempFloat - 0.5) * MAX24),-MAX23,MAX23_1);
                    }
                    break;
                  case MACRODOUBLEFORMAT :
                    if (fread (&tempDouble,
                      sizeof (tempDouble),1,tempFile) != 1)
                      finished = TRUE;
                    else
                    {
                      tempDouble = limitDValue (tempDouble,0.0,1.0);
                      outputSample = limitSLValue (
                        (long)((tempDouble - 0.5) * MAX24),-MAX23,MAX23_1);
                    }
                    break;
                  default :
                    break;
                }
              }
              if (!finished)
                tempSample.setFrame24Extra (i,j,outputSample);
            }
            if (!finished) i++;
          }
        }
        else
        {
          while (!finished)
          {
            for (j=0; j<outputChannels && !finished; j++)
            {
              if (macro [macroNumber].dataSigned == MACROSIGNED)
              {
                switch (macro [macroNumber].dataFormat)
                {
                  case MACROBYTEFORMAT :
                    if (fscanf (tempFile,"%ld",&tempSignedLong) != 1)
                      finished = TRUE;
                    else
                    {
                      tempSignedChar = limitSLValue (tempSignedLong,-MAX7,MAX7_1);
                      outputSample = tempSignedChar * 65536;
                    }
                    break;
                  case MACROSHORTFORMAT :
                    if (fscanf (tempFile,"%ld",&tempSignedLong) != 1)
                      finished = TRUE;
                    else
                    {
                      tempSignedShort = limitSLValue (tempSignedLong,-MAX15,MAX15_1);
                      outputSample = tempSignedShort * 256;
                    }
                    break;
                  case MACROLONG24FORMAT :
                    if (fscanf (tempFile,"%ld",&tempSignedLong) != 1)
                      finished = TRUE;
                    else
                    {
                      tempSignedLong = limitSLValue (tempSignedLong,-MAX23,MAX23_1);
                      outputSample = tempSignedLong;
                    }
                    break;
                  case MACROLONG32FORMAT :
                    if (fscanf (tempFile,"%ld",&tempSignedLong) != 1)
                      finished = TRUE;
                    else
                      outputSample = tempSignedLong / 256;
                    break;
                  case MACROFLOATFORMAT :
                    if (fscanf (tempFile,"%f",&tempFloat) != 1)
                      finished = TRUE;
                    else
                    {
                      tempFloat = limitFValue (tempFloat,-1.0,1.0);
                      outputSample = limitSLValue (
                        (long)(tempFloat * MAX23),-MAX23,MAX23_1);
                    }
                    break;
                  case MACRODOUBLEFORMAT :
                    if (fscanf (tempFile,"%lf",&tempDouble) != 1)
                      finished = TRUE;
                    else
                    {
                      tempDouble = limitDValue (tempDouble,-1.0,1.0);
                      outputSample = limitSLValue (
                        (long)(tempDouble * MAX23),-MAX23,MAX23_1);
                    }
                    break;
                  default :
                    break;
                }
              }
              else
              {
                switch (macro [macroNumber].dataFormat)
                {
                  case MACROBYTEFORMAT :
                    if (fscanf (tempFile,"%lu",&tempUnsignedLong) != 1)
                      finished = TRUE;
                    else
                    {
                      tempUnsignedChar = limitULValue (tempUnsignedLong,0,MAX8_1);
                      outputSample = (tempUnsignedChar - MAX7) * 65536;
                    }
                    break;
                  case MACROSHORTFORMAT :
                    if (fscanf (tempFile,"%lu",&tempUnsignedLong) != 1)
                      finished = TRUE;
                    else
                    {
                      tempUnsignedShort = limitULValue (tempUnsignedLong,0,MAX16_1);
                      outputSample = (tempUnsignedShort - MAX15) * 256;
                    }
                    break;
                  case MACROLONG24FORMAT :
                    if (fscanf (tempFile,"%lu",&tempUnsignedLong) != 1)
                      finished = TRUE;
                    else
                    {
                      tempUnsignedLong = limitULValue (tempUnsignedLong,0,MAX24_1);
                      outputSample = tempUnsignedLong - MAX23;
                    }
                    break;
                  case MACROLONG32FORMAT :
                    if (fscanf (tempFile,"%lu",&tempUnsignedLong) != 1)
                      finished = TRUE;
                    else
                      outputSample = (tempUnsignedLong - MAX31) / 256;
                    break;
                  case MACROFLOATFORMAT :
                    if (fscanf (tempFile,"%f",&tempFloat) != 1)
                      finished = TRUE;
                    else
                    {
                      tempFloat = limitFValue (tempFloat,0.0,1.0);
                      outputSample = limitSLValue (
                        (long)((tempFloat - 0.5) * MAX24),-MAX23,MAX23_1);
                    }
                    break;
                  case MACRODOUBLEFORMAT :
                    if (fscanf (tempFile,"%lf",&tempDouble) != 1)
                      finished = TRUE;
                    else
                    {
                      tempDouble = limitDValue (tempDouble,0.0,1.0);
                      outputSample = limitSLValue (
                        (long)((tempDouble - 0.5) * MAX24),-MAX23,MAX23_1);
                    }
                    break;
                  default :
                    break;
                }
              }
              if (!finished)
                tempSample.setFrame24Extra (i,j,outputSample);
            }
            if (!finished) i++;
          }
        }
       
        if (!feof (tempFile))
          fl_show_alert ("Warning","Premature end of data while",
          "reading macro output",TRUE);
 
        tempSampleLength = i;
        fclose (tempFile);
        
        // Trim temp sample to right length
        DPSample tempClip;
        tempSample.consolidate ();
        tempSample.removeExtra ();
        tempSample.setRange (tempSampleLength,tempSample.getFrames ());
        tempSample.cut (&tempClip,2);

        if (macro [macroNumber].input == MACRONONEINPUT ||
          macro [macroNumber].input == MACROBUFFERINPUT ||
          (macro [macroNumber].input == MACROEITHERINPUT &&
          !sample->getRangeValid ()))
        {
          // Whole buffer
          sample->cloneData (tempSample);
        }
        else
        {
          // Range only
          DPSample tempClip;
          long rangeStartBak   = sample->getRangeStart ();
          long displayStartBak = sample->getDisplayStart ();
          long displayEndBak   = sample->getDisplayEnd ();
          sample->cut (&tempClip,2);
          sample->setRange (rangeStartBak,rangeStartBak);
          sample->paste (&tempSample,2);
          sample->setDisplay (displayStartBak,displayEndBak);
        }
      }

      updateSample (sample);

      if (remove (tempFileName) == -1)
        fl_show_alert ("Warning","Unable to remove temporary file",
        tempFileName,TRUE);
      delete [] tempFileName;
      return 0;;
    }
    
    case MACROTEMPAIFFMODE :
    case MACROSAVEAIFFMODE :
    {
      // Save AIFF file
      char *filename;
      char *reloadFilename;
      char *tempOutputFilename;
      char *error;
      int rangeOnly;

      if (macro [macroNumber].processMode == MACROSAVEAIFFMODE ||
        macro [macroNumber].input == MACRONONEINPUT)
        rangeOnly = FALSE;
      else if (macro [macroNumber].input == MACROBUFFERINPUT ||
        (macro [macroNumber].input == MACROEITHERINPUT &&
        !sample->getRangeValid ()))
        rangeOnly = FALSE;
      else
        rangeOnly = TRUE;
      
      // Generate temp filenames
      const char *tempDir = fl_get_sample_tempdir (mainForm->sample);
      char *tempNameTemplate;
      char *mktempFileName;

      // Filename used for reload
      tempNameTemplate = new char [strlen (tempDir) + 9];
      strcpy (tempNameTemplate,tempDir);
      if (strlen (tempNameTemplate) > 0 &&
        tempNameTemplate [strlen (tempNameTemplate) - 1] != '/')
        strcat (tempNameTemplate,"/");
      strcat (tempNameTemplate,"aXXXXXX");
      mktempFileName = mktemp (tempNameTemplate);
      if (!mktempFileName)
      {
        delete [] tempNameTemplate;
        return "Unable to create temporary filename";
      }
      reloadFilename = new char [strlen (tempNameTemplate) + 1];
      strcpy (reloadFilename,tempNameTemplate);

      // Filename used for temporary file output
      strcpy (tempNameTemplate,tempDir);
      if (strlen (tempNameTemplate) > 0 &&
        tempNameTemplate [strlen (tempNameTemplate) - 1] != '/')
        strcat (tempNameTemplate,"/");
      strcat (tempNameTemplate,"bXXXXXX");
      mktempFileName = mktemp (tempNameTemplate);
      if (!mktempFileName)
      {
        delete [] reloadFilename;
        delete [] tempNameTemplate;
        return "Unable to create temporary filename (for temp output)";
      }
      tempOutputFilename = new char [strlen (tempNameTemplate) + 1];
      strcpy (tempOutputFilename,tempNameTemplate);

      // Filename used for saving
      if (macro [macroNumber].processMode == MACROSAVEAIFFMODE)
      {
        if (sample->getFilename ())
        {
          filename = new char [strlen (sample->getFilename ()) + 1];
          strcpy (filename,sample->getFilename ());
        }
        else
        {
          filename = 0;
        }
      }
      else
      {
        strcpy (tempNameTemplate,tempDir);
        if (strlen (tempNameTemplate) > 0 &&
          tempNameTemplate [strlen (tempNameTemplate) - 1] != '/')
          strcat (tempNameTemplate,"/");
        strcat (tempNameTemplate,"cXXXXXX");
        mktempFileName = mktemp (tempNameTemplate);
        if (!mktempFileName)
        {
          delete [] reloadFilename;
          delete [] tempOutputFilename;
          delete [] tempNameTemplate;
          return "Unable to create temporary filename (for reload)";
        }
        filename = new char [strlen (tempNameTemplate) + 1];
        strcpy (filename,tempNameTemplate);
      }
      
      delete [] tempNameTemplate;
      
      if (!filename)
      {
        char *tempFilename;
        fl_use_fselector (1);
        if (first1)
        {
          tempFilename = (char *) fl_show_fselector (
            "Please enter the filename to save",".","*.aiff",0);
          first1 = 0;
        }
        else
        {
          tempFilename = (char *) fl_show_fselector (
            "Please enter the filename to save",0,0,0);
        }

        updateSampleFormatCompression (sample);

        if (tempFilename)
        {
          int irixFile = open (tempFilename,O_RDONLY);
          if (irixFile != -1)
          {
            close (irixFile);
            if (!fl_show_question_old
              (tempFilename,"exists already - overwrite ?",0))
              ENDAIFFMACRO
          }
          filename = new char [strlen (tempFilename) + 1];
          strcpy (filename,tempFilename);
        }
        else
        {
          ENDAIFFMACRO
        }
      }

      if (error = sample->saveAIFF
      (filename,rangeOnly,fl_get_sample_copysave (mainForm->sample),FALSE))
      {
        fl_show_alert ("Warning - Could not save file",
          filename,error,TRUE);
        ENDAIFFMACRO
      }
      else
      {
        sample->consolidate ();
        sample->removeExtra ();
      }

      // Execute command
      char *command = macro [macroNumber].command;
      char *tempCommand;
      int tempCommandLen;
      int reloadUsed    = FALSE;
      int fReplacements = 0;
      int gReplacements = 0;
      int percentFound;
      int i;
      int j;
      
      // Count number of replacements needed
      // If %g found reloadUsed = TRUE;
      percentFound = FALSE;
      for (i=0; i<strlen (command); i++)
      {
        if (command [i] == '%')
        {
          percentFound = TRUE;
        }
        else
        {
          if (command [i] == 'f' && percentFound)
            fReplacements++;
          else if (command [i] == 'g' && percentFound)
            gReplacements++;
          percentFound = FALSE;
        }
      }
      
      if (gReplacements) reloadUsed = TRUE;

      tempCommandLen = strlen (command) +
        (fReplacements * (strlen (filename) - 2)) +
        (gReplacements * (strlen (reloadFilename) - 2)) +
        strlen (tempOutputFilename) + 3;

      tempCommand = new char [tempCommandLen];

      // Replace %f in command with filename
      // Replace %g in command with reload filename
      
      j = 0;
      percentFound = FALSE;
      for (i=0; i<strlen (command); i++)
      {
        if (command [i] == '%')
        {
          if (percentFound) tempCommand [j++] = '%';
          percentFound = TRUE;
        }
        else
        {
          if (command [i] == 'f' && percentFound)
          {
            tempCommand [j] = 0;
            strcat (tempCommand,filename);
            j = strlen (tempCommand);
          }
          else if (command [i] == 'g' && percentFound)
          {
            tempCommand [j] = 0;
            strcat (tempCommand,reloadFilename);
            j = strlen (tempCommand);
          }
          else
          {
            if (percentFound) tempCommand [j++] = '%';
            tempCommand [j++] = command [i];
          }
          percentFound = FALSE;
        }
      }
      if (percentFound) tempCommand [j++] = '%';
      tempCommand [j] = 0;

      if (macro [macroNumber].output == MACRODIALOGOUTPUT)
      {
        strcat (tempCommand," >");
        strcat (tempCommand,tempOutputFilename);
      }

      // New command in tempCommand
      system (tempCommand);
      delete [] tempCommand;

      if (macro [macroNumber].output == MACRODIALOGOUTPUT)
      {
        if (!fl_load_browser (macrooutputForm->browser,tempOutputFilename))
        {
          REMOVETEMPFILES
          delete [] reloadFilename;
          delete [] tempOutputFilename;
          if (macro [macroNumber].processMode == MACROTEMPAIFFMODE)
            delete [] filename;
          return "Cannot load temp file into browser";
        }
        fl_show_form (macrooutputForm->macrooutputForm,
          FL_PLACE_MOUSE | FL_FREE_SIZE,FL_FULLBORDER,
          "Command Output");
      }
      else if (macro [macroNumber].output == MACROBUFFEROUTPUT)
      {
        // Reload sample or new temporary filename (as reloadUsed)

        char *reloadName;
        if (reloadUsed)
          reloadName = reloadFilename;
        else
          reloadName = filename;

        // Possibly as range only so be careful !! (as rangeOnly)

        DPSample tempSample;

        if (error = tempSample.loadAIFF (reloadName))
        {
          fl_show_alert ("Warning - Could not reload file",
            reloadName,error,TRUE);
          REMOVETEMPFILES
          ENDAIFFMACRO
        }          

        if (!rangeOnly)
        {
          tempSample.setFilename (sample->getFilename ());
          *sample = tempSample;
        }
        else
        {
          DPSample tempClip;
          long rangeStartBak   = sample->getRangeStart ();
          long displayStartBak = sample->getDisplayStart ();
          long displayEndBak   = sample->getDisplayEnd ();
          sample->cut (&tempClip,2);
          sample->setRange (rangeStartBak,rangeStartBak);
          sample->paste (&tempSample,2);
          sample->setDisplay (displayStartBak,displayEndBak);
        }

        updateSample (sample);
      }

      REMOVETEMPFILES
      delete [] reloadFilename;
      delete [] tempOutputFilename;
      delete [] filename;
      return 0;
    }

    default :
    {
      return "Unknown macro process mode";
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION findNonSpace
---------------------------------------------------------------------------*/
int findNonSpace (const char *name)
{
  int i;
  if (!name) return 0;
  i = 0;
  while (name [i] && isspace (name [i])) i++;
  return i;
}

/***************************************************************************/
