/* vi:set ts=8 sts=0 sw=8:
 * $Id: misc.c,v 1.19 2000/02/19 22:58:06 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <glib.h>
#include "main.h"
#include "file.h"
#include "misc.h"
#include "prefs.h"
#include "gtkefilesel.h"
#include "xpm/tb_blank.xpm"


/*** global function definitions ***/
char *
handle_null_string(char *s)
{
	return (s) ? s : "(null)";
} /* handle_null_string */


char *
my_tempname(void)
{
	char *fname;

	/*
	 * is tempnam() portable?  if so, we could do:
	 *	fname = tempnam(tmpdir, ".gnp");
	 * but i don't think it's ANSI compliant...
	 */
	fname = g_new(char, 255);
	g_snprintf(fname, 255, "%s/.gnp%d%d",
		   prefs.tmpdir, (int)time(NULL), (int)getpid());

	return fname;
} /* my_tempnam() */


#ifndef GTK_HAVE_FEATURES_1_1_0
char *
my_basename(const char *name)
{
	register gchar *base;

	if (name == NULL)
		return NULL;

	base = strrchr(name, '/');
	if (base)
		return base + 1;

	return (char *)name;
} /* my_basename */


char *
my_dirname(const char *fname)
{
	register char *base;
	register unsigned len;

	g_return_val_if_fail(fname != NULL, NULL);

	if (!(base = strrchr(fname, '/')))
		return g_strdup(".");

	while (base > fname && *base == '/')
		base--;

	len = (unsigned)(1 + (base - fname));

	base = g_new(char, len + 1);
	g_memmove(base, fname, len);
	base[len] = 0;

	return base;
}
#endif	/* gtk 1.0.x */


GtkWidget *
misc_open_dialog_create(void *wdata, char *title,
			void (*fsel_ok)(GtkWidget *, gpointer),
			void (*fsel_cancel)(GtkWidget *, gpointer),
			void (*fsel_destroy)(GtkWidget *, gpointer))
{
	GtkWidget *fs;

	fs = gtk_efilesel_new(title);
	gtk_efilesel_hide_fileop_buttons(GTK_EFILESEL(fs));
	(void)gtk_signal_connect(GTK_OBJECT(fs),
			"destroy", GTK_SIGNAL_FUNC(fsel_destroy), wdata);
	(void)gtk_signal_connect(GTK_OBJECT(GTK_EFILESEL(fs)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(fsel_ok), wdata);
	(void)gtk_signal_connect(GTK_OBJECT(GTK_EFILESEL(fs)->cancel_button),
			"clicked", GTK_SIGNAL_FUNC(fsel_cancel), wdata);
	return fs;
} /* misc_open_dialog_create */


/*
 * PUBLIC: misc_new_pixmap
 *
 * taken and modified from testgtk.c.  If the initial pathname isn't found,
 * then a retry is done using the subdirectory "xpm" from the current
 * directory.  If that isn't found, then a blank pixmap is used.
 *
 * If pixmap_p and mask_p are NOT null, then don't use a blank pixmap, and
 * don't make the final call to gtk_pixmap_new().  Instead, return the
 * corresponding data filled in from gdk_pixmap_create_from_xpm_d().
 */
GtkWidget *
misc_new_pixmap(char *fname, GtkWidget *parent, GdkPixmap **pixmap_p,
		GdkBitmap **mask_p)
{
	extern char *pkgdatadir;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	char *buf;
	int len;

	g_assert((pixmap_p && mask_p) || (!pixmap_p && !mask_p));

	len = strlen(pkgdatadir) + strlen(fname) + 6;
	buf = g_new(char, len);
	g_snprintf(buf, len, "%s/xpm/%s", pkgdatadir, fname);
	if (!file_exist(buf)) {
		g_warning("'%s' not found", buf);
		g_snprintf(buf, len, "xpm/%s", fname);
		g_warning("attempting '%s'", buf);
		GNPDBG_TOOLBAR(("new_pixmap: buf = '%s'\n", buf));
		if (!file_exist(buf) && !pixmap_p && !mask_p) {
			g_warning("'%s' not found (using blank image)", buf);
			pixmap = gdk_pixmap_create_from_xpm_d(
					parent->window,
					&mask,
					&parent->style->bg[GTK_STATE_NORMAL],
					tb_blank);
			goto done;
		}
	}

	pixmap = gdk_pixmap_create_from_xpm(
			parent->window,
			&mask,
			&parent->style->bg[GTK_STATE_NORMAL],
			buf);
done:
	g_free(buf);
	if (!pixmap_p && !mask_p)
		return gtk_pixmap_new(pixmap, mask);

	*pixmap_p = pixmap;
	*mask_p = mask;
	return NULL;
} /* misc_new_pixmap */



/*
 * PUBLIC: misc_button_new_w_label
 *
 * convenience routine for creating button with label
 */
GtkWidget *
misc_button_new_w_label(char *labelname, char *gnomename,
			GtkSignalFunc cbfunc,
			gpointer cbdata,
			GtkWidget *box, int opts, int pad)
{
	GtkWidget *b;

	g_assert((opts & PACK_START) || (opts & PACK_END));

#ifdef USE_GNOME
	if (gnomename)
		b = gnome_stock_button(gnomename);
	else
#endif
	b = gtk_button_new_with_label(labelname);
	if (cbfunc) {
		if (opts & CONNECT_OBJ)
			(void)gtk_signal_connect_object(GTK_OBJECT(b),
							"clicked",
							GTK_SIGNAL_FUNC(cbfunc),
							cbdata);
		else
			(void)gtk_signal_connect(GTK_OBJECT(b), "clicked",
						 GTK_SIGNAL_FUNC(cbfunc),
						 cbdata);
	}

	if (opts & PACK_START)
		gtk_box_pack_start(GTK_BOX(box), b,
				   opts & PACK_EXPAND, opts & PACK_FILL, pad);
	else if (opts & PACK_END)
		gtk_box_pack_end(GTK_BOX(box), b,
				 opts & PACK_EXPAND, opts & PACK_FILL, pad);

	if (opts & CANCEL_DEFAULT)
		GTK_WIDGET_SET_FLAGS(b, GTK_CAN_DEFAULT);

	if (opts & GRAB_DEFAULT) {
		g_assert(opts & CANCEL_DEFAULT);
		gtk_widget_grab_default(b);
	}

#ifdef GTK_HAVE_FEATURES_1_1_0
	if (opts & NO_RELIEF)
		gtk_button_set_relief(GTK_BUTTON(b), GTK_RELIEF_NONE);
#endif

	if (opts & SHOW_BUTTON)
		gtk_widget_show(b);

	return b;
} /* misc_button_new_w_label */


/*
 * PUBLIC: misc_entry_w_label_create
 *
 * convenience routine for creating a text entry widget and label
 */
GtkWidget *
misc_entry_w_label_create(
	GtkWidget *parent,
	char *labeltext,
	char *entrytext,
	GtkWidget *table,
	int lft1, int rht1, int top1, int bot1,
	int lft2, int rht2, int top2, int bot2)
{
	GtkWidget *tmp, *hbox = NULL;

	tmp = gtk_label_new(labeltext);
	if (table) {
		gtk_misc_set_alignment(GTK_MISC(tmp), 0, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table),
					  tmp, lft1, rht1, top1, bot1);
	} else {
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(parent), hbox, FALSE, FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	}

	tmp = gtk_entry_new();
	if (table) {
		gtk_table_attach_defaults(GTK_TABLE(table),
					  tmp, lft2, rht2, top2, bot2);
	} else {
		gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	}

	if (entrytext)
		gtk_entry_set_text(GTK_ENTRY(tmp), entrytext);

	return tmp;
} /* misc_entry_w_label_create */


/*
 * PUBLIC: misc_show_and_raise
 *
 * If the widget exists, then show it if it's hidden, and then raise it.
 */
gboolean
misc_show_and_raise(GtkWidget *wgt)
{
	if (wgt && GTK_IS_WIDGET(wgt)) {
		if (!GTK_WIDGET_VISIBLE(wgt))
			gtk_widget_show(wgt);
		gdk_window_raise(wgt->window);
		return TRUE;
	}

	return FALSE;
} /* misc_show_and_raise */


#ifdef USE_FILEINFO
/*
 * PUBLIC: ttoa
 *
 * converts time_t to an ascii string.  non-reentrant because ctime() is
 * non-reentrant.
 */
char *
ttoa(time_t curtime, bool_t append_nl)
{
	char *timestr;

	timestr = ctime(&curtime);
	if (!append_nl) {
		if (timestr[strlen(timestr) - 1] == '\n')
			timestr[strlen(timestr) - 1] = '\0';
	}

	return timestr;
} /* ttoa */


/*
 * PUBLIC: ltoa
 *
 * converts a (long) number to an ascii string.  non-reentrant because i use a
 * static local variable.
 */
char *
ltoa(long num)
{
	static char str[MAXPATH];

	g_snprintf(str, MAXPATH, "%ld", num);
	return str;
} /* ltoa */
#endif	/* USE_FILEINFO */


#ifdef USE_SEARCH
/*
 * PUBLIC: strcasestr
 *
 * strcasestr() is basically the code for strstr() taken from glibc.
 * toupper() statements are used where needed so that comparisons between
 * differing case no longer matter.  the original author's comment follows:
 */
/*
 * My personal strstr() implementation that beats most other algorithms.
 * Until someone tells me otherwise, I assume that this is the
 * fastest implementation of strstr() in C.
 * I deliberately chose not to comment it.  You should have at least
 * as much fun trying to understand it, as I had to write it :-).
 *
 * Stephen R. van den Berg, berg@pool.informatik.rwth-aachen.de	*/
typedef unsigned chartype;
char *
strcasestr(const char *phaystack, const char *pneedle)
{
	register const unsigned char *haystack, *needle;
	register chartype b, c;

	haystack = (const unsigned char *)phaystack;
	needle = (const unsigned char *)pneedle;

	b = *needle;
	if (b != '\0')
	{
		haystack--;	/* possible ANSI violation */
		do
		{
			c = *++haystack;
			if (c == '\0')
				goto ret0;
		}
		while (toupper(c) != toupper(b));

		c = *++needle;
		if (c == '\0')
			goto foundneedle;
		++needle;
		goto jin;

		for (;;)
		{
			register chartype a;
			register const unsigned char *rhaystack, *rneedle;

			do
			{
				a = *++haystack;
				if (a == '\0')
					goto ret0;
				if (toupper(a) == toupper(b))
					break;
				a = *++haystack;
				if (a == '\0')
					goto ret0;
shloop:				if (0) goto shloop;	/* avoid ANSI warning */
			}
			while (toupper(a) != toupper(b));

jin:	  a = *++haystack;
			if (a == '\0')
				goto ret0;

			if (toupper(a) != toupper(c))
				goto shloop;

			rhaystack = haystack-- + 1;
			rneedle = needle;
			a = *rneedle;

			if (toupper(*rhaystack) == toupper(a))
				do
				{
					if (a == '\0')
						goto foundneedle;
					++rhaystack;
					a = *++needle;
					if (toupper(*rhaystack) != toupper(a))
						break;
					if (a == '\0')
						goto foundneedle;
					++rhaystack;
					a = *++needle;
				}
				while (toupper(*rhaystack) == toupper(a));

			needle = rneedle;/* took the register-poor approach */

			if (a == '\0')
				break;
		}
	}
foundneedle:
	return (char*) haystack;
ret0:
	return 0;
} /* strcasestr */
#endif	/* USE_SEARCH */


/*
 * PUBLIC: strrepl
 *
 * given "orig" string, look for substring "pat", and if found, replace it with
 * string "repl".  returns a newly allocated buffer if "pat" was found, NULL if
 * not.
 */
char *
strrepl(char *orig, char *pat, char *repl)
{
	int p1, p2, p3;
	char *needle;
	char *buf;

	if (!orig || !pat || ((needle = strstr(orig, pat)) == NULL))
		return NULL;

	p1 = (needle - orig);
	p2 = (repl) ? strlen(repl) : 0;
	p3 = strlen(orig) - (p1 + strlen(pat));

	buf = g_new(char, p1 + p2 + p3 + 1);
	strncpy(buf, orig, p1);
	buf[p1] = '\0';
	if (repl)
		strcat(buf, repl);
	strcat(buf, needle + strlen(pat));

	return buf;
} /* strrepl */


/* the end */
