/* vi:set ts=8 sts=0 sw=8:
 * $Id: srcctrl.c,v 1.11 2000/02/19 07:47:06 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include "main.h"
#ifdef USE_SOURCE_CTRL
# ifdef APP_GNP
#include "doc.h"
#include "dialog.h"
#include "prefs.h"
#include "file.h"
#include "misc.h"
#include "srcctrl.h"

#include "gnpintl.h"


/*** global variables ***/
char *scc_cmd_desc[SCC_NUM_CMDS] = {
	N_("Check In (locked)"),
	N_("Check In (unlocked)"),
	N_("Check In (initial)"),
	N_("Check Out (unlocked)"),
	N_("Check Out (locked)"),
	N_("Undo Check Out"),
	N_("Revision History")
};


/*** local function prototypes ***/
static void scc_menu_cb_common(int cmdtype, gpointer cbdata);
static void scc_dlg_ok(GtkWidget *wgt, gpointer cbdata);
static void scc_dlg_cancel(GtkWidget *wgt, gpointer cbdata);
static void scc_dlg_destroy(GtkWidget *wgt, gpointer cbdata);


/*** global function definitions ***/

#if defined(GTK_HAVE_FEATURES_1_1_0) && !defined(USE_GNOME)
void
scc_menu_cb(GtkWidget *wgt, gpointer cbdata, guint cbaction)
{
	scc_menu_cb_common((int)cbaction, cbdata);
}
#endif

#if defined(USE_GNOME) || !defined(GTK_HAVE_FEATURES_1_1_0)
void
scc_checkin_locked(GtkWidget *widget, gpointer cbdata)
{
	scc_menu_cb_common(SCC_CI_LOCK, cbdata);
}


void
scc_checkin_unlocked(GtkWidget *widget, gpointer cbdata)
{
	scc_menu_cb_common(SCC_CI_UNLOCK, cbdata);
}


void
scc_checkin_initial(GtkWidget *widget, gpointer cbdata)
{
	scc_menu_cb_common(SCC_CI_INIT, cbdata);
}


void
scc_checkout_unlocked(GtkWidget *widget, gpointer cbdata)
{
	scc_menu_cb_common(SCC_CO_UNLOCK, cbdata);
}


void
scc_checkout_locked(GtkWidget *widget, gpointer cbdata)
{
	scc_menu_cb_common(SCC_CO_LOCK, cbdata);
}


void
scc_checkout_undo(GtkWidget *widget, gpointer cbdata)
{
	scc_menu_cb_common(SCC_CO_UNDO, cbdata);
}


void
scc_rev_history(GtkWidget *widget, gpointer cbdata)
{
	scc_menu_cb_common(SCC_REVHIST, cbdata);
}
#endif	/* defined(USE_GNOME) || !defined(GTK_HAVE_FEATURES_1_1_0) */


/*** local function definitions ***/
/*
 *  --- TITLE: Check in file ---
 *  0 "Check In (locked)"
 *  1
 *  2 Filename (editable)      : /tmp/newfile.txt
 *  3 Command (editable)       : sccs delta -y$s $F
 *  4 Comments (editable)      : <empty>
 *
 *
 *  --- Check out file ---
 *  0 Filename (editable)      : /tmp/blahfile.doc
 *  1 Command (editable)       : sccs delget $F
 */
typedef struct scc_dlg_s {
	GtkWidget *toplev;	/* top level dialog widget */
	GtkWidget *fname;	/* entry widget.  contains filename */
	GtkWidget *cmd;		/* entry widget.  source code control cmd */
	GtkWidget *comment;	/* entry widget.  checkin comment, if any */
	int cmdtype;
	win_t *w;
} scc_dlg_t;

/*
 * create dialog box to confirm command.  also allows user to change the
 * entries.
 */
static void
scc_menu_cb_common(int cmdtype, gpointer cbdata)
{
	GtkWidget *hbox, *vbox, *tmp, *table;
	scc_dlg_t *sdlg;
	char *fullpath;
	win_t *w = (win_t *)cbdata;
	doc_t *d = DOC_CURRENT(w);

	if (d->changed &&
	    (cmdtype == SCC_CI_LOCK || cmdtype == SCC_CI_UNLOCK ||
	     cmdtype == SCC_CI_INIT || cmdtype == SCC_CO_UNLOCK ||
	     cmdtype == SCC_CO_LOCK || cmdtype == SCC_CO_UNDO)) {

		do_dialog_ok(" Document not saved ",
			     " You must save the file before\n"
			     " checking it in/out. ");
		return;
	}

	sdlg = g_new(scc_dlg_t, 1);
	sdlg->cmdtype = cmdtype;
	sdlg->w = w;
	sdlg->toplev = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(sdlg->toplev),
			     _("gnotepad+ : Source Control Options"));
	(void)gtk_signal_connect(GTK_OBJECT(sdlg->toplev), "destroy",
				 GTK_SIGNAL_FUNC(scc_dlg_destroy), sdlg);
	gtk_container_border_width(GTK_CONTAINER(sdlg->toplev), 5);

	vbox = GTK_DIALOG(sdlg->toplev)->vbox;
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);

	table = gtk_table_new(6, 2, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

	/*  0 "Check In (locked)"
	 *  1					*/
	tmp = gtk_label_new(gettext(scc_cmd_desc[cmdtype]));
	gtk_table_attach_defaults(GTK_TABLE(table), tmp, 0, 2, 0, 1);
	tmp = gtk_label_new(" ");
	gtk_table_attach_defaults(GTK_TABLE(table), tmp, 0, 2, 1, 2);

	/*  2 Filename (editable)      : /tmp/newfile.txt */
	fullpath = file_full_pathname_make(d->fname);
	sdlg->fname = misc_entry_w_label_create(vbox, _("Filename"), fullpath,
						table, 0, 1, 2, 3, 1, 2, 2, 3);
	g_free(fullpath);

	/*  3 Command (editable)       : sccs delta -y$s $F */
	sdlg->cmd = misc_entry_w_label_create(vbox, _("Command"),
					prefs.scc[prefs.scc_selected][cmdtype],
					table, 0, 1, 3, 4, 1, 2, 3, 4);

	/*  4 Comments (editable)      : <empty> */
	if (cmdtype == SCC_CI_LOCK || cmdtype == SCC_CI_UNLOCK ||
	    cmdtype == SCC_CI_INIT) {
		sdlg->comment = misc_entry_w_label_create(vbox, _("Comments"),
					NULL, table, 0, 1, 4, 5, 1, 2, 4, 5);
	} else
		sdlg->comment = NULL;

	tmp = gtk_label_new(" ");
	gtk_table_attach_defaults(GTK_TABLE(table), tmp, 0, 2, 5, 6);

	/* the buttons */
	hbox = GTK_DIALOG(sdlg->toplev)->action_area;
	(void)misc_button_new_w_label(_("Ok"), GNOME_STOCK_BUTTON_OK,
				      GTK_SIGNAL_FUNC(scc_dlg_ok), sdlg, hbox,
				      PACK_START | PACK_EXPAND | PACK_FILL |
				      CANCEL_DEFAULT, 0);
	(void)misc_button_new_w_label(_("Cancel"), GNOME_STOCK_BUTTON_CANCEL,
				      GTK_SIGNAL_FUNC(scc_dlg_cancel),
				      sdlg, hbox,
				      PACK_START | PACK_EXPAND | PACK_FILL |
				      CANCEL_DEFAULT | GRAB_DEFAULT, 0);
	gtk_widget_show_all(sdlg->toplev);
} /* scc_menu_cb_common */


static void
scc_dlg_ok(GtkWidget *wgt, gpointer cbdata)
{
	scc_dlg_t *sdlg = (scc_dlg_t *)cbdata;
	char *fname, *cmnt, *tmp1, *tmp2;
	doc_t *d = DOC_CURRENT(sdlg->w);
	char *cmd = NULL, *outfile = NULL;
	int err = ESUCCESS;

	fname = gtk_entry_get_text(GTK_ENTRY(sdlg->fname));
	if (!fname || strcmp(fname, "") == 0)
		goto out;

	cmd = gtk_entry_get_text(GTK_ENTRY(sdlg->cmd));
	if (!cmd || strcmp(cmd, "") == 0)
		goto out;

	tmp1 = g_strdup(cmd);
	tmp2 = strrepl(tmp1, "$F", fname);
	g_free(tmp1);

	if (sdlg->comment) {
		cmnt = gtk_entry_get_text(GTK_ENTRY(sdlg->comment));

		if (cmnt && strcmp(cmnt, "")) {
			tmp1 = g_new(char, strlen(cmnt) + 3);
			g_snprintf(tmp1, strlen(cmnt) + 3, "\"%s\"", cmnt);
			cmd = strrepl(tmp2, "$s", tmp1);
			g_free(tmp1);
		} else {
			cmd = strrepl(tmp2, "$s", "\"No comment entered\"");
			g_free(tmp2);
		}
	} else
		cmd = tmp2;

	outfile = my_tempname();
	cmd = realloc(cmd, strlen(cmd) + strlen(outfile) + 3);
	strcat(cmd, " >");
	strcat(cmd, outfile);
	GNPDBG_SRCCTRL(("scc_dlg_ok: cmd = '%s'\n", cmd));
	err = system(cmd);

out:
	/* if there was no error, unload and reload file as needed */
	if (!err) {
		switch (sdlg->cmdtype) {
		case SCC_CI_LOCK:
		case SCC_CI_INIT:
		case SCC_CO_UNLOCK:
		case SCC_CO_LOCK:
		case SCC_CO_UNDO:
			doc_unload(d);
			doc_load(d);
			doc_info_label_update(sdlg->w);
			break;
		case SCC_CI_UNLOCK:
			if (file_exist(d->fname)) {
				doc_unload(d);
				doc_load(d);
			} else {
				doc_flags_set(d, FORCE_READONLY | NEVER_UNLOAD);
			}
			doc_info_label_update(sdlg->w);
			break;
		case SCC_REVHIST:
			doc_new(sdlg->w, outfile, DocText,
				DO_LOAD | UPDATE_TITLE | FORCE_READONLY |
					  NEVER_UNLOAD);
			break;
		}
	}

	/* it is now safe to delete the temporary output file */
	(void)unlink(outfile);
	g_free(outfile);

	gtk_widget_destroy(sdlg->toplev);

	if (err) {
		tmp1 = g_new(char, strlen(cmd) + 64);
		g_snprintf(tmp1, strlen(cmd) + 64,
			   "'%s' failed\nRet = %d, errno = %d",
			   cmd, err, errno);
		do_dialog_error("Error in command", tmp1);
		g_free(tmp1);
	}

	g_free(cmd);
} /* scc_dlg_ok */


static void
scc_dlg_cancel(GtkWidget *wgt, gpointer cbdata)
{
	scc_dlg_t *sdlg = (scc_dlg_t *)cbdata;

	gtk_widget_destroy(sdlg->toplev);
} /* scc_dlg_cancel */


static void
scc_dlg_destroy(GtkWidget *wgt, gpointer cbdata)
{
	scc_dlg_t *sdlg = (scc_dlg_t *)cbdata;

	g_free(sdlg);
} /* scc_dlg_destroy */


# endif	/* APP_GNP */
#endif	/* USE_SOURCE_CTRL */

/* the end */
