/* Implements "lispish" nested lists of strings.
 *
 * gsumi version 0.5
 *
 * Copyright 1997 Owen Taylor <owt1@cornell.edu>
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "lispish.h"

#define BUFSIZE 256

/* create a new list node */
Node *
node_list_new()
{
  Node *node = malloc(sizeof(Node));
  assert(node);
  node->type = NODE_LIST;
  node->next = NULL;
  node->c.l = NULL;

  return node;
}

/* create a new string node */
Node *
node_string_new(char *str)
{
  Node *node = malloc(sizeof(Node));
  assert(node);
  node->type = NODE_STRING;
  node->next = NULL;
  if (str)
    {
      node->c.s = malloc(sizeof(char)*(1+strlen(str)));
      assert(node->c.s);
      strcpy(node->c.s,str);
    }
  else
    node->c.s = 0;

  return node;
}

/* append a node onto a list */
void
node_list_append(Node *list, Node *new)
{
  Node *n;

  assert(list);
  assert(list->type == NODE_LIST);
  assert(new);

  n = list->c.l;
  if (!n)
    {
      list->c.l = new;
    }
  else
    {
      while (n->next) 
	n = n->next;
      n->next = new;
    }
}

/* make a two element list from a string and a double */
Node *
node_pair_double(char *key, double val)
{
  char buffer[20];
  Node *node = node_list_new();

  sprintf(buffer,"%g",val);
  node_list_append(node,node_string_new(key));
  node_list_append(node,node_string_new(buffer));

  return node;
}

/* make a two element list from two strings */
Node *
node_pair_string(char *key, char *val)
{
  Node *node = node_list_new();
  node_list_append(node,node_string_new(key));
  node_list_append(node,node_string_new(val));

  return node;
}

/* does real work of matching list beginnings. 
   Used by node_list_matches, node_list_find_match, node_list_remove_match */
static int
vnode_list_matches(Node *node, va_list args)
{
  int matched = 0;
  Node *n;

  assert(node);

  n = node->c.l;

  if (node->type == NODE_LIST)
    while (n)
      {
	char *str;
	int is_case;
	str = va_arg(args,char *);
	if (!str) 
	  {
	    matched = 1;
	    break;
	  }
	is_case = va_arg(args,int);
	
	if (n->type != NODE_STRING) break;
	
	if (is_case)
	  {
	    if (strcasecmp(str,n->c.s))
	      break;
	  }
	else
	  {
	    if (strcmp(str,n->c.s))
	      break;
	  }
	n = n->next;
      }
  return matched;
}

/* check if node is a list with specified starting sequence.
   node_list_matches(Node *node, char *key1, int is_case1, ... , NULL)
   where is_case[n] specifies whether the n'th argument should be
   matched case insensitively */
int
node_list_matches(Node *node, ...)
{
  va_list args;
  int matched;

  assert(node);

  va_start (args, node);

  matched = vnode_list_matches(node,args);

  va_end (args);

  return matched;
}


/* finds a sublist with specified starting sequence (as in node_list_matches),
   removes it and returns it. Returns NULL if no match */
Node *
node_list_remove_match(Node *list, ...)
{
  va_list args;
  Node *nn, *n;

  assert(list);
  assert(list->type == NODE_LIST);

  nn = list->c.l;
  n = 0;

  while (nn)
    {
      va_start (args, list);
      if (vnode_list_matches(nn,args))
	{
	  if (n)
	    n->next = nn->next;
	  else
	    list->c.l->next = nn->next;
	  va_end (args);
	  break;
	}
      va_end (args);
      n = nn;
      nn = nn->next;
    }

  return nn;
}

/* finds a sublist with specified starting sequence (as in node_list_matches),
   and returns it. Returns NULL if no match */
Node *
node_list_find_match(Node *list, ...)
{
  va_list args;

  Node *n;

  assert(list);
  assert(list->type == NODE_LIST);

  n = list->c.l;
  
  while (n)
    {
      va_start (args, list);
      if (vnode_list_matches(n,args))
	{
	  if (n)
	    n->next = n->next;
	  else
	    list->c.l->next = n->next;
	  va_end (args);
	  break;
	}
      va_end (args);
      n = n->next;
    }

  return n;
}
/* finds a sublist which begins with specified key, removes it 
   and returns it (returns NULL if no match) */
Node *
node_list_remove_pair(Node *list, char *key)
{
  assert (key);
  return node_list_remove_match(list, key, 0, NULL);
}

/* finds two-element sublist which begins with specified key, returns
   the string value of the second element (returns NULL if no match,
   or matched pair has a string as second element) */
char *
node_list_lookup_pair(Node *list, char *key)
{
  Node *n;

  assert(list);
  assert(list->type == NODE_LIST);
  assert(key);

  n = list->c.l;
  
  while (n)
    {
      if (node_list_matches(n,key,0,NULL))
	{
	  if ((n->c.l->next) &&
	      (n->c.l->next->type == NODE_STRING))
	    return n->c.l->next->c.s;
	  else
	    return NULL;
	}
      n = n->next;
    }

  return NULL;
}

/* frees a node */
void
node_free(Node *node)
{
  Node *n,*nn;

  assert(node);
  
  if (node->type == NODE_STRING)
    if (node->c.s)
      free(node->c.s);
  else if (node->type == NODE_LIST)
    {
      nn = node->c.l;
      while (nn)
	{
	  n = nn;
	  nn = n->next;
	  free(n);
	}
    }
  free(node);
}

/* Reads a string element from a file String elements are split on
   whitespace unless quoted, in which case they can contain
   whitespace, and escaped quotes and backslashes. All escaped
   characters are de-escaped on reading */
Node *
node_string_from_file(FILE *file) 
{ 
  int c; 
  int escape; 
  char buf[BUFSIZE]; int i; int len;
  
  Node *node = node_string_new("");
  len = 0;
  
  c = getc(file);
  
  while (c != EOF && isspace(c))
    c = getc(file);

  if (c == EOF)
    {
      node_free(node);
      return NULL;
    }

  if (c == '"')
    {
      c = getc(file);
      i = 0;
      escape = 0;
      while (!((c == EOF) || (!escape && c == '"')))
	{
	  if (!escape && c == '\\')
	    {
	      escape = 1;
	    }
	  else
	    {
	      escape = 0;
	      buf[i++] = c;
	      if (i==BUFSIZE-1)
		{
		  buf[i] = 0;
		  node->c.s = realloc(node->c.s,
					     sizeof(char)*(i+len+1));
		  strcat(node->c.s,buf);
		  len += i;
		  i = 0;
		}
	    }
	  c = getc(file);
	}
      if (c == EOF)
	{
	  fprintf(stderr,"Unterminated string\n");
	  node_free(node);
	  return NULL;
	}
    }
  else
    {
      i = 0;
      while (!((c == EOF) || isspace(c) || c == '(' || c == ')'))
	{
	  buf[i++] = c;
	  if (i==BUFSIZE-1)
	    {
	      buf[i] = 0;
	      node->c.s = realloc(node->c.s,
					 sizeof(char)*(i+len+1));
	      strcat(node->c.s,buf);
	      len += i;
	      i = 0;
	    }
	  c = getc(file);
	}
      if (c != EOF)
	ungetc(c,file);
    }

  buf[i] = 0;
  node->c.s = realloc(node->c.s,
			     sizeof(char)*(i+len+1));
  strcat(node->c.s,buf);

  return node;
}

/* reads a list node from a text representation in file. Returns NULL on
   error. Strings area as described in node_string_from_file. if
   is_toplevel, ends on EOF, otherwise ends on ')'. Removes trailing
   ')' */
Node *
node_list_from_file(FILE *file, int is_toplevel)
{
  int c;
  Node *node = node_list_new();
  Node *n = NULL;
  
  c = getc(file);
  while (c != EOF && !(c == ')'))
    {
      if (!isspace(c))
	{
	  Node *nn;
	  if (c == '(')
	    nn = node_list_from_file(file,0);
	  else
	    {
	      ungetc(c,file);
	      nn = node_string_from_file(file);
	    }
	  if (!nn)		/* propogate errors downward */
	    {
	      node_free(node);
	      return NULL;
	    }
	  if (n)
	    n->next = nn;
	  else
	    node->c.l = nn;
	  n = nn;
	}
      c = getc(file);
    }

  if (is_toplevel)
    {
      if (c == ')')
	{
	  fprintf(stderr,"Unmatched ')'\n");
	  node_free(node);
	  return NULL;
	}
    }
  else
    {
      if (c == EOF)
	{
	  fprintf(stderr,"Unmatched '('\n");
	  node_free(node);
	  return NULL;
	}
    }
  return node;
}

/* Writes a string node to file. String will be faithfully reproduced
   by node_string_from_file */
void
node_string_to_file(Node *node,FILE *file)
{
  assert(node);
  assert(node->type == NODE_STRING);
  
  if (node->c.s)
    {
      int needs_quote = 0;
      char *p;
      if (!node->c.s || !*node->c.s)
	needs_quote = 1;
      for (p=node->c.s;*p;p++)
	{
	  if (*p == '"' || *p == '\\' || isspace(*p))
	    needs_quote = 1;
	}
      if (needs_quote) putc('"',file);
      if (node->c.s)
	for (p=node->c.s;*p;p++)
	  {
	    if (*p == '"' || *p == '\\')
	      putc('\\',file);
	    putc(*p,file);
	  }
      if (needs_quote) putc('"',file);
    }
}

/* Writes a list node to list. String will be faithfully reproduced
   by node_list_from_file. Does not write enclosing parens. level
   specifies indenting. (4 spaces per level in front of each sublist;
   sublists are put on a new line) */
void
node_list_to_file(Node *node, FILE *file, int level)
{
  Node *n;
  int i;

  assert(node);
  assert(node->type == NODE_LIST);
  
  n = node->c.l;

  while (n)
    {
      if (n->type == NODE_STRING)
	node_string_to_file(n,file);
      else if (n->type == NODE_LIST)
	{
	  putc('(',file);
	  node_list_to_file(n,file,level+1);
	  putc(')',file);
	}
      /* indent lists */
      n = n->next;
      if (n)
	if (n->type == NODE_LIST)
	  {
	    putc('\n',file);
	    for (i=0; i<level; i++)
	      fprintf(file,"    ");
	  }
	else
	  putc(' ',file);
    }
  if (level == 0)
    putc('\n',file);
}
