/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Uses:      GTK+ 1.2, Imlib, libpng, libtiff                               *
 * Modul:     paul.h                                                         *
 *            Definition of structures, constants and prototypes             *
 * Author:    Andreas Tille                                                  *
 * Date:      18.02.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#if !defined(_paul_h_)
#define _paul_h_
#include <gtk/gtk.h>
#include <gdk_imlib.h>
#include <png.h>
#include "gtkintl.h"

#define  PAUL_ID      42
#define  PICTURE_ID   17

typedef enum {
   CT_LTEXT,            /* not editable text which could be longer than one line */
   CT_INT,              /* integer                                               */
   CT_DOUBLE,           /* double                                                */
   CT_STRING            /* editable string                                       */
} C_TYPE;

typedef union {
   char  *s;
   int    i;
   double d;
} C_VALUE;

typedef struct { const char *key;
                 C_VALUE     val; 
                 C_TYPE      typ;
                 char        flag;
               } CHUNK;

/* possible masks for CHUNK->valid */
#define INVALID    0
#define VALID      1
#define EDITABLE   2
#define IN_CHANGE  4
#define IsValid(chunk)           ( (chunk->flag) & VALID     )
#define IsEditable(chunk)        ( (chunk->flag) & EDITABLE  )
#define IsInChange(chunk)        ( (chunk->flag) & IN_CHANGE )


#ifdef PHYSICAL_PAUL
/* Physical data */
typedef struct {
   CHUNK         *spec;     /* all the damn data which are specified in physical.c       */
   char          *file;     /* file with physical data                                   */
   double        *D;        /* data proportional D                 			 */
   double        *Dt;       /* data proportional dD/dt               			 */
} PHYS_GLOBAL;

PHYS_GLOBAL *PG;            /* global physical data                    			 */
#endif

typedef struct { 
   int            id;       /* identify stucture as PICTURE                              */
   GdkImlibImage *im;       /* ImLib struktur                     			 */
   unsigned char *data;     /* RGB-data                              			 */
   int            width;    /* image width                             			 */
   int            height;   /* image height                            			 */
   char          *file;     /* filename (without dir and extension)   			 */
   char          *ext;      /* extension                              			 */
   char          *dir;      /* directory                              			 */
   int            size;     /* = width * height                       			 */
   char           mark;     /* is image selected in filelist (NOTSEL, TMPSEL, SEL)       */
   char           spp;      /* samples per Pixel (monochrom = 1, else != 1 (== 3)	 */
   char           storepix; /* monochrome image in 1 or 3 byte         			 */
   char           trans;    /* transparency                            			 */
   unsigned int   res;      /* scanner resolution in DPI              			 */
   unsigned long  x_offset, /* x coordinate of left upper corner while scanning        	 */
                  y_offset; /* y coordinate of left upper corner while scanning        	 */
                            /* SANE (Nikon Coolscan) works with 65536 pixel per mm   	 */
   unsigned long *his;      /* histogramm                             			 */
#ifdef HAVE_LIBFFTW
   double        *fftprof;  /* profile of FFT                                            */
#endif
   int            n_gamma;  /* number of elements of the gamma-table  			 */
   png_color     *gamma;    /* gamma-table                          			 */
   time_t         zeit;     /* time, when the image was created    			 */
   CHUNK         *spec;     /* spezifying data                    			 */
   int            roff_x;   /* x-offset to reference image             			 */
   int            roff_y;   /* y-offset to reference image             			 */
   long           flag;     /* flag for different purpose             			 */
} PICTURE;

typedef struct { int x1, y1, x2, y2; } BOX;

typedef struct { 
   long           f;      /* flag what to do with this garbage                           */
   BOX           *cut;    /* corners of area to cut                   		         */
   unsigned char  eps,    /* "noise step" for difference images       		         */
                  fordn,  /* order of filtering                       		         */
                  filter, /* type of filtering                        		         */
                  greps,  /* allowed difference between RGB values    		         */
                  shr,    /* step of shrinkage                        		         */
                  terr,   /* allowed error when deleting borders      		         */
                  thresh; /* threshold for deleting borders           		         */
   int            offset, /* offset for enhancing brightness          		         */
                  xi, yi; /* coordinates for image to insert          		         */
   char          *bright, /* name of textfile to output brightness                       */
                 *html,   /* name of HTML file to output                                 */
                 *link;   /* name of link to point to when clicked the inserted image    */
   double         sek,    /* animation delay time between two images                     */
                  scale;  /* factor for contrast                                         */
} OPTIONS;

typedef struct {
   int             id;      /* identify stucture as PAUL structure                       */
   GList          *activ,   /* currently displayed image                   		 */
                  *piclist, /* list of all images to display               		 */
                  *spiclist;/* copy of piclist if necessary in callbacks                 */
   PICTURE        *op;      /* operation image                             		 */
   GtkWidget      *filelist;/* list of filenames (contains p as user_data)		 */
   GtkWidget      *show;    /* window to show images                       		 */
   GtkWidget      *opi;     /* window to show operation image (list field for operation  * 
                             * image name set as user data)  			         */
   GtkWidget      *w;       /* file list window (contains info as user_d.)		 */
   GtkWidget      *ctrl;    /* animation buttons and accept moved images button          */
#ifdef PHYSICAL_PAUL
   GtkWidget      *curve;   /* GtkDatabox widget to show phase curve                     */
#endif
   GtkItemFactory *items;   /* itemfactory widget to change sensitivities of entries     */
   OPTIONS        *opt;     /* alle Mglichen Optionen                     		 */
} PAUL;

typedef enum {
   P_S_CHUNK = 0,
   P_I_CHUNK,
   P_INT,
   P_LONG,
   P_GAMMA,
   P_UCHAR,
   P_DOUBLE
} PARM_TYPE;

#define  BILD(pl)           _BILD(pl, __FILE__, __LINE__)
#define  IS_PAUL(p)         ( (p)    && ((PAUL *)(p))->id       == PAUL_ID )
#define  IS_PICTURE(bild)   ( (bild) && ((PICTURE *)(bild))->id == PICTURE_ID )
#define  CheckPicList(pl)   _CheckPicList(pl, __FILE__, __LINE__)
/* eventuell einfach im-> davorsetzen */
#define  DATA       data               
#define  W          width
#define  H          height

#define iswap(x, y)  {register int zw = x; x = y; y = zw;}

extern const char *ChunkNames[],
            ChunkNameSource[],        /* source file                                       */
            ChunkNameSoftware[],      /* software which created the image (paul/paulscan)  */
            ChunkNameAuthor[],	      /* who created the image (g_get_user_name())         */
            ChunkNameDescription[],   /* detailed description about pauls changes          */
            ChunkNameTyp[],	      /* type of pauls changes                             */
            ChunkFilmType[],          /* negativ or positiv when scanning image            */
            ChunkBrightness[],        /* brightness of scanning process                    */
	    ChunkContrast[],          /* contrast of scanning process                      */
#ifdef PHYSICAL_PAUL
            ChunkPhysicalDataFile[],  /* filename of physical data file                    */
	    ChunkPhi[],               /* phase position of the image            	   */
            ChunkRelTime[],           /* time delay related to first image                 */
#endif
            ChunkReferenceFile[];     /* reference image of scanning process               */

/* Default values to delete border */
#define  DEFAULT_THRESHOLD       20
#define  DEFAULT_ERROR           50
#define  DEFAULT_SCAN_THRESHOLD  68   /* top/bottom:  56, (after rotating) lift-right: 63  *
                                       * PCD images from convert need at least 67          */
#define  DEFAULT_SCAN_ERROR      82   /*              81,                      92          */

/* possible return values of paul functions */
#define RET_OK       0
#define RET_ERR     -1
#define RET_SPECIAL 42

/* selection of an image */
#define NOSEL        0
#define TMPSEL       1
#define SEL          2

extern  char TypAusschnitt[],
             TypDifferenzBild[],
             TypDelRand[],
             TypDelScanRand[],
             TypFilter[],
             TypFFT[],
             TypGray[],
             TypInsert[],
             TypKontrastBild[],
             TypNegativ[],
             TypMirror[],
             TypMove[],
             TypRotate[],
             TypScan[],
             TypShrink[];

extern  const char *exename; /* "paul" ... or how to call the program */
#define HISTFILE   "histogram"
#define VPROFILE   "vertical_profile"
#define BRIGHTFILE "bright.dat"

#define DIFFERENZ          0x00000001
#define MAKEGRAY           0x00000002
#define HISTOGRAMM         0x00000004
#define INFO               0x00000008
#define SAVEMAXIMA         0x00000010
#define SAVEMINIMA         0x00000020
#define SAVEEXTREMA        0x00000030
#define BRIGHT             0x00000040
#define SHRINK             0x00000080
#define FILTER             0x00000100
#define MATCHPICTURES      0x00000200
#define QUIET              0x00000400
#define NOTMONOCHROM       0x00000800
#define DELBORDER          0x00001000
#define ONLYLEFTRIGHT      0x00002000
#define DELSCANBORDER      0x00004000
#define NEGATIV            0x00008000
#define PRINTGAMMA         0x00010000
#define SAFEFLAG           0x00020000
#define PNGORTIFF          0x00040000
#define MIRROR_H           0x00080000
#define MIRROR_V           0x00100000
#define ROT_90             0x00200000
#define ROT_180            0x00400000
#define ROT_270            (ROT_90 | ROT_180)
#define FFT                0x00800000
#define HTMLOUTPUT         0x01000000

#define Differenz(flag)          ( flag & DIFFERENZ  )
#define DoMakeGray(flag)         ( flag & MAKEGRAY )
#define DoMatchPictures(flag)    ( flag & MATCHPICTURES )
#define DelBorder(flag)          ( flag & DELBORDER )
#define OnlyLeftRight(flag)      ( flag & ONLYLEFTRIGHT )
#define DelScanBorder(flag)      ( flag & DELSCANBORDER )
#define Bright(flag)             ( flag & BRIGHT )
#define Negativ(flag)            ( flag & NEGATIV )
#define HistogrammFlag(flag)     ( flag & HISTOGRAMM )
#define NotMonoChrom(flag)       ( flag & NOTMONOCHROM )
#define SaveExtrema(flag)        ( flag & SAVEEXTREMA )
#define SaveMinima(flag)         ( flag & SAVEMINIMA )
#define SaveMaxima(flag)         ( flag & SAVEMAXIMA )
#define Filter(flag)             ( flag & FILTER )
#define SaveBild(flag)           ( flag & SAFEFLAG )
#define SavePNG(flag)            ( SaveBild(flag) && !(flag & PNGORTIFF) ) 
#define SaveTIFF(flag)           ( SaveBild(flag) &&  (flag & PNGORTIFF) )
#define SetPNG(flag)             ( flag |= SAFEFLAG )
#define SetTIFF(flag)            ( flag |= SAFEFLAG + PNGORTIFF )
#define DontSaveBild(flag)       ( flag &= ~SAFEFLAG )
#define Info(flag)               ( flag & INFO )
#define Shrink(flag)             ( flag & SHRINK )
#define HMirror(flag)            ( flag & MIRROR_H )
#define VMirror(flag)            ( flag & MIRROR_V )
#define Rotate90(flag)           ( (flag & ROT_90)  && !(flag & ROT_180) )
#define Rotate180(flag)          ( (flag & ROT_180) && !(flag & ROT_90)  )
#define Rotate270(flag)          ( flag & (ROT_90 | ROT_180) )
#define Rotate(flag)             ( (flag & ROT_90) | (flag & ROT_180) )
#define Mirror(flag)             ( flag & (MIRROR_H | MIRROR_V) )
#define OnlyInfo(flag)           ( !((0xFFFFFFFF&(~INFO)&(~QUIET)) & flag) && (flag&QUIET) )
#define DoFFT(flag)              ( flag & FFT )
#define HtmlOutput(flag)         ( flag & HTMLOUTPUT )
#define IsAnimation(opt)         ( opt->sek > -1 )

#define IsMonochrom(bild)        ( bild->spp == 1 )
#define RED    0
#define GREEN  1
#define BLUE   2

int     usage(void);

#define APPCUT           "_c"
#define APPFILTER        "_f"
#define APPFFT           "_fft"
#define APPGRAY          "_g"
#define APPINSERT        "_i"
#define APPKONTRAST      "_k"
#define APPMOVE          "_m"
#define APPMIN           "_min"
#define APPMAX           "_max"
#define APPNEGATIV       "_n"
#define APPROTATE        "_o"
#define APPMIRROR        "_p"
#define APPDELBORDER     "_r"
#define APPDELSCANBORDER "_rs"
#define APPSHRINK        "_s"    /* level of shrinkage appended */

#define FREE(ptr)          { if ( ptr )  { g_free(ptr); ptr = NULL; } }
#define G_LIST_FREE(list)  { if ( list ) { g_list_free(list); list = NULL; } }

extern     int             UPDATE_IMAGE;
extern     int             SHOW_SLOW;
extern     int             ANIMATION_SPEED;  
#define    SINGLE          0  /* no animation activ                                             */
#define    SLOW_ANIMATION  1  /* animation is slow enought to show name of picture in title bar */
#define    FAST_ANIMATION  2  /* suppress updating title bar                                    */

/* bright.c */
unsigned   long brightness(unsigned char *buf, int x, int y, int step, int storepix);
unsigned   long Xsum(unsigned char *buf, int len, int storepix);
unsigned   long Ysum(unsigned char *buf, int len, int step);
int        AnalyseBrightness(PAUL *p);

/* cutarea.c */
int        CropImage(PICTURE *bild, int x, int y, int w, int h);
gint       CutArea(PICTURE *bild, BOX *cut, glong flag, gchar *was, gchar *ext, gint del_old);
int        CutAreas(GList *piclist, OPTIONS *p);

/* datauti.c */
void       CurveWidgetDestroy(GtkWidget *window, void *data);
GtkWidget *CreateDataboxFrame(GtkWidget *curve, char *title);

/* delbordr.c */
int        SelectPictureWithoutBorder(PAUL *p);

/* delscanb.c */
int        DeleteScannerBorder(PAUL *p);

/* difpic.c */
int        DifferenceQueue(PAUL *p);
int        MakeDifference(PAUL *p);

/* extrema.c */
int        Extrema(PAUL *p);

/* fft.c */
int        FastFourierTransform(PAUL *p);

/* filesel.c */
char      *ChdirToLoadDir(char *filepath);
void       FileSelectionLoadSave(void *data, GtkSignalFunc func, char *name, const char *title, 
                                 const char *pattern);
char      *FilenameExtractFromSelection(GtkFileSelection *fs, PICTURE *bild);
GtkWidget *FileEntryOrBrowse(char *desc, char *init);
void       FileSelectEntryChanged(GtkWidget *text, GtkWidget *button);
void       FileSelectBrowse(GtkWidget *button, GtkWidget *text);
void       FileSelectChanged(GtkButton *button, char **file);

/* filter.c */
int        FilterBilder(PAUL *p);

/* gray.c */
int        MakeGray(PAUL *p);
void       MakeSingleGreen(PICTURE *bild, int ch);

/* gtkpaul.c */
int        GtkInterface(PAUL *p);

/* gtkuseful.c */
void       AreYouSure(void *data, GtkSignalFunc func, char *question);
GtkWidget *CreateCancelButton(GtkWidget *window, GtkWidget *box);
GtkWidget *CreateOkButton(GtkWidget *window, GtkWidget *box, void *userdata, 
                          GtkSignalFunc f1, void *d1, GtkSignalFunc f2, void *d2);

/* histogrm.c */
int        PrintHistograms(GList *piclist);
int        ShowHistograms(GList *piclist);

/* insert.c */
int        InsertImage(PAUL *p);
int        InsertSingleImage(PICTURE *target, PAUL *p);

/* matchpic.c */
int        MakeMove(PAUL *p);
int        AutoMatchPictures(PAUL *p);
int        CreateMatchPictures(PAUL *p);
GList     *EndMovedPictures(GList *piclist, GList *srclist, PICTURE *op, long flag);
int        MovePic(PAUL *p, int v, int h, int mod);
void       SetMoveTitle(PAUL *p);
int        ShiftValue(int mod);

/* messages.c */
void       MessagesInit(void);

/* mirror.c */
int        MakeMirror(PAUL *p);

/* negativ.c */
void       NegativBuffer(unsigned char *buf, int size, int step);
int        MakeNegative(GList *piclist);

/* paulload.c */
PICTURE   *LoadPicture(PICTURE *bild, char *file, long flag);
GdkImlibImage *at_imlib_set_image_from_data(unsigned char **data, gint w, gint h);

/* paulinfo.c */
void       PaulInfo(GList *piclist, long flag);
void       InfoFrame(GtkWidget *child, PICTURE *bild);
#ifdef PHYSICAL_PAUL
void       PhysicalInfoFrame(GtkWidget *box, PHYS_GLOBAL **pg);
#endif

/* paulspec.c */
int        Char2Chunk(CHUNK *chunk, char *buf);
char      *Chunk2Char(CHUNK *chunk);
int        Chunk2Double(CHUNK *chunk, const char *key, double *val);
void       CopyChunk(CHUNK *dest, CHUNK *src);
char      *GetCharSpec(CHUNK *chunks, const char *spec);
int        CopySpec(CHUNK *chunks, const char *spec, C_VALUE value);
int        SetDoubleSpec(CHUNK *chunks, const char *spec, double d);
int        SetSpec(CHUNK *chunks, const char *spec, char *buf);
PICTURE   *InitSpec(void);
int        SpecIsValid(CHUNK *chunks, const char *key);
int        NumValidChunks(CHUNK *chunks);
void       GetPictureAuthor(PICTURE *bild);
void       GetPictureSpecs(PICTURE *bild, char *file);
time_t     convert_png_time_to_time_t(png_time *ptime);
void       CreateDescription(PICTURE *bild, char *was);

#ifdef PHYSICAL_PAUL
/* physical.c */
void       FreePhysical(PHYS_GLOBAL **pg);
int        LoadPhysicalParameters(char *filename);
void       LoadPhysical(PAUL *p);
PHYS_GLOBAL *PhysicalInit(char *filename);
int        ShowPhaseCurve(PAUL *p, PHYS_GLOBAL *pg);
int        ShowPhasePoint(PAUL *p, PICTURE *bild);
#endif

/* picuti.c */
PICTURE   *_BILD(GList *l, char *file, int line);
int        _CheckPicList(GList *pl, char *file, int line);
int        CheckOperating(PAUL *p);
void       Backup(const char *name);
char      *CheckFile(const char *file);
int        CreateShowPixmaps(GList *piclist);
void       FreeBild(PICTURE *bild);
void       FreePiclist(GList *piclist);
void       FreePaul(PAUL *p);
void       FreeSpecs(CHUNK *spec);
int        GetListPosition(GList *list, GList *elem);
gchar     *ImgFileName(PICTURE *bild);
void       ImgChunksUpdate(PICTURE *bild, char *type, char *desc, char *app, long flag);
int        NBILDER(GList *l);
char      *NewFileName(PICTURE *bild, char *appendix);
int        NewImage(PICTURE *bild, int w_new, int h_new, unsigned char *data);
char      *RemoveUnderscore(char *string);
BOX       *ReadBoxFromString(char *string);
PICTURE   *ReadPic(char *file, OPTIONS *p);
GList     *ReadPicFiles(int nfiles, char **efiles, OPTIONS *p);
char      *time_t2string(time_t zeit);

/* pngload.c */
int        PaulLoadPNG(PICTURE *bild, FILE *fp, long flag, int *read_anything);

/* pngsave.c */
int        WritePNGImage(PICTURE *bild, char *filename);

/* rotate.c */
int        MakeRotate(PAUL *p);

/* savepics.c */
void       SavePictures(GList *piclist, long flag);

/* scale.c */
int        SkaliereBilder(PAUL *p);

/* schrink.c */
int        ShrinkPicture(PICTURE *bild, unsigned char shr);
int        ShrinkPictures(PAUL *p);

/* screen.c */
int        ApplyPicture(GdkWindow *win, PICTURE *bild);
void       GrowPic(GdkWindow *win, PICTURE *bild);
void       ShrinkPic(GdkWindow *win, PICTURE *bild);

/* tifload.c */
int        PaulLoadTIFF(PICTURE *bild, FILE *fp, char *file, long flag);

/* tifsave.c */
int        GetTag(const char *key);
int        WriteTIFFImage(PICTURE *bild, char *filename);

#include "gmalloc.h"

#endif


