// ---------------------------------------------------------------------------
// - Interp.cpp                                                              -
// - aleph engine - interpreter class implementation                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Real.hpp"
#include "Meta.hpp"
#include "List.hpp"
#include "Interp.hpp"
#include "Return.hpp"
#include "Reader.hpp"
#include "System.hpp"
#include "Method.hpp"
#include "Library.hpp"
#include "Printer.hpp"
#include "Builtin.hpp"
#include "Boolean.hpp"
#include "Function.hpp"
#include "InputFile.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // this procedure initialize an integer register bank
  static t_long* long_bank_init (const int size) {
    t_long* bank = new t_long[size];
    for (int i = 0; i < size; i++) bank[i] = 0;
    return bank;
  }
  // this procedure initialize a real register bank
  static double*  real_bank_init (const int size) {
    double* bank = new double[size];
    for (int i = 0; i < size; i++) bank[i] = 0.0;
    return bank;
  }
  // this procedure initialize a character register bank
  static char* char_bank_init (const int size) {
    char* bank = new char[size];
    for (int i = 0; i < size; i++) bank[i] = nilc;
    return bank;
  }
  // this procedure initialize a boolean register bank
  static bool* bool_bank_init (const int size) {
    bool* bank = new bool[size];
    for (int i = 0; i < size; i++) bank[i] = false;
    return bank;
  }

  // this procedure initialize the interpreter global nameset
  static void gset_init (Interp* interp) {
    // bind standard constants
    Nameset* nset = interp->getnset ();
    nset->symcst  ("...",       nset);
    nset->symcst  ("nil",       (Object*) nilp);
    nset->symcst  ("true",      new Boolean (true));
    nset->symcst  ("false",     new Boolean (false));

    //  builtin functions
    interp->mkrsv ("if",        new Function (builtin_if));
    interp->mkrsv ("do",        new Function (builtin_do));
    interp->mkrsv ("for",       new Function (builtin_for));
    interp->mkrsv ("try",       new Function (builtin_try));
    interp->mkrsv ("eval",      new Function (builtin_eval));
    interp->mkrsv ("const",     new Function (builtin_const));
    interp->mkrsv ("trans",     new Function (builtin_trans));
    interp->mkrsv ("class",     new Function (builtin_class));
    interp->mkrsv ("block",     new Function (builtin_block));
    interp->mkrsv ("while",     new Function (builtin_while));
    interp->mkrsv ("gamma",     new Function (builtin_gamma));
    interp->mkrsv ("throw",     new Function (builtin_throw));
    interp->mkrsv ("lambda",    new Function (builtin_lambda));
    interp->mkrsv ("switch",    new Function (builtin_switch));
    interp->mkrsv ("return",    new Function (builtin_return));
    interp->mkrsv ("protect",   new Function (builtin_protect));
    interp->mkrsv ("nameset",   new Function (builtin_nameset));

    // builtin operators
    interp->mkrsv ("+",         new Function (builtin_add));
    interp->mkrsv ("-",         new Function (builtin_sub));
    interp->mkrsv ("*",         new Function (builtin_mul));
    interp->mkrsv ("/",         new Function (builtin_div));
    interp->mkrsv ("==",        new Function (builtin_eql));
    interp->mkrsv ("!=",        new Function (builtin_neq));
    interp->mkrsv (">=",        new Function (builtin_geq));
    interp->mkrsv (">",         new Function (builtin_gth));
    interp->mkrsv ("<=",        new Function (builtin_leq));
    interp->mkrsv ("<",         new Function (builtin_lth));
    interp->mkrsv ("assert",    new Function (builtin_assert));

    // logical operators
    interp->mkrsv ("or",        new Function (builtin_or));
    interp->mkrsv ("not",       new Function (builtin_not));
    interp->mkrsv ("and",       new Function (builtin_and));
    
    // standard printer objects
    interp->mkrsv ("print",     new Printer  (Printer::OUTPUT));
    interp->mkrsv ("println",   new Printer  (Printer::OUTPUTLN));
    interp->mkrsv ("error",     new Printer  (Printer::ERROR));
    interp->mkrsv ("errorln",   new Printer  (Printer::ERRORLN));

    // standard predicates
    interp->mkrsv ("nil-p",       new Function (builtin_nilp));
    interp->mkrsv ("cons-p",      new Function (builtin_consp));
    interp->mkrsv ("list-p",      new Function (builtin_listp));
    interp->mkrsv ("real-p",      new Function (builtin_realp));
    interp->mkrsv ("class-p",     new Function (builtin_clsp));
    interp->mkrsv ("string-p",    new Function (builtin_strp));
    interp->mkrsv ("vector-p",    new Function (builtin_vecp));
    interp->mkrsv ("number-p",    new Function (builtin_nump));
    interp->mkrsv ("boolean-p",   new Function (builtin_boolp));
    interp->mkrsv ("integer-p",   new Function (builtin_intp));
    interp->mkrsv ("literal-p",   new Function (builtin_litp));
    interp->mkrsv ("nameset-p",   new Function (builtin_nstp));
    interp->mkrsv ("instance-p",  new Function (builtin_instp));
    interp->mkrsv ("hashtable-p", new Function (builtin_ashp));
    interp->mkrsv ("character-p", new Function (builtin_charp));

    // standard objects
    interp->mkrsv ("Real",        new Meta (Real::mknew));
    interp->mkrsv ("Cons",        new Meta (Cons::mknew));
    interp->mkrsv ("List",        new Meta (List::mknew));
    interp->mkrsv ("String",      new Meta (String::mknew));
    interp->mkrsv ("Vector",      new Meta (Vector::mknew));
    interp->mkrsv ("Integer",     new Meta (Integer::mknew));
    interp->mkrsv ("Boolean",     new Meta (Boolean::mknew));
    interp->mkrsv ("Character",   new Meta (Character::mknew));

    // bind the interpreter
    interp->mkrsv ("interp",      interp);    
  }

  // this procedure look for a shared library
  static Library* getshl (Vector* shlib, const String& name) {
    long len = (shlib == nilp) ? 0 : shlib->length ();
    if (len == 0) return nilp;
    for (long i = 0; i < len; i++) {
      Library* lib = dynamic_cast <Library*> (shlib->get (i));
      if (lib == nilp) continue;
      if (lib->getname () == name) return lib;
    }
    return nilp;
  }

  // create a default interpreter
  
  Interp::Interp (void) {
    // create a default terminal
    p_term = new Terminal;
    Object::iref (p_term);
    // save streams
    p_is = p_term; Object::iref (p_is);
    p_os = p_term; Object::iref (p_os);
    p_es = new OutputTerm (OutputTerm::ERROR);
    Object::iref (p_es);
    // initialize the register bank
    d_li = long_bank_init (Interp::MAX_REGS);
    d_lr = real_bank_init (Interp::MAX_REGS);
    d_lc = char_bank_init (Interp::MAX_REGS);
    d_lb = bool_bank_init (Interp::MAX_REGS);
    // initialize the arguments vector
    p_argv = new Vector;
    Object::iref (p_argv);
    // initialize the global nameset
    p_gset = new Globalset;
    Object::iref (p_gset);
    gset_init (this);
    // create the execution stack
    p_stk = new Stack;
    // create the dynamic library vector
    p_shlib = new Vector;
  }

  // create a new interpreter with the specified streams

  Interp::Interp (Input* is, Output* os, Output* es) {
    // reset the terminal
    p_term = nilp;
    // save streams
    p_is = is; Object::iref (is);
    p_os = os; Object::iref (os);
    p_es = es; Object::iref (es);
    // initialize the register bank
    d_li = long_bank_init (Interp::MAX_REGS);
    d_lr = real_bank_init (Interp::MAX_REGS);
    d_lc = char_bank_init (Interp::MAX_REGS);
    d_lb = bool_bank_init (Interp::MAX_REGS);
    // initialize the arguments vector
    p_argv = new Vector;
    Object::iref (p_argv);
    // initialize the global nameset
    p_gset = new Globalset;
    Object::iref (p_gset);
    gset_init (this);
    // create the execution stack
    p_stk = new Stack;
    // create the dynamic library vector
    p_shlib = new Vector;  }

  // delete this interpreter

  Interp::~Interp (void) {
    delete [] d_li;
    delete [] d_lr;
    delete [] d_lc;
    delete [] d_lb;
    delete    p_stk;
    delete    p_shlib;
    Object::dref (p_is);
    Object::dref (p_os);
    Object::dref (p_es);
    Object::dref (p_term);
    Object::dref (p_gset);
    Object::dref (p_argv);
  }

  // return the class name

  String Interp::repr (void) const {
    return "Interp";
  }

  // set the assert flag

  void Interp::setassert (const bool flag) {
    d_assert = flag;
  }

  // set the interpreter arguments

  void Interp::setargs (const Strvec& args) {
    long len = args.length ();
    for (long i = 0; i < len; i++) 
      p_argv->append (new String (args.get (i)));
  }

  // create a new reserved entry in the global nameset
  
  void Interp::mkrsv (const String& name, Object* object) {
    p_gset->symcst (name, object);
    Token::mkrsv   (name);
  }

  // create a new nameset in reference to another one

  Nameset* Interp::mknset (const String& name, Nameset* parent) {
    // look for parent
    if (parent == nilp) parent = p_gset;
    // look for an existing one
    if (parent->exists (name) == true) {
      Object* object = parent->eval (this, parent, name);
      Nameset* nset = dynamic_cast <Nameset*> (object);
      if (nset == nilp)
	throw Exception ("type-error", "name does not evaluate as a nameset",
			 Object::repr (object));
      return nset;
    }
    Nameset* result = new Globalset (parent);
    parent->symcst (name, result);
    return result;
  }

  // open a new dynamic library by name
 
  Object* Interp::library (const String& name, Cons* args) {
    // check if the library already exists
    Library* lib = getshl (p_shlib, name);
    if (lib != nilp) return lib;
    // the library does no exists, so create it
    lib = new Library (name);
    p_shlib->append (lib);
    // call the initialize procedure now
    Object::cref (lib->apply (this, p_gset, "initialize", args));
    return lib;
  }
 
  // loop on the standard input by doing a read-eval loop
  // this procedure return false is something bad happen

  bool Interp::loop (void) {
    bool status = true;
    // create a new reader 
    Reader* rd = new Reader (p_is);

    // loop until we have an eof
    while (true) {
      try {
	Cons* cons = rd->parse ();
	if (cons == nilp) break;
	Object::cref (cons->eval (this,p_gset));
	Object::dref (cons);
      } catch (const Exception& e) {
	status = false;
	p_es->writeln (e);
	if (e.getabf () == true) System::exit (1);
      } catch (const Return& r) {
      } catch (...) {
	status = false;
	p_es->writeln ("fatal: unknwon exception trapped");
	break;
      }
    }
    // clean and return
    delete rd;
    return status;
  }

  // loop with an input file input by doing a read-eval loop
  // this procedure return false is something bad happen
  
  bool Interp::loop (const String& fname) {
    // try to open this file
    InputFile* is = nilp;
    try {
      is = new InputFile (fname);
      Object::iref (is);
    } catch (const Exception& e) {
      p_es->writeln (e);
      delete is;
      if (e.getabf () == true) System::exit (1);
      return false;
    }
    
    // create a new reader 
    Reader* rd = new Reader (is);

    // loop until we have an eof
    bool status = true;
    while (true) {
      try {
	Cons* cons = rd->parse ();
	if (cons == nilp) break;
	Object::cref (cons->eval (this,p_gset));
	Object::dref (cons);
      } catch (const Exception& e) {
	status = false;
	p_es->writeln (e, fname, rd->getlnum ());
	if (e.getabf () == true) System::exit (1);
	break;
      } catch (const Return& r) {
      } catch (...) {
	status = false;
	p_es->writeln ("fatal: unknwon exception trapped");
	break;
      }
    }
    // clean and return
    delete rd;
    Object::dref (is);
    return status;
  }
  
  // evaluate an interpreter method name

  Object* Interp::eval (Interp* interp, Nameset* nset, const String& name) {
    if (name == "argv") return p_argv;
    return new Method (name,this);
  }

  // apply this interpreter with a set of arguments and a method name

  Object* Interp::apply (Interp* interp, Nameset* nset, const String& name,
			 Cons* args) {
    // special case for library since we do not evaluate here
    if (name == "library") {
      long len = (args == nilp) ? 0 : args->length ();
      if (len < 1) 
	throw Exception ("argument-error", "missing arguments with library");
      Object* car = args->getcar ();
      Object* obj = (car == nilp) ? nilp : car->eval (interp, nset);
      String* str = dynamic_cast <String*> (obj);
      if (str == nilp) 
	throw Exception ("type-error", "invalid object with library",
			 Object::repr (obj));
      Cons* argl = args->getcdr ();
      return library (*str, argl);
    }
      
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if ((name == "get-real-precision") && (argc == 0)) {
      delete argv;
      return new Real (Real::getprecision ());
    }

    // check for one argument
    if ((name == "set-real-precision") && (argc == 1)) {
      t_real val = argv->getreal (0);
      Real::setprecision (val);
      delete argv;
      return nilp;
    }
    if ((name == "load") && (argc == 1)) {
      String val = argv->getstring (0);
      this->loop (val);
      delete argv;
      return nilp;
    }

    // call the object method
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }
}
