// ---------------------------------------------------------------------------
// - Character.cpp                                                           -
// - standard object library - character class implementation                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Method.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new default character

  Character::Character (void) {
    d_value = nilc;
  }

  // create a new character from a native character

  Character::Character (const char value) {
    d_value = value;
  }
  
  // create a new character from a string representation

  Character::Character (const String& value) {
    if (value.length () == 1) {
      d_value = value[0];
    } else if ((value.length () == 3) && (value[0] == '\'') && 
	       (value[2] == '\'')) {
      d_value = value[1];
    } else
      throw Exception ("format-error",
		       "illegal character representation",value);
  }

  // copy constructor for this character class

  Character::Character (const Character& that) {
    d_value = that.d_value;
  }

  // return the class name

  String Character::repr (void) const {
    return "Character";
  }

  // return a literal representation of this character

  String Character::toLiteral (void) const {
    String quote = "'";
    return (quote + d_value + quote);
  }

  // return a string representation of this character

  String Character::toString (void) const {
    String result = d_value;
    return result;
  }

  // return a native character

  char Character::toCharacter (void) const {
    return d_value;
  }

  // set an character with a native value

  Character& Character::operator = (const char value) {
    d_value = value;
    return *this;
  }

  // set an character with a character

  Character& Character::operator = (const Character& value) {
    d_value = value.d_value;
    return *this;
  }

  // compare an character with a native value
  
  bool Character::operator == (const char value) const {
    return (d_value == value);
  }
  
  bool Character::operator != (const char value) const {
    return (d_value != value);
  }
  
  // compare two character characters
  
  bool Character::operator == (const Character& value) const {
    return (d_value == value.d_value);
  }
  
  bool Character::operator != (const Character& value) const {
    return (d_value != value.d_value);
  }
  
  bool Character::operator < (const Character& value) const {
    return (d_value < value.d_value);
  }
  
  bool Character::operator <= (const Character& value) const {
    return (d_value <= value.d_value);
  }
  
  bool Character::operator > (const Character& value) const {
    return (d_value > value.d_value);
  }
  
  bool Character::operator >= (const Character& value) const {
    return (d_value >= value.d_value);
  }
  
  // return true if the character is alpha

  bool Character::isalpha (void) const {
    bool result = ((d_value > 'a') && (d_value < 'z')) ||
      ((d_value > 'A') && (d_value < 'Z'));
    return result;
  }

  // return true if the character is a digit

  bool Character::isdigit (void) const {
    return ((d_value >= '0') && (d_value <= '9'));
  }

  // return true if the character is a blank or tab
  
  bool Character::isblank (void) const {
    return ((d_value == ' ') || (d_value == '\t'));
  }

  // return true if the character is an eol

  bool Character::iseol (void) const {
    return (d_value == eolc);
  }
  
  // return true if the character is an eof

  bool Character::iseof (void) const {
    return (d_value == eofc);
  }

  // return true if the character is nil

  bool Character::isnil (void) const {
    return (d_value == nilc);
  }

  // operate this character with another object

  Object* Character::oper (t_oper type, Object* object) {
    Integer*   iobj = dynamic_cast <Integer*>   (object);
    Character* cobj = dynamic_cast <Character*> (object);
    switch (type) {
    case Object::ADD:
      if (iobj != nilp) return new Character (d_value + iobj->toInteger ());
      break;
    case Object::SUB:
      if (iobj != nilp) return new Character (d_value - iobj->toInteger ());
      break;
    case Object::EQL:
      if (cobj != nilp) return new Boolean (d_value == cobj->d_value);
      break;
    case Object::NEQ:
      if (cobj != nilp) return new Boolean (d_value != cobj->d_value);
      break;
    case Object::GEQ:
      if (cobj != nilp) return new Boolean (d_value >= cobj->d_value);
      break;
    case Object::GTH:
      if (cobj != nilp) return new Boolean (d_value > cobj->d_value);
      break;
    case Object::LEQ:
      if (cobj != nilp) return new Boolean (d_value <= cobj->d_value);
      break;
    case Object::LTH:
      if (cobj != nilp) return new Boolean (d_value < cobj->d_value);
      break;
    default:
      throw Exception ("operator-error", "unsupported character operator");
    }
    throw Exception ("type-error", "invalid operand with character",
		     Object::repr (object));
  }

  // create a new character in a generic way

  Object* Character::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Character;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with character constructor");
    // try to map the character argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new Character;

    // try an integer object
    Integer* ival = dynamic_cast <Integer*> (obj);
    if (ival != nilp) return new Character (ival->toInteger ());

    // try a character object
    Character* cval = dynamic_cast <Character*> (obj);
    if (cval != nilp) return new Character (*cval);

    // try a string object
    String* sval = dynamic_cast <String*> (obj);
    if (sval != nilp) return new Character (*sval);

    // illegal object
    throw Exception ("type-error", "illegal object with character constructor",
		     obj->repr ());
  }

  // set an object to this character

  Object* Character::vdef (Interp* interp, Nameset* nset, Object* object) {
    Character* cobj = dynamic_cast <Character*> (object);
    if (cobj != nilp) {
      d_value = cobj->d_value;
      return this;
    }
    throw Exception ("type-error", "invalid object with character vdef",
		     Object::repr (object));
  }

  // evaluate this character with a member name

  Object* Character::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply this character with a method name

  Object* Character::apply (Interp* interp, Nameset* nset, const String& name,
			    Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) delete argv;
    if ((name == "to-string") && (argc == 0)) 
      return new String (toString ());
    if ((name == "to-integer") && (argc == 0)) 
      return new Integer (toCharacter ());
    if ((name == "++") && (argc == 0)) {
      d_value++;
      return this;
    }
    if ((name == "--") && (argc == 0)) {
      d_value--;
      return this;
    }
    if ((name == "alpha-p") && (argc == 0)) {
      return new Boolean (isalpha ());
    }
    if ((name == "digit-p") && (argc == 0)) {
      return new Boolean (isdigit ());
    }
    if ((name == "blank-p") && (argc == 0)) {
      return new Boolean (isblank ());
    }
    if ((name == "eol-p") && (argc == 0)) {
      return new Boolean (iseol ());
    }
    if ((name == "eof-p") && (argc == 0)) {
      return new Boolean (iseof ());
    }
    if ((name == "nil-p") && (argc == 0)) {
      return new Boolean (isnil ());
    }

    // dispatch one argument
    if ((name == "+") && (argc == 1)) {
      Object* result = oper (Object::ADD, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "-") && (argc == 1)) {
      Object* result = oper (Object::SUB, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "+=") && (argc == 1)) {
      t_long val = argv->getint (0);
      d_value += val;
      delete argv;
      return this;
    }
    if ((name == "-=") && (argc == 1)) {
      t_long val = argv->getint (0);
      d_value -= val;
      delete argv;
      return this;
    }
    if ((name == "==") && (argc == 1)) {
      Object* result = oper (Object::EQL, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "!=") && (argc == 1)) {
      Object* result = oper (Object::NEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "<") && (argc == 1)) {
      Object* result = oper (Object::LTH, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "<=") && (argc == 1)) {
      Object* result = oper (Object::LEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == ">") && (argc == 1)) {
      Object* result = oper (Object::GTH, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == ">=") && (argc == 1)) {
      Object* result = oper (Object::GEQ, argv->get (0));
      delete argv;
      return result;
    }

    // call the object method
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }
}
