// ---------------------------------------------------------------------------
// - Character.hpp                                                           -
// - standard object library - character class definition                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_CHARACTER_HPP
#define ALEPH_CHARACTER_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

#ifndef  ALEPH_VECTOR_HPP
#include "Vector.hpp"
#endif

namespace aleph {

  /// The Character class is the object version of the basic 8-bit character.
  /// The Character class is derived from the literal class and implements
  /// standard comparision operators.
  /// @author amaury darsch

  class Character: public Literal {
  private:
    /// the character representation
    char d_value;

  public:
    /// create a new default character
    Character (void);

    /// create a new character from a native character
    /// @param value the value to create
    Character (const char value);

    /// create a new character from a string representation
    /// @param value the string to convert
    Character (const String& value);

    /// copy constructor for this character class
    /// @param that the character class to copy
    Character (const Character& that);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this character
    String toLiteral (void) const;

    /// @return a string representation of this character
    String toString (void) const;

    /// @return a native character
    char toCharacter (void) const;

    /// assign a character with a native value
    /// @param value the value to assign
    Character& operator = (const char value);

    /// assign a character with a character value
    /// @param value the value to assign
    Character& operator = (const Character& value);

    /// compare this character with a native value
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const char value) const;

    /// compare this character with a native value
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const char value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are less
    bool operator < (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are less or equal
    bool operator <= (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are greater
    bool operator > (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are greater or equal    
    bool operator >= (const Character& value) const;

    /// @return true if the character is an alphanumeric
    bool isalpha (void) const;

    /// @return true if the character is a digit
    bool isdigit (void) const;

    /// @return true if the character is a blank or tab
    bool isblank (void) const;

    /// @return true if the character is an eol
    bool iseol (void) const;

    /// @return true if the character is an eof
    bool iseof (void) const;

    /// @return true if the character is nil
    bool isnil (void) const;

    /// operate this character with another object
    /// @param type the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);

    /// generate a new character
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// set an object to this character
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate a character member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this character with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);
  };
}

#endif
