// ---------------------------------------------------------------------------
// - Exception.hpp                                                           -
// - standard object library - exception class definition                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_EXCEPTION_HPP
#define ALEPH_EXCEPTION_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The Exception class is the base class for any exception generated by 
  /// the standard object library. The Exception holds a short string as a 
  /// descriptor and eventually a message. An object can be attached to the 
  /// exception for further reference.
  /// @author amaury darsch

  class Exception : public virtual Object {
  private:
    /// the exception id
    String d_eid;
    /// the exception reason
    String d_reason;
    /// the exception client object
    Object* p_object;
    /// the newline flag
    bool d_nlf;
    /// the abort flag
    bool d_abf;

  public:
    /// Create a exception with an id
    /// @param eid the exception id
    explicit Exception (const String& eid);

    /// Create a exception with an id and a reason
    /// @param eid    the exception id
    /// @param reason the reason for this exception
    explicit Exception (const String& eid, const String& reason);

    /// create an exception with an id, a reason and a name
    /// @param eid    the exception eid
    /// @param reason the reason for this exception
    /// @param name   the name for the reason
    explicit Exception (const String& eid, const String& reason,
			const String& name);

    /// create an exception with an eid an object
    /// @param eid    the exception id
    /// @param object the exception client object
    explicit Exception (const String& eid, Object* object);

    /// create an exception with an id, a reason and an object
    /// @param eid    the exception id
    /// @param reason the reason for this exception
    /// @param object the exception client object
    explicit Exception (const String& eid, const String& reason,
			Object* object);

    /// Copy constructor for this exception class.
    /// @param that the exception to copy
    Exception (const Exception& that);

    /// @return the class name
    String repr (void) const;

    /// @return the exception id
    inline String geteid (void) const {
      return d_eid;
    }

    /// @return the exception reason
    inline String getval (void) const {
      return d_reason;
    }

    /// @return the exception object
    inline Object* getobj (void) const {
      return p_object;
    }

    /// set the new line flag
    void setnlf (const bool flag);

    /// @return the new line flag
    inline bool getnlf (void) const {
      return d_nlf;
    }

    /// set the abort flag
    void setabf (const bool flag);

    /// @return the new line flag
    inline bool getabf (void) const {
      return d_abf;
    }

    /// evaluate an exception member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

  private:
    // make the assignment operator private
    Exception& operator = (const Exception&);
  };
}

#endif
