// ---------------------------------------------------------------------------
// - Output.cpp                                                              -
// - standard object library - output stream class implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Output.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // return true if we have a tty - by default is false

  bool Output::istty (void) const {
    return false;
  }

  // write an array of character to the output stream

  void Output::write (const char* value, const long size) {
    for (long i = 0; i < size; i++)
      write (value[i]);
  }

  // write a string to the output stream

  void Output::write (const String& value) {
    // get a c-string from this string
    char* cstrg = value.toChar ();
    if (cstrg == nilp) return;
    // write string content
    write (cstrg);
    delete cstrg;
  }

  // write a newline to this output stream

  void Output::newline (void) {
    write (eolc);
  }

  // write a string followed by a new line. This is a convenient function

  void Output::writeln (const String& line) {
    write (line);
    write ('\n');
  }

  // write an exception string on this output stream

  void Output::writeln (const Exception& e) {
    String what = "exception : ";
    String resm = "reason    : ";
    String resv = e.getval ();
    
    if (e.getnlf () == true) write ('\n');
    write (what + e.geteid ());
    write ('\n');
    if (resv.length () > 0) {
      write (resm + resv);
      write ('\n');
    }
  }

  // write an exception string on this output stream

  void Output::writeln (const Exception& e, const String& name, 
			const long lnum) {
    String what = "exception : ";
    String file = "in file   : "; 
    String resm = "reason    : ";
    String resv = e.getval ();

    if (e.getnlf () == true) write ('\n');
    write (what + e.geteid ());
    write ('\n');
    write (file + name + " at or around line " + lnum);
    write ('\n');
    if (resv.length () > 0) {
      write (resm + resv);
      write ('\n');
    }
  }

  // write a character on the output stream - no exception is thrown

  Output& Output::operator << (const char value) {
    try {
      this->write (value);
    } catch (const Exception& e) {};
    return *this;
  }

  // write a string on the output stream - no exception is thrown
  
  Output& Output::operator << (const String& value) {
    try {
      this->write (value);
    } catch (const Exception& e) {};
    return *this;
  }

  // write an integer on the output stream - no exception is thrown

  Output& Output::operator << (const long value) {
    try {
      this->write (value);
    } catch (const Exception& e) {};
    return *this;
  }

  // apply this output class with a method name

  Object* Output::apply (Interp* interp, Nameset* nset, const String& name,
			 Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch one argument
    if ((name == "write") && (argc == 1)) {
      Object* obj = argv->get (0);
      // check for character
      Character* cobj = dynamic_cast <Character*> (obj);
      if (cobj != nilp) {
	char c = cobj->toCharacter ();
	write (c);
	return obj;
      }
      String* sobj = dynamic_cast <String*> (obj);
      if (sobj != nilp) {
	write (*sobj);
	return obj;
      }
      throw Exception ("type-error", "invalid object to write", 
		       Object::repr (obj));
    }
    if ((name == "writeln") && (argc == 1)) {
      String val = argv->getstring (0);
      writeln (val);
      return nilp;
    }
    if ((name == "newline") && (argc == 0)) {
      newline ();
      return nilp;
    }
    // apply with object
    return Object::apply (interp, nset, name, argv);
  }
}
