// ---------------------------------------------------------------------------
// - OutputFile.cpp                                                          -
// - standard object library - outputfile class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cio.hxx"
#include "Method.hpp"
#include "Exception.hpp"
#include "OutputFile.hpp"

namespace aleph {

  // create a new file output stream by name. If the file cannot be created
  // an exception is thrown.

  OutputFile::OutputFile (const String& name) {  
    // copy file name and check
    d_name  = name;
    if (name.length () == 0) throw Exception ("name-error", "null file name");
    // try to open the file
    char* fname = name.toChar ();
    if ((d_sid = c_openw (fname)) == -1) {
      delete [] fname;
      throw Exception ("open-error","cannot open file for writing",name);
    }
    delete [] fname;
  }

  // destroy this class by closing this file

  OutputFile::~OutputFile (void) {
    if (d_sid >= 0) c_close (d_sid);
  }

  // return the class name

  String OutputFile::repr (void) const {
    return "OutputFile";
  }

  // write one character to the output stream
  
  void OutputFile::write (const char value) {
    // write one character
    long count = c_write (d_sid, &value,1);
    if (count < 0) throw Exception ("write-error", "cannot write character");
  }

  // write a character string to the output stream

  void OutputFile::write (const char* value) {
    long size = String::length (value);
    if (size == 0) return;
    // write the string
    long count = c_write (d_sid, value,size);
    if (count < 0) throw Exception ("write-error");
  }

  // write an integer to the output stream
  
  void OutputFile::write (const long value) {
    long count = c_write (d_sid, value);
    if (count < 0) throw Exception ("write-error");
  }

  // return the file name associated with this stream
  
  String OutputFile::getFileName (void) const {
    return d_name;
  }

  // close this output file
  
  void OutputFile::close (void) {
    c_close (d_sid);
    d_sid = -1;
  }

  // create a new output file in a generic way

  Object* OutputFile::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) 
      throw Exception ("argument-error", 
		       "invalid arguments with with output file"); 

    // try to map the file name
    String name = argv->getstring (0);
    return new OutputFile (name);
  }

  // evaluate this output file with a member name

  Object* OutputFile::eval (Interp* interp, Nameset* nset, 
			    const String& name) {
    return new Method (name, this);
  }

  // apply this output file with a method name

  Object* OutputFile::apply (Interp* interp, Nameset* nset, const String& name,
			     Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "close") && (argc == 0)) {
      delete argv;
      close ();
      return nilp;
    }
    if ((name == "get-name") && (argc == 0)) {
      delete argv;
      return new String (d_name);
    }

    // output method call
    Object* result = nilp;
    try {
      result =  Output::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    delete argv;
    return result;
  }
}
