// ---------------------------------------------------------------------------
// - OutputTerm.cpp                                                          -
// - standard object library - terminal output stream class implementation   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cio.hxx"
#include "cterm.hxx"
#include "cstring.hxx"
#include "Method.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"
#include "OutputTerm.hpp"

namespace aleph {

  // Create a new default output stream. By default the output stream is
  // mapped to the default standard output

  OutputTerm::OutputTerm (void)  {
    d_sid    = c_stdout ();
    p_tinfo  = c_rtinfo (false);
    d_insert = true;
  }

  // Create a new default output stream according to the mode.
  
  OutputTerm::OutputTerm (t_mode mode)  {
    switch (mode) {
    case OutputTerm::OUTPUT:
      d_sid    = c_stdout ();
      break;
    case OutputTerm::ERROR:
      d_sid    = c_stderr ();
      break;
    }
    p_tinfo  = c_rtinfo (false);
    d_insert = true;
  }

  // destroy this output terminal

  OutputTerm::~OutputTerm (void) {
    for (int i = 0; i < OTERM_MAX; i++)
      delete [] p_tinfo[i];
    delete [] p_tinfo;
  }

  // return the class name

  String OutputTerm::repr (void) const {
    return "OutputTerm";
  }

  // return true if we have a tty

  bool OutputTerm::istty (void) const {
    return c_istty (d_sid);
  }

  // write one character to the output stream
  
  void OutputTerm::write (const char value) {
    long count = c_write (d_sid, &value,1);
    if (count < 0) throw Exception ("write-error","cannot write character");
  }

  // write a character string to the output stream
  
  void OutputTerm::write (const char* value) {
    long size = c_strlen (value);
    if (size == 0) return;
    // write the string
    long count = c_write (d_sid, value,size);
    if (count < 0) throw Exception ("write-error","cannot write string");
  }

  // write an integer to the output stream
  
  void OutputTerm::write (const long value) {
    long count = c_write (d_sid, value);
    if (count < 0) throw Exception ("write-error", "cannot write integer");
  }

  // insert a character based on the mode

  void OutputTerm::insert (const char c) {
    // check that we have terminal capabilities
    if (p_tinfo == nilp) {
      write (c);
      return;
    }
    
    // check for mode and perform insert
    if (d_insert == true) {
      if (p_tinfo[OTERM_INSERT_CHAR] != nilp) {
	c_tparm (d_sid,p_tinfo, OTERM_INSERT_CHAR);
	write (c);
      } else {
	c_tparm (d_sid,p_tinfo, OTERM_IMODE_START);
	write (c);
	c_tparm (d_sid,p_tinfo, OTERM_IMODE_END);
      }
    } else
      write (c);
  }
  
  // erase on the left a certain amount of characters
  
  void OutputTerm::erasel (const long num) {
    if (num <= 0) return;
    for (long i = 0; i < num; i++) {
      c_tparm (d_sid,p_tinfo, OTERM_MOVE_LEFT);
      c_tparm (d_sid,p_tinfo, OTERM_DELETE_CHAR);
    }
  }
  
  // delete one character at the cursor position
  
  void OutputTerm::del (void) {
    c_tparm (d_sid,p_tinfo, OTERM_DELETE_CHAR);
  }
  
  // move to the left by a certain number of characters
  
  void OutputTerm::movel (const long num) {
    if (num <= 0) return;
    for (long i = 0; i < num; i++) {
      c_tparm (d_sid,p_tinfo, OTERM_MOVE_LEFT);
    }
  }
  
  // move to the right by a certain amount
  
  void OutputTerm::mover (const long num) {
    if (num <= 0) return;
    for (long i = 0; i < num; i++) {
      c_tparm (d_sid,p_tinfo, OTERM_MOVE_RIGHT);
    }
  }

  // create a new output term in a generic way

  Object* OutputTerm::mkout (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 0) 
      throw Exception ("argument-error", 
		       "invalid arguments with with output term"); 
    return new OutputTerm (OutputTerm::OUTPUT);
  }

  // create a new error term in a generic way

  Object* OutputTerm::mkerr (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 0) 
      throw Exception ("argument-error", 
		       "invalid arguments with with error term"); 
    return new OutputTerm (OutputTerm::ERROR);
  }

  // evaluate this output term with a member name

  Object* OutputTerm::eval (Interp* interp, Nameset* nset, 
			    const String& name) {
    return new Method (name, this);
  }

  // apply this output term with a method name

  Object* OutputTerm::apply (Interp* interp, Nameset* nset, const String& name,
			     Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);

    // output method call
    Object* result = nilp;
    try {
      result =  Output::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    delete argv;
    return result;
  }
}
