// ---------------------------------------------------------------------------
// - Real.hpp                                                                -
// - standard object library - real class definition                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_REAL_HPP
#define ALEPH_REAL_HPP

#ifndef  ALEPH_INTEGER_HPP
#include "Integer.hpp"
#endif

namespace aleph {

  /// The Real class is a the object version of the native double floating
  /// point number. The real number is implemented like the integer class and
  /// is derived from the Literal class. The class implements also a bunch of
  /// method for floating point operations.
  /// @author amaury darsch

  class Real : public Literal {
  private:
    // the precision for comparision
    static t_real d_precision;

    // the real representation is a double
    t_real d_value;

  public:
    /// create a new default real
    Real (void);

    /// create a new real from a default native double
    /// @param value the value to create
    Real (const t_real value);

    /// create a new real from an integer class
    /// @param value the value to create
    Real (const Integer& value);

    /// create a new real from a string
    /// @param value the value to convert
    Real (const String& value);

    /// copy constructor for this real
    /// @param that the real class to copy
    Real (const Real& that);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this real
    String toLiteral (void) const;

    /// @return a string representation of this real
    String toString (void) const;

    /// @return a formated string based on the precision
    String format (const long precision) const;

    /// @return a double value for this real
    t_real toReal (void) const;

    /// @return an integer representation from this real
    t_long toInteger (void) const;

    /// assign an real with a native value
    /// @param value the value to assign
    Real& operator = (const t_real value);

    /// assign an real with a native value
    /// @param value the value to assign
    Real& operator = (const Real& value);

    /// compare this real with a native value
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const long value) const;

    /// compare this real with a native value
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const long value) const;

    /// compare two reals
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const Real& value) const;

    /// compare two reals
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const Real& value) const;

    /// compare two reals
    /// @param value the value to compare
    /// @return true if they are less
    bool operator < (const Real& value) const;

    /// compare two reals
    /// @param value the value to compare
    /// @return true if they are less or equal
    bool operator <= (const Real& value) const;

    /// compare two reals
    /// @param value the value to compare
    /// @return true if they are greater
    bool operator > (const Real& value) const;

    /// compare two reals
    /// @param value the value to compare
    /// @return true if they are greater or equal
    bool operator >= (const Real& value) const;

    /// add two double reals together
    /// @param x the first argument to add
    /// @param y the second argument to add
    /// @return a new double sum of the previous one
    friend Real operator + (const Real& x, const Real& y);

    /// add a real to this one
    /// @param x the argument to add
    /// @return this added real
    Real& operator += (const Real& x);

    /// substract a real to this one
    /// @param x the argument to substract
    /// @return this added real
    Real& operator -= (const Real& x);

    /// substract two double reals together
    /// @param x the first argument to subsract
    /// @param y the second argument to substract
    /// @return a new double difference of the previous one
    friend Real operator - (const Real& x, const Real& y);

    /// compute the opposite of the double real
    /// @param x the real to oppose
    /// @return a new real opposite of the argument
    friend Real operator - (const Real& x);

    /// multiply two double reals together
    /// @param x the first argument to multiply
    /// @param y the second argument to multiply
    /// @return a new double product of the previous one
    friend Real operator * (const Real& x, const Real& y);

    /// multiply a real with this one
    /// @param x the argument to multiply
    /// @return this multiplied real
    Real& operator *= (const Real& x);

    /// divide two double reals together
    /// @param x the numerator
    /// @param y the denumerator
    /// @return the division of the arguments
    friend Real operator / (const Real& x, const Real& y);

    /// @return true if the number is nan
    bool isnan (void) const;
  
    /// @return the ceiling of this number
    Real ceiling (void) const;

    /// @return the floor of this number
    Real floor (void) const;

    /// @return the absolute value of this number
    Real abs (void) const;

    /// @return the remainder of this value with the argument
    Real mod (const Real& x) const;

    /// @return the square root of this real
    Real sqrt (void) const;

    /// @return the natural logarithm of this number
    Real log (void) const;

    /// @return the exponential of this number
    Real exp (void) const;

    /// @return the power of this number with its argument
    Real pow (const Real& x) const;

    /// @return the sine of this number
    Real sin (void) const;

    /// @return the cosine of this number
    Real cos (void) const;

    /// @return the tangent of this number
    Real tan (void) const;

    /// @return the arc sine of this number
    Real asin (void) const;

    /// @return the arc cosine of this number
    Real acos (void) const;

    /// @return the arc tangent of this number
    Real atan (void) const;

    /// @return the hyperbolic sine of this number
    Real sinh (void) const;

    /// @return the hyperbolic cosine of this number
    Real cosh (void) const;

    /// @return the hyperbolic tangent of this number
    Real tanh (void) const;

    /// @return the hyperbolic arc sine of this number
    Real asinh (void) const;

    /// @return the hyperbolic arc cosine of this number
    Real acosh (void) const;

    /// @return the hyperbolic arc tangent of this number
    Real atanh (void) const;

    /// set the precision for the ?= operator
    static void setprecision (t_real precision) {
      Real::d_precision = precision;
    }

    /// @return the real precision for the ?= operator
    static t_real getprecision (void) {
      return Real::d_precision;
    }

    /// operate this real with another object
    /// @param type the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);

    /// generate a new real
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// set an object to this real
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate a real member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this real with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);

  private:
    // make the integer class a friend
    friend class Integer;
  };
}

#endif
