// ---------------------------------------------------------------------------
// - Terminal.cpp                                                            -
// - aleph standard library - terminal class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cterm.hxx"
#include "Method.hpp"
#include "Boolean.hpp"
#include "Terminal.hpp"
#include "Character.hpp"
#include "Exception.hpp"

// default prompts
static const char* DEFAULT_PROMPT1 = "aleph >";
static const char* DEFAULT_PROMPT2 = "..... >";

namespace aleph {

  // create a default terminal

  Terminal::Terminal (void) {
    // prompt initialize
    d_prt1 = DEFAULT_PROMPT1;
    d_prt2 = DEFAULT_PROMPT2;
    // save the input stream state
    save ();
  }

  // destroy this terminal

  Terminal::~Terminal (void) {
    // restore terminal state
    restore ();
  }

  // return the class name

  String Terminal::repr (void) const {
    return "Terminal";
  }

  // read a line and return a string

  String Terminal::readline (bool pflag) {
    // reset the readline buffer
    d_lbuf.reset ();
    
    // check for the eof
    if (d_eof == true) return eofc;
    
    // turn off canonical mode
    bool canon = nocanon ();
    
    // ok, print the prompt
    pflag ? Output::write (d_prt1) : Output::write (d_prt2);
    
    // loop until we have an eol/eof
    while (1) {
      t_word w = wread ();
      
      // check for backspace
      if (InputTerm::isbs (w) == true) {
	bool status = d_lbuf.erase ();
	if (status == true) erasel (1);
	continue;
      }
      
      // check for delete
      if (InputTerm::isdel (w) == true) {
	bool status = d_lbuf.del ();
	if (status == true) del ();
	continue;
      }
      
      // check for insert
      if (InputTerm::isins (w) == true) {
	d_insert = !d_insert;
	d_lbuf.setimode (d_insert);
	continue;
      }
      
      // check for move to begin line
      if (w == InputTerm::WIDE_STDBL) {
	long num = d_lbuf.setcb ();
	movel (num);
	continue;
      }
      
      // check for move to end of line
      if (w == InputTerm::WIDE_STDEL) {
	long num = d_lbuf.setce ();
	mover (num);
	continue;
      }
      
      // check for kill everything
      if (w == InputTerm::WIDE_STDKB) {
	long num = d_lbuf.setce ();
	mover (num);
	num = d_lbuf.length ();
	erasel (num);
	d_lbuf.reset ();
	continue;
      }
      
      // check for kill to eol
      if (w == InputTerm::WIDE_STDKL) {
	long num = d_lbuf.setce ();
	mover (num);
	d_lbuf.kill (num);
	erasel (num);
	continue;
      }
      
      // check for move to left
      if (InputTerm::isle (w) == true) {
	bool status = d_lbuf.movel ();
	if (status == true) movel  (1);
	continue;
      }
      
      // check for move to right
      if (InputTerm::isri (w) == true) {
	bool status = d_lbuf.mover ();
	if (status == true) mover  (1);
	continue;
      }

      // make sure the cursor is reset to the end for a eol/eof
      if (w == InputTerm::WIDE_STDNL) d_lbuf.setce ();
      if (w == InputTerm::WIDE_STDEF) d_lbuf.setce ();
      
      // if we have a wide character here - we ignore it
      if (InputTerm::iswide (w) == true) continue;
      
      // check for printable - if not we reject - we accept eol/eof as well
      if (InputTerm::isprintable (w) == false) continue;
      
      // normal character
      char c = InputTerm::wtoc (w);
      d_lbuf.add (c);
      if ((c == '\n') || (c == eofc)) {
	write ('\n');
	break;
      }
      if (canon == true) insert (c);
    }
    
    // this is it - return the string buffer
    restore ();
    return d_lbuf.toString ();
  }

  // set the primary prompt

  void Terminal::setpprompt (const String& value) {
    d_prt1 = value;
  }

  // set the seconday prompt

  void Terminal::setsprompt (const String& value) {
    d_prt1 = value;
  }

  // create a new terminal in a generic way

  Object* Terminal::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 0) 
      throw Exception ("argument-error", "invalid arguments with terminal"); 
    return new Terminal;
  }

  // evaluate this terminal with a member name

  Object* Terminal::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply this terminal with a method name

  Object* Terminal::apply (Interp* interp, Nameset* nset, const String& name,
			   Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "read") && (argc == 0)) {
      Character* result = new Character (read ());
      delete argv;
      return result;
    }
    if ((name == "readline") && (argc == 0)) {
      String* result = new String (readline (true));
      delete argv;
      return result;
    }
    if ((name == "readln") && (argc == 0)) {
      String* result = new String (readln ());
      delete argv;
      return result;
    }
    if ((name == "newline") && (argc == 0)) {
      newline ();
      return nilp;
    }
    if ((name == "valid-p") && (argc == 0)) {
      delete argv;
      return new Boolean (!iseof ());
    }
    if ((name == "eof-p") && (argc == 0)) {
      delete argv;
      return new Boolean (iseof ());
    }
    if ((name == "get-primary") && (argc == 0)) {
      delete argv;
      return new String (getpprompt ());
    }
    if ((name == "get-secondary") && (argc == 0)) {
      delete argv;
      return new String (getsprompt ());
    }

    // dispatch one argument
    if ((name == "readline") && (argc == 1)) {
      bool flag = argv->getbool (0);
      String* result = new String (readline (flag));
      delete argv;
      return result;
    }
    if ((name == "write") && (argc == 1)) {
      Object* obj = argv->get (0);
      // check for character
      Character* cobj = dynamic_cast <Character*> (obj);
      if (cobj != nilp) {
	char c = cobj->toCharacter ();
	write (c);
	delete argv;
	return obj;
      }
      String* sobj = dynamic_cast <String*> (obj);
      if (sobj != nilp) {
	Output::write (*sobj);
	delete argv;
	return obj;
      }
      throw Exception ("type-error", "invalid object to write", 
		       Object::repr (obj));
    }
    if ((name == "writeln") && (argc == 1)) {
      String val = argv->getstring (0);
      writeln (val);
      delete argv;
      return nilp;
    }
    if ((name == "pushback") && (argc == 1)) {
      Object* obj = argv->get (0);
      // check for a character
      Character* c = dynamic_cast <Character*> (obj);
      if (c != nilp) {
	pushback (c->toCharacter ());
	delete argv;
	return nilp;
      }
      String* s = dynamic_cast <String*> (obj);
      if (s != nilp) {
	pushback (*s);
	delete argv;
	return nilp;
      }
      throw Exception ("type-error", "invalid object with pushback method",
		       Object::repr (obj));
    }
    if ((name == "set-primary") && (argc == 1)) {
      String val = argv->getstring (0);
      setpprompt (val);
      delete argv;
      return nilp;
    }
    if ((name == "set-secondary") && (argc == 1)) {
      String val = argv->getstring (0);
      setsprompt (val);
      delete argv;
      return nilp;
    }

    // object method call
    Object* result = nilp;
    try {
      result = Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    delete argv;
    return result;
  }
}
