// ---------------------------------------------------------------------------
// - Vector.hpp                                                              -
// - standard object library - dynamic vector class definition               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_VECTOR_HPP
#define  ALEPH_VECTOR_HPP

#ifndef  ALEPH_ITERABLE_HPP
#include "Iterable.hpp"
#endif

namespace aleph {

  /// The Vector class implementes a dynamic array. It is by far less efficent
  /// than an array where the size of the array is known in advance. This
  /// class should be used when an indexed container is required.
  /// @author amaury darsch

  class Vector : public Iterable {
  private:
    /// the current size of this vector
    long d_length;
    /// the size of the allocated vector
    long d_size;
    /// the allocated vector
    Object** p_vector;

  public:
    /// create an empty vector.
    Vector (void);

    /// create a vector with a predefined allocated size
    /// @param size the requested size
    Vector (const long size);

    /// copy constructor for this vector
    /// @param that the vector to copy 
    Vector (const Vector& that);

    /// destroy this vector
    ~Vector (void);

    /// @return the class name
    String repr (void) const;

    /// assign a vector to this one
    /// @param that the vector to assign
    Vector& operator = (const Vector& that);

    /// append an element in this vector
    /// @param object the object to add
    void append (Object* object);

    /// set an object at a given position
    /// @param index the vector index
    /// @param object the object to set
    void set (const long index, Object* object);

    /// get an object at a certain index
    /// @param index the vector index
    /// @return the object at this position
    Object* get (const long index) const;

    /// @return the first object in this vector
    inline Object* first (void) const {
      return get (0);
    }

    /// @return the last object in this vector
    inline Object* last (void) const {
      return get (d_length-1);
    }
    /// @return the number of objects in this vector
    long length (void) const;

    /// @return a new iterator for this vector
    Iterator* makeit (void);

    /// return a long integer value by object index
    t_long getint (const long index) const;

    /// return a real value by object index
    t_real getreal (const long index) const;

    /// return a real value by object index, even from an integer
    t_real getireal (const long index) const;

    /// return a boolean value by object index
    bool getbool (const long index) const;

    /// return a character value by object index
    char getchar (const long index) const;

    /// return a string value by object index
    String getstring (const long index) const;

    /// generate a vector of evaluated arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param args   the arguments to evaluate
    /// @return a vector of evaluated argument
    static Vector* eval (Interp*interp, Nameset* nset, Cons* args);

    /// generate a new vector
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// evaluate a vector member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this vector with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);

  private:
    // make the vector iterator a friend
    friend class VectorIterator;
  };

  /// The VectorIterator class is the iterator for the vector class. Such 
  /// iterator is constructed with the "makeit" vector method. The iterator
  /// is reset to the beginning of the vector.
  /// @author amaury darsch

  class VectorIterator : public Iterator {
  private:
    /// the vector to iterate
    Vector* p_vector;
    /// the current vector index
    long d_index;

  public:
    /// create a new iterator from a vector
    /// @param vec the vector to iterate
    VectorIterator (Vector* vec);

    /// destroy this vectoriterator
    ~VectorIterator (void);

    /// @return the class name
    String repr (void) const;

    /// reset the iterator to the begining
    void begin (void);

    /// reset the iterator to the end
    void end (void);

    /// move the list iterator to the next position
    void next (void);

    /// move the list iterator to the previous position
    void prev (void);

    /// @return the object at the current position
    Object* getobj (void);

    /// @return true if the iterator is at the end
    bool isend (void);

  private:
    // make the copy constructor private
    VectorIterator (const VectorIterator&);
    // make the assignment operator private
    VectorIterator& operator = (const VectorIterator&);
  };
}

#endif
