# vm.asm
#
# The kForth Virtual Machine
#
# Copyright (c) 1998--1999 Krishna Myneni, Creative Consulting for
#   Research and Education
#
# This software is provided under the terms of the General Public License.
#
# Usage from C++
#
#       extern "C" int vm (byte* ip);
#       ecode = vm(ip);
#
# Originally written for the A386 assembler
# Revisions:
#	3-21-98
#	8-25-1998 ported to GNU assembler under Linux
#	9-8-1998  additional functions added: c@, c!, f=, f<>
#	9-10-1998 added: and, or, not, xor
#	9-11-1998 added: = , <>, <, >, <=, >=
#	9-15-1998 added: +!, fsqrt, fsin, fcos, -rot, sp@, rp@
#	9-17-1998 added: a@
#	9-27-1998 added: mod, 2*, 2/
#	10-1-1998 added: pick
#	10-4-1998 fixed signed integer division and mod; added /MOD
#	10-6-1998 added ?dup
#	10-14-1998 added count
#	10-16-1998 fixed L_div error
#	10-19-1998 added 0<, 0=, 0>, true, false
#	10-20-1998 added 2+, 2-
#	02-09-1999 added execute
#	03-01-1999 added open, lseek, close, read, write
#	03-02-1999 added ioctl
#	03-03-1999 added usleep	
#	03-07-1999 added fill, cmove
#	03-09-1999 interchanged meaning of execute and call
#		   to be consistent with ANS Forth
#	03-27-1999 added +loop, unloop
#	03-29-1999 added roll
#	03-31-1999 added cmove>, key
#	05-05-1999 fixed +loop
#	05-06-1999 added fround
#	05-15-1999 added floor
#	05-26-1999 added fatan2, lshift, rshift
#	05-27-1999 added u<, quit, base
#	06-02-1999 added */, */mod
#	06-09-1999 call CPP functions from vm
#	
.equ WSIZE,	4

.equ OP_ADDR,	65
.equ OP_FVAL,	70
.equ OP_IVAL,	73
.equ OP_RET,	238

# Error Codes

.equ E_NOT_ADDR,	1
.equ E_DIV_ZERO,	4
.equ E_RET_STK_CORRUPT,	5
.equ E_UNKNOWN_OP,	6

.data
FCONST_180: .double 180.
JumpTable: .int L_false, L_true, L_nop, L_nop     # 0 -- 3
           .int L_nop, L_nop, L_nop, L_nop        # 4 -- 7
           .int L_nop, L_nop, C_open, C_lseek     # 8 -- 11
           .int C_close, C_read, C_write, C_ioctl # 12 -- 15
           .int L_usleep, L_nop, L_nop, L_nop     # 16 -- 19
           .int L_fill, L_cmove, L_cmovefrom, L_nop # 20 -- 23
           .int L_nop, L_nop, L_nop, L_nop        # 24 -- 27
           .int L_nop, L_nop, CPP_cr__Fv, L_nop       # 28 -- 31
           .int CPP_spaces__Fv, L_store, L_nop, L_nop # 32 -- 35
           .int L_nop, L_mod, L_and, L_nop        # 36 -- 39
           .int L_nop, L_nop, L_mul, L_add        # 40 -- 43
           .int L_nop, L_sub, CPP_dot__Fv, L_div        # 44 -- 47
           .int L_nop, L_nop, L_nop, L_nop        # 48 -- 51
           .int L_nop, L_nop, L_nop, L_nop        # 52 -- 55
           .int L_nop, L_nop, L_nop, L_nop        # 56 -- 59
           .int L_lt, L_eq, L_gt, L_nop           # 60 -- 63
           .int L_fetch, L_addr, L_base, L_call   # 64 -- 67
           .int L_nop, L_nop, L_fval, L_nop       # 68 -- 71
           .int L_nop, L_ival, L_nop, C_key       # 72 -- 75
           .int L_lshift, L_slashmod, C_numberquery, L_nop # 76 -- 79
           .int L_nop, L_nop, L_rshift, CPP_dots__Fv  # 80 -- 83
           .int C_accept, L_nop, L_nop, L_nop     # 84 -- 87
           .int L_starslash, L_starslashmod, L_nop, L_nop  # 88 -- 91
           .int L_nop, L_nop, L_xor, L_nop        # 92 -- 95
           .int L_nop, CPP_allot__Fv, L_binary, L_count # 96 -- 99
           .int L_decimal, CPP_emit__Fv, CPP_fdot__Fv, L_nop # 100 -- 103
           .int L_hex, L_i, L_j, L_nop            # 104 -- 107
           .int L_nop, L_nop, L_nop, L_nop        # 108 -- 111
           .int L_nop, L_nop, L_nop, L_nop        # 112 -- 115
           .int CPP_type__Fv, CPP_udot__Fv, L_nop, CPP_words__Fv # 116 -- 119
           .int L_nop, L_nop, L_nop, L_nop        # 120 -- 123
           .int L_or, L_nop, L_not, L_nop         # 124 -- 127
           .int L_fsin, L_fcos, C_ftan, C_fasin   # 128 -- 131
           .int C_facos, C_fatan, C_fexp, C_fln   # 132 -- 135
           .int C_flog, L_fatan2, L_nop, L_nop    # 136 -- 139
           .int C_fmin, C_fmax, L_floor, L_fround # 140 -- 143
           .int L_nop, L_nop, L_nop, L_nop        # 144 -- 147
           .int L_nop, L_nop, L_stof, L_ftos      # 148 -- 151
           .int L_degtorad, L_radtodeg, L_nop, L_nop  # 152 -- 155
           .int L_nop, L_nop, L_nop, L_nop        # 156 -- 159
           .int L_inc, L_dec, L_abs, L_neg        # 160 -- 163
           .int L_min, L_max, L_twostar, L_twodiv # 164 -- 167
           .int L_twoplus, L_twominus, L_cfetch, L_cstore  # 168 -- 171
           .int L_wfetch, L_wstore, L_dffetch, L_dfstore  # 172 -- 175
           .int L_sffetch, L_sfstore, L_spfetch, L_plusstore # 176 -- 179
           .int L_fadd, L_fsub, L_fmul, L_fdiv    # 180 -- 183
           .int L_fabs, L_fneg, C_fpow, L_fsqrt   # 184 -- 187
           .int L_nop, L_nop, L_feq, L_fne        # 188 -- 191
           .int L_flt, L_fgt, L_fle, L_fge        # 192 -- 195
           .int L_nop, L_nop, L_nop, L_nop        # 196 -- 199
           .int L_drop, L_dup, L_swap, L_over     # 200 -- 203
           .int L_rot, L_minusrot, L_nip, L_tuck  # 204 -- 207
           .int L_pick, L_roll, L_2drop, L_2dup   # 208 -- 211
           .int L_2swap, L_2over, L_2rot, L_depth # 212 -- 215
           .int L_querydup, L_nop, L_nop, L_nop   # 216 -- 219
           .int L_push, L_pop, L_puship, L_rfetch # 220 -- 223
           .int L_rpfetch, L_afetch, L_nop, L_nop # 224 -- 227
           .int L_nop, L_nop, L_jz, L_nop         # 228 -- 231
           .int L_jmp, L_loop, L_plusloop, L_unloop  # 232 -- 235
           .int L_execute, L_nop, L_ret, L_abort  # 236 -- 239
           .int L_quit, L_nop, L_ge, L_le         # 240 -- 243
           .int L_ne, L_zerolt, L_zeroeq, L_zerogt # 244 -- 247
           .int L_ult, L_nop, L_nop, L_nop        # 248 -- 251
           .int L_nop, L_nop, L_nop, L_nop        # 252 -- 255



.text
	.align 4
.global JumpTable
.global L_depth, L_quit, L_abort
.global vm
	.type	vm,@function
vm:	
        pushl %ebp
        pushl %ebx
        pushl %ecx
        pushl %edx
        movl %esp, %ebp
        movl 20(%ebp), %ebx     # load the Forth instruction pointer
        movl %ebx, GlobalIp     
next:
        movl GlobalIp, %ebx
        xor %eax, %eax
        movb (%ebx), %al         # get the opcode
        sall $2, %eax            # determine offset of opcode
        movl $JumpTable, %ebx
        add %eax, %ebx           # address of machine code
	xor %eax, %eax            # error code is zero
        call *(%ebx)              # execute one instruction
        cmpl $0, %eax              # check for error
        jnz exitloop            
        incl GlobalIp           # increment the Forth instruction ptr
        jmp next
exitloop:
        cmpl $OP_RET, %eax         # return from vm?
        jnz vmexit
        xor %eax, %eax            # clear the error
vmexit: 
	pop %edx
        pop %ecx
        pop %ebx
        pop %ebp
        ret
L_nop:
        movl $E_UNKNOWN_OP, %eax   # unknown operation
        ret
L_quit:
	movl $8, %eax		# exit the virtual machine
	ret
L_abort:
	movl BottomOfStack, %eax
	movl %eax, GlobalSp
	movl BottomOfTypeStack, %eax
	movl %eax, GlobalTp
	jmp L_quit
L_base:	
	movl GlobalSp, %ebx
	movl $Base, %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl GlobalTp
	xor %eax, %eax
	ret
L_binary:
	movl $Base, %ebx
	movl $2, (%ebx)
	ret
L_decimal:	
	movl $Base, %ebx
	movl $10, (%ebx)
	ret
L_hex:	
	movl $Base, %ebx
	movl $16, (%ebx)
	ret		
L_false:
	movl GlobalSp, %ebx
	movl $0, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	ret
L_true:
	movl GlobalSp, %ebx
	movl $-1, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	ret
L_ret:
	movl $WSIZE, %eax
        addl %eax, GlobalRp
        incl GlobalRtp
        movl GlobalRtp, %ebx
        cmpl BottomOfReturnTypeStack, %ebx
        jle ret1
        movl $OP_RET, %eax             # exhausted the return stack so exit vm
        jmp retexit
ret1:   movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jz ret2
        movl $E_RET_STK_CORRUPT, %eax   # indicate return stack corrupted
        jmp retexit
ret2:   movl GlobalRp, %ebx
        movl (%ebx), %eax
	movl %eax, GlobalIp		# reset the instruction ptr
        xor %eax, %eax
retexit:
        ret
L_usleep:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	call usleep
	popl %eax
	xor %eax, %eax
	ret
L_fill:
	call L_swap
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz fill2
	popl %ebx
	popl %ebx
	movl $E_NOT_ADDR, %eax
	jmp fillexit
fill2:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call memset
	addl $12, %esp
	xor %eax, %eax
fillexit:	
	ret
L_cmove:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call L_swap
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmove2
	popl %ebx
	movl $E_NOT_ADDR, %eax
	ret
cmove2:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmove3
	popl %ebx
	popl %ebx
	movl $E_NOT_ADDR, %eax
	ret
cmove3:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call memcpy
	addl $12, %esp
	xor %eax, %eax				
	ret		
L_cmovefrom:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx	# load count register
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmovefrom2
	movl $E_NOT_ADDR, %eax						
	ret
cmovefrom2:
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	movl %ecx, %eax
	decl %eax
	addl %eax, %ebx
	movl %ebx, %edx		# dest addr in %edx
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmovefrom3
	movl $E_NOT_ADDR, %eax
	ret
cmovefrom3:
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	movl %ecx, %eax
	decl %eax
	addl %eax, %ebx		# src addr in %ebx
cmovefromloop:	
	movb (%ebx), %al
	decl %ebx
	xchgl %ebx, %edx
	movb %al, (%ebx)
	decl %ebx
	xchgl %ebx, %edx
	loop cmovefromloop	
	xor %eax, %eax
	ret
L_call:	
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	call *(%ebx)
	xor %eax, %eax
	ret 		
L_push:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        movl GlobalRp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalRp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        movl GlobalRtp, %ebx
        movb %al, (%ebx)
        decl GlobalRtp
        xor %eax, %eax
        ret
L_pop:
	movl $WSIZE, %eax
        addl %eax, GlobalRp
        incl GlobalRtp
        movl GlobalRp, %ebx
        mov (%ebx), %eax
        movl GlobalSp, %ebx
        mov %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalRtp, %ebx
        movb (%ebx), %al
        movl GlobalTp, %ebx
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
        ret
L_puship:
        movl GlobalIp, %eax
        movl GlobalRp, %ebx
        mov %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalRp
        movl GlobalRtp, %ebx
	movb $OP_ADDR, %al
        movb %al, (%ebx)
        decl GlobalRtp
        xor %eax, %eax
        ret
L_execute:	
        movl GlobalIp, %eax
        movl GlobalRp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax 
        subl %eax, GlobalRp
        movl GlobalRtp, %ebx
	movb $OP_ADDR, %al
        movb %al, (%ebx)
        decl GlobalRtp
	movl $WSIZE, %eax
        addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
        movl (%ebx), %eax
	decl %eax
	movl %eax, GlobalIp
        xor %eax, %eax
        ret
L_rfetch:
        movl GlobalRp, %ebx
        addl $WSIZE, %ebx
        movl (%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        movl GlobalTp, %ebx
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
	ret
L_rpfetch:
	movl GlobalRp, %eax
	addl $WSIZE, %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	ret
L_spfetch:
	movl GlobalSp, %eax
	movl %eax, %ebx
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax 
	ret
L_i:
        movl GlobalRtp, %ebx
        movb 3(%ebx), %al
        movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl GlobalTp
        movl GlobalRp, %ebx
        movl 3*WSIZE(%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        xor %eax, %eax
        ret
L_j:
        movl GlobalRtp, %ebx
        movb 6(%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl GlobalTp
        movl GlobalRp, %ebx
        movl 6*WSIZE(%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        xor %eax, %eax
        ret
L_loop:
        movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz loopbad
        movl GlobalRp, %ebx
	movl $WSIZE, %eax
	sall $1, %eax
	addl $WSIZE, %eax
	addl %eax, %ebx
        movl (%ebx), %eax
        incl %eax
        subl $WSIZE, %ebx
        cmp (%ebx), %eax
        jz L_unloop
loop1:	addl $WSIZE, %ebx          # set loop counter to incremented value
        movl %eax, (%ebx)
        subl $WSIZE, %ebx
	subl $WSIZE, %ebx
        movl (%ebx), %eax
        movl %eax, GlobalIp      # set instruction ptr to start of loop
        xor %eax, %eax
        ret
L_unloop:  
	movl $WSIZE, %eax
	imul $3, %eax
	addl %eax, GlobalRp  # terminal count reached, discard top 3 items
	movl $3, %eax
        addl %eax, GlobalRtp
        xor %eax, %eax
        ret
loopbad:
        movl $E_RET_STK_CORRUPT, %eax
        ret	
L_plusloop:
	movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz loopbad
        movl GlobalRp, %ebx
	movl $WSIZE, %eax
	sall $1, %eax
	addl $WSIZE, %eax
	addl %eax, %ebx
        movl (%ebx), %eax
	pushl %eax
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	popl %eax
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	addl %ebx, %eax	
	movl GlobalRp, %ebx
	addl $WSIZE, %ebx
	addl $WSIZE, %ebx
	cmpl (%ebx), %eax
	jge L_unloop
	jmp loop1					
L_jz:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        cmpl $0, %eax
        jz jz1
	movl $4, %eax
        addl %eax, GlobalIp       # do not jump
        jmp jzexit
jz1:    movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax          # get the relative jump count
        decl %eax
        addl %eax, GlobalIp
jzexit: xor %eax, %eax
        ret
L_jmp:
        movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax          # get the relative jump count
        addl %eax, %ebx
        subl $2, %ebx
        movl %ebx, GlobalIp      # set instruction ptr
	xor %eax, %eax
        ret
L_count:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz counterror
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	xor %eax, %eax
	movb (%ebx), %al
	movl GlobalSp, %ebx
	incl WSIZE(%ebx)
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	xor %eax, %eax
	ret
counterror:
	movl $E_NOT_ADDR, %eax
	ret
L_ival:
        movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	subl $WSIZE, %ebx
        movl %ebx, GlobalSp    # decrement the stack ptr
        movl GlobalTp, %ebx
	movb $OP_IVAL, %al
        movb %al, (%ebx)     # set type and dec
        decl GlobalTp
	movl $WSIZE, %eax
        addl %eax, GlobalIp    # advance the instruction ptr
        xor %eax, %eax            # no errors
        ret
L_addr:
        call L_ival
        movl GlobalTp, %ebx
	movb $OP_ADDR, %al
        movb %al, 1(%ebx)   # set type
	xor %eax, %eax
        ret
L_fval:
        movl GlobalIp, %ebx
        incl %ebx
        fldl (%ebx)
        movl GlobalSp, %ebx
        subl $WSIZE, %ebx
        fstpl (%ebx)
	movl $WSIZE, %eax
	sall $1, %eax
        subl %eax, GlobalSp
        movl GlobalTp, %ebx
        movb $OP_FVAL, (%ebx)
        decl %ebx
        movb $OP_FVAL, (%ebx)
	decl GlobalTp
	decl GlobalTp
        addl %eax, GlobalIp
	xor %eax, %eax
        ret
L_and:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	pushl %ebx
	movl (%ebx), %ebx
	andl %ebx, %eax
	popl %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_or:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	orl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_not:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	notl %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	ret
L_xor:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	xorl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_lshift:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx
	movl WSIZE(%ebx), %eax
	shll %cl, %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	ret
L_rshift:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx
	movl WSIZE(%ebx), %eax
	shrl %cl, %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	ret
L_eq:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %ebx, %eax
	jne eq2
eq1:	movl GlobalSp, %ebx
	movl $-1, WSIZE(%ebx)
	jmp eq3
eq2:    movl GlobalSp, %ebx
	movl $0, WSIZE(%ebx)
eq3:    movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_ne:
	call L_eq
	call L_not
	ret
L_ult:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jae eq2
	jmp eq1
	ret
L_lt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jge eq2
	jmp eq1
L_gt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jle eq2
	jmp eq1
L_le:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jg eq2
	jmp eq1
L_ge:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jl eq2
	jmp eq1
L_zerolt:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	jl zerolt2
zerolt1:
	movl $0, WSIZE(%ebx)
	jmp zeroltexit
zerolt2:
	movl $-1, WSIZE(%ebx)
zeroltexit:
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_zeroeq:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je zerolt2
	jmp zerolt1
L_zerogt:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	jg zerolt2
	jmp zerolt1 	
L_querydup:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je L_querydupexit
	movl %eax, (%ebx)
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	movb %al, (%ebx)
	decl GlobalTp
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	xor %eax, %eax
L_querydupexit:
	ret	
L_drop:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	xor %eax, %eax
        ret
L_dup:
        movl GlobalSp, %ebx
        movl WSIZE(%ebx), %eax
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalTp, %ebx
        movb 1(%ebx), %al
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
        ret
L_swap:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        movl WSIZE(%ebx), %eax
        xchgl (%ebx), %eax
        xchgl WSIZE(%ebx), %eax
        movl GlobalTp, %ebx
        incl %ebx
        movb 1(%ebx), %al
        xchgb (%ebx), %al
        xchgb 1(%ebx), %al
        xor %eax, %eax
        ret
L_over:
        movl GlobalSp, %ebx
        movl 2*WSIZE(%ebx), %eax
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalTp, %ebx
        movb 2(%ebx), %al
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
        ret
L_rot:
        call L_swap
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        movl 2*WSIZE(%ebx), %eax
        xchgl (%ebx), %eax
        xchgl 2*WSIZE(%ebx), %eax
        movl GlobalTp, %ebx
        incl %ebx
        movb  2(%ebx), %al
        xchgb (%ebx), %al
        xchgb 2(%ebx), %al
        xor %eax, %eax
        ret
L_minusrot:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	movl -2*WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	movb %al, (%ebx)
	incl %ebx
	movw 1(%ebx), %ax
	movw %ax, (%ebx)
	movb -1(%ebx), %al
	movb %al, 2(%ebx)
	xor %eax, %eax
	ret
L_nip:
        call L_swap
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	xor %eax, %eax
        ret
L_tuck:
        call L_swap
        call L_over
        ret
L_pick:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	addl $2, %eax
	imul $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	movl WSIZE(%ebx), %eax
	addl $2, %eax
	movl GlobalTp, %ebx
	addl %eax, %ebx
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, 1(%ebx)
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	ret
L_roll:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx 
	movl (%ebx), %eax
	pushl %eax
	pushl %ebx
	incl %eax
	imul $WSIZE, %eax
	addl %eax, %ebx		# addr of item to roll
	movl (%ebx), %eax
	popl %ebx
	movl %eax, (%ebx)
	popl %eax
	incl %eax		# number of cells to copy
	movl %eax, %ecx
	imul $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, %edx		# dest addr
	subl $WSIZE, %ebx	# src addr
rollloop:
	movl (%ebx), %eax
	sub $WSIZE, %ebx
	xchgl %ebx, %edx
	movl %eax, (%ebx)
	sub $WSIZE, %ebx
	xchgl %ebx, %edx
	loop rollloop	
	xor %eax, %eax
	ret
L_depth:
	movl GlobalSp, %ebx
	movl BottomOfStack, %eax
	subl %ebx, %eax
	movl $WSIZE, (%ebx)
	movl $0, %edx
	idivl (%ebx)
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	xor %eax, %eax
        ret
L_2drop:
	movl $WSIZE, %eax
	addl %eax, %eax
	addl %eax, GlobalSp
        addl $2, GlobalTp
	xor %eax, %eax
        ret
L_2dup:
	movl GlobalSp, %ebx
	movl 2*WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl 2*WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb 2(%ebx), %al
	movb %al, (%ebx)
	decl %ebx
	movb 2(%ebx), %al
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
        ret
L_2swap:
        call L_rot              # not very efficient --- recode later
        call L_push
        call L_rot
        call L_pop
        ret
L_2over:
        call L_push             # not very efficient --- recode later
        call L_push
        call L_2dup
        call L_pop
        call L_pop
        call L_2swap
        ret
L_2rot:
        call L_push             # not very efficient --- recode later
        call L_push
        call L_2swap
        call L_pop
        call L_pop
        call L_2swap
        ret
L_fetch:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_IVAL, (%ebx)
        decl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %ebx
        movl (%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
	xor %eax, %eax
	ret
fetcherror:
        movl $E_NOT_ADDR, %eax
        ret
L_store:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        pushl %eax                # push address onto stack
        addl $WSIZE, %ebx
        movl (%ebx), %eax
        pop %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        xor %eax, %eax
        ret
L_afetch:
	call L_fetch
	movl GlobalTp, %ebx
	movb $OP_ADDR, 1(%ebx)
	ret
L_cfetch:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	movb (%ebx), %al
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
        ret
L_cstore:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	pop %ebx
	movb %al, (%ebx)
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	xor %eax, %eax
	ret	
L_wfetch:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movb $OP_IVAL, 1(%ebx)
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	movw (%ebx), %ax
	cwde
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
        ret
L_wstore:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	pop %ebx
	movw %ax, (%ebx)
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	xor %eax, %eax
        ret
L_sffetch:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_FVAL, (%ebx)
        decl %ebx
        movb $OP_FVAL, (%ebx)
        decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %ebx
        flds (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalSp, %ebx
        fstpl (%ebx)
        subl %eax, GlobalSp
	xor %eax, %eax
        ret
L_sfstore:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)              # load the f number into NDP
        subl $WSIZE, %ebx
        movl (%ebx), %ebx          # load the dest address
        fstps (%ebx)             # store as single precision float
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        ret
L_dffetch:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_FVAL, (%ebx)
        decl %ebx
        movb $OP_FVAL, (%ebx)
        decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %ebx                  # get address of fval
        fldl (%ebx)                      # load fval into the NDP
        movl GlobalSp, %ebx
        subl $WSIZE, %ebx
        fstpl (%ebx)
	movl $WSIZE, %eax
	sall $1, %eax
        subl %eax, GlobalSp
	xor %eax, %eax
        ret
L_dfstore:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        pushl %eax                # push address onto stack
        addl $WSIZE, %ebx
        fldl (%ebx)
        pop %ebx
        fstpl (%ebx)
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
        xor %eax, %eax
        ret
L_inc:
        movl GlobalSp, %ebx
        incl WSIZE(%ebx)
        ret
L_dec:
        movl GlobalSp, %ebx
        decl WSIZE(%ebx)
        ret
L_twoplus:
	movl GlobalSp, %ebx
	incl WSIZE(%ebx)
	incl WSIZE(%ebx)
	ret
L_twominus:
	movl GlobalSp, %ebx
	decl WSIZE(%ebx)
	decl WSIZE(%ebx)
	ret
L_abs:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	cmpl $0, %eax
	jl abs1
	xor %eax, %eax
	ret
abs1:	negl %eax
	movl %eax, (%ebx)
	xor %eax, %eax
        ret
L_neg:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	negl %eax
	movl %eax, (%ebx)
	xor %eax, %eax
        ret
L_max:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jl max1
	movl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	jmp maxexit
max1:
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
maxexit:
	incl GlobalTp
	xor %eax, %eax
        ret
L_min:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jg min1
	movl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	jmp minexit
min1:
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
minexit:
	incl GlobalTp
	xor %eax, %eax
        ret
L_twostar:
	movl GlobalSp, %ebx
	sall $1, WSIZE(%ebx)
	ret
L_twodiv:
	movl GlobalSp, %ebx
	sarl $1, WSIZE(%ebx)
	ret
L_add:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        addl %eax, WSIZE(%ebx)
        incl GlobalTp
        xor %eax, %eax
        ret
L_sub:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        subl %eax, WSIZE(%ebx)
        incl GlobalTp
        xor %eax, %eax
        ret
L_mul:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        addl $WSIZE, %ebx
        imull (%ebx)
        movl %eax, (%ebx)
        incl GlobalTp
        xor %eax, %eax
        ret
L_div:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        cmpl $0, %eax
        jnz div1
        movl $E_DIV_ZERO, %eax
        jmp divexit
div1:	
	addl $WSIZE, %ebx
        movl (%ebx), %eax
	cdq
        idivl -WSIZE(%ebx)
        movl %eax, (%ebx)
	xor %eax, %eax
divexit:
        ret
L_mod:
	call L_div
	movl %edx, (%ebx)
	ret
L_slashmod:
	call L_div
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_swap
	ret
L_starslash:
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        movl WSIZE(%ebx), %eax
        imull (%ebx)
	idivl -WSIZE(%ebx)
	movl %eax, WSIZE(%ebx)
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax		
	ret
L_starslashmod:
	call L_starslash
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_swap
	ret
L_plusstore:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	push %ebx
	push %ebx
	push %ebx
	movl WSIZE(%ebx), %ebx
	movl (%ebx), %eax
	pop %ebx
	movl 2*WSIZE(%ebx), %ebx
	addl %ebx, %eax
	pop %ebx
	movl WSIZE(%ebx), %ebx
	movl %eax, (%ebx)
	pop %ebx
	movl $WSIZE, %eax
	sall $1, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
	ret 		
L_stof:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        fildl (%ebx)
        movl GlobalTp, %ebx
        movb $OP_FVAL, (%ebx)
        decl %ebx
        movb $OP_FVAL, (%ebx)
	decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
	movl $WSIZE, %eax
        subl %eax, %ebx
        fstpl (%ebx)
	sall $1, %eax
        subl %eax, GlobalSp
	xor %eax, %eax
        ret
L_ftos:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
        addl %eax, %ebx
        fistpl (%ebx)
        incl GlobalTp
        movl GlobalTp, %ebx
        incl %ebx
        movb $OP_IVAL, (%ebx)
	xor %eax, %eax
        ret
L_degtorad:
	fldl FCONST_180
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fdivp %st, %st(1)
        fldpi
        fmulp %st, %st(1)
        fstpl (%ebx)
        ret
L_radtodeg:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fldpi
	fxch
        fdivp %st, %st(1)
        fldl FCONST_180
        fmulp %st, %st(1)
        fstpl (%ebx)
        ret
L_fne:
	call L_feq
	movl GlobalSp, %ebx
	notl WSIZE(%ebx)
	ret
L_feq:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	fucompp
	fnstsw %ax
	andb $69, %ah
	xorb $64, %ah
	jne flt2
	jmp flt1
L_flt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fcompl (%ebx)
	fnstsw %ax
	andb $69, %ah
	jne flt2
flt1:	movl GlobalSp, %ebx
	movl $-1, WSIZE(%ebx)
	jmp fltexit
flt2:   movl GlobalSp, %ebx
	movl $0, WSIZE(%ebx)
fltexit:
	movl $4, %eax
	addl %eax, GlobalTp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	xor %eax, %eax
	ret
L_fgt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fcompl (%ebx)
	fnstsw %ax
	andb $69, %ah
	cmpb $1, %ah
	jne flt2
	jmp flt1
L_fle:
	call L_2over
	call L_2over
	call L_feq
	call L_push
	call L_flt
	call L_pop
	call L_or
	ret
L_fge:
	call L_2over
	call L_2over
	call L_feq
	call L_push
	call L_fgt
	call L_pop
	call L_or
	ret
L_fadd:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	sall $1, %eax
        addl %eax, %ebx
        faddl (%ebx)
        fstpl (%ebx)
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        ret
L_fsub:
	movl $WSIZE, %eax
	sall $1, %eax
	addl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	subl $WSIZE, %eax
        subl %eax, %ebx
        fsubl (%ebx)
        addl %eax, %ebx
        fstpl (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        ret
L_fmul:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	sall $1, %eax
        addl %eax, %ebx
        fmull (%ebx)
        fstpl (%ebx)
	sarl $1, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        ret
L_fdiv:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
        ftst
        jnz fdiv1
        subl %eax, GlobalSp
        movl $E_DIV_ZERO, %eax
        jmp fdivexit
fdiv1:  addl $WSIZE, %ebx
	addl $WSIZE, %ebx
        fdivrl (%ebx)
        fstpl (%ebx)
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
fdivexit:
	xor %eax, %eax
	ret
L_fabs:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fabs
        fstpl (%ebx)
        ret
L_fneg:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fchs
        fstpl (%ebx)
        ret
L_floor:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	pushl %eax
	movl (%ebx), %eax
	pushl %eax
	call floor
	addl $8, %esp
	fstpl (%ebx)
	xor %eax, %eax		
	ret
L_fround:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	fldl (%ebx)
	frndint
	fstpl (%ebx)
	ret
L_fsqrt:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fsqrt
	fstpl WSIZE(%ebx)
	ret
L_fcos:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fcos
	fstpl WSIZE(%ebx)
	ret
L_fsin:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fsin
	fstpl WSIZE(%ebx)
	ret
L_fatan2:
	movl GlobalSp, %ebx
	addl $2*WSIZE, %ebx
	fldl WSIZE(%ebx)
	fldl -WSIZE(%ebx)
	fpatan
	fstpl WSIZE(%ebx)
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	ret
L_fpow:
#        add GlobalSp, WSIZE
#        mov ebx, GlobalSp
#        FLD Q[ebx]
#        add ebx, 2*WSIZE
#        FLD Q[ebx]
#        FYL2X
#        # FLD1
#        # FSCALE
#        FSTP Q[ebx]
#        add GlobalSp, WSIZE
#        add GlobalTp, 2
        ret
#

	.comm GlobalSp,4,4
	.comm GlobalTp,4,4
	.comm GlobalIp,4,4
	.comm GlobalRp,4,4
	.comm GlobalRtp,4,4
	.comm BottomOfStack,4,4
	.comm BottomOfReturnStack,4,4
	.comm BottomOfTypeStack,4,4
	.comm BottomOfReturnTypeStack,4,4
	.comm Base,4,4
	.comm State,4,4
	.comm To_In,4,4
	.comm TIB,256,1
