/* File "scanner.h":
 * Defines lexical analysis of malaga source files. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* constants ================================================================*/

/* The next token is read by "read_next_token".
 * Its value is stored in <next_token>.
 * A token that consists of one character has the code of that character.
 * A token value of EOF stands for end-of-file.
 * All other tokens are as follows: */
enum
{
  TOK_STRING = 256,      /* a string */
  TOK_IDENT,             /* an identifier */
  TOK_VARIABLE,          /* a variable name */
  TOK_CONSTANT,          /* a constant name */
  TOK_NUMBER,            /* a floating number */ 
  TOK_ASSIGN,            /* := */
  TOK_ASSIGN_PLUS,       /* :=+ */
  TOK_ASSIGN_MINUS,      /* :=- */
  TOK_ASSIGN_ASTERISK,   /* :=* */
  TOK_ASSIGN_SLASH,      /* :=/ */
  TOK_NOT_EQUAL,         /* /= */
  TOK_NOT_CONGRUENT,     /* /~ */
  TOK_ACCEPT,            /* accept */
  TOK_ALLO_RULE,         /* allo_rule */
  TOK_AND,               /* and */
  TOK_ASSERT,            /* assert */
  TOK_CHOOSE,            /* choose */
  TOK_COMBI_RULE,        /* combi_rule */
  TOK_DEFINE,            /* define */
  TOK_ELSE,              /* else */
  TOK_ELSEIF,            /* elseif */
  TOK_END,               /* end */
  TOK_END_RULE,          /* end_rule */
  TOK_ERROR,             /* error */
  TOK_FAIL,              /* fail */
  TOK_FILTER_RULE,       /* filter_rule */
  TOK_FOREACH,           /* foreach */
  TOK_GREATER,           /* greater */
  TOK_GREATER_EQUAL,     /* greater_equal */
  TOK_IF,                /* if */
  TOK_IN,                /* in */
  TOK_INCLUDE,           /* include */
  TOK_INITIAL,           /* initial */
  TOK_INPUT_RULE,        /* input_rule */
  TOK_LESS,              /* less */
  TOK_LESS_EQUAL,        /* less_equal */
  TOK_MATCHES,           /* matches */
  TOK_NOT,               /* not */
  TOK_OR,                /* or */
  TOK_PARALLEL,          /* parallel */
  TOK_PRUNING_RULE,      /* pruning_rule */
  TOK_REQUIRE,           /* require */
  TOK_RESULT,            /* result */
  TOK_RETURN,            /* return */
  TOK_ROBUST_RULE,       /* robust_rule */
  TOK_RULES,             /* rules */
  TOK_SUBRULE,           /* subrule */
  TOK_THEN               /* then */
};

#define FIRST_KEYWORD TOK_ACCEPT
#define LAST_KEYWORD TOK_THEN

#define NUMBER_OF_KEYWORDS (LAST_KEYWORD - FIRST_KEYWORD + 1)

#define TOKEN_NAME_MAX 512

/* variables ================================================================*/

GLOBAL short_t next_token; /* Next token that is to be consumed by parser. */

GLOBAL char token_name[TOKEN_NAME_MAX];
/* If <next_token> == TOK_IDENT or <next_token> == TOK_STRING,
 * its content is in <token_name>. */

GLOBAL double token_number;
/* If <next_token> == TOK_NUMBER, its content is in <token_number>. */

/* functions ================================================================*/

extern void set_scanner_input (string_t input);
/* Let the scanner use <input> as scanner input.
 * <input> must remain valid until the scanner has done its work. */

extern void include_file (string_t file_name, long_t file_name_index);
/* Open a new level of inclusion and read tokens from <file_name>.
 * <file_name_index> is index of the current file name in the string pool. */ 

extern void end_include (void);
/* Stop reading from current source stream and read from former file. */

extern void stop_scanner (void);
/* Stop the scanner in case of an emergency. */

extern string_t current_file_name (void);
/* Return the name of the file reading from or NULL. */

extern long_t current_file_name_index (void);
/* Return the name index of the file reading from or -1. */

extern long_t current_line_number (void);
/* Return the line number where the last char has been read or -1. */

extern long_t current_column (void);
/* Return the column where the last char has been read or -1. */

extern void read_next_token (void);
/* Read the next token from current source into <next_token>.
 * If end of input stream is reached, return EOF. */

extern void test_token (short_t token);
/* Test if <token> is the next token. If it's not, report an error. */

extern void parse_token (short_t token);
/* Test if <token> is the next token and read next token. */

extern string_t token_as_text (short_t token);
/* Return <token> as a string readable for humans.
 * Note that the string is only valid until this function is called again. */

extern void delete_escapes (string_t string);
/* Delete any "\" in <string> (except it is preceeded by a "\"). */

/*---------------------------------------------------------------------------*/
