/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysvideo.c,v 1.4 1999/07/15 18:42:02 slouken Exp $";
#endif

/* SVGAlib based SDL video driver implementation.
*/

#include <stdlib.h>
#include <stdio.h>

#include <vga.h>
#include <vgamouse.h>
#include <vgakeyboard.h>

#include "SDL.h"
#include "SDL_error.h"
#include "SDL_video.h"
#include "SDL_mouse.h"
#include "SDL_sysvideo.h"
#include "SDL_video_c.h"
#include "SDL_pixels_c.h"
#include "SDL_sysmouse_c.h"
#include "SDL_events_c.h"
#include "SDL_lowvideo.h"


/* Hardware surface functions */
static int SDL_AllocHWSurface(SDL_Surface *surface);
static int SDL_LockHWSurface(SDL_Surface *surface);
static void SDL_UnlockHWSurface(SDL_Surface *surface);
static int SDL_FlipHWSurface(SDL_Surface *surface);
static void SDL_FreeHWSurface(SDL_Surface *surface);

/* Private display data */
#define NUM_MODELISTS	4		/* 8, 16, 24, and 32 bits-per-pixel */

static int SDL_nummodes[NUM_MODELISTS];
static SDL_Rect **SDL_modelist[NUM_MODELISTS];
static int *SDL_vgamode[NUM_MODELISTS];

static int AddMode(int mode, int actually_add)
{
	vga_modeinfo *modeinfo;

	modeinfo = vga_getmodeinfo(mode);
	if ( modeinfo->flags & CAPABLE_LINEAR ) {
		int i, j;

		i = modeinfo->bytesperpixel-1;
		if ( actually_add ) {
			SDL_Rect saved_rect[2];
			int      saved_mode[2];
			int b;

			/* Add the mode, sorted largest to smallest */
			b = 0;
//printf("Level %d: ", i);
			j = 0;
			while ( (SDL_modelist[i][j]->w > modeinfo->width) ||
			        (SDL_modelist[i][j]->h > modeinfo->height) ) {
//printf("%dx%d ", SDL_modelist[i][j]->w, SDL_modelist[i][j]->h);
				++j;
			}
			/* Skip modes that are already in our list */
			if ( (SDL_modelist[i][j]->w == modeinfo->width) &&
			     (SDL_modelist[i][j]->h == modeinfo->height) ) {
//printf("%dx%d -- skipped\n", SDL_modelist[i][j]->w, SDL_modelist[i][j]->h);
				return(0);
			}
			/* Insert the new mode */
			saved_rect[b] = *SDL_modelist[i][j];
			saved_mode[b] = SDL_vgamode[i][j];
			SDL_modelist[i][j]->w = modeinfo->width;
			SDL_modelist[i][j]->h = modeinfo->height;
			SDL_vgamode[i][j] = mode;
//printf("^%dx%d %d^ ", SDL_modelist[i][j]->w, SDL_modelist[i][j]->h, modeinfo->colors);
			/* Everybody scoot down! */
			if ( saved_rect[b].w && saved_rect[b].h ) {
			    for ( ++j; SDL_modelist[i][j]->w; ++j ) {
				saved_rect[!b] = *SDL_modelist[i][j];
				saved_mode[!b] = SDL_vgamode[i][j];
				*SDL_modelist[i][j] = saved_rect[b];
				SDL_vgamode[i][j] = saved_mode[b];
				b = !b;
//printf("%dx%d ", SDL_modelist[i][j]->w, SDL_modelist[i][j]->h);
			    }
			    *SDL_modelist[i][j] = saved_rect[b];
			    SDL_vgamode[i][j] = saved_mode[b];
//printf("%dx%d ", SDL_modelist[i][j]->w, SDL_modelist[i][j]->h);
			}
//printf("\n");
		} else {
			++SDL_nummodes[i];
		}
	}
	return(modeinfo->flags & CAPABLE_LINEAR);
}

static void SDL_UpdateVideoInfo(SDL_VideoInfo *info)
{
	vga_modeinfo *modeinfo;

	info->hw_available = 1;
	modeinfo = vga_getmodeinfo(vga_getcurrentmode());
	info->video_mem = (modeinfo->maxpixels/1024);
	if ( modeinfo->bytesperpixel > 0 ) {
		info->video_mem *= modeinfo->bytesperpixel;
	}
	/* FIXME: Add hardware accelerated blit information */
//printf("Video memory available: %dK\n", info->video_mem);
//printf("Hardware accelerated blit: %savailable\n", modeinfo->haveblit ? "" : "not ");
}

int SDL_SYS_VideoInit(SDL_PixelFormat *vformat)
{
	extern int _SDL_lockSO;
	int keyboard;
	int i, j;
	int mode, total_modes;

	/* Initialize all variables that we clean on shutdown */
	for ( i=0; i<NUM_MODELISTS; ++i ) {
		SDL_nummodes[i] = 0;
		SDL_modelist[i] = NULL;
		SDL_vgamode[i] = NULL;
	}

	/* Initialize the library */
	vga_disabledriverreport();
	if ( vga_init() < 0 ) {
		SDL_SetError("Unable to initialize SVGAlib");
		return(-1);
	}
	_SDL_lockSO = 1;  /* SVGAlib sets an atexit() handler */

	/* Enable mouse and keyboard support */
	vga_setmousesupport(1);
	keyboard = keyboard_init_return_fd();
	if ( keyboard < 0 ) {
		SDL_SetError("Unable to initialize keyboard");
		return(-1);
	}
	SDL_vgainitkeymaps(keyboard);
	keyboard_seteventhandler(SDL_vgakeyboardcallback);

	/* Determine the screen depth (use default 8-bit depth) */
	vformat->BitsPerPixel = 8;

	/* Enumerate the available fullscreen modes */
	total_modes = 0;
	for ( mode=vga_lastmodenumber(); mode; --mode ) {
		if ( vga_hasmode(mode) ) {
			if ( AddMode(mode, 0) ) {
				++total_modes;
			}
		}
	}
	if ( total_modes == 0 ) {
		SDL_SetError("No linear video modes available");
		return(-1);
	}
	for ( i=0; i<NUM_MODELISTS; ++i ) {
		SDL_vgamode[i] = (int *)malloc(SDL_nummodes[i]*sizeof(int));
		if ( SDL_vgamode[i] == NULL ) {
			SDL_OutOfMemory();
			return(-1);
		}
		SDL_modelist[i] = (SDL_Rect **)
				malloc((SDL_nummodes[i]+1)*sizeof(SDL_Rect *));
		if ( SDL_modelist[i] == NULL ) {
			SDL_OutOfMemory();
			return(-1);
		}
		for ( j=0; j<SDL_nummodes[i]; ++j ) {
			SDL_modelist[i][j]=(SDL_Rect *)malloc(sizeof(SDL_Rect));
			if ( SDL_modelist[i][j] == NULL ) {
				SDL_OutOfMemory();
				return(-1);
			}
			memset(SDL_modelist[i][j], 0, sizeof(SDL_Rect));
		}
		SDL_modelist[i][j] = NULL;
	}
	for ( mode=vga_lastmodenumber(); mode; --mode ) {
		if ( vga_hasmode(mode) ) {
			AddMode(mode, 1);
		}
	}
	/* Free extra (duplicated) modes */
	for ( i=0; i<NUM_MODELISTS; ++i ) {
		j = 0;
		while ( SDL_modelist[i][j] && SDL_modelist[i][j]->w ) {
			j++;
		}
		while ( SDL_modelist[i][j] ) {
//printf("Freeing unused mode in level %d\n", i);
			free(SDL_modelist[i][j]);
			SDL_modelist[i][j] = NULL;
			j++;
		}
	}

	/* Fill in our hardware acceleration capabilities */
	SDL_UpdateVideoInfo(&SDL_HWCaps.info);
	SDL_HWCaps.AllocHWSurface = SDL_AllocHWSurface;
	SDL_HWCaps.LockHWSurface = SDL_LockHWSurface;
	SDL_HWCaps.UnlockHWSurface = SDL_UnlockHWSurface;
	//SDL_HWCaps.FlipHWSurface = SDL_FlipHWSurface;
	SDL_HWCaps.FreeHWSurface = SDL_FreeHWSurface;

	/* We're done! */
	return(0);
}

SDL_Rect **SDL_SYS_ListModes(SDL_Surface *screen,
				SDL_PixelFormat *format, Uint32 flags)
{
	return(SDL_modelist[((format->BitsPerPixel+7)/8)-1]);
}

/* Various screen update functions available */
static void SDL_DirectUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);
static void SDL_BankedUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);

SDL_Surface *SDL_SYS_SetVideoMode(SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	int mode;
	vga_modeinfo *modeinfo;

	/* Since SVGAlib 1.40 leaks the mouse here, we have to close it */
	mouse_close();

	/* Try to set the requested linear video mode */
	bpp = (bpp+7)/8-1;
	for ( mode=0; SDL_modelist[bpp][mode]; ++mode ) {
		if ( (SDL_modelist[bpp][mode]->w == width) &&
		     (SDL_modelist[bpp][mode]->h == height) ) {
			break;
		}
	}
	if ( SDL_modelist[bpp][mode] == NULL ) {
		SDL_SetError("Couldn't find requested mode in list");
		return(NULL);
	}
//printf("Setting video mode %d\n", SDL_vgamode[bpp][mode]);
	vga_setmode(SDL_vgamode[bpp][mode]);
	vga_setpage(0);
	if ( vga_setlinearaddressing() < 0 ) {
		SDL_SetError("Unable to set linear addressing");
		return(NULL);
	}
	modeinfo = vga_getmodeinfo(SDL_vgamode[bpp][mode]);

	/* Update hardware acceleration info */
	SDL_UpdateVideoInfo(&SDL_HWCaps.info);

	/* Get the new pixel format for the screen (is this right?) */
	SDL_FreeFormat(current->format);
	bpp = (bpp+1)*8;
	if ( (bpp == 16) && (modeinfo->colors == 32768) ) {
		bpp = 15;
	}
	current->format = SDL_AllocFormat(bpp, 0, 0, 0, 0);
	if ( current->format == NULL ) {
		return(NULL);
	}

	/* Set up the new mode framebuffer */
	current->flags = (SDL_FULLSCREEN|SDL_HWSURFACE);
	current->w = width;
	current->h = height;
	current->pitch = modeinfo->linewidth;
	current->pixels = vga_getgraphmem();
//printf("Set mode %dx%dx%d with pitch %d and pixels %p\n", current->w, current->h, current->format->BitsPerPixel, current->pitch, current->pixels);

	/* Set the blit function */
	SDL_SYS_UpdateRects = SDL_DirectUpdate;

	/* Set up the mouse handler again (buggy SVGAlib 1.40) */
	mouse_seteventhandler(SDL_vgamousecallback);

	/* We're done */
	return(current);
}

/* We don't actually allow hardware surfaces other than the main one */
static int SDL_AllocHWSurface(SDL_Surface *surface)
{
	return(-1);
}
static void SDL_FreeHWSurface(SDL_Surface *surface)
{
	return;
}

/* We need to wait for vertical retrace on page flipped displays */
static int SDL_LockHWSurface(SDL_Surface *surface)
{
	if ( (surface->flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF ) {
		vga_waitretrace();
	}
	return(0);
}
static void SDL_UnlockHWSurface(SDL_Surface *surface)
{
	return;
}

/* FIXME: How is this done with SVGAlib? */
static int SDL_FlipHWSurface(SDL_Surface *surface)
{
	return(0);
}

static void SDL_DirectUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	return;
}

/* FIXME: Can this be used under SVGAlib? */
static void SDL_BankedUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	return;
}

int SDL_SYS_SetColors(SDL_Surface *screen, int firstcolor, int ncolors)
{
	SDL_Color *colors;
	int i, which;

	colors = screen->format->palette->colors;
	for ( i=ncolors, which=firstcolor; i; --i, ++which ) {
		vga_setpalette(which, colors[which].r>>2,
		                      colors[which].g>>2,
		                      colors[which].b>>2);
	}
	return(1);
}

/* Note:  If we are terminated, this could be called in the middle of
   another SDL video routine -- notably SDL_SYS_UpdateRects.
*/
void SDL_SYS_VideoQuit(SDL_Surface *screen)
{
	int i, j;

	/* Reset the console video mode */
	if ( screen && (screen->w && screen->h) ) {
		vga_setmode(TEXT);
	}
	keyboard_close();

	/* Free video mode lists */
	for ( i=0; i<NUM_MODELISTS; ++i ) {
		if ( SDL_modelist[i] != NULL ) {
			for ( j=0; SDL_modelist[i][j]; ++j )
				free(SDL_modelist[i][j]);
			free(SDL_modelist[i]);
			SDL_modelist[i] = NULL;
		}
		if ( SDL_vgamode[i] != NULL ) {
			free(SDL_vgamode[i]);
			SDL_vgamode[i] = NULL;
		}
	}
	if ( screen && (screen->flags & SDL_HWSURFACE) ) {
		/* Direct screen access, no memory buffer */
		screen->pixels = NULL;
	}
}
void SDL_SYS_FinalQuit(void) { }
