/*								-*- C++ -*-
 * $Id: DLG_file.cpp,v 1.2 1998/04/16 19:22:05 wg Exp $
 *
 * Purpose: file load and save dialogs
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#include <limits.h> // for PATH_MAX
#include <dirent.h> // for opendir, readdir, etc.
#include <unistd.h> // for getcwd
#include <string.h> // for strncpy

#ifndef PATH_MAX
#define PATH_MAX 256
#endif

#define  Uses_wxDialogBox
#define  Uses_wxStringList
#define  Uses_wxButton
#define  Uses_wxListBox
#define  Uses_wxMessage
#define  Uses_wxText
#define  Uses_wxChoice
#include "wx.h"

//-----------------------------------------------------------------------------
// wxFileDialogBox, Dialog for file selection
//-----------------------------------------------------------------------------

class wxFileDialogBox : public wxDialogBox {
public:
    wxFileDialogBox(const char *message, const char *caption,
		    wxWindow *parent, const char *default_path,
		    const char *default_filename, const char *default_extension,
		    const char *wildcard, int flags);
    ~wxFileDialogBox(void);

    Bool OnCharHook(wxKeyEvent &event);
    Bool OnClose(void);
    void OnCommand(wxWindow &win, wxCommandEvent &event);
    void OnDefaultAction(wxItem *item);
    void OnPaint(void);
    void OnSize(int width, int height);
    void ReadDir(void);
    char *filename;
private:
    wxText	*file;
    wxText	*dir;
    wxButton	*ok;
    wxButton	*cancel;
    wxListBox	*dir_box;
    wxListBox	*file_box;
    wxChoice	*choice;
    wxButton	*mkdir_but;
    char	*wildcard;
    char	*extension;
    int		flags;
    char	*title;
};

// the global history list for directories
static const int dir_history_n = 16;
static char*     dir_history[dir_history_n] = { "./" };

//-----------------------------------------------------------------------------
// implementation of wxFileDialogBox
//-----------------------------------------------------------------------------

wxFileDialogBox::wxFileDialogBox(const char *WXUNUSED(message),
				 const char *caption, wxWindow *parent,
				 const char *default_path,
				 const char *default_filename,
				 const char *default_extension,
				 const char *_wildcard, int _flags)
{
    // initialize defaults if necessary
    char buf[PATH_MAX+1];
    if (!default_path)
	if ((default_path=getcwd(buf, PATH_MAX)) == NULL)
	    default_path = ".";
    if (!default_filename)
	if ((default_filename=_wildcard) == NULL)
	    default_filename = "";

    wildcard  = copystring(_wildcard);
    flags     = _flags;
    filename  = NULL;
    extension = NULL;
    title     = copystring(caption);

    // adjust extension to be NULL or ".???"
    if (default_extension) {
	if (*default_extension == '.')
	    ++default_extension;
	int ext_len = strlen(default_extension);
	extension = new char[ext_len+2]; // one for '.' and one for '\0'
	*extension = '.';
	strcpy(extension, default_extension);
    }

    file    = dir      = NULL;
    ok      = cancel   = mkdir_but = NULL;
    dir_box = file_box = NULL;
    choice = NULL;

    // create caption with wildcard hint
    char *capt = new char[strlen(title) + strlen(wildcard) + 10];
    strcpy(capt, title); strcat(capt, " (");
    strcat(capt, wildcard); strcat(capt, ")");
    // create dialogbox
    Create(parent, capt, TRUE, -1, -1, wxFSB_WIDTH, wxFSB_HEIGHT,
	   wxDEFAULT_DIALOG_STYLE|wxNO_DC, "fileDialogBox");
    delete[] capt;
    // set minimal
    SetMinSize(wxFSB_WIDTH/2, wxFSB_HEIGHT/2);
    // label positioning and spacing
    SetLabelPosition(wxVERTICAL);
    h_margin = 10;
    v_margin = h_space  = v_space  = 5;
    // create panel items
    dir      = wxNEW wxText   (this, NULL, "Directory", default_path);
    file     = wxNEW wxText   (this, NULL, "File / Filter", default_filename);
    dir_box  = wxNEW wxListBox(this, NULL, NULL);
    file_box = wxNEW wxListBox(this, NULL, NULL);
    int count;
    for (count=1; count<dir_history_n && dir_history[count]; count++);
    SetLabelPosition(wxHORIZONTAL);
    choice   = wxNEW wxChoice (this, NULL, "History", -1, -1, -1, -1,
			       count, dir_history);
    mkdir_but = wxNEW wxButton(this, NULL, "New directory");
    ok       = wxNEW wxButton (this, NULL, "Ok");
    cancel   = wxNEW wxButton (this, NULL, "Cancel");
}

wxFileDialogBox::~wxFileDialogBox(void)
{
    if (wildcard)  delete[] wildcard;
    if (extension) delete[] extension;
    if (filename)  delete[] filename;
    if (title)     delete[] title;
}

//-----------------------------------------------------------------------------
// read directory
//-----------------------------------------------------------------------------

static int CopyPathAdjusted(char *dest, char* source)
{
    int len;

    wxExpandPath(dest, source);
    len = strlen(dest);
    if (dest[len-1] != '/')
	dest[len++] = '/'; // add trailing /
    dest[len] = '\0';

    return len;
}

static Bool IsSubdirectory(char *name)
{
    DIR* teststream;
    Bool is_subdir = FALSE;

    if ( (teststream = opendir(name)) ) {
	if (readdir(teststream)) {
	    is_subdir = TRUE;
	}
	closedir(teststream);
    }

    return is_subdir;
}

#define DO_NOT_SHOW_DOT_FILES(fn, fl) \
    ((fn[0] == '.') && \
     ((fl&wxHIDE_DOT_FILES) || (fn[1]=='\0') || \
      ((fn[1]=='.') && (fn[2]=='\0'))))

#define FILE_MATCHED(fn, wild, fl) \
    (wild && wxMatchWild(wild, fn) && \
     (!(fl & wxHIDE_READONLY) || access(fn, W_OK) == 0))

void wxFileDialogBox::ReadDir(void)
{
    char           buf[PATH_MAX+1];
    int            pathlen;
    wxStringList   files, dirs;
    struct dirent* dir;
    DIR*           dirstream;

    pathlen = CopyPathAdjusted(buf, this->dir->GetValue());

    if ( (dirstream = opendir(buf)) ) {
	// busy cursor
	wxBeginBusyCursor();
	// parent directory
	if (strcmp(buf, "/"))
	    dirs.Add("..");
	while ( (dir = readdir(dirstream)) ) {
	    // skip "." and ".." and ".*" if wxHIDE_DOT_FILES
	    if (DO_NOT_SHOW_DOT_FILES(dir->d_name, flags))
		continue;
	    // complete path
	    strcpy(buf+pathlen, dir->d_name);
	    // decide, to which list the name should be appended
	    if (IsSubdirectory(buf))
		dirs.Add(dir->d_name);
	    else if (FILE_MATCHED(dir->d_name, wildcard, flags))
		files.Add(dir->d_name);
	}
	// sort file and directories, set list boxes
	dirs. Sort(); dir_box ->Set(&dirs);  dir_box ->SetSelection(0, TRUE);
	files.Sort(); file_box->Set(&files); file_box->SetSelection(0, TRUE);
	// close dirstream
	closedir(dirstream);
	// un-busy cursor
	wxEndBusyCursor();
    } else {
	wxMessageBox("Specified Dir not readable", "Warning",
		     wxOK|wxCENTRE, this);
	return;
    }
}

//-----------------------------------------------------------------------------
// event handlers
//-----------------------------------------------------------------------------

Bool wxFileDialogBox::OnCharHook(wxKeyEvent &event)
{
    if (event.KeyCode() == WXK_ESCAPE) {
	OnClose();
	return TRUE;
    }
    // maybe, that other keys are handled (e.g. F1 by wxApp for help)
    return wxDialogBox::OnCharHook(event);
}

Bool wxFileDialogBox::OnClose(void)
{
    if (filename) {
	delete[] filename;
	filename = NULL;
    }
    Show(FALSE);
    return FALSE;
}

void wxFileDialogBox::OnCommand(wxWindow &win, wxCommandEvent &event)
{
    wxItem *item = (wxItem*)(&win);

    // if a new item was selected in the file_box => do update text field
    if (item == file_box)
	file->SetValue(event.commandString);
    // if cancel was pressed, close box and return
    if (item == cancel)	{
	OnClose();
	return;
    }
    // if ok was pressed, execute default action
    if (item == ok)
	OnDefaultAction(item);
    // if directory history item selected, update directory
    if (item == choice && choice->GetSelection()>0) {
	dir->SetValue(choice->GetStringSelection());
	choice->SetSelection(0);
	OnDefaultAction(dir);
    }
    // create new directory if user requests it
    if (item == mkdir_but) {
	int pathlen;
	char buf[PATH_MAX+1];
	char* name = wxGetTextFromUser("Directory name:", "New directory");

	if (name) {
	    pathlen = CopyPathAdjusted(buf, dir->GetValue());
	    strncpy(buf+pathlen, name, PATH_MAX-pathlen);
	    buf[PATH_MAX] = '\0';
	    if (!wxMkdir(buf))
		wxMessageBox("Cannot create directory.", "Warning",
			     wxOK|wxCENTRE, this);
	    else
		OnDefaultAction(dir);
	}
    }
}

void wxFileDialogBox::OnDefaultAction(wxItem *item)
{
    Bool reload_dir = FALSE;
    char fpath[PATH_MAX+1];
    int  pathlen;

    // copy path to fpath and length to pathlen, '/' is the last character
    pathlen = CopyPathAdjusted(fpath, dir->GetValue());
    if (item == dir) {
	// a new path was specified in the text field => reload directory
	reload_dir = TRUE;
    }
    if (item == dir_box) {
	// a new directory was selected in the dir listbox => append subdir
	if(dir_box->GetSelection() < 0)
	    return; // My fix is here 
	strcpy(fpath + pathlen, dir_box->GetStringSelection());
	reload_dir = TRUE;
    }
    if (wxIsWild(file->GetValue())) {
	// file is wildcard => reload directory
	delete wildcard;
	wildcard = copystring(file->GetValue());
	reload_dir = TRUE;
	// set new title
	char *capt = new char[strlen(title) + strlen(wildcard) + 10];
	strcpy(capt, title); strcat(capt, " (");
	strcat(capt, wildcard); strcat(capt, ")");
	SetTitle(capt);
	delete[] capt;
    }
    // reload dir or exit with filename
    if (reload_dir) {
	// set expanded path
	char fpath2[PATH_MAX+1];
	CopyPathAdjusted(fpath2, fpath); // expand again (e.g. appended "..")
	dir->SetValue(fpath2);
	ReadDir();
    } else {
	if (choice->FindString(fpath) < 0) { // add to history
	    if (dir_history[dir_history_n-1])
		delete[] dir_history[dir_history_n-1];
	    for (int i=dir_history_n-1; i>1; --i)
		dir_history[i] = dir_history[i-1];
	    dir_history[1] = copystring(fpath);
	}
	strcpy(fpath + pathlen, file->GetValue());
	if (wxFileExists(fpath)) {
	    if (flags & (wxSAVE | wxOVERWRITE_PROMPT)) {
		// overwrite warning
		char buf[PATH_MAX+100];
		sprintf(buf, "Do you want to replace\n%s?", fpath);
		if (wxMessageBox(buf, "Overwrite Warning",
				 wxYES_NO|wxCENTRE, this) == wxNO)
		    return;
	    }
	} else {
	    if ((flags & wxOPEN ) && extension) {
		// try to append the default extension
		char *fpath_end = strchr(fpath, '\0');
		strcpy(fpath_end, extension);
		if (!wxFileExists(fpath))
		    *fpath_end = '\0'; // extension did not help ...
	    }
	}
	// try default extension if file does not exist
	filename = copystring(fpath);
	Show(FALSE);
    }
}

void wxFileDialogBox::OnPaint(void)
{
    if (ok && cancel) {
	int x, y, width, height;
	GetSize(&width, &height);
	ok->GetPosition(&x, &y);
	DrawStaticLine(0, y - 1, width);
    }
    wxDialogBox::OnPaint();
}

void wxFileDialogBox::OnSize(int width, int height)
{
    if (!file || !dir || !ok || !cancel || !file_box || !dir_box ||
	!choice || !mkdir_but || width <= 0 || height <= 0)
	return;
    // I need the client size
    GetClientSize(&width, &height);
    // item sizes (queried with GetSize)
    int iwidth, iheight;
    // width of a column
    int halfwidth  = width / 2 - h_space;
    // position of right column
    int right = halfwidth + h_space;
    // line position
    int line;
    // button width
    const int buttonwidth = 70;
    // set directory and file text item
    dir  -> GetSize(&iwidth, &iheight);
    dir  -> SetSize(0,     0, halfwidth, iheight);
    file -> SetSize(right, 0, halfwidth, iheight);
    line = iheight + v_space;
    // set ok and cancel button items
    ok->GetSize(&iwidth, &iheight);
    height -= iheight;
    ok     -> SetSize(halfwidth - buttonwidth, height, buttonwidth, iheight);
    cancel -> SetSize(right,                   height, buttonwidth, iheight);
    // set history choice item
    choice->GetSize(&iwidth, &iheight);
    height -= iheight + 2*v_space;
    choice->SetSize(0, height, halfwidth, iheight);
    mkdir_but->SetSize(right, height, -1, -1);
    // set directory and file listbox items
    iheight = height - line - v_space;
    dir_box  -> SetSize(0,     line, halfwidth, iheight);
    file_box -> SetSize(right, line, halfwidth, iheight);
    // show end of directory and file
    dir->ShowEnd();
    file->ShowEnd();
}

//-----------------------------------------------------------------------------
// wxFileSelector
//-----------------------------------------------------------------------------

char *wxFileSelector(const char *message, const char *default_path,
		     const char *default_filename,
		     const char *default_extension,
		     const char *wildcard, int flags,
		     wxWindow *parent, int x, int y)
{
    // busy cursor
    wxBeginBusyCursor();
    // create dialog box
    wxFileDialogBox *box = wxNEW wxFileDialogBox(NULL, message, parent,
						 default_path, default_filename,
						 default_extension, wildcard,
						 flags);
    // read directory
    box->ReadDir();
    // centre box
    if (x < 0) box->Centre(wxHORIZONTAL);
    if (y < 0) box->Centre(wxVERTICAL);
    // show frame
    box->Show(TRUE);
    // un-busy cursor
    wxEndBusyCursor();

    // close box and return values
    char *ret_value = NULL;
    if (box->filename) {
	strncpy(wxBuffer, box->filename, wxBufferSize);
	ret_value = wxBuffer;
    }
    delete box;
    return ret_value;
}

char *wxLoadFileSelector(const char *what, const char *extension,
			 const char *default_name, wxWindow *parent)
{
    char wild[60], prompt[60];

    sprintf(prompt, wxSTR_LOAD_FILE, what);
    sprintf(wild, "*.%s", (*extension == '.' ? ++extension : extension));
    return wxFileSelector(prompt,
			  wxPathOnly((char*)default_name),
			  wxFileNameFromPath((char*)default_name),
			  extension, wild, wxOPEN, parent);
}

char *wxSaveFileSelector(const char *what, const char *extension,
			 const char *default_name, wxWindow *parent)
{
    char wild[60], prompt[60];

    sprintf(prompt, wxSTR_SAVE_FILE, what);
    sprintf(wild, "*.%s", (*extension == '.' ? ++extension : extension));
    return wxFileSelector(prompt,
			  wxPathOnly((char*)default_name),
			  wxFileNameFromPath((char*)default_name),
			  extension, wild, wxSAVE|wxOVERWRITE_PROMPT, parent);
}
