# This module is part of ML and does all list processing functions

package List;

use strict;

=head1 CONSTRUCTOR

=item new( [PHRASE] )

 List->new();

Creates a new object which will be used for a list and
eventually loads the list if a name is given. Returns
a List object.

=back

=head1 METHODS

=over 4

=item load ( LIST )

Loads the indicated list into the object.

=item save ( LIST )

Saves the indicated list object to the disk files.

=item savestats ()

Saves updates the statistics file on disk.

=item update_stats( BYTES )

Updates the stats, argument is number of bytes, returns the next
sequence number. Does nothing if no stats.

=item send_sub_to_owner ( WHO, COMMENT )
Send a message to the list owners telling that someone
wanted to subscribe to the list.

=item send_to_editor ( MSG )

Send a Mail::Internet type object to the editor (for approval).

=item send_msg ( MSG )

Sends the Mail::Internet message to the list.

=item send_file ( FILE, USER, GECOS )

Sends the file to the USER. FILE may only be welcome for now.

=item delete_user ( ARRAY )

Delete the indicated users from the list.
 
=item get_cookie ()

Returns the cookie for a list, if available.

=item get_max_size ()

Returns the maximum allowed size for a message.

=item get_reply_to ()

Returns an array with the Reply-To values.

=item get_default_user_options ()

Returns a default option of the list for subscription.

=item get_total ()

Returns the number of subscribers to the list.

=item get_user ( USER )

Returns a hash with the informations regarding the indicated
user.

=item get_first_user ()

Returns a hash to the first user on the list.

=item get_next_user ()

Returns a hash to the next users, until we reach the end of
the list.

=item update_user ( USER, HASHPTR )

Sets the new values given in the hash for the user.

=item add_user ( USER, HASHPTR )

Adds a new user to the list. May overwrite existing
entries.

=item is_user ( USER )

Returns true if the indicated user is member of the list.
 
=item am_i ( FUNCTION, USER )

Returns true is USER has FUNCTION (owner, editor) on the
list.

=item get_state ( FLAG )

Returns the value for a flag : sig or sub.

=item may_do ( ACTION, USER )

Chcks is USER may do the ACTION for the list. ACTION can be
one of following : send, review, index, getm add, del,
reconfirm, purge.

=item is_moderated ()

Returns true if the list is moderated.

=item archive_exist ( FILE )

Returns true if the indicated file exists.

=item archive_send ( WHO, FILE )

Send the indicated archive file to the user, if it exists.

=item archive_ls ()

Returns the list of available files, if any.

=item archive_msg ( MSG )

Archives the Mail::Internet message given as argument.

=item is_archived ()

Returns true is the list is configured to keep archives of
its messages.

=item get_stats ( OPTION )

Returns either a formatted printable strings or an array whith
the statistics. OPTION can be 'text' or 'array'.

=item print_info ( FDNAME )

Print the list informations to the given file descriptor, or the
currently selected descriptor.

=cut

use Carp;

use Mail::Header;
use Mail::Internet;
use Archive;
use Language;
use Log;
use Conf;
use Time::Local;
use MIME::QuotedPrint;
use MIME::Entity;

## RCS identification.
#my $id = '@(#)$Id: List.pm,v 1.3 1998/12/22 13:52:39 sympa Exp $';


## Database and SQL statement handlers
my ($dbh, $sth, @sth_stack, $use_db);
my %date_format = (
		   'read' => {
		       'Pg' => 'date_part(\'epoch\',%s)',
		       'mysql' => 'UNIX_TIMESTAMP(%s)',
		       'Oracle' => '((to_number(to_char(%s,\'J\')) - to_number(to_char(to_date(\'01/01/1970\',\'dd/mm/yyyy\'), \'J\'))) * 86400) +to_number(to_char(%s,\'SSSSS\'))',
		       'Sybase' => 'datediff(second, "01/01/1970",%s)'
		       },
		   'write' => {
		       'Pg' => '\'epoch\'::datetime + \'%d sec\'',
		       'mysql' => 'FROM_UNIXTIME(%d)',
		       'Oracle' => 'to_date(to_char(round(%s/86400) + to_number(to_char(to_date(\'01/01/1970\',\'dd/mm/yyyy\'), \'J\'))) || \':\' ||to_char(mod(%s,86400)), \'J:SSSSS\')',
		       'Sybase' => 'dateadd(second,%s,"01/01/1970")'
		       }
	       );

## This is the generic hash which keeps all lists in memory.
my %list_of_lists = ();

use Fcntl;
use DB_File;

$DB_BTREE->{compare} = '_compare_addresses';

## Connect to Database
sub db_connect {
    do_log('debug2', 'List::db_connect');

    my $connect_string;

    unless (require DBI) {
	do_log ('info',"enable to use DBI library, install DBI (CPAN) first");
	return undef;
    }

    ## Do we have db_xxx required parameters
    foreach my $db_param ('db_type','db_name','db_host','db_user') {
	unless ($Conf{$db_param}) {
	    do_log ('info','Missing parameter %s for DBI connection', $db_param);
	    return undef;
	}
    }

    if ($Conf{'db_type'} eq 'Oracle') {
	## Oracle uses sids instead of dbnames
	$connect_string = sprintf 'DBI:%s:sid=%s;host=%s', $Conf{'db_type'}, $Conf{'db_name'}, $Conf{'db_host'};

    }elsif ($Conf{'db_type'} eq 'Pg') {
	$connect_string = sprintf 'DBI:%s:dbname=%s\@%s', $Conf{'db_type'}, $Conf{'db_name'}, $Conf{'db_host'};

    }else {
	$connect_string = sprintf 'DBI:%s:dbname=%s;host=%s', $Conf{'db_type'}, $Conf{'db_name'}, $Conf{'db_host'};
    }

    unless ( $dbh = DBI->connect($connect_string, $Conf{'db_user'}, $Conf{'db_passwd'}) ) {
	do_log ('notice','Can\'t connect to Database %s as %s', $connect_string, $Conf{'db_user'});
	return undef;
    }

    if ($Conf{'db_type'} eq 'Pg') { # Configure Postgres to use ISO format dates
       $dbh->do ("SET DATESTYLE TO 'ISO';");
    }

    ## added sybase support
    if ($Conf{'db_type'} eq 'Sybase') { # Configure to use sympa database 
	my $dbname;
	$dbname="use $Conf{'db_name'}";
        $dbh->do ($dbname);
    }

    do_log('debug','Connected to Database %s',$Conf{'db_name'});

    return 1;
}

## Disconnect from Database
sub db_disconnect {
    do_log('debug2', 'List::db_disconnect');

    unless ($dbh->disconnect()) {
	do_log ('notice','Can\'t disconnect from Database %s : %s',$Conf{'db_name'}, $dbh->errstr);
	return undef;
    }

    return 1;
}

## Creates an object.
sub new {
    my($pkg, $name) = @_;
    my $liste={};
    do_log('debug2', 'List::new(%s)', $name);

    ## Only process the list if the name is valid.
    unless ($name and ($name =~ /^[a-z0-9][a-z0-9\-\+\._]*$/io) ) {
	&do_log('info', 'Incorrect listname %s',  $name);
	return undef;
    }
    ## Lowercase the list name.
    $name =~ tr/A-Z/a-z/;
    
    if ($list_of_lists{$name}){
	# use the current list in memory and update it
	$liste=$list_of_lists{$name};
    }else{
	do_log('debug', 'List object %s created', $name) if $main::opt_d; ##TEMP

	# create a new object list
	bless $liste, $pkg;
    }
    return undef unless ($liste->load($name));
    return $liste;
}

## Saves the statistics data to disk.
sub savestats {
    my $self = shift;
    do_log('debug2', 'List::savestats');
   
    ## Be sure the list has been loaded.
    my $name = $self->{'name'};
    return undef unless ($list_of_lists{$name});
    
   _save_stats_file("$name/stats", $self->{'stats'}, $self->{'total'});
    
    return 1;
}

## Update the stats struct 
## Input  : num of bytes of msg
## Output : num of msgs sent
sub update_stats {
    my($self, $bytes) = @_;
    do_log('debug2', 'List::savestats(%d)', $bytes);

    my $stats = $self->{'stats'};
    $stats->[0]++;
    $stats->[1] += $self->{'total'};
    $stats->[2] += $bytes;
    $stats->[3] += $bytes * $self->{'total'};
    return $stats->[0];
}

## Dumps a copy of lists to disk, in text format
sub dump {
    do_log('debug2', 'List::dump');

    my $listname;

    foreach $listname (keys %list_of_lists) {
	
	my $list = new List($listname);
	my $user_file_name;

	if ($list->{'admin'}{'user_data_source'} eq 'database') {
            do_log('debug', 'Dumping list %s',$listname);
	    $user_file_name = "$listname/subscribers.db.dump";
	    $list->_save_users_file($user_file_name);
	    $list->{'mtime'} = [ (stat("$listname/config"))[9], (stat("$listname/subscribers"))[9], (stat("$listname/stats"))[9] ];
	}

    }
    return 1;
}

## Saves a copy of the list to disk. Does not remove the
## data.
sub save {
    my $self = shift;
    do_log('debug2', 'List::save');

    my $name = $self->{'name'};    
 
    return undef 
	unless ($list_of_lists{$name});
 
    my $user_file_name;

    if ($self->{'admin'}{'user_data_source'} eq 'file') {
	$user_file_name = "$name/subscribers";

        unless ($self->_save_users_file($user_file_name)) {
	    &do_log('info', 'unable to save user file %s', $user_file_name);
	    return undef;
	}
        $self->{'mtime'} = [ (stat("$name/config"))[9], (stat("$name/subscribers"))[9], (stat("$name/stats"))[9] ];
    }
    
    return 1;
}

## Loads the administrative data for a list
sub load {
    my ($self, $name) = @_;
    do_log('debug2', 'List::load(%s)', $name);

    my $users;

    ## Check if we have the directory
    unless (-d "$name") {
	&do_log('info', 'No directory for list %s', $name);
	return undef ;
    }

    ## ...and the config file is readable
    unless (-r "$name/config") {
	&do_log('info', 'No config file for list %s', $name);
	return undef;
    }
    
    my $m1 = (stat("$name/config"))[9];
    my $m2 = (stat("$name/subscribers"))[9];
    my $m3 = (stat("$name/stats"))[9];

    my $admin = _load_admin_file("$name/config") 
	if ($self->{'name'} ne $name || $m1 > $self->{'mtime'}->[0]);
    $self->{'admin'} = $admin if ($admin);


    ## Only load total of users from a Database
    if ($self->{'admin'}->{'user_data_source'} eq 'database') {
	$users->{'total'} = _load_total_db($name)
	    unless $self->{'total'};
    }elsif($self->{'admin'}->{'user_data_source'} eq 'file') { 
	
	## Touch subscribers file if not exists
	unless ( -r "$name/subscribers") {
	    open L, ">$name/subscribers" or return undef;
	    close L;
	    do_log('info','No subscribers file, creating %s',"$name/subscribers");
	}

	$users = _load_users_file("$name/subscribers") 
	    if ($self->{'name'} ne $name || $m2 > $self->{'mtime'}->[1]);
    }elsif($self->{'admin'}->{'user_data_source'} eq 'include') {

    ## include other subscribers as defined in include directives (list|ldap|sql|file)
	unless ( defined ($self->{'admin'}{'include'}) ) {
	    &do_log('notice', 'Include paragraph missing in configuration file');
	    return undef;
	}

	$m2 = $self->{'mtime'}->[1]; 
	## if (first time ) OR (Config has changed) OR( TTL has expired ) then reload
	if ( (!$self->{'mtime'}->[0]) || ( $m1 > $self->{'mtime'}->[0]) || (time > ($self->{'mtime'}->[1] + $self->{'admin'}{'ttl'}))) {
	    $users = _load_users_include($self->{'admin'});
	    $m2 = time;
	}

    }else { do_log ('debug','internal error report to authors');}

    my $stats = _load_stats_file("$name/stats")  if ($self->{'name'} ne $name || $m3 > $self->{'mtime'}->[2]);

    $self->{'name'}  = $name if (-d "$name");
    $self->{'stats'} = $stats if ($stats);
    $self->{'users'} = $users->{'users'} if ($users);
    $self->{'ref'}   = $users->{'ref'} if ($users);
    $self->{'total'} = $users->{'total'} if ($users);
    $self->{'mtime'} = [ $m1, $m2, $m3 ];

    $list_of_lists{$name} = $self;
    return $self;
}

## Alert owners
sub send_alert_to_owner {
    my($self, $alert) = @_;
    do_log('debug2', 'List::send_alert_to_owner(%s)', $alert);
 
    my ($i, @rcpt);
    my $admin = $self->{'admin'}; 
    my $name = $self->{'name'};
    my $host = $admin->{'host'};

    return unless ($name && $admin);
    
    foreach $i (@{$admin->{'owner'}}) {
	next if ($i->{'reception'} eq 'nomail');
	push(@rcpt, $i->{'email'}) if ($i->{'email'});
    }

    do_log('notice', 'Warning : no owner defined or  all of them use nomail option in list %s', $name ) unless (@rcpt);

    my $hdr = new Mail::Header;
    $hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
    $hdr->add('To', "\"". sprintf(Msg(8, 1, "Owners of list %s :"), $name)."\" <$name-request\@$host>");
    
    if ($alert eq 'bounce_rate') {
	my $rate = $self->get_total_bouncing() * 100 / $self->{'total'};
	$rate = int ($rate * 10) / 10;

	$hdr->add('Subject', sprintf(Msg(8, 28, "WARNING: bounce rate too high in list %s"), $name));
	
	*DESC = smtp::smtpto($Conf{'request'}, \@rcpt);
	
	$hdr->print(\*DESC);
	print DESC "\n";
	printf DESC Msg(8, 27, "Bounce rate in list %s is %d%%.\nYou should delete bouncing subscribers : %s/reviewbouncing/%s"), $name, $rate, $Conf{'wwsympa_url'}, $name ;
	close(DESC);
    }else {
	do_log('info', 'Unknown alert %s', $alert);
    }

    return 1;
}

## Send a sub/sig notice to the owners.
sub send_notify_to_listmaster {
    my ($operation, $param) = @_;
    do_log('debug2', 'List::send_notify_to_listmaster(%s, %s)', $operation, $param);
    
    ## Loop detected in Sympa
    if ($operation eq 'loop_command') {
	my $file = $param;

	my $notice = build MIME::Entity (From => $Conf{'sympa'},
					 To => $Conf{'listmaster'},
					 Subject => 'Loop detected',
					 Data => 'A loop has been detected with the following message');

	$notice->attach(Path => $file,
			Type => 'message/rfc822');

	## Send message
	my $rcpt = $Conf{'listmaster'};
	*FH = &smtp::smtpto($Conf{'request'}, \$rcpt);
	$notice->print(\*FH);
	close FH;
    }
    
    return 1;
}

## Send a sub/sig notice to the owners.
sub send_notify_to_owner {
    my($self, $who, $gecos, $operation, $by) = @_;
    do_log('debug2', 'List::send_notify_to_owner(%s, %s, %s, %s)', $who, $gecos, $operation, $by);
    
    my ($i, @rcpt);
    my $admin = $self->{'admin'}; 
    my $name = $self->{'name'};
    my $host = $admin->{'host'};

    return unless ($name && $admin && $who);
    
    foreach $i (@{$admin->{'owner'}}) {
	next if ($i->{'reception'} eq 'nomail');
	push(@rcpt, $i->{'email'}) if ($i->{'email'});
    }

    do_log('notice', 'Warning : no owner defined or  all of them use nomail option in list %s', $name ) unless (@rcpt);
    
    my $hdr = new Mail::Header;
    $hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
    $hdr->add('To', "\"". sprintf(Msg(8, 1, "Owners of list %s :"), $name)."\" <$name-request\@$host>");
    
    if ($operation eq 'warn-signoff') {
	$hdr->add('Subject', sprintf(Msg(8, 21, "WARNING: %s list %s from %s %s"), $operation, $name, $who, $gecos));
	
	*DESC = smtp::smtpto($Conf{'request'}, \@rcpt);
	
	$hdr->print(\*DESC);
	print DESC "\n";
	printf DESC Msg(8, 23, "WARNING : %s %s failed to signoff from %s
because his address was not found in the list\n (You may help this person)\n"),$who, $gecos, $name ;
	close(DESC);
    }else{
	
	$hdr->add('Subject', sprintf(Msg(8, 21, "FYI: %s list %s from %s %s"), $operation, $name, $who, $gecos));
	
	*DESC = smtp::smtpto($Conf{'request'}, \@rcpt);
	$hdr->print(\*DESC);
	print DESC "\n";
	if ($by) {
	    printf DESC Msg(8, 26, "FYI command %s list %s from %s %s validated by %s\n (no action needed)\n"),$operation, $name, $who, $gecos, $by ;
	}else{
	    printf DESC Msg(8, 22, "FYI command %s list %s from %s %s \n (no action needed)\n"),$operation, $name, $who, $gecos ;
	}
	close(DESC);
    }
}

## Send a subscription request to the owners.
sub send_sub_to_owner {
   my($self, $who, $gecos) = @_;
   do_log('debug2', 'List::send_sub_to_owner(%s, %s)', $who, $gecos);

   my($i, @rcpt);
   my $admin = $self->{'admin'}; 
   my $name = $self->{'name'};
   my $host = $admin->{'host'};

   return unless ($name && $admin && $who);

   foreach $i (@{$admin->{'owner'}}) {
        next if ($i->{'reception'} eq 'nomail');
        push(@rcpt, $i->{'email'}) if ($i->{'email'});
   }
   do_log('notice', 'Warning : no owner defined or  all of them use nomail option in list %s', $name ) unless (@rcpt) ;
   my $hdr = new Mail::Header;
   $hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
   $hdr->add('To', "\"". sprintf(Msg(8, 1, "Owners of list %s :"), $name)."\" <$name-request\@$host>");
   $hdr->add('Subject', sprintf(Msg(8, 2, "%s subscription request"), $name));

   *DESC = smtp::smtpto($Conf{'request'}, \@rcpt);
   $hdr->print(\*DESC);
   printf DESC Msg(8, 3, $msg::sub_owner), $name, $name, $who, $gecos;
   close(DESC);
}

## Send a notification to authors of messages sent to editors
sub notify_sender{
   my($self, $sender) = @_;
   do_log('debug2', 'List::notify_sender(%s)', $sender);

   my $admin = $self->{'admin'}; 
   my $name = $self->{'name'};
   return unless ($name && $admin && $sender);

   my $hdr = new Mail::Header;
   $hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
   $hdr->add('To', $sender);

   *DESC = smtp::smtpto($Conf{'request'}, \$sender);
   $hdr->print(\*DESC);
   printf DESC Msg(4, 38, "Your message for list %s has been forwarded to editor(s)\n"), $name;
   close DESC;
}

## Send a Unsubscription request to the owners.
sub send_sig_to_owner {
    my($self, $who) = @_;
    do_log('debug2', 'List::send_sig_to_owner(%s)', $who);
    
    my($i, @rcpt);
    my $admin = $self->{'admin'}; 
    my $name = $self->{'name'};
    my $host = $admin->{'host'};
    
    return unless ($name && $admin && $who);
    
    foreach $i (@{$admin->{'owner'}}) {
        next if ($i->{'reception'} eq 'nomail');
        push(@rcpt, $i->{'email'}) if ($i->{'email'});
    }
    do_log('notice', 'Warning : no owner defined or  all of them use nomail option in list %s', $name ) unless (@rcpt) ;
    my $hdr = new Mail::Header;
   $hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
    $hdr->add('To', "\"". sprintf(Msg(8, 1, "Owners of list %s :"), $name)."\" <$name-request\@$host>");
    $hdr->add('Subject', sprintf(Msg(8, 24, "%s UNsubscription request"), $name));
    
    *DESC = smtp::smtpto($Conf{'request'}, \@rcpt);
    $hdr->print(\*DESC);
    printf DESC Msg(8, 25, $msg::sig_owner), $name, $name, $who;
    close(DESC);
}

## Send a message to the editor
sub send_to_editor {
   my($self, $method, $msg) = @_;
   do_log('debug2', "List::send_to_editor, method : $method");

   my($i, @rcpt);
   my $admin = $self->{'admin'};
   my $name = $self->{'name'};
   my $host = $admin->{'host'};
   my $modqueue= $Conf{'queuemod'};
   return unless ($name && $admin);
  
   srand (time());
   my @now=localtime(time);
   my $messageid=$now[6].$now[5].$now[4].$now[3].$now[2].$now[1]."."
                 .int(rand(6)).int(rand(6)).int(rand(6)).int(rand(6)).int(rand(6)).int(rand(6))."\@".$host;
   my $modkey=MD5->hexhash(join('/', $self->get_cookie(),$messageid));
   my $boundary ="----------------- Message-Id: \<$messageid\>" 
       if ($method eq 'md5');
   
   if($method eq 'md5'){  
       open(OUT, ">$modqueue\/$name\_$modkey") || return;
       $msg->print(\*OUT);
       close(OUT);
   }
   foreach $i (@{$admin->{'editor'}}) {
      next if ($i->{'reception'} eq 'nomail');
      push(@rcpt, $i->{'email'}) if ($i->{'email'});
   }
   unless (@rcpt) {
       foreach $i (@{$admin->{'owner'}}) {
	   next if ($i->{'reception'} eq 'nomail');
	   push(@rcpt, $i->{'email'}) if ($i->{'email'});
       }

       do_log('notice','Warning : no editor defined for list %s, contacting owners', $name );
   }
   my $hdr = new Mail::Header;
   $hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
   $hdr->add('To', "\"".sprintf(Msg(8, 4, "Editors of list %s :"), $name)."\" <$name-editor\@$host>");
   $hdr->add('Subject', sprintf Msg(8, 5, '[%s] Article to approve'), $name);
   if ($method eq 'md5') {
      $hdr->add('MIME-Version', "1.0");
      $hdr->add('Content-Type',"multipart/mixed; boundary=\"$boundary\"");
   }
   *DESC = smtp::smtpto($Conf{'request'}, \@rcpt);
   $hdr->print(\*DESC);
   print DESC "\n";
   if($method eq 'md5'){
	print DESC "--$boundary\n";
	print DESC "Content-type: text/plain\n";
        print DESC "Content-Transfert-Encoding: 8bit\n\n";

	printf DESC Msg(8, 10,"To distribute the following message into list %s: mailto:%s?subject=DISTRIBUTE%%20%s%%20%s\n\n"), $name, $Conf{'sympa'}, $name, $modkey;
	printf DESC  Msg(8, 11, "To refuse it (delete it) :\nmailto:%s?subject=REJECT%20%s%20%s\n"), $Conf{'sympa'}, $name, $modkey;
   }   
   print DESC "\n";
   print DESC "--$boundary\n" if ($method eq 'md5');
   print DESC "Content-Type: message/rfc822\n\n" if ($method eq 'md5');
   $msg->print(\*DESC);

   close(DESC);

   return $modkey;
}

## Send an authentification message
sub send_auth {
   my($self, $sender, $msg) = @_;
   do_log('debug2', 'List::send_auth(%s)', $sender);

   my($i, @rcpt);
   my $admin = $self->{'admin'};
   my $name = $self->{'name'};
   my $host = $admin->{'host'};
   my $authqueue = $Conf{'queueauth'};
   return undef unless ($name && $admin);
  
   srand (time());
   my @now = localtime(time);
   my $messageid = $now[6].$now[5].$now[4].$now[3].$now[2].$now[1]."."
                   .int(rand(6)).int(rand(6)).int(rand(6)).int(rand(6))
		   .int(rand(6)).int(rand(6))."\@".$host;
   my $modkey = MD5->hexhash(join('/', $self->get_cookie(),$messageid));
   my $boundary = "----------------- Message-Id: \<$messageid\>" ;
   my $contenttype = "Content-Type: message\/rfc822";
     
   open(OUT, ">$authqueue\/$name\_$modkey") || return;
   $msg->print(\*OUT);
   close(OUT);
 
   my $hdr = new Mail::Header;
   $hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
   $hdr->add('To', $sender );
#   $hdr->add('Subject', Msg(8, 16, "Authentification needed"));
   $hdr->add('Subject', "confirm $modkey");
   $hdr->add('MIME-Version', "1.0");
   $hdr->add('Content-Type',"multipart/mixed; boundary=\"$boundary\"") ;
   $hdr->add('Content-Transfert-Encoding', "8bit");
   
   *DESC = smtp::smtpto($Conf{'request'}, \$sender);
   $hdr->print(\*DESC);
   print DESC "\n";
   print DESC "--$boundary\n";
   print DESC "Content-Type: text/plain\n\n";
   printf DESC Msg(8, 12,"In order to broadcast the following message into list %s, please reply to %s :\n"), $name, $Conf{'sympa'}, $modkey ;
   print DESC "--$boundary\n";
   print DESC "Content-Type: message/rfc822\n\n";
   $msg->print(\*DESC);
   close(DESC);

   return $modkey;
}

## Distribute a message to the list
sub distribute_msg {
    my($self, $msg, $bytes) = @_;
    do_log('debug2', 'List::distribute_msg');
    
    my ($i, $numsmtp);
    my $hdr = $msg->head();
    my $listname = $self->{'name'};
    my $host = $self->{'admin'}{'host'};
    
    ## Change the reply-to header if necessary. Never change it if already there.
    my $reply = $self->get_reply_to();
    if ($reply && !$hdr->get('Reply-To') && $reply !~ /sender/io) {
	    if ($reply =~ /^list$/io) {
		$reply = "$listname\@$host";
	    }
	    $hdr->add('Reply-To', $reply);
    }
    
    ## Remove unwanted headers if present.
    $hdr->delete('Return-Receipt-To');
    $hdr->delete('Precedence');
    $hdr->delete('X-Sequence');
    $hdr->add('X-Loop', "$listname\@$host");
    
    ## Update the stats, and returns the new X-Sequence, if any.
    $hdr->add('X-Sequence', $self->update_stats($bytes));
    $self->savestats();
    ## Add other useful headers
    $hdr->add('Precedence', 'list');
    foreach $i (@{$self->{'admin'}{'custom_header'}}) {
	$hdr->add($1, $2) if ($i=~/^([\S\-\:]*)\s(.*)$/);
    }
    
    ## Blindly send the message to all users.
    unless ($numsmtp = $self->send_msg($msg)) {
	do_log('info','Unable to send message to list %s', $listname);
	return undef;
    }
    
    ## Append to the digest
    if ($self->is_digest()){
	$self->archive_msg_digest($msg);
    }
    
    $self->archive_msg($msg);

    return $numsmtp;
}

## Send a message to the list
sub send_msg {
    my($self, $msg) = @_;
    do_log('debug2', 'List::send_msg');
    
    my $hdr = $msg->head;
    my $name = $self->{'name'};
    my $admin = $self->{'admin'};
    my $total = $self->{'total'};

    unless ($total > 0) {
	&do_log('info', 'No subscriber in list %s', $name);
	return undef;
    }

    ## Bounce rate
    ## Available in database mode only
    if ($List::use_db) {
	if (($self->get_total_bouncing() * 100 / $total) > $self->{'admin'}{'bounce'}{'warn_rate'}) {
	    $self->send_alert_to_owner('bounce_rate');
	}
    }

    ## Add Custom Subject
    if ($admin->{'custom_subject'}) {
	my $subject_field = $msg->head->get('Subject');
	$subject_field =~ s/^\s*(.*)\s*$/$1/;
	if (index($subject_field, $admin->{'custom_subject'}) <0) {
	    $msg->head->delete('Subject');
	    $msg->head->add('Subject', $admin->{'custom_subject'}." ".$subject_field);
	}
    }
    
    ## Add a footer
    $msg = _add_parts($msg,  $name, $self->{'admin'}{'footer_type'});

    ## Who is the enveloppe sender ?
    my $host = $self->{'admin'}{'host'};
    my $from = "$name-owner\@$host";
    
    my @tabrcpt;

    for ( my $user = $self->get_first_user(); $user; $user = $self->get_next_user() ){
	push @tabrcpt, $user->{'email'} unless 
	    ( ($user->{'reception'} eq 'digest') 
	      or ($user->{'reception'} eq 'nomail') );
    }

    unless (@tabrcpt) {
	&do_log('info', 'No subscriber for sending msg in list %s', $name);
	return undef;
    }

    ## verifications much later.
    smtp::mailto($msg, $from, @tabrcpt );
}

## Add footer/header to a message
sub _add_parts {
    my ($msg, $listname, $type) = @_;
    do_log('debug2', 'List:_add_parts(%s, %s)', $listname, $type);

    ## Add footer
    my $footer = "$listname/message.footer";
    my  $footermime = $footer . ".mime";

    my $header = "$listname/message.header";
    my $headermime = $header . ".mime";

    my (@footer_msg, @header_msg);

    ## No footer/header
    unless (-f $footermime or -f $footer or -f $headermime or -f $header) {
	return $msg;
    }
    
    ## Perl module not available
    unless (require MIME::Parser) {
	do_log ('debug',"enable to use MIME::Parser library, install MIME/Tools (CPAN) first");
	return $msg;
    }
    
    my $parser = new MIME::Parser output_dir     => '/tmp',
                                  output_to_core => 'ALL';
    
    ## Msg Content-Type
    my $content_type = $msg->head->get('Content-Type');
    
    ## Brutal force !!!
    if (($type eq 'append') and 
	(!$content_type or $content_type =~ /^text\/plain/i)) {
	
	if (-r $header) {
	    open HEADER, $header;
	    @header_msg = <HEADER>;
	    close HEADER;
	}
	
	if (-r $footer) {
	    open FOOTER, $footer;
	    @footer_msg = <FOOTER>;
	    close FOOTER;
	}
    }
    
    unless ($msg =~ /^MIME::Entity/) {
	my @lines = (@{$msg->header}, "\n", @header_msg, @{$msg->body}, @footer_msg);
	$msg = $parser->parse_data(\@lines);
    }
    
    ## MIME footer/header
    unless (($type eq 'append') or 
	    ($content_type =~ /^multipart\/alternative/i)) {
	
	if (-r $headermime) {
	    
	    my $header_part = $parser->parse_in($headermime);    
	    $msg->make_multipart unless $msg->is_multipart;
	    $msg->add_part($header_part, 0); ## Add AS FIRST PART (0)
	    
	    ## text/plain header
	}elsif (-r $header ) {
	    
	    $msg->make_multipart unless $msg->is_multipart;
	    my $header_part = build MIME::Entity Path        => $header,
	                                         Type        => "text/plain",
			                         Encoding    => "8bit";
	    $msg->add_part($header_part, 0);
	}
	
	if (-r $footermime) {
	    
	    my $footer_part = $parser->parse_in($footermime);    
	    $msg->make_multipart unless $msg->is_multipart;
	    $msg->add_part($footer_part);
	    
	## text/plain footer
	}elsif (-r $footer ) {
	    
	    $msg->make_multipart unless $msg->is_multipart;
	    $msg->attach(Path        => $footer,
			 Type        => "text/plain",
			 Encoding    => "8bit"
			 );
	}
    }
    return $msg;
}

## Send a digest message to the subscribers with reception=digest
sub send_msg_digest {
    my($self, $listname) = @_;
    do_log('debug2', 'List:send_msg_digest(%s)', $listname);
    
    my $filename = "$Conf{'queuedigest'}/$listname";
    
    ## Who is the enveloppe sender ?
    my $admin = $self->{'admin'};
    my $host = $admin->{'host'};
    my $name = "$self->{'name'}";
    my $from = "$name-request\@$host";
    my $returnpath = "$name-owner\@$host";
    my $reply = $from;
    my $to = "$name\@$host";

    if ($self->get_reply_to() =~ /^list$/io) {
	$reply = "$name\@$host";
    }

    my @tabrcpt;
    my $i;
    
    my $subject;
    my ($mail, @list_of_mail);

    ## Check the list
    return undef unless ($listname eq $name);

    ## Create the list of subscribers in digest mode
    for (my $user = $self->get_first_user(); $user; $user = $self->get_next_user()) {
	push @tabrcpt, $user->{'email'} 
	     if $user->{'reception'} eq "digest";
    }
    return if ($#tabrcpt == -1);

    my $old = $/;
    $/ = "\n\n" . $msg::separator . "\n\n";

    ## Digest split in individual messages
    open DIGEST, $filename or return undef;
    foreach (<DIGEST>){
	
	my @text = split /\n/;
	pop @text; pop @text;

	## Restore carriage returns
	foreach $i (0 .. $#text) {
	    $text[$i] .= "\n";
	}

	$mail = new Mail::Internet \@text;

	push @list_of_mail, $mail;
	
    }
    close DIGEST;
    $/ = $old;

    ## Deletes the introduction part
    splice @list_of_mail, 0, 1;

    ## Index construction
    foreach $i (0 .. $#list_of_mail){
	my $mail = $list_of_mail[$i];

	## Subject cleanup
	$subject = $mail->head->get('Subject') or next;
	$mail->head->replace('Subject', &tools::decode_string($subject));
    }

    ## Digest Headers
    my $hdr = new Mail::Header;
    $hdr->add('Reply-to', "$reply");
    $hdr->add('From', "$from");
    $hdr->add('To', "$to");
    $hdr->add('Subject',sprintf(Msg(8, 9, "Digest of list %s"),$listname));

    ## Digest construction
    *DESC = smtp::smtpto($returnpath, \@tabrcpt);
    $hdr->print(\*DESC);

    my @topics;
    push @topics, sprintf(Msg(8, 13, "Table of content"));
    push @topics, sprintf(" :\n\n");

    ## Digest index
    foreach $i (0 .. $#list_of_mail){
	my $mail = $list_of_mail[$i];	
	my $subject = $mail->head->get('Subject') || "\n";
	
	push @topics, sprintf ' ' x (2 - length($i)) . "%d. %s", $i+1, $subject;
    }
    
    my $msg = MIME::Entity->build (To         => $to,
				   From       => $from,
				   'Reply-to' => $reply,
				   Type       => 'multipart/mixed',
				   Subject    => sprintf(Msg(8, 9, "Digest of list %s"),$listname)
				   );
    
    my $table_of_content = MIME::Entity->build (Type        => 'text/plain; charset=iso-8859-1',
						Description => sprintf(Msg(8, 13, 'Table of content')),
						Data        => \@topics
						);
    
    $msg->add_part($table_of_content);
    
    my $digest = MIME::Entity->build (Type     => 'multipart/digest',
				      Boundary => '__--__--'
				      );
    ## Digest messages
    foreach $mail (@list_of_mail) {
	$mail->tidy_body;
	$mail->remove_sig;
	
	$digest->attach(Type     => 'message/rfc822',
			Disposition => 'inline',
			Data        => $mail->as_string
			);
    }
    
    my @now  = localtime(time);
    my $footer = sprintf Msg(8, 14, "End of %s Digest"), $listname;
    $footer .= sprintf " - %s\n", POSIX::strftime("%a %b %e %H:%M:%S %Y", @now);

    $digest->attach(Type        => 'text/plain',
		    Disposition => 'inline',
		    Data        => $footer
		    );
    
    $msg->add_part($digest); 

    ## Add a footer
    $msg = _add_parts($msg, $name, $self->{'admin'}{'footer_type'});

    $msg->print (\*DESC);
    close(DESC); 
}

## Send a global (not relative to a list) file to a user
sub send_global_file {
    my($action, $who, $context) = @_;
    do_log('debug2', 'List::send_global_file(%s, %s)', $action, $who);

    my $filename;
    my $data = $context;

    unless ($data->{'user'}) {
	unless ($data->{'user'} = &get_user_db($who)) {
	    $data->{'user'}{'email'} = $who;
	}

	unless ($data->{'user'}{'lang'}) {
	    $data->{'user'}{'lang'} = $Language::sympa_lang;
	}
    }

    ## What file   
    if (-r "$Conf{'etc'}/templates/$action.tpl") {
	$filename = "$Conf{'etc'}/templates/$action.tpl";
    }elsif (-r "--DIR--/bin/templates/$action.tpl") {
	$filename = "--DIR--/bin/templates/$action.tpl";
    }else{
	$filename = '' ;
	do_log ('err',"enable to open file $Conf{'etc'}/templates/$action.tpl NOR --DIR--/bin/templates/$action.tpl");
    }
    $data->{'conf'}{'email'} = $Conf{'email'};
    $data->{'conf'}{'host'} = $Conf{'host'};
    $data->{'conf'}{'sympa'} = $Conf{'sympa'};
    $data->{'conf'}{'listmaster'} = $Conf{'listmaster'};
    $data->{'conf'}{'wwsympa_url'} = $Conf{'wwsympa_url'};
    $data->{'conf'}{'version'} = $main::Version;
    $data->{'from'} = $Conf{'request'};
    $data->{'return_path'} = $Conf{'request'};

    mail::mailfile($filename, $who, $data);

    return 1;
}

## Send a file to a user
sub send_file {
    my($self, $action, $who, $context) = @_;
    do_log('debug2', 'List::send_file(%s, %s)', $action, $who);

    my $name = $self->{'name'};
    my $filename;

    my $data = $context;

    ## Change to list directory
    unless (chdir $name) {
	&do_log('info', 'Cannot chdir to %s', $name); 
    }

    unless ($data->{'user'}) {
	unless ($data->{'user'} = &get_user_db($who)) {
	    $data->{'user'}{'email'} = $who;
	    $data->{'user'}{'lang'} = $self->{'admin'}{'lang'};
	}
    }

    ## What file   
    if (-r "$action.tpl") {
	$filename = "$action.tpl";
    }elsif (-r "$action.mime") {
	$filename = "$action.mime";
    }elsif (-r "$action") {
	$filename = "$action";
    }elsif (-r "$Conf{'etc'}/templates/$action.tpl") {
	$filename = "$Conf{'etc'}/templates/$action.tpl";
    }elsif (-r "$Conf{'home'}/$action.mime") {
	$filename = "$Conf{'home'}/$action.mime";
    }elsif (-r "$Conf{'home'}/$action") {
	$filename = "$Conf{'home'}/$action";
    }elsif (-r "--DIR--/bin/templates/$action.tpl") {
	$filename = "--DIR--/bin/templates/$action.tpl";
    }else {
	$filename = '';
	do_log ('err',"enable to open file $action.tpl in list directory NOR $Conf{'etc'}/templates/$action.tpl NOR --DIR--/bin/templates/$action.tpl");
    }

    ## Unique return-path
    if ((($self->{'admin'}{'welcome_return_path'} eq 'unique') && ($action eq 'welcome')) ||
	(($self->{'admin'}{'remind_return_path'} eq 'unique') && ($action eq 'remind')))  {
	my $escapercpt = $who ;
	$escapercpt =~ s/\@/\/a\//;
	$data->{'return_path'} = "bounce+$escapercpt\/\/$name\@$self->{'admin'}{'host'}";
    }else {
	$data->{'return_path'} = "$name-owner\@$self->{'admin'}{'host'}";
    }
    
    $data->{'conf'}{'email'} = $Conf{'email'};
    $data->{'conf'}{'host'} = $Conf{'host'};
    $data->{'conf'}{'sympa'} = $Conf{'sympa'};
    $data->{'conf'}{'listmaster'} = $Conf{'listmaster'};
    $data->{'conf'}{'wwsympa_url'} = $Conf{'wwsympa_url'};
    $data->{'list'}{'lang'} = $self->{'admin'}{'lang'};
    $data->{'list'}{'name'} = $name;
    $data->{'list'}{'host'} = $self->{'admin'}{'host'};
    $data->{'list'}{'subject'} = $self->{'admin'}{'subject'};
    $data->{'list'}{'owner'} = $self->{'admin'}{'owner'};
    $data->{'from'} = "$name-request\@$data->{'list'}{'host'}";
    
    foreach my $key (keys %{$context}) {
	$data->{'context'}{$key} = $context->{$key};
    }

    mail::mailfile($filename, $who, $data);

    chdir $Conf{'home'};

    return 1;
}

## Delete the indicate users from the list.
sub delete_user {
    my($self, @u) = @_;
    do_log('debug2', 'List::delete_user');

    my $who;
    my $name = $self->{'name'};

    foreach $who (@u) {
        $who = lc($who);
	if ($self->{'admin'}{'user_data_source'} eq 'database') {
	    my $statement;

	    ## Check database connection
	    unless ($dbh and $dbh->ping) {
		return undef unless &db_connect();
	    }
	    
	    ## Delete record in SUBSCRIBER
	    $statement = sprintf "DELETE FROM subscriber_table WHERE (user_subscriber=%s AND list_subscriber=%s)",$dbh->quote($who), $dbh->quote($name);
	    
	    unless ($dbh->do($statement)) {
		do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
		return undef;
	    }   

	    $self->{'total'}--;

	}else {
	    my $users = $self->{'users'};

	    delete $self->{'users'}{$who};
	    $self->{'total'}-- unless (exists $users->{$who});
	}
    }

    $self->savestats();

    return 1;
}

## Returns the cookie for a list, if any.
sub get_cookie {
   return shift->{'admin'}{'cookie'};
}

## Returns the maximum size allowed for a message to the list.
sub get_max_size {
   return shift->{'admin'}{'max_size'};
}

## Returns an array with the Reply-To data
sub get_reply_to {
   return (shift->{'admin'}{'reply_to'});
}

## Returns a default user option
sub get_default_user_options {
    my $self = shift->{'admin'};
    my $what = shift;
    do_log('debug2', 'List::get_default_user_options(%s)', $what);

    if ($self) {
	return $self->{'default_user_options'};
    }
    return undef;
}

## Returns the number of subscribers to the list
sub get_total {
   return shift->{'total'};
}

## Returns a hash for a given user
sub get_user_db {
    my $who = lc(shift);
    do_log('debug2', 'List::get_user_db(%s)', $who);

    my $statement;
 
    unless ($List::use_db) {
	&do_log('info', 'Sympa not setup to use DBI');
	return undef;
    }

    ## Check database connection
    unless ($dbh and $dbh->ping) {
	return undef unless &db_connect();
    }

    if ($Conf{'db_type'} eq 'Oracle') {
	## "AS" not supported by Oracle
	$statement = sprintf "SELECT lower(email_user) \"email\", gecos_user \"gecos\", password_user \"password\", cookie_delay_user \"cookie_delay\", lang_user \"lang\" FROM user_table WHERE lower(email_user) = %s ", $dbh->quote($who);
    }else {
	$statement = sprintf "SELECT lower(email_user) AS email, gecos_user AS gecos, password_user AS password, cookie_delay_user AS cookie_delay, lang_user AS lang FROM user_table WHERE lower(email_user) = %s ", $dbh->quote($who);
    }
    
    push @sth_stack, $sth;

    unless ($sth = $dbh->prepare($statement)) {
	do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
	return undef;
    }
    
    unless ($sth->execute) {
	do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	return undef;
    }
    
    my $user = $sth->fetchrow_hashref;
 
    $sth->finish();

    $sth = pop @sth_stack;

    return $user;
}

## Returns a subscriber of the list.
sub get_subscriber {
    my  $self= shift;
    my  $email = lc(shift);
    
    do_log('debug2', 'List::get_subscriber');

    if ($self->{'admin'}{'user_data_source'} eq 'database') {

	my $name = $self->{'name'};
	my $statement;
	my $date_field = sprintf $date_format{'read'}{$Conf{'db_type'}}, 'date_subscriber', 'date_subscriber';

	## Check database connection
	unless ($dbh and $dbh->ping) {
	    return undef unless &db_connect();
	}

	if ($Conf{'db_type'} eq 'Oracle') {
	    ## "AS" not supported by Oracle
	    $statement = sprintf "SELECT email_user \"email\", gecos_user \"gecos\", lang_user \"lang\", bounce_subscriber \"bounce\", reception_subscriber \"reception\", visibility_subscriber \"visibility\", %s \"date\"  FROM user_table, subscriber_table WHERE (email_user = %s AND list_subscriber = %s AND user_subscriber = %s)", $date_field, $dbh->quote($email), $dbh->quote($name), $dbh->quote($email);
	}else {
	    $statement = sprintf "SELECT email_user AS email, gecos_user AS gecos, lang_user AS lang, bounce_subscriber AS bounce, reception_subscriber AS reception, visibility_subscriber AS visibility, %s AS date  FROM user_table, subscriber_table WHERE (email_user = %s AND list_subscriber = %s AND user_subscriber = %s)", $date_field, $dbh->quote($email), $dbh->quote($name), $dbh->quote($email);
	}

	push @sth_stack, $sth;

	unless ($sth = $dbh->prepare($statement)) {
	    do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
	    return undef;
	}
	
	unless ($sth->execute) {
	    do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	    return undef;
	}
	
	my $user = $sth->fetchrow_hashref;

	$sth->finish();

	$sth = pop @sth_stack;

	return $user;
    }else {
	my $i;
	return undef 
	    unless $self->{'users'}{$email};

	my %user = split(/\n/, $self->{'users'}{$email});
	
	return \%user;
    }
}


## Returns the first user for the list.
sub get_first_user {
    my $self = shift;
    do_log('debug2', 'List::get_first_user');

    if ($self->{'admin'}{'user_data_source'} eq 'database') {

	my $name = $self->{'name'};
	my $statement;
	my $date_field = sprintf $date_format{'read'}{$Conf{'db_type'}}, 'date_subscriber', 'date_subscriber';
	
	## Check database connection
	unless ($dbh and $dbh->ping) {
	    return undef unless &db_connect();
	}

	if ($Conf{'db_type'} eq 'Oracle') {
	    ## "AS" not supported by Oracle
	    $statement = sprintf "SELECT email_user \"email\", gecos_user \"gecos\", reception_subscriber \"reception\", visibility_subscriber \"visibility\", cookie_delay_user \"cookie_delay\", lang_user \"lang\", bounce_subscriber \"bounce\", password_user \"password\", %s \"date\", substr(user_subscriber,instr(user_subscriber,'\@')+1) \"dom\" FROM user_table,subscriber_table WHERE (list_subscriber = %s AND email_user = user_subscriber) ORDER BY \"dom\"", $date_field, $dbh->quote($name);

	}elsif ($Conf{'db_type'} eq 'Sybase'){


	    $statement = sprintf "SELECT email_user \"email\", gecos_user \"gecos\", reception_subscriber \"reception\", visibility_subscriber \"visibility\", cookie_delay_user \"cookie_delay\", lang_user \"lang\", bounce_subscriber \"bounce\", password_user \"password\", %s \"date\", substring(user_subscriber,charindex('\@',user_subscriber)+1,100) \"dom\" FROM user_table,subscriber_table WHERE (list_subscriber = %s AND email_user = user_subscriber) ORDER BY \"dom\"", $date_field, $dbh->quote($name)
    ;
	}else {
	    $statement = sprintf "SELECT email_user AS email, gecos_user AS gecos, reception_subscriber AS reception, visibility_subscriber AS visibility, cookie_delay_user AS cookie_delay, lang_user AS lang, bounce_subscriber AS bounce, password_user AS password, %s AS date, SUBSTRING(user_subscriber FROM position('\@' IN user_subscriber) FOR 50) AS dom FROM user_table,subscriber_table WHERE (list_subscriber = %s AND email_user = user_subscriber) ORDER BY dom", $date_field, $dbh->quote($name);
	}

	push @sth_stack, $sth;

	unless ($sth = $dbh->prepare($statement)) {
	    do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
	    return undef;
	}
	
	unless ($sth->execute) {
	    do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	    return undef;
	}
	
	my $user = $sth->fetchrow_hashref;

	return $user;
    }else {
	my ($i, $j);
	my $ref = $self->{'ref'};
	
	if ($ref->seq($i, $j, R_FIRST) == 0)  {
	    my %user = split(/\n/, $j);
	    return \%user;
	}
	return undef;
    }
}

## Loop for all subsequent users.
sub get_next_user {
    my $self = shift;
    do_log('debug2', 'List::get_next_user');

    if ($self->{'admin'}{'user_data_source'} eq 'database') {
	my $user = $sth->fetchrow_hashref;

	unless (defined $user) {
	    $sth->finish;
	    $sth = pop @sth_stack;
	}

#	$self->{'total'}++;

	return $user;
    }else {
	my ($i, $j);
	my $ref = $self->{'ref'};
	
	if ($ref->seq($i, $j, R_NEXT) == 0) {
	    my %user = split(/\n/, $j);
	    return \%user;
	}
	return undef;
    }
}

## Returns the first bouncing user
sub get_first_bouncing_user {
    my $self = shift;
    do_log('debug2', 'List::get_first_bouncing_user');

    unless ($self->{'admin'}{'user_data_source'} eq 'database') {
	&do_log('info', 'Function available for lists in database mode only');
	return undef;
    }
    
    my $name = $self->{'name'};
    my $statement;
    my $date_field = sprintf $date_format{'read'}{$Conf{'db_type'}}, 'date_subscriber', 'date_subscriber';
    
    ## Check database connection
    unless ($dbh and $dbh->ping) {
	return undef unless &db_connect();
    }

    if ($Conf{'db_type'} eq 'Oracle') {
	## "AS" not supported by Oracle
	$statement = sprintf "SELECT email_user \"email\", gecos_user \"gecos\", reception_subscriber \"reception\", visibility_subscriber \"visibility\", cookie_delay_user \"cookie_delay\", lang_user \"lang\", bounce_subscriber \"bounce\", %s \"date\" FROM user_table, subscriber_table WHERE (list_subscriber = %s AND bounce_subscriber != 'NULL' AND email_user = user_subscriber )", $date_field, $dbh->quote($name);
    }else {
	$statement = sprintf "SELECT email_user AS email, gecos_user AS gecos, reception_subscriber AS reception, visibility_subscriber AS visibility, cookie_delay_user AS cookie_delay, lang_user AS lang, bounce_subscriber AS bounce, %s AS date FROM user_table, subscriber_table WHERE (list_subscriber = %s AND bounce_subscriber != 'NULL' AND email_user = user_subscriber )", $date_field, $dbh->quote($name);
    }

    push @sth_stack, $sth;

    unless ($sth = $dbh->prepare($statement)) {
	do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
	return undef;
    }
    
    unless ($sth->execute) {
	do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	return undef;
    }
    
    my $user = $sth->fetchrow_hashref;
    
    return $user;
}

## Loop for all subsequent bouncing users.
sub get_next_bouncing_user {
    my $self = shift;
    do_log('debug2', 'List::get_next_bouncing_user');

    unless ($self->{'admin'}{'user_data_source'} eq 'database') {
	&do_log('info', 'Function available for lists in database mode only');
	return undef;
    }

    my $user = $sth->fetchrow_hashref;
    
    unless (defined $user) {
	$sth->finish;
	$sth = pop @sth_stack;
    }

    return $user;
}

## Total bouncing subscribers
sub get_total_bouncing {
    my $self = shift;
    do_log('debug2', 'List::get_total_boucing');

    unless ($self->{'admin'}{'user_data_source'} eq 'database') {
	&do_log('info', 'Function available for lists in database mode only');
	return undef;
    }

    my $name = $self->{'name'};
    my $statement;
   
    ## Check database connection
    unless ($dbh and $dbh->ping) {
	return undef unless &db_connect();
    }	   
    
    ## Query the Database
    $statement = sprintf "SELECT count(*) FROM subscriber_table WHERE (list_subscriber = %s  AND bounce_subscriber != 'NULL')", $dbh->quote($name);
    
    push @sth_stack, $sth;

    unless ($sth = $dbh->prepare($statement)) {
	do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
	return undef;
    }
    
    unless ($sth->execute) {
	do_log('debug','Unable to execute SQL statement: %s', $dbh->errstr);
	return undef;
    }
    
    my $total =  $sth->fetchrow;

    $sth->finish();

    $sth = pop @sth_stack;

    return $total;
}

## Is the person in user table (db only)
sub is_user_db {
   my $who = lc(pop);
   do_log('debug2', 'List::is_user_db(%s)', $who);

   return undef unless ($who);

   unless ($List::use_db) {
       &do_log('info', 'Sympa not setup to use DBI');
       return undef;
   }

   my $statement;
   
   ## Check database connection
   unless ($dbh and $dbh->ping) {
       return undef unless &db_connect();
   }	   
   
   ## Query the Database
   $statement = sprintf "SELECT count(*) FROM user_table WHERE email_user = %s", $dbh->quote($who);
   
   push @sth_stack, $sth;

   unless ($sth = $dbh->prepare($statement)) {
       do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
       return undef;
   }
   
   unless ($sth->execute) {
       do_log('debug','Unable to execute SQL statement: %s', $dbh->errstr);
       return undef;
   }
   
   my $is_user = $sth->fetchrow();

   $sth->finish();
   
   $sth = pop @sth_stack;

   return $is_user;
}

## Is the indicated person a subscriber to the list ?
sub is_user {
    my ($self, $who) = @_;
    $who= lc($who);
    do_log('debug2', 'List::is_user(%s)', $who);

   return undef unless ($self && $who);

   if ($self->{'admin'}{'user_data_source'} eq 'database') {
   
       my $statement;
       my $name = $self->{'name'};

       ## Check database connection
       unless ($dbh and $dbh->ping) {
	   return undef unless &db_connect();
       }	   

       ## Query the Database
       $statement = sprintf "SELECT count(*) FROM subscriber_table WHERE (list_subscriber = %s AND user_subscriber = %s)",$dbh->quote($name), $dbh->quote($who);
       
       push @sth_stack, $sth;

       unless ($sth = $dbh->prepare($statement)) {
	   do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
	   return undef;
       }
       
       unless ($sth->execute) {
	   do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	   return undef;
       }
       
       my $is_user = $sth->fetchrow;

       $sth->finish();

       $sth = pop @sth_stack;

       return $is_user;
   }else {
       my $users = $self->{'users'};
       return 0 unless ($users);
       
       return 1 if ($users->{$who});
       return 0;
   }
}

## Sets new values for the given user (except gecos)
sub update_user {
    my($self, $who, $values) = @_;
    do_log('debug2', 'List::update_user(%s)', $who);
    $who = lc($who);    

    my ($field, $value);
    
    if ($self->{'admin'}{'user_data_source'} eq 'database') {
	
	my ($user, $statement, $table);
	my $name = $self->{'name'};
	
	## mapping between var and field names
	my %map_field = ( reception => 'reception_subscriber',
			  visibility => 'visibility_subscriber',
			  date => 'date_subscriber',
			  gecos => 'gecos_user',
			  password => 'password_user',
			  bounce => 'bounce_subscriber'
			  );
	
	## mapping between var and tables
	my %map_table = ( reception => 'subscriber_table',
			  visibility => 'subscriber_table',
			  date => 'subscriber_table',
			  gecos => 'user_table',
			  password => 'user_table',
			  bounce => 'subscriber_table');
	
	## Check database connection
	unless ($dbh and $dbh->ping) {
	    return undef unless &db_connect();
	}	   
	
	## Update each table
	foreach $table ('user_table','subscriber_table') {
	    
	    my @set_list;
	    while (($field, $value) = each %{$values}) {
		next unless ($map_field{$field} and $map_table{$field});
		if ($map_table{$field} eq $table) {
		    if ($field eq 'date') {
			$value = sprintf $date_format{'write'}{$Conf{'db_type'}}, $value, $value;
		    }elsif ($value eq 'NULL'){
			$value = '\N';
		    }else {
			$value = $dbh->quote($value);
		    }
		    my $set = sprintf "%s=%s", $map_field{$field}, $value;
		    push @set_list, $set;
		}
	    }
	    next unless @set_list;
	    
	    ## Update field
	    if ($table eq 'user_table') {
		$statement = sprintf "UPDATE %s SET %s WHERE (email_user=%s)", $table, join(',', @set_list), $dbh->quote($who); 
	    }elsif ($table eq 'subscriber_table') {
		$statement = sprintf "UPDATE %s SET %s WHERE (user_subscriber=%s AND list_subscriber=%s)", $table, join(',', @set_list), $dbh->quote($who), $dbh->quote($name);
	    }
	    
	    unless ($dbh->do($statement)) {
		do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
		return undef;
	    }
	}
    }else {
	my $user = $self->{'users'}->{$who};
	return undef unless $user;
	
	my %u = split(/\n/, $user);
	my ($i, $j);
	$u{$i} = $j while (($i, $j) = each %{$values});
	
	while (($field, $value) = each %{$values}) {
	    $u{$field} = $value;
	}
	
	$user = join("\n", %u);      
	$self->{'users'}->{$who} = $user;
    }
    
    return 1;
}

## Sets new values for the given user in the Database
sub update_user_db {
    my($who, $values) = @_;
    do_log('debug2', 'List::update_user_db(%s)', $who);
    $who = lc($who);

    unless ($List::use_db) {
	&do_log('info', 'Sympa not setup to use DBI');
	return undef;
    }

    my ($field, $value);
    
    my ($user, $statement, $table);
    
    ## mapping between var and field names
    my %map_field = ( gecos => 'gecos_user',
		      password => 'password_user',
		      cookie_delay => 'cookie_delay_user',
		      lang => 'lang_user'
		      );
    
    ## Check database connection
    unless ($dbh and $dbh->ping) {
	return undef unless &db_connect();
    }	   
    
    ## Update each table
    my @set_list;
    while (($field, $value) = each %{$values}) {
	next unless ($map_field{$field});
	my $set;
	
	if ($map_field{$field} eq 'cookie_delay_user')  {
	    $set = sprintf '%s=%s', $map_field{$field}, $value;
	}else { 
	    $set = sprintf '%s=%s', $map_field{$field}, $dbh->quote($value);
	}

	push @set_list, $set;
    }
    
    return undef 
	unless @set_list;
    
    ## Update field
    $statement = sprintf "UPDATE user_table SET %s WHERE (email_user=%s)"
	, join(',', @set_list), $dbh->quote($who); 
    
    unless ($dbh->do($statement)) {
	do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	return undef;
    }
    
    return 1;
}

## Adds a new user to Database (in User table)
sub add_user_db {
    my($values) = @_;
    do_log('debug2', 'List::add_user_db');

   my ($field, $value);
   my ($user, $statement, $table);
   
   unless ($List::use_db) {
       &do_log('info', 'Sympa not setup to use DBI');
       return undef;
   }

   return undef unless (my $who = lc($values->{'email'}));

   return undef if (is_user_db($who));

   ## mapping between var and field names
   my %map_field = ( email => 'email_user',
		     gecos => 'gecos_user',
		     password => 'password_user',
		     cookie_delay => 'cookie_delay_user',
		     lang => 'lang_user'
		     );

   ## Check database connection
   unless ($dbh and $dbh->ping) {
       return undef unless &db_connect();
   }	   

   ## Update each table
   my (@insert_field, @insert_value);
   while (($field, $value) = each %{$values}) {
       next unless ($map_field{$field});

       my $insert = sprintf "%s", $dbh->quote($value);
       push @insert_value, $insert;
       push @insert_field, $map_field{$field}
   }

   return undef 
       unless @insert_field;

   ## Update field
   $statement = sprintf "INSERT INTO user_table (%s) VALUES (%s)"
       , join(',', @insert_field), join(',', @insert_value); 
   
   unless ($dbh->do($statement)) {
       do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
       return undef;
   }

   return 1;
}

## Adds a new user, no overwrite.
sub add_user {
    my($self, $values) = @_;
    do_log('debug2', 'List::add_user');

    my $date_field = sprintf $date_format{'write'}{$Conf{'db_type'}}, $values->{'date'}, $values->{'date'};
    
    return undef
	unless (my $who = lc($values->{'email'}));
    
    if ($self->{'admin'}{'user_data_source'} eq 'database') {
	
	my $name = $self->{'name'};
	my $statement;
	
	## Check database connection
	unless ($dbh and $dbh->ping) {
	    return undef unless &db_connect();
	}	   
	
	## Is the email in user table ?
	if ($self->is_user($who)) {
	    
	    ## Update User in Table
	    $statement = sprintf "UPDATE user_table SET email_user=%s, gecos_user=%s WHERE email_user=%s", $dbh->quote($who), $dbh->quote($values->{'gecos'}), $dbh->quote($who);
	    
	    unless ($dbh->do($statement)) {
		do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
		return undef;
	    }
	}elsif (! is_user_db($who)) {
	    ## Insert User in Table
	    $statement = sprintf "INSERT INTO user_table (email_user, gecos_user) VALUES (%s,%s)",$dbh->quote($who), $dbh->quote($values->{'gecos'});
	    
	    unless ($dbh->do($statement)) {
		do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	       return undef;
	    }
	}

	## Update Subscriber Table
	$statement = sprintf "INSERT INTO subscriber_table (user_subscriber, list_subscriber, date_subscriber, reception_subscriber, visibility_subscriber) VALUES (%s, %s, %s, %s, %s)", $dbh->quote($who), $dbh->quote($name), $date_field, $dbh->quote($values->{'reception'}), $dbh->quote($values->{'visibility'});
       
       unless ($dbh->do($statement)) {
	   do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	   return undef;
       }

	$self->{'total'}++;

   }else {
          my (%u, $i, $j);

	  $self->{'total'}++ unless ($self->{'users'}->{$who});
	  $u{$i} = $j while (($i, $j) = each %{$values});
	  $self->{'users'}->{$who} = join("\n", %u);
   }

   $self->savestats();

   return 1;
}

## Is the user listmaster
sub is_listmaster {
    my $who = shift;
    
    $who =~ y/A-Z/a-z/;
    foreach my $listmaster (@{$Conf{'listmasters'}}){
	return 1 if ($listmaster =~ /^\s*$who\s*$/i);
    }    
    return 0;
}

## Does the user have a particular function in the list ?
sub am_i {
    my($self, $function, $who) = @_;
    do_log('debug2', 'List::am_i(%s, %s)', $function, $who);

    my $u;
    
    return undef unless ($self && $who);;
    $function =~ y/A-Z/a-z/;
    $who =~ y/A-Z/a-z/;
    chomp($who);
    
    ## Listmaster has all privileges
    return 1 if (($function eq 'owner') and &is_listmaster($who));

    ## Check owners | editor
    if ($function =~ /^(editor|owner)$/i){
	return undef unless ($self->{'admin'} && $self->{'admin'}{$function});
	
	foreach $u (@{$self->{'admin'}{$function}}) {
	    return 1 if ($u->{'email'} =~ /^$who$/i);
	}
    }
    elsif ($function =~ /^privileged_owner$/i) {
	foreach $u (@{$self->{'admin'}{'owner'}}) {
	    return 1 if (($u->{'email'} =~ /^$who$/i) && ($u->{'profile'} =~ 'privileged'));
	}
    }
    return undef;
}

## Return the state for simple functions
sub get_state {
    my($self, $action) = @_;
    do_log('debug2', 'List::get_state(%s)', $action);
    
    my $i;
    
    my $admin = $self->{'admin'};
    if ($action =~ /^sig$/io) {
	$i = $admin->{'unsubscribe'}{'name'};
	return 'open' if ($i =~ /^(open|public)$/io);
	return 'closed' if ($i =~ /^closed$/io);
	return 'auth' if ($i =~ /^auth$/io);
	return 'owner' if ($i =~ /^owner$/io);
	
    }elsif ($action =~ /^sub$/io) {
	$i = $admin->{'subscribe'}{'name'};
	return 'open' if ($i =~ /^(open|public)$/io);
	return 'auth' if ($i =~ /^auth$/io);
	return 'owner' if ($i =~ /^owner$/io);
	return 'closed' if ($i =~ /^closed$/io);
    }
    
    return undef;
}

## Return the action to perform for 1 sender using 1 auth method to perform 1 operation
sub get_action {
    my $operation = shift;
    do_log('debug2', 'List::get_action(%s)', $operation);

    ## following parameters depends on operation type
    my ($list,$auth_method);
    my $context={};
    my (@rules, $name) ;

    if($operation =~ /^(subscribe|remind|review|del|info|visibility|invite)/) {
	$context->{'listname'} = shift ;
	$context->{'sender'} = shift;
	$auth_method = shift;
#	&do_log('debug', "-------------- operation $operation ,listname : $context->{'listname'}, sender : $context->{'sender'}, auth : $auth_method");
	
    }elsif($operation =~ /^(unsubscribe|add|set)/) {
	$context->{'listname'}=shift;
	$context->{'sender'}= shift;
	$context->{'email'}= shift;
	$auth_method = shift;
    }elsif($operation =~ /^send/) {
	$context->{'listname'}=shift;
	$context->{'sender'}= shift;
	$auth_method = shift;
	$context->{'hdr'} = shift;
    }elsif($operation =~ /^edit_list|create_list|global_remind/) {
        $context->{'sender'}= shift;
	$auth_method = shift ;
    }else{
	do_log ('info',"fatal error : unknown operation $operation in List::get_action");
	return undef;
    }

    unless ( $auth_method =~ /^(smtp|md5|pgp|smime)/) {
	do_log ('info',"fatal error : unknown auth method $auth_method in List::get_action");
	return undef;
    }

    if ($context->{'listname'}) {
        unless ( $list = new List ($context->{'listname'}) ){
	    do_log ('info',"get_action :  unable to create object $context->{'listname'}");
	    return undef ;
	}

	@rules = @{$list->{'admin'}{$operation}{'rules'}};
	$name = $list->{'admin'}{$operation}{'name'};
    }else{
	my $scenario = &_load_scenario_file ($operation,$Conf{$operation});
        @rules = @{$scenario->{'rules'}};
	$name = $scenario->{'name'};
    }

    unless ($name) {
	do_log ('info',"internal error : configuration for operation $operation is not yet performed by scenario");
	return undef;
    }
    foreach my $rule (@rules) {
	next if ($rule eq 'scenario');
	if ($auth_method eq $rule->{'auth_method'}) {
	    my $result =  &verify ($context,$rule->{'condition'});

	    if (! defined ($result)) {
		do_log ('notice',"error in $rule->{'condition'},$rule->{'auth_method'},$rule->{'action'}" );
		return ("error-performing-condition : $rule->{'condition'}",$rule->{'auth_method'},'reject') ;
	    }
	    if ($result == -1) {
		do_log ('debug2',"rule $rule->{'condition'},$rule->{'auth_method'},$rule->{'action'} rejected");
		next;
	    }
	    if ($result == 1) {
		do_log ('debug2',"rule $rule->{'condition'},$rule->{'auth_method'},$rule->{'action'} accepted");
		return ($rule->{'condition'},$rule->{'auth_method'},$rule->{'action'});
	    }
	}
    }
    do_log ('debug2',"no rule match, reject");
    return ('default','default','reject');
}


## check if email respect some condition
sub verify {
    my ($context, $condition) = @_;
    do_log('debug2', 'List::verify(%s)', $condition);

#    while (my($k,$v) = each %{$context}) {
#	do_log ('debug2',"verify: context->{$k} = $v");
#    }

    unless (defined($context->{'sender'} )) {
	do_log('info',"internal error, no sender find in List::verify, report authors");
	return undef;
    }

    unless ($condition =~ /(\!)?\s*(true|is_listmaster|is_editor|is_owner|is_subscriber|match|equal|message)\s*\(\s*(.*)\s*\)\s*/i) {
	&do_log('info', "error rule syntaxe: unknown condition $condition");
	return undef;
    }
    my $negation = 1 ;
    if ($1 eq '!') {
	$negation = -1 ;
    }

    my $condition_key = lc($2);
    my $arguments = $3;
    my @args;
    my $list;   

    while ($arguments =~ s/^\s*(\[\w+(\-\>[\w\-]+)?\]|([\w\-\.]+)|\/([^\/\\]+|\\\/|\\)+[^\\]+\/)\s*,?//) {
	my $value=$1;
	if ($value =~ /^\[(\w+)\]$/i) {
	    if (defined ($context->{$1})) {
		$value=$context->{$1};
	    }else{
		do_log('notice',"unkown variable context \[$1\] in rule $condition");
		return undef;
	    }
	}elsif($value =~ /^\[header\-\>([\w\-]+)\]$/i) {
	    if (defined ($context->{'hdr'})) {
		my $header = $context->{'hdr'};
		$value = $header->get($1);
	    }else{
		do_log('notice',"unkown variable context \[$1\] in rule $condition");
		return undef;
	    }
	}
	push (@args,$value);
	
    }
    # condition that require 0 argument
    if ($condition_key eq 'true') {
	unless ($#args == -1){ 
	    do_log('notice',"error rule syntaxe : incorrect number of argument or incorrect argument syntaxe $condition") ; 
	    return undef ;
	}
	# condition that require 1 argument
    }elsif ($condition_key eq 'is_listmaster') {
	unless ($#args == 0) { 
	     do_log ('notice',"error rule syntaxe : incorrect argument number for condition $condition_key") ; 
	    return undef ;
	}
	# condition that require 2 args
#
    }elsif ($condition_key =~ /^is_owner|is_editor|is_subscriber|match|equal|message$/i) {
	unless ($#args == 1) {
	    do_log ('notice',"error rule syntaxe : incorrect argument number for condition $condition_key") ; 
	    return undef ;
	}
    }else{
	do_log('notice', "error rule syntaxe : unknown condition $condition_key");
	return undef;
    }
    ## Now eval the condition
    ##### condition : true
    if ($condition_key =~ /\s*(true|any)\s*/i) {
	return $negation;
    }
    ##### condition is_listmaster
    if ($condition_key eq 'is_listmaster') {
	if ( &is_listmaster($args[0])) {
	    return $negation;
	}else{
	    return -1 * $negation;
	}
    }
    ##### condition is_owner, is_subscriber and is_editor
    if ($condition_key =~ /is_owner|is_subscriber|is_editor/i) {
	$list = new List ($args[0]);
	if (! $list) {
	    do_log('info',"unable to create list object \"$args[0]\"");
	    return undef;
	}
    }
    if ($condition_key eq 'is_subscriber') {
	if ($list->is_user($args[1])) {
	    return $negation ;
	}else{
	    return -1 * $negation ;
	}
    }
    if ($condition_key eq 'is_owner') {
	if ($list->am_i('owner',$args[1])) {
	    return $negation ;
	}else{
	    return -1 * $negation ;
	}
    }
    if ($condition_key eq 'is_editor') {
	if ($list->am_i('editor',$args[1])) {
	    return $negation ;
	}else{
	    return -1 * $negation ;
	}
    }
    ##### match
    if ($condition_key eq 'match') {
	unless ($args[1] =~ /^\/(.*)\/$/) {
	    &do_log('info', 'Match parameter %s is not a regexp', $args[1]);
	    return undef;
	}
	my $regexp = $1;
	
	if ($regexp =~ /\[host\]/) {
	    my $reghost = $Conf{'host'};
            $reghost =~ s/\./\\./g ;
            $regexp =~ s/\[host\]/$reghost/g ;
	}

	if ($args[0] =~ /$regexp/i) {
	    return $negation ;
	}else{
	    return -1 * $negation ;
	}
    }

    ## equal
    if ($condition_key eq 'equal') {
	if ($args[0] =~ /^$args[1]$/i) {
	    return $negation ;
	}else{
	    return -1 * $negation ;
	}
    }
    return undef;
}



## May the indicated user edit the indicated list parameter or not ?
sub may_edit {

    my($self,$parameter, $who) = @_;
    do_log('debug2', 'List::may_edit(%s, %s)', $parameter, $who);

    my $role;

    return undef unless ($self);

    my $edit_conf = &tools::load_edit_list_conf;

    if ( &is_listmaster($who)) {
	## listmaster has read write acces on any parameter
	return 'write';
    }
    
    if ( $self->am_i('privileged_owner',$who) ) {
	return ($edit_conf->{$parameter}{'privileged_owner'}) 
	    if $edit_conf->{$parameter}{'privileged_owner'};

	return ($edit_conf->{'default'}{'privileged_owner'}) 
	    if ($edit_conf->{'default'}{'privileged_owner'});
    }

    if ( $self->am_i('owner',$who) ) {

	return ($edit_conf->{$parameter}{'owner'}) 
	    if ($edit_conf->{$parameter}{'owner'});

	return ($edit_conf->{'default'}{'owner'}) 
	    if ($edit_conf->{'default'}{'owner'});
    }

    if ( $self->am_i('editor',$who) ) {
	return ($edit_conf->{$parameter}{'editor'}) 
	    if ($edit_conf->{$parameter}{'editor'});

	return ($edit_conf->{'default'}{'editor'}) 
	    if ($edit_conf->{'default'}{'editor'});
    }

    if ( $self->am_i('subscriber',$who) ) {
	return ($edit_conf->{$parameter}{'subscriber'}) 
	    if ($edit_conf->{$parameter}{'subscriber'});

	return ($edit_conf->{'default'}{'subscriber'}) 
	    if ($edit_conf->{'default'}{'subscriber'});
    }

    return ($edit_conf->{$parameter}{'default'}) 
	if ($edit_conf->{$parameter}{'default'});

    return ($edit_conf->{'default'}{'default'}) 
	if ($edit_conf->{'default'}{'default'});
    
    return 'hidden';
}


## May the indicated user edit a paramter while creating a new list
sub may_create_parameter {

    my($parameter, $who) = @_;
    do_log('debug2', 'List::may_create_parameter(%s, %s)', $parameter, $who);

    if ( &is_listmaster($who)) {
	return 1;
    }
    my $edit_conf = &tools::load_edit_list_conf;
    $edit_conf->{$parameter} ||= $edit_conf->{'default'};
    if (! $edit_conf->{$parameter}) {
	do_log('notice','tools::load_edit_list_conf privilege for parameter $parameter undefined');
	return undef;
    }
    if ($edit_conf->{$parameter}  =~ /^(owner)||(privileged_owner)$/i ) {
	return 1;
    }else{
	return 0;
    }

}


## May the indicated user do something with the list or not ?
## Action can be : send, review, index, get
##                 add, del, reconfirm, purge
sub may_do {
   my($self, $action, $who) = @_;
   do_log('debug2', 'List::may_do(%s, %s)', $action, $who);

   my $i;

   ## Just in case.
   return undef unless ($self && $action);
   my $admin = $self->{'admin'};
   return undef unless ($admin);

   $action =~ y/A-Z/a-z/;
   $who =~ y/A-Z/a-z/;

   if ($action =~ /^access_web_archive$/io) {

       my $web_arc_access = $admin->{'web_archive'}{'access'};
       if ($web_arc_access =~ /^public$/io)  {
	   return 1;
       }elsif ($web_arc_access =~ /^private$/io) {
	   return 1 if ($self->is_user($who)
			or $self->am_i('owner', $who)
			or $self->am_i('editor', $who));
       }elsif ($web_arc_access =~ /^owner$/io) {
	   return 1 if ($self->am_i('owner', $who)
			or $self->am_i('editor', $who));
       }elsif ($web_arc_access =~ /^closed$/io) {
	   return 1 if &is_listmaster($who);
       }
       return undef;

   }

   if ($action =~ /^(index|get)$/io) {
       my $arc_access = $admin->{'archive'}{'access'};
       if ($arc_access =~ /^public$/io)  {
	   return 1;
       }elsif ($arc_access =~ /^private$/io) {
	   return 1 if ($self->is_user($who));
	   return $self->am_i('owner', $who);
       }elsif ($arc_access =~ /^owner$/io) {
	   return $self->am_i('owner', $who);
       }
       return undef;
   }

   if ($action =~ /^(review)$/io) {
       foreach $i (@{$admin->{'review'}}) {
	   if ($i =~ /^public$/io) {
	       return 1;
	   }elsif ($i =~ /^private$/io) {
	       return 1 if ($self->is_user($who));
	       return $self->am_i('owner', $who);
	   }elsif ($i =~ /^owner$/io) {
	       return $self->am_i('owner', $who);
	   }
	   return undef;
       }
   }

   if ($action =~ /^send$/io) {
      if ($admin->{'send'} =~/^(private|privateorpublickey|privateoreditorkey)$/i) {

         return undef unless ($self->is_user($who) || $self->am_i('owner', $who));
      }elsif ($admin->{'send'} =~ /^(editor|editorkey|privateoreditorkey)$/i) {
         return undef unless ($self->am_i('editor', $who));
      }elsif ($admin->{'send'} =~ /^(editorkeyonly|publickey|privatekey)$/io) {
         return undef;
      }
      return 1;
   }

   if ($action =~ /^(add|del|remind|reconfirm|purge|expire)$/io) {
      return $self->am_i('owner', $who);
   }

   if ($action =~ /^(modindex)$/io) {
       return undef unless ($self->am_i('editor', $who));
       return 1;
   }

   if ($action =~ /^auth$/io) {
       if ($admin->{'send'} =~ /^(privatekey)$/io) {
	   return 1 if ($self->is_user($who) || $self->am_i('owner', $who));
       } elsif ($admin->{'send'} =~ /^(privateorpublickey)$/io) {
	   return 1 unless ($self->is_user($who) || $self->am_i('owner', $who));
       }elsif ($admin->{'send'} =~ /^(publickey)$/io) {
	   return 1;
       }
       return undef; #authent
   } 
   return undef;
}

## Is the list moderated ?
sub is_moderated {
   return (shift->{'admin'}{'send'}{'name'}=~/^(editor|editorkey|editorkeyonly)$/);
}

## Is the list moderated with a key?
sub is_moderated_key {
   return (shift->{'admin'}->{'send'}{'name'}=~/^(editorkeyonly|editorkey|privateoreditorkey)$/);
}

## Is the list moderated with a key?
sub is_privateoreditorkey {
   return (shift->{'admin'}{'send'}{'name'}=~/^privateoreditorkey$/);
}

## Is the list auth with a key?
sub is_private_key {
   return (shift->{'admin'}->{'send'}{'name'}=~/^privatekey$/);
}

## Is the list auth with a key?
sub is_public_key {
   return (shift->{'admin'}{'send'}{'name'}=~/^publickey$/);
}

## Is the list auth with a key?
sub is_authentified {
   return (shift->{'admin'}{'send'}{'name'}=~/^(publickey|privatekey|privateorpublickey)$/);
}

## Does the list support digest mode
sub is_digest {
   return (shift->{'admin'}{'digest'});
}

## Does the file exist ?
sub archive_exist {
   my($self, $file) = @_;
   do_log('debug2', 'List::archive_exist(%s)', $file);

   return undef unless ($self->is_archived());
   Archive::exist("$self->{'name'}/archives", $file);
}

## Send an archive file to someone
sub archive_send {
   my($self, $who, $file) = @_;
   do_log('debug2', 'List::archive_send(%s, %s)', $who, $file);

   return unless ($self->is_archived());
   my $i;
   if ($i = Archive::exist("$self->{'name'}/archives", $file)) {
      mail::mailarc($i, Msg(8, 7, "File") . " $self->{'name'} $file",$who );
   }
}

## List the archived files
sub archive_ls {
   my $self = shift;
   do_log('debug2', 'List::archive_ls');

   Archive::list("$self->{'name'}/archives") if ($self->is_archived());
}

## Archive 
sub archive_msg {
   my($self, $msg) = @_;
   do_log('debug2', 'List::archive_msg');

   Archive::store("$self->{'name'}/archives",$self->is_archived() , $msg) if ($self->is_archived());

   Archive::outgoing("$Conf{'queueoutgoing'}","$self->{'name'}\@$self->{'admin'}{'host'}",$msg) if ($self->is_web_archived());
}

sub archive_msg_digest {
   my($self, $msg) = @_;
   do_log('debug2', 'List::archive_msg_digest');

   $self->store_digest( $msg) if ($self->{'name'});
}

## Is the list archived ?
sub is_archived {
   return shift->{'admin'}{'archive'}{'period'};
}

## Is the list web archived ?
sub is_web_archived {

    return shift->{'admin'}{'web_archive'}{'access'} ;
   
}

## Returns statistics about a given list
sub get_stats {
   my @st = @{shift->{'stats'}};
   my $type = shift;
   do_log('debug2', 'List::get_stats(%s)', $type);

   if ($type eq 'text') {
      return sprintf(Msg(8, 8, $msg::stats_detail), @st[0..1], $st[2] / 1024 / 1024, $st[3] / 1024 / 1024);
   }
   return @st;
}

## Returns 1 if the  digest  must be send 
sub get_nextdigest {
    my $self = shift;
    do_log('debug2', 'List::get_nextdigest');

    my $digest = $self->{'admin'}{'digest'};
    my $listname = $self->{'name'};
    my ($hh, $mm, @days); 

    unless (-e "$Conf{'queuedigest'}/$listname") {
	return undef;
    }

    unless ($digest) {
	return undef;
    }
    
    @days = @{$digest->{'days'}};
    ($hh, $mm) = ($digest->{'hour'}, $digest->{'minute'});
       
    # get the current date
    my @now  = localtime(time);
    # compare dates
    my $i = $now[6]; # current day
    my @timedigest = localtime( (stat "$Conf{'queuedigest'}/$listname")[9]);
    my $indice;
    my $ind;

    foreach $ind (@days){
	$indice=$ind;
	last if($ind== $i);
    }
    if (($indice == $i) and 
	($now[2] * 60 + $now[1]) >= ($hh * 60 + $mm) and 
	(timelocal(0, $mm, $hh, $now[3], $now[4], $now[5]) > timelocal(0, $timedigest[1], $timedigest[2], $timedigest[3], $timedigest[4], $timedigest[5]))){
	
	# la date >  we can send the digest  
	
	return 1;
    }

    return undef;
}

## Returns the administrative informations about a list.
sub print_info {
    my $self = shift;
    local *INF = shift || select;
    do_log('debug2', 'List::print_info');

    my $admin = $self->{'admin'};

    my $i;
    my $lang = $admin->{'lang'};
    my @result;

    push @result, sprintf Msg(9, 12, "Subject            : %s\n"), $admin->{'subject'};
    
    foreach $i (@{$admin->{'owner'}}) {
	push @result, sprintf Msg(9, 1, "Owner              : %s\n"), $i->{'email'};
    }
    foreach $i (@{$admin->{'editor'}}) {
	push @result, sprintf Msg(9, 2, "Editor             : %s\n"), $i->{'email'};
    }
    push @result, sprintf Msg(9, 3, "Subscription       : %s\n")
	, $admin->{'subscribe'}{'title'}{$lang};
    
    push @result, sprintf Msg(9, 4, "Unsubscribe       : %s\n")
	, $admin->{'unsubscribe'}{'title'}{$lang};
    
    push @result, sprintf Msg(9, 5, "Send is            : %s\n"), $admin->{'send'}{'title'}{$lang};
    
    push @result, sprintf Msg(9, 6, "Review is          : %s\n")
	, $admin->{'review'}{'title'}{$lang};
    
    push @result, sprintf Msg(9, 7, "Reply-to           : %s\n"), $admin->{'reply_to'} 
    if ($admin->{'reply_to'});
    
    push @result, sprintf Msg(9, 8, "Maximum size       : %d\n"), $admin->{'max_size'} 
    if ($admin->{'max_size'});
    
    foreach $i (@{$admin->{'custom_header'}}) {
	push @result, sprintf Msg(9, 9, "URL of the liste       : %s\n"), $i 
	    if ($admin->{'custom_header'});
    }
    
    if ($admin->{'digest'}) {
	my $digest = sprintf '%s %d:%d', join(',',@{$admin->{'digest'}{'days'}}), $admin->{'digest'}{'hour'}, $admin->{'digest'}{'minute'};
	push @result, sprintf Msg(9, 10, "DIGEST       : %s\n"), $digest;
    }
    
    push @result, sprintf Msg(9, 11, "custom_subject    : %s\n"), $admin->{'custom_subject'} 
    if ($admin->{'custom_subject'});
    
    push @result, "\n";
    
    if (open FILE, "$self->{'name'}/info") {
	while (<FILE>) {
	    push @result, $_;
	}
    }
    return @result;
}

## load a scenario if not inline (in the list configuration file)
sub _load_scenario_file {
    my ($function, $name)= @_;
    do_log('debug2', 'List::_load_scenario_file(%s, %s)', $function, $name);

    my $scenario_file = "$Conf{'etc'}/scenari/$function.$name";
    my $structure;

    unless (open SCENARI, $scenario_file) {
        unless (open SCENARI,"--DIR--/bin/scenari/$function.$name") {
	    do_log ('info',"Unable to open scenario $scenario_file nor --DIR--/bin/scenari/$function.$name, please report to listmaster");
	    return &_load_scenario ($function,$name,'true() smtp -> reject');
	}
    }

    my $paragraph= join '',<SCENARI>;
    close SCENARI;
    unless ($structure = &_load_scenario ($function,$name,$paragraph)) { 
	do_log ('info',"error in $function scenario $scenario_file ");
    }

    return $structure ;
}

sub _load_scenario {
    my ($function, $scenario_name, $paragraph ) = @_;
    do_log('debug2', 'List::_load_scenario(%s,%s)', $function,$scenario_name);

    my $structure = {};
    $structure->{'name'} = $scenario_name ;
    my @scenario;
    my @old = ($*, $/);
    my @rules = split /\n/, $paragraph;

    ## Following lines are ordered
    
    ($*, $/) = (0,'\n');
    
    push(@scenario, 'scenario');
    foreach (@rules) {
	next if (/^\s*\w+\s*$/o); # skip paragraph name
	my $rule = {};
	s/\#.*$//;         # remove comments
        next if (/^\s*$/); # reject empty lines
	if (/^\s*title\.(\w+)\s+(.*)\s*$/i) {
	    $structure->{'title'}{$1} = $2;
	    next;
	}
        
        if (/^\s*include\s*(.*)\s*$/i) {
        ## introducing in few common rules using include
	    my $include = &_load_scenario_file ('include',$1);
            push(@scenario,@{$include->{'rules'}});
	    next;
	}

	unless (/^\s*(.*)\s+(md5|pgp|smtp|smime)\s*->\s*(.*)\s*$/i) {
	    do_log ('notice', "error rule syntaxe in scenario $function rule line $. expected : <condition> <auth_mod> -> <action>");
	    do_log ('debug',"error parsing $rule");
	    return undef;
	}
	$rule->{condition}=$1;
	$rule->{auth_method}=$2 || 'smtp';
	$rule->{action}=$3 ;

	push(@scenario,$rule);
	
    }
    
    ## Restore paragraph mode
    ($*, $/) = @old;
    $structure->{'rules'} = \@scenario;
    return $structure; 
}

## Loads all administrative and configuration data for a list.
sub _load_admin_file {
    my $file = shift;
    do_log('debug2', 'List::_load_admin_file(%s)', $file);
    
    open(L, $file) || return undef;
    my @old = ($*, $/);
    $* = 1; $/ = '';
    
    ## Create the initial admin array.
    my $admin = {};
    my $content;


    ## Process the file in paragraph mode.
    while (<L>) {

	# next if /^\s*\#/;
        # remove comments and then recognize empty line as correct
	s/\#.*\n//g;
	s/^\s*$//g;
	next if /^$/ ;

	if (/^\s*(owner|editor)\s*$/o) {
	    my $function = $1;
            my $value ;
	    my $user = {};
	    $user->{'email'} = $1 if (/^\s*email\s+(.+)\s*$/o);
	    $user->{'email'} =~ s/\s*$// ;
	    $user->{'gecos'} = $1 if (/^\s*gecos\s+(.+)\s*$/o);
	    $user->{'info'} = $1 if (/^\s*info\s+(.+)\s*$/o);
	    $user->{'reception'} = $1 if (/^\s*reception\s+(nomail)\s*$/o);
	    $user->{'profile'} = $1 if (/^\s*profile\s+(privileged)\s*$/o);
	    $user->{'auth'} = $1 if (/^\s*auth\s+(\S+)\s*$/o);
	    $user->{'password'} = $1 if (/^\s*password\s+(.+)\s*$/o);
	    $user->{'regexp'} = $1 if (/^\s*regexp\s+(.+)\s*$/o);
	    push(@{$admin->{$function}}, $user);
	    next;
	}

	## Lists defined with include_sql_query, load SQL parameters
	if (/^\s*include_sql_query\s*$/o) {
	    my $sql_args = {};
            $sql_args->{'type'} = 'sql_query';
	    $sql_args->{'db_type'} = $1 if (/^\s*db_type\s+(.+)\s*$/o);
	    $sql_args->{'db_name'} = $1 if (/^\s*db_name\s+(.+)\s*$/o);
	    $sql_args->{'host'} = $1 if (/^\s*host\s+(.+)\s*$/o);
	    $sql_args->{'user'} = $1 if (/^\s*user\s+(.+)\s*$/o);
	    $sql_args->{'passwd'} = $1 if (/^\s*passwd\s+(.+)\s*$/o);
	    $sql_args->{'sql_query'} = $1 if (/^\s*sql_query\s+(.+)\s*$/o);
	    $sql_args->{'f_dir'} = $1 if (/^\s*f_dir\s+(.+)\s*$/o);
	    push @{$admin->{'include'}}, $sql_args;
	    next;
	}	

	## Lists defined with include_ldap_query, load LDAP parameters
	if (/^\s*include_ldap_query\s*$/o) {
	    my $ldap_args = {};
            $ldap_args->{'type'} = 'ldap_query';
	    $ldap_args->{'host'} = $1 if (/^\s*host\s+(.+)\s*$/o);
	    $ldap_args->{'port'} = $1 if (/^\s*port\s+(.+)\s*$/o);
	    $ldap_args->{'user'} = $1 if (/^\s*user\s+(.+)\s*$/o);
	    $ldap_args->{'passwd'} = $1 if (/^\s*passwd\s+(.+)\s*$/o);
	    $ldap_args->{'suffix'} = $1 if (/^\s*suffix\s+(.+)\s*$/o);
	    $ldap_args->{'filter'} = $1 if (/^\s*filter\s+(.+)\s*$/o);
	    push @{$admin->{'include'}}, $ldap_args;
	    next;
	}

	if (/^\s*(archive)\s*$/o) {
	    $admin->{'archive'}{'period'} = $1 if (/^\s*period\s+(day|week|month|quarter|year)\s*$/o);
	    $admin->{'archive'}{'access'} = $1 if (/^\s*access\s+(public|private|owner|closed)\s*$/o);
	    next;
	}

	if (/^\s*(web_archive)\s*$/o) {
	    $admin->{'web_archive'}{'access'} = $1 if (/^\s*access\s+(public|private|owner|closed|listmaster)\s*$/o);
	    next;
	}

	if (/^\s*(default_user_options)\s*$/o) {
	    my $function = $1;
	    $admin->{$function}{'reception'} = $1 if (/^\s*reception\s+(digest|nomail)\s*$/o);
	    $admin->{$function}{'visibility'} = $1 if (/^\s*visibility\s+(conceal)\s*$/o);
	    next;
	}
	
	## Lists defined with include_sql_query, load SQL parameters
	if (/^\s*bounce\s*$/o) {
	    $admin->{'bounce'}{'halt_rate'} = $1 if (/^\s*halt_rate\s+(\d+)\s*$/o);
	    $admin->{'bounce'}{'warn_rate'} = $1 if (/^\s*warn_rate\s+(\d+)\s*$/o);
	    next;
	}	

	if (/^\s*(subscribe|subscription)\s*$/o) {
	    
	    unless ($admin->{'subscribe'} = &_load_scenario ('subscribe','inline',$_)) { 
		do_log ('info',"error in subscribe scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

        if (/^\s*(subscribe|subscription)\s+(\S+)\s*$/o) {
            my $mode = $2;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'subscribe'} = &_load_scenario_file ('subscribe',$mode)) { 
		do_log ('info',"error in subscribe scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}
	if (/^\s*add\s*$/o) {
	    
	    unless ($admin->{'add'} = &_load_scenario ('add','inline',$_)) { 
		do_log ('info',"error in add scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

        if (/^\s*add\s+(\S+)\s*$/o) {
            my $mode = $1;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'add'} = &_load_scenario_file ('add',$mode)) { 
		do_log ('info',"error in add scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}
	if (/^\s*invite\s*$/o) {
	    
	    unless ($admin->{'invite'} = &_load_scenario ('invite','inline',$_)) { 
		do_log ('info',"error in invite scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

        if (/^\s*invite\s+(\S+)\s*$/o) {
            my $mode = $1;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'invite'} = &_load_scenario_file ('invite',$mode)) { 
		do_log ('info',"error in invite scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}
	if (/^\s*del\s*$/o) {
	    
	    unless ($admin->{'del'} = &_load_scenario ('del','inline',$_)) { 
		do_log ('info',"error in del scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

        if (/^\s*del\s+(\S+)\s*$/o) {
            my $mode = $1;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'del'} = &_load_scenario_file ('del',$mode)) { 
		do_log ('info',"error in del scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

	if (/^\s*info\s*$/o) {
	    
	    unless ($admin->{'info'} = &_load_scenario ('info','inline',$_)) { 
		do_log ('info',"error in info scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

        if (/^\s*info\s+(\S+)\s*$/o) {
            my $mode = $1;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'info'} = &_load_scenario_file ('info',$mode)) { 
		do_log ('info',"error in info scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

	if (/^\s*(unsubscribe|unsubscribtion|sig)\s*$/o) {
	    
	    unless ($admin->{'unsubscribe'} = &_load_scenario ('unsubscribe','inline',$_)) { 
		do_log ('info',"error in unsubscribe scenario in %s configuration file ",$admin->{'name'});
		return undef;
	    }
	    next;
	}
	
	if (/^\s*(unsubscribe|unsubscribtion|sig)\s+(\S+)\s*$/o) {
	    my $mode = $2;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'unsubscribe'} = &_load_scenario_file ('unsubscribe',$mode)) { 
		do_log ('info',"error in subscribe scenario in %s configuration file ",$admin->{'name'});
		return undef;
	    }
	    next;
	}

	if (/^\s*review\s*$/o) {
	    
	    unless ($admin->{'review'} = &_load_scenario ('review','inline',$_)) { 
		do_log ('info',"error in review scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

        if (/^\s*review\s+(\S+)\s*$/o) {
            my $mode = $1;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'review'} = &_load_scenario_file ('review',$mode)) { 
		do_log ('info',"error in review scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

	if (/^\s*visibility\s*$/o) {
	    
	    unless ($admin->{'visibility'} = &_load_scenario ('visibility','inline',$_)) { 
		do_log ('info',"error in visibility scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

        if (/^\s*visibility\s+(\S+)\s*$/o) {
            my $mode = $1;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'visibility'} = &_load_scenario_file ('visibility',$mode)) { 
		do_log ('info',"error in visibility scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}

	if (/^\s*cookie\s+(.+)\s*$/o) {
	    $admin->{'cookie'} = $1;
	    next;
	}


	if (/^\s*(send)\s*$/o) {
	    
	    unless ($admin->{'send'} = &_load_scenario ('send','inline',$_)) { 
		do_log ('info',"error in send scenario in %s configuration file ",$admin->{'name'});
		return undef;
	    }
	    next;
	}
	
	if (/^\s*(send)\s+(\S+)\s*$/o) {
	    my $mode = $2;
	    $mode =~ s/\,/\_/g;
	    unless ($admin->{'send'} = &_load_scenario_file ('send',$mode)) { 
		do_log ('info',"error in send scenario in %s configuration file ",$admin->{'name'});
		return undef;
	    }
	    next;
	}

	if (/^\s*(reply-to|reply_to|replyto)\s+(sender|list|.*)\s*$/o) {
	    $admin->{'reply_to'} = $2;
	    next;
	}

	## Compatibility with version <= 2.2.5 
	if (/^\s*archive\s*(day|week|month|quarter|year)\s*$/o) {
	    $admin->{'archive'}{'period'} = $1;
	    next;
	}

	if (/^\s*max_size\s+(\d+)\s*$/o) {
	    $admin->{'max_size'} = $1;
	    next;
	}

	if (/^\s*subject\s+(.*)\s*$/o) {
	    $admin->{'subject'} = $1;
	    next;
	}

	if (/^\s*anonymous_sender\s+(.+)\s*$/o) {
	    $admin->{'anonymous_sender'} = $1;
	    next;
	}

	if (/^\s*host\s*(\S+)\s*$/o) {
	    $admin->{'host'} = $1 ;
	    next;
	}

	if (/^\s*(custom-header|custom_header)\s+(.+)$/o) {
	    push(@{$admin->{'custom_header'}},$2);
	    next;
	}

	if (/^\s*digest\s+(.+)\s+(\d+):(\d+)\s*$/o) {
	    $admin->{'digest'}{'hour'} = $2;
	    $admin->{'digest'}{'minute'} = $3;
	    my $days = $1;
	    $days =~ s/\s//go;
	    @{$admin->{'digest'}{'days'}} = split /,/, $days;
	    next;
	}

	if (/^\s*clean_delay_queuemod\s+(\d+)$/o) {
	    $admin->{'clean_delay_queuemod'} = $1;
	    next;
	}

	if (/^\s*custom_subject\s+(.*)\s*$/o) {
	    $admin->{'custom_subject'} = $1;
	    next;
	}
	
	if (/^\s*footer_type\s+(.*)\s*$/o) {
	    $admin->{'footer_type'} = $1;
	    next;
	}

	## Structure of subscriber data (default file)
	if (/^\s*user_data_source\s+(file|database|include)\s*$/o) {
	    $admin->{'user_data_source'} = $1;
	    next;
	}

	## Time To Leave for include data source (seconds)
	if (/^\s*ttl\s+(\d+)\s*$/o) {
	    $admin->{'ttl'} = $1;
	    next;
	}

	if (/^\s*remind\s*$/o) {

	    unless ($admin->{'remind'} = &_load_scenario ('remind','inline',$_)) { 
		do_log ('info',"error in remind scenario in %s configuration file ",$admin->{'name'});
                return undef;
	    }
	    next;
	}
        if (/^\s*remind\s+(\w+)\s*$/o) {
	    $admin->{'remind'} = &_load_scenario_file ('remind',$1);
	    next;
	}

        ## Include_list just need listname
	if (/^\s*include_list\s+([\w\-\.]+)\s*$/o) {
            my $list_args = {};
            $list_args->{'type'} = 'list';
            $list_args->{'list'} = $1;
            push  @{$admin->{'include'}}, $list_args;
	    next;
        }
        ## Include_file just need the path
	if (/^\s*include_file\s+(\S+)\s*$/o) {
            my $list_args = {};
            $list_args->{'type'} = 'file';
            $list_args->{'file'} = $1;
            push  @{$admin->{'include'}}, $list_args;
	    next;
        }

	## List priority
	if (/^\s*priority\s+(\w)\s*$/o) {
	    $admin->{'priority'} = $1;
	    next;
	}

	## Account
	if (/^\s*account\s+(\S+)\s*$/o) {
	    $admin->{'account'} = $1;
	    next;
	}

	## Topics
	if (/^\s*topics\s+(.+)\s*$/o) {
	    my $topics = $1;
	    $topics =~ s/\s//g;
	    @{$admin->{'topics'}} = split /,/, $topics;
	    next;
	}
	## welcome message return path format
	if (/^\swelcome_return_path\s+(\w+)\s*$/o) {
	    if ($1 =~ /unique|owner/i) {
		$admin->{'welcome_return_path'} = $1 ;
	    }
	}
	## remind message return path format
	if (/^\sremind_return_path\s+(\w+)\s*$/o) {
	    if ($1 =~ /unique|owner/i) {
		$admin->{'remind_return_path'} = $1 ;
	    }
	}
	## creation 
	if (/^\screation/o) {
	    $admin->{'creation'}{'date'} = $1 if (/^\s*date\s+(\d+)\s*$/o);
	    $admin->{'creation'}{'by'} = $1 if (/^\s*by\s+(.+)\s*$/o);
	}
	## last modification
	if (/^\supdate/o) {
	    $admin->{'update'}{'date'} = $1 if (/^\s*date\s+(\d+)\s*$/o);
	    $admin->{'update'}{'by'} = $1 if (/^\s*by\s+(.+)\s*$/o);
	    $admin->{'update'}{'serial'} = $1 if (/^\s*serial\s+(\d+)\s*$/o);
	}
        

	## Language
	if (/^\s*lang\s+(\w+)\s*$/o) {
	    $admin->{'lang'} = $1;
	    next;
	}

	## Unknown list parameter
	do_log('info', 'Unknown list parameter in file %s : %s', $file, $_);

    }
    close(L);
    ($*, $/) = @old;
    
    ## serial number of the configuration file
    $admin->{'update'}{'serial'} ||= '0' ;

    ## set default value 
    $admin->{'ttl'} ||= '3600' ;
     
    ## Cookie
    $admin->{'cookie'} ||= $Conf{'cookie'};

    ## Bounce default
    $admin->{'bounce'}{'warn_rate'} ||= $Conf{'bounce_warn_rate'};
    $admin->{'bounce'}{'halt_rate'} ||= $Conf{'bounce_halt_rate'};

    $admin->{'host'} ||= $Conf{'host'};

    $admin->{'reply_to'} ||= 'sender';

    $admin->{'priority'} ||= $Conf{'default_list_priority'};

    ## Default for user_data_source is 'file'
    ## if not using a RDBMS
    if ($List::use_db) {
	$admin->{'user_data_source'} ||= 'database';
    }else {
	$admin->{'user_data_source'} ||= 'file';
    }
 
    $admin->{'footer_type'} ||= 'mime';

    $admin->{'welcome_return_path'} ||= $Conf{'welcome_return_path'} ;

    $admin->{'remind_return_path'} ||= $Conf{'remind_return_path'} ;

    $admin->{'lang'} ||= $Conf{'lang'};

    ## Default for archives/access is now owner (was is same as review)
    $admin->{'archive'}{'access'} ||= 'owner';

    
    ## Subscription and unsubscribe add and del are closed 
    ## if subscribers are extracted via external include method
    ## (current version external method are SQL or LDAP query
    if ($admin->{'user_data_source'} eq 'include') {
	$admin->{'subscribe'} = &_load_scenario_file ('subscribe','closed');
	$admin->{'add'} = &_load_scenario_file ('del','closed');
	$admin->{'invite'} = &_load_scenario_file ('invite','closed');
	$admin->{'unsubscribe'} = &_load_scenario_file ('add','closed');
	$admin->{'del'} = &_load_scenario_file ('unsubscribe','closed');
    }

    ## Default scenarii
    foreach my $param ('info','review','subscribe','unsubscribe','send','add','del','remind','visibility','invite') {
	unless ($admin->{$param}) {
	    $admin->{$param} = &_load_scenario_file ($param, 'default');
	}
    }

    ## Return the anonymous hash.
    $admin;
}


## Loads the statistics informations
sub _load_stats_file {
    my $file = shift;
    do_log('debug2', 'List::_load_stats_file(%s)', $file);

   ## Create the initial stats array.
   my $stats;
 
   if (open(L, $file)){     
       if (<L> =~ /^(\d+)\s+(\d+)\s+(\d+)\s+(\d+)/) {
	   $stats = [ $1, $2, $3, $4 ];
       } else {
	   $stats = [ 0, 0, 0, 0 ];
       }
       close(L);
   } else {
       $stats = [ 0, 0, 0, 0 ];
   }

   ## Return the array.
   $stats;
}


## Loads the list of subscribers.
sub _load_users_file {
    my $file = shift;
    do_log('debug2', 'List::_load_users_file(%s)', $file);
    
    ## Open the file and switch to paragraph mode.
    open(L, $file) || return undef;
    my @old = ($*, $/);
    $* = 1; $/ = '';
    
    ## Create the in memory btree using DB_File.
    my %users;
    my $btree = new DB_File::BTREEINFO;
    return undef unless ($btree);
    $btree->{'compare'} = '_compare_addresses';
    my $ref = tie %users, 'DB_File', undef, O_CREAT|O_RDWR, 0600, $btree;
    return undef unless ($ref);
    
    ## Counters.
    my $total = 0;
    
    ## Process the lines
    while (<L>) {
	my(%user, $email);
	$user{'email'} = $email = $1 if (/^\s*email\s+(.+)\s*$/o);
	$user{'gecos'} = $1 if (/^\s*gecos\s+(.+)\s*$/o);
	$user{'options'} = $1 if (/^\s*options\s+(.+)\s*$/o);
	$user{'auth'} = $1 if (/^\s*auth\s+(\S+)\s*$/o);
	$user{'password'} = $1 if (/^\s*password\s+(.+)\s*$/o);
	$user{'stats'} = "$1 $2 $3" if (/^\s*stats\s+(\d+)\s+(\d+)\s+(\d+)\s*$/o);
	$user{'firstbounce'} = $1 if (/^\s*firstbounce\s+(\d+)\s*$/o);
	$user{'date'} = $1 if (/^\s*date\s+(\d+)\s*$/o);
	$user{'reception'} = $1 if (/^\s*reception\s+(digest|nomail)\s*$/o);
	$user{'visibility'} = $1 if (/^\s*visibility\s+(conceal|noconceal)\s*$/o);
	unless ($users{$email}) {
	    $total++;
	    $users{$email} = join("\n", %user);
	}
    }
    close(L);
    
    ($*, $/) = @old;
    
    my $l = {
	'ref'	=>	$ref,
	'users'	=>	\%users,
	'total'	=>	$total
	};
    
    $l;
}



## include a list as subscribers.
sub _include_users_list {
    my ($users, $param) = @_;
    do_log('debug2', 'List::_include_users_list');

    my $includelistname = $param->{'list'};
    my $total = 0;
    
    my $includelist = new List ($includelistname);
    unless ($includelist) {
	do_log('info', 'Included list %s unknown' , $includelistname);
	return undef;
    }
    
    for (my $user = $includelist->get_first_user(); $user; $user = $includelist->get_next_user()) {
	my %u;
	my $email =  $u{'email'} = $user->{'email'};
	$u{'gecos'} = $user->{'gecos'};
	unless ($users->{$email}) {
	    $total++;
	    $users->{$email} = join("\n", %u);
	}
    }
    do_log ('info',"Include %d subscribers from list %s",$total,$includelistname);
    return $total ;
}

sub _include_users_file {
    my ($users, $param) = @_;
    do_log('debug2', 'List::_include_users_file');

    my $filename = $param->{'file'};
    my $total = 0;
    
    unless (open(INCLUDE, "$filename")) {
	do_log('info', 'enable to open file "%s"' , $filename);
	return undef;
    }
    do_log('debug','including file %s' , $filename) if ($main::opt_d);
    
    while (<INCLUDE>) {
	$_ =~ /^\s*([^\#]+)\s*\#?/;
	my $email =$1;
	$email  =~ /(.*)\s*$/;
	$email =$1;
	my %u;
	$u{'email'} = $email;
	
	if ($email) {
	    $total++;
	    $users->{$email} = join("\n", %u);
	}
    }
    close INCLUDE ;
    
    do_log ('info',"include %d subscribers from file %s",$total,$filename);
    return $total ;
}


## Returns a list of subscribers extracted from a remote LDAP Directory
sub _include_users_ldap {
    my ($users, $param) = @_;
    do_log('debug2', 'List::_include_users_ldap');
    
    unless (require Net::LDAP) {
	do_log ('debug',"enable to use LDAP library, install perl-ldap (CPAN) first");
	return undef;
    }

    my $host = $param->{'host'};
    my $port = $param->{'port'} || '389';
    my $user = $param->{'user'};
    my $passwd = $param->{'passwd'};
    my $ldap_suffix = $param->{'suffix'};
    my $ldap_filter = $param->{'filter'};
    
#    my $default_reception = $admin->{'default_user_options'}{'reception'};
#    my $default_visibility = $admin->{'default_user_options'}{'visibility'};

    ## LDAP and query handler
    my ($ldaph, $fetch);
 
    unless ($ldaph = Net::LDAP->new($host, port => "$port")) {
	do_log ('notice',"Can\'t connect to LDAP server '$host' '$port' : $@");
	return undef;
    }
    
    do_log('debug', "Connected to LDAP server $host:$port") if ($main::opt_d);
    
    unless ($ldaph->bind ("$user", password => "$passwd")) {
	do_log ('notice',"Can\'t bind with server $host:$port as user '$user' : $@");
	return undef;
    }

    do_log('debug', "Binded to LDAP server $host:$port ; user : '$user'") if ($main::opt_d);
    
    do_log('debug', 'Searching in base %s ; filter: %s', $ldap_suffix, $ldap_filter) if ($main::opt_d);
    unless ($fetch = $ldaph->search ( base => "$ldap_suffix",
                                      filter => "$ldap_filter" )) {
        do_log('debug',"Unable to perform LDAP search in $ldap_suffix for $ldap_filter : $@");
        return undef;
    }
    
    ## Counters.
    my $total = 0;
    my $dn; 
   
    ## returns a reference to a HASH where the keys are the DNs
    ##  the second level hash's hold the attributes
    my $all_entries = $fetch->as_struct ;

    foreach $dn (keys %$all_entries) { 
	my $entry = $all_entries->{$dn};
#	my $emails = $entry->{'mail'};
#	foreach my $email (@$emails){
	
	my %u;
	$u{'email'} = $entry->{'mail'}[0];
	$u{'gecos'} = $entry->{'cn'}[0];
	$u{'date'} = time;
	## should consult user default options
	unless ($users->{$u{'email'}}) {
	    $total++;
	    $users->{$u{'email'}} = join("\n", %u);
	}
#    }
    }
    
    unless ($ldaph->unbind) {
	do_log('notice','Can\'t unbind from  LDAP server %s:%s',$host,$port);
	return undef;
    }
    do_log ('debug',"unbinded from LDAP server %s:%s ",$host,$port) if ($main::opt_d);
    do_log ('debug','%d subscribers included from LDAP query',$total);

    return $total;
}

## Returns a list of subscribers extracted from an remote Database
sub _include_users_sql {
    my ($users, $param) = @_;


    unless ( require DBI ){
	do_log('notice',"Intall module DBI (CPAN) before using include_sql_query");
	return undef ;
    }

    my $db_type = $param->{'db_type'};
    my $db_name = $param->{'db_name'};
    my $host = $param->{'host'};
    my $user = $param->{'user'};
    my $passwd = $param->{'passwd'};
    my $sql_query = $param->{'sql_query'};

    ## For CSV (Comma Separated Values) 
    my $f_dir = $param->{'f_dir'}; 

#    my $default_reception = $admin->{'default_user_options'}{'reception'};
#    my $default_visibility = $admin->{'default_user_options'}{'visibility'};
    
    my ($dbh, $sth);
    my $connect_string;

    if ($f_dir) {
	$connect_string = "DBI:CSV:f_dir=$f_dir";
    }elsif ($db_type eq 'Oracle') {
	$connect_string = "DBI:$db_type:host=$host;sid=$db_name";
    }elsif ($db_type eq 'Pg') {
	$connect_string = "DBI:$db_type:dbname=$db_name\@$host";
    }else {
	$connect_string = "DBI:$db_type:$db_name:$host";
    }

    unless ($dbh = DBI->connect($connect_string, $user, $passwd)) {
	do_log ('notice','Can\'t connect to Database %s',$db_name);
	return undef;
    }
    do_log('debug','Connected to Database %s',$db_name);
    
    unless ($sth = $dbh->prepare($sql_query)) {
        do_log('debug','Unable to prepare SQL query : %s', $dbh->errstr);
        return undef;
    }
    unless ($sth->execute) {
        do_log('debug','Unable to perform SQL query %s : %s ',$sql_query, $dbh->errstr);
        return undef;
    }
    
    ## Counters.
    my $total = 0;
    
    ## Process the SQL results
    while (my $email = $sth->fetchrow) {
	my %u;

	$u{'email'} = $email;

	$u{'date'} = time;
	## should be set to list user default options
#	$u{'reception'} = $default_reception;
#	$u{'visibility'} = $default_visibility;
	unless ($users->{$email}) {
	    $total++;
	    $users->{$email} = join("\n", %u);
	}
    }
    $sth->finish ;
    $dbh->disconnect();

    do_log ('debug','%d included subscribers from SQL query', $total);
    return $total;
}

## Loads the list of subscribers from an external include source
sub _load_users_include {
    my $admin = shift ;
    do_log('debug2', 'List::_load_users_include');

    my (%users, $total);

    ## Create in memory btree using DB_File.
    my $btree = new DB_File::BTREEINFO;
    return undef unless ($btree);
    $btree->{'compare'} = '_compare_addresses';
    my $ref = tie %users, 'DB_File', undef, O_CREAT|O_RDWR, 0600, $btree;
    return undef unless ($ref);
    foreach my $incl (@{$admin->{'include'}}) {
	## get the list of users
	if ($incl->{'type'} eq 'sql_query') {
	    $total += _include_users_sql(\%users, $incl);
	}elsif ($incl->{'type'} eq 'ldap_query') {
	    $total += _include_users_ldap(\%users, $incl);
	}elsif ($incl->{'type'} eq 'list') {
	    $total += _include_users_list (\%users, $incl);
	}elsif ($incl->{'type'} eq 'file') {
	    $total += _include_users_file (\%users, $incl);
	}
    }

    my $l = {
	'ref'	=>	$ref,
	'users'	=>	\%users,
	'total'	=>	$total
	};

    $l;
}

sub _load_total_db {
    my $name = shift;
    do_log('debug2', 'List::_load_total_db(%s)', $name);

    unless ($List::use_db) {
	&do_log('info', 'Sympa not setup to use DBI');
	return undef;
    }
    
    my ($statement);

    ## Check database connection
    unless ($dbh and $dbh->ping) {
	return undef unless &db_connect();
    }	   

    ## Query the Database
    $statement = sprintf "SELECT count(*) FROM subscriber_table WHERE list_subscriber = %s", $dbh->quote($name);
       
    push @sth_stack, $sth;

    unless ($sth = $dbh->prepare($statement)) {
	do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
	return undef;
    }
    
    unless ($sth->execute) {
	do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	return undef;
    }
    
    my $total = $sth->fetchrow;

    $sth->finish();

    $sth = pop @sth_stack;

    return $total;
}

## Writes to disk the stats data for a list.
sub _save_stats_file {
    my $file = shift;
    my $stats = shift;
    my $total = shift;
    do_log('debug2', 'List::_save_stats_file(%s, %d)', $file, $total);
    
    open(L, "> $file") || return undef;
    printf L "%d %d %d %d %d\n", @{$stats}, $total;
    close(L);
}

## Writes to disk the administrative informations for a list.
sub _save_admin_file {
    my($file, $admin) = @_;
    do_log('debug2', 'List::_save_admin_file(%s)', $file);

    my $param;

#    rename("$file", "$file.old");
    unless (open CONFIG, "> $file.new") {
	&do_log('info', 'Cannot create %s', $file);
	return undef;
    }

    ## Single line param
    foreach $param ('subject','max_size','custom_subject',
		    'footer_type','host','user_data_source','priority',
		    'reply_to','account') {
	if ($admin->{$param}) {
	    printf CONFIG "%s %s\n\n", $param, $admin->{$param};
	}
    }

    ## Owners & Editors
    foreach my $role ('owner','editor') {
	foreach my $entry (@{$admin->{$role}}) {
	    printf CONFIG "%s\n", $role;
	    foreach my $attr (keys %{$entry}) {
		printf CONFIG "%s %s\n", $attr, $entry->{$attr}; 
	    }
	    print CONFIG "\n";
	}
    }

    ## Scenarized params
    foreach $param  ('visibility','send','subscribe','unsubscribe','review','remind','add','del') {
	if ($admin->{$param}{'name'} eq 'inline') {
	    printf CONFIG "%s\n", $param;
	    foreach my $rule (@{$admin->{$param}{'rules'}}) {
		print CONFIG $rule;
	    }
	}else {
	    next if ($admin->{$param}{'name'} eq 'default');
	    ## Scenario file
	    printf CONFIG "%s %s\n", $param, $admin->{$param}{'name'};
	}
	print CONFIG "\n";
    }

    ## Digest
    if ($admin->{'digest'}) {
	printf CONFIG "digest %s %d:%d\n\n", join (',', @{$admin->{'digest'}{'days'}})
	    , $admin->{'digest'}{'hour'}, $admin->{'digest'}{'minute'};
    }

    ## Archives
    foreach $param ('archive','web_archive') {
	next unless $admin->{$param};

	printf CONFIG "%s\n", $param;
	foreach my $key (keys %{$admin->{$param}}) {
	    printf CONFIG "%s %s\n", $key, $admin->{$param}{$key};
	}
	print CONFIG "\n";
    }

    ## Default user options
    if ($admin->{'default_user_options'}) {
	print CONFIG "default_user_options\n";
	foreach my $key (keys %{$admin->{'default_user_options'}}) {
	    printf CONFIG "%s %s\n", $key, $admin->{'default_user_options'}{$key};
	}
	print CONFIG "\n";
    }

    ## Custom headers
    if ($admin->{'custom_header'}) {
	foreach my $header (@{$admin->{'custom_header'}}) {
	    printf CONFIG "custom_header %s\n\n", $header;
	}
    }

    ## Topics
    if ($admin->{'topics'}) {
	printf CONFIG "topics %s\n\n", join (',', @{$admin->{'topics'}});
    }

    close CONFIG;
    return 1;
}

## Writes the user list to disk
sub _save_users_file {
    my($self, $file) = @_;
    do_log('debug2', 'List::_save_users_file(%s)', $file);
    
    my($k, $s);
    
    do_log('debug','Saving user file %s', $file) if ($main::opt_d);
    
    rename("$file", "$file.old");
    open SUB, "> $file" or return undef;
    
    for ($s = $self->get_first_user(); $s; $s = $self->get_next_user()) {
	foreach $k ('date','email','gecos','reception','visibility') {
	    printf SUB "%s %s\n", $k, $s->{$k} unless ($s->{$k} eq '');
	    
	}
	print SUB "\n";
    }
    close SUB;
    return 1;
}

sub _compare_addresses {
   my ($a, $b) = @_;

   my ($ra, $rb);

   $a =~ tr/A-Z/a-z/;
   $b =~ tr/A-Z/a-z/;

   $ra = reverse $a;
   $rb = reverse $b;

   return ($ra cmp $rb);
}

sub _compare_addresses_old {
   my ($a, $b) = @_;
   my ($pa,$pb); 
   $a =~ tr/A-Z/a-z/;
   $b =~ tr/A-Z/a-z/;
   $a =~ /\.(\w*)$/;
   my $ra = $1;
   $b =~ /\.(\w*)$/;
   my $rb = $1;
   ($Conf{'poids'}{$ra} and $pa=$Conf{'poids'}{$ra}) or  $pa=$Conf{'poids'}{'*'};
   ($Conf{'poids'}{$rb} and $pb=$Conf{'poids'}{$rb}) or  $pb=$Conf{'poids'}{'*'};

   $pa != $pb and return ($pa cmp $pb);

   $ra = join('.', reverse(split(/[@\.]/, $a)));
   $rb = join('.', reverse(split(/[@\.]/, $b)));

   return ($ra cmp $rb);
}

## Does the real job : stores the message given as an argument into
## the digest of the list.
sub store_digest {
    my($self,$msg) = @_;
    do_log('debug2', 'List::store_digest');

    my($filename, $newfile);
    my $separator = $msg::separator;  

    unless ( -d "$Conf{'queuedigest'}") {
	return;
    }
    
    my @now  = localtime(time);
    $filename = "$Conf{'queuedigest'}/$self->{'name'}";
    $newfile = !(-e $filename);
    my $oldtime=(stat $filename)[9] unless($newfile);
  
    open(OUT, ">> $filename") || return;
    if ($newfile) {
	## create header
	printf OUT "\nThis digest for list has been created on %s\n\n",
      POSIX::strftime("%a %b %e %H:%M:%S %Y", @now);
	print OUT "------- THIS IS A RFC934 COMPLIANT DIGEST, YOU CAN BURST IT -------\n\n";
	print OUT "\n$separator\n\n";

       # send the date of the next digest to the users
    }
    $msg->head->delete('Received');
    $msg->print(\*OUT);
    print OUT "\n$separator\n\n";
    close(OUT);
    
    #replace the old time
    utime $oldtime,$oldtime,$filename   unless($newfile);
}

## List of lists hosted by Sympa
sub get_lists {
   my(@lists, $l);
   do_log('debug2', 'List::get_lists()');

   unless (-d $Conf{'home'}) {
       do_log('debug',"no such directory $Conf{'home'}");
       return undef ;
   }
   
   unless (opendir(DIR, $Conf{'home'})) {
       do_log('debug',"unable to open $Conf{'home'}");
       return undef;
   }
   foreach $l (sort readdir(DIR)) {
      next unless (($l !~ /^\./o) and (-d $l) and (-f "$l/config"));
      push @lists, $l ;
   }
   return @lists;
}

## List of lists in database mode which e-mail parameter is member of
sub get_which_db {
    my $email = shift;
    do_log('debug2', 'List::get_which_db(%s)', $email);

    unless ($List::use_db) {
	&do_log('info', 'Sympa not setup to use DBI');
	return undef;
    }
    
    my ($l, %which, $statement);

    ## Check database connection
    unless ($dbh and $dbh->ping) {
	return undef unless &db_connect();
    }	   

    $statement = sprintf "SELECT list_subscriber FROM subscriber_table WHERE user_subscriber = %s",$dbh->quote($email);

    push @sth_stack, $sth;

    unless ($sth = $dbh->prepare($statement)) {
	do_log('debug','Unable to prepare SQL statement : %s', $dbh->errstr);
	return undef;
    }

    unless ($sth->execute) {
	do_log('debug','Unable to execute SQL statement : %s', $dbh->errstr);
	return undef;
    }

    while ($l = $sth->fetchrow) {
	$l =~ s/\s*$//;  ## usefull for PostgreSQL
	$which{$l} = 1;
    }

    $sth->finish();

    $sth = pop @sth_stack;

    return \%which;
}

## List of lists where $1 (an email) is $2 (owner, editor or subscriber)
sub get_which {
    my $email = shift;
    my $function = shift;
    do_log('debug2', 'List::get_which(%s, %s)', $email, $function);

    my ($l, @which);

    ## WHICH in Database
    my $db_which = {};

    if (($function eq 'member') and (defined $Conf{'db_type'})) {
	if ($List::use_db) {
	    $db_which = &get_which_db($email);
	}
    }

    foreach $l (get_lists()){
 
	my $list = new List ($l);
	next unless ($list);
        if ($function eq 'member') {
	    if ($list->{'admin'}{'user_data_source'} eq 'database') {
		if ($db_which->{$l}) {
		    push @which, $l ;
		}
	    }else {
		push @which, $list->{'name'} if ($list->is_user($email));
	    }
	}elsif ($function eq 'owner') {
	    push @which, $list->{'name'} if ($list->am_i('owner',$email));
	}elsif ($function eq 'editor') {
	    push @which, $list->{'name'} if ($list->am_i('editor',$email));
	}else {
	    do_log('debug',"Internal error, unknown or undefined parameter $function  in get_which");
            return undef ;
	}
    }
    
    return @which;
}


## send auth request to $request 
sub request_auth {
    my ($email, $cmd, $listname, @param) = @_;
    do_log('debug2', 'List::request_auth(%s, %s, %s, %s)',$email, $cmd, $listname, join (',', @param));

    my $keyauth;
    my ($body, $command);

    if ($cmd eq 'signoff'){
	$keyauth = compute_auth ($email, $cmd, $listname);
	$command = "auth $keyauth $cmd $listname $email";
        my $url = "mailto:$Conf{'sympa'}?subject=$command";
        $url =~ s/\s/%20/g;
        $body = sprintf Msg(6, 261, $msg::signoff_need_auth ),
	        $listname, $Conf{'sympa'},$command, $url;
 
    }elsif ($cmd eq 'sub'){
        $keyauth = compute_auth ($email, $cmd, $listname);
	$command = "auth $keyauth $cmd $listname $param[0]";
        my $url = "mailto:$Conf{'sympa'}?subject=$command";
        $url =~ s/\s/%20/g;
        $body = sprintf Msg(6, 260, $msg::subscription_need_auth)
	    ,$listname,  $Conf{'sympa'}, $command, $url ;
    }elsif ($cmd eq 'add'){
        $keyauth = compute_auth ($param[0],$cmd,$listname);
	$command = "auth $keyauth $cmd $listname $param[0] $param[1]";
        $body = sprintf Msg(6, 39, $msg::adddel_need_auth),$listname
	    , $Conf{'sympa'}, $command;
    }elsif ($cmd eq 'del'){
        my $keyauth = compute_auth($param[0],$cmd,$listname);
	$command = "auth $keyauth $cmd $listname $param[0]";
        $body = sprintf Msg(6, 39, $msg::adddel_need_auth),$listname
	    , $Conf{'sympa'}, $command;
    }elsif ($cmd eq 'remind'){
        my $keyauth = compute_auth('',$cmd,$listname);
	$command = "auth $keyauth $cmd $listname";
        $body = sprintf Msg(6, 79, $msg::remind_need_auth),$listname
	    , $Conf{'sympa'}, $command;
    }

    mail::mailback (\$body, $command, $email);

    return 1;
}

## genererate a md5 checksum using private cookie and parameters
sub compute_auth {
    my ($email, $cmd, $listname) = @_;
    do_log('debug2', 'List::compute_auth(%s, %s, %s)', $email, $cmd, $listname);

    my ($cookie, $key) ;

    $email =~ y/[A-Z]/[a-z]/;
    $cmd =~ y/[A-Z]/[a-z]/;
    $listname =~ y/[A-Z]/[a-z]/;

    my $list = new List ($listname);
    if ($list){
        $cookie = $list->get_cookie() || "this should be set in sympa.conf";
    }else {
	$cookie = "this should be set in sympa.conf";
    }

    $key = substr(MD5->hexhash(join('/', $cookie, $listname, $email, $cmd)), -8) ;

    return $key;
}

## return total of messages awaiting moderation
sub get_mod_spool_size {
    my $self = shift;
    do_log('debug2', 'List::get_mod_spool_size()');    
    my @msg;
    
    unless (opendir SPOOL, $Conf{'queuemod'}) {
	&do_log('info', 'Unable to read spool %s', $Conf{'queuemod'});
	return undef;
    }

    @msg = sort grep(/^$self->{'name'}\_\w+$/, readdir SPOOL);

    return ($#msg + 1);
}

sub probe_db {
    do_log('debug2', 'List::probe_db()');    
    my (%checked, $table);

    ## Is the Database defined
    unless (defined($Conf{'db_name'})) {
	&do_log('info', 'No db_name defined in configuration file');
	return undef;
    }

    unless ($dbh and $dbh->ping) {
	return undef unless &db_connect();
    }
	
    my @tables;
    if ($Conf{'db_type'} eq 'mysql') {
	unless (@tables = $dbh->func( '_ListTables' )) {
	    &do_log('info', 'Can\'t load tables list from database %s : %s', $Conf{'db_name'}, $dbh->errstr);
	    return undef;
	}

    }elsif ($Conf{'db_type'} eq 'Pg') {
	
	unless (@tables = $dbh->tables) {
	    &do_log('info', 'Can\'t load tables list from database %s', $Conf{'db_name'});
	    return undef;
	}

    }elsif ($Conf{'db_type'} eq 'Oracle') {
 	
 	my $statement = "SELECT table_name FROM user_tables";	 

	push @sth_stack, $sth;

	unless ($sth = $dbh->prepare($statement)) {
	    do_log('debug','Unable to prepare SQL query : %s', $dbh->errstr);
	    return undef;
     	}

       	unless ($sth->execute) {
	    &do_log('debug','Can\'t load tables list from database and Unable to perform SQL query %s : %s ',$statement, $dbh->errstr);
	    return undef;
     	}
 
	## Process the SQL results
     	while (my $table= $sth->fetchrow()) {
	    push @tables, lc ($table);   	
	}
	
     	$sth->finish();

	$sth = pop @sth_stack;

    }elsif ($Conf{'db_type'} eq 'Sybase') {
  
	my $statement = "SELECT name FROM sympa..sysobjects WHERE type='U'";     
 
	push @sth_stack, $sth;
	unless ($sth = $dbh->prepare($statement)) {
	    do_log('debug','Unable to prepare SQL query : %s', $dbh->errstr);
	    return undef;
	}
	unless ($sth->execute) {
	    &do_log('debug','Can\'t load tables list from database and Unable to perform SQL query %s : %s ',$statement, $dbh->errstr);
	    return undef;
	}

	## Process the SQL results
	while (my $table= $sth->fetchrow()) {
	    push @tables, lc ($table);   
	}
	
	$sth->finish();
	$sth = pop @sth_stack;
    }
    
    foreach $table ( @tables ) {
	$checked{$table} = 1;
    }
    
    foreach $table('user_table', 'subscriber_table') {
	unless ($checked{$table}) {
	    &do_log('info', 'Table %s not found in database %s', $table, $Conf{'db_name'});
	    return undef;
	}
    }

    return 1;
}

## Packages must return true.
1;
