


/*
 *  Dr Geo an interactive geometry software
 * (C) Copyright Hilaire Fernandes  1997-1999
 * hilaire.fernandes@iname.com 
 * 
 *
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public Licences as by published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Publis License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef point_h
#define point_h

#include <string.h>
#include <libintl.h>
#include <math.h>

#include "classbase.h"
#include "complexe.h"
#include "mode_obj.h"
#include "graphic.h"
#include "traite.h"

// Class derived from the base class point_c
// Class point1, point defined by two numbers  (its coordinates)
class point1:public point_c
{
  public:
  valeur_c * s1, *s2;		// pointers to two class value

  point1 (void):point_c ()
  {
  }
  point1 (liste_elem & lp):point_c ()
  {
    classe = POINT_CO;
    s1 = (valeur_c *) lp.lire (1);
    s2 = (valeur_c *) lp.lire (2);
    actualise ();
    init_nom ();
  }
  void actualise (void);
  void move (int mx, int my)
  {
  };
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// Class point2, free point on the plane
class point2:public point_c
{
  public:
  point2 (void):point_c ()
  {
  }
  point2 (double x, double y):point_c ()
  {
    p.x = x;
    p.y = y;
    couleur = ROUGE;
    classe = POINT_PL;
    init_nom ();
  }
  point2 (double x, double y, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    p.x = x;
    p.y = y;
    classe = POINT_PL;
    init_nom ();
  }
  void actualise ();
};
// Class point_sur_xxx, free point on object xxx
class point_sur_droite:public point_c
{
  public:
  double xm;
  droite_c *parent;		// point on a line

    point_sur_droite (void):point_c ()
  {
    classe = POINT_DR;
  }
  point_sur_droite (double x, double y, liste_elem & lp):point_c ()
  {
    droite_s co;
    vecteur_s u, am;
    double pente;
    classe = POINT_DR;
    parent = (droite_c *) lp.lire (1);
    co = parent->coordonnees ();
    u = parent->directeur ();
    if (u.x == 0 && u.y == 0)
      {
	p.x = co.a.x;
	p.y = co.a.y;
	couleur = ROUGE;
	xm = 0;
	return;
      }
    pente = parent->pente ();
    if (fabs (pente) > 1)
      {
	p.y = y;
	if (pente != 1E100)
	  p.x = (y - co.a.y) / pente + co.a.x;
	else
	  p.x = co.a.x;
      }
    else
      {
	p.x = x;
	p.y = pente * (x - co.a.x) + co.a.y;
      }
    am.x = -co.a.x + p.x;
    am.y = -co.a.y + p.y;
    xm = sqrt (am.x * am.x + am.y * am.y);
    if (am.x * u.x + am.y * u.y < 0)
      xm = -xm;
    couleur = ROUGE;
    init_nom ();
  }
  point_sur_droite (double x, double y, liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    droite_s co;
    vecteur_s u, am;
    double pente;
    classe = POINT_DR;
    parent = (droite_c *) lp.lire (1);;
    co = parent->coordonnees ();
    u = parent->directeur ();
    if (u.x == 0 && u.y == 0)
      {
	p.x = co.a.x;
	p.y = co.a.y;
	couleur = ROUGE;
	xm = 0;
	return;
      }
    pente = parent->pente ();
    if (fabs (pente) > 1)
      {
	p.y = y;
	if (pente != 1E100)
	  p.x = (y - co.a.y) / pente + co.a.x;
	else
	  p.x = co.a.x;
      }
    else
      {
	p.x = x;
	p.y = pente * (x - co.a.x) + co.a.y;
      }
    am.x = -co.a.x + p.x;
    am.y = -co.a.y + p.y;
    xm = sqrt (am.x * am.x + am.y * am.y);
    if (am.x * u.x + am.y * u.y < 0)
      xm = -xm;
    init_nom ();
  }
  void move (int mx, int my);
  void actualise ();
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
class point_sur_demi_droite:public point_c
{
  public:
  double xm;			// point's abscissa on the halt-line

  droite_c *parent;		// point on  halt-line

    point_sur_demi_droite (void):point_c ()
  {
    classe = POINT_DD;
  }
  point_sur_demi_droite (double x, double y, liste_elem & lp):point_c ()
  {
    classe = POINT_DD;
    parent = (droite_c *) lp.lire (1);
    xm = 0;
    actualise ();
    move (UNIT_MONtoECR (x - p.x), -UNIT_MONtoECR (y - p.y));
    init_nom ();
  }
  point_sur_demi_droite (double x, double y, liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    classe = POINT_DD;
    parent = (droite_c *) lp.lire (1);
    xm = 0;
    actualise ();
    move (UNIT_MONtoECR (x - p.x), -UNIT_MONtoECR (y - p.y));
    init_nom ();
  }
  void move (int mx, int my);
  void actualise ();
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
class point_sur_lieu:public point_c
{
  public:
  int xm;			// position on the locus

  lieu_point_c *parent;
    point_sur_lieu (void):point_c ()
  {
    classe = POINT_LI;
  }
  point_sur_lieu (double x, double y, liste_elem & lp):point_c ()
  {
    int nb;
    double d, d1;
    point_s l, pl, *adr;
    classe = POINT_LI;
    parent = (lieu_point_c *) lp.lire (1);
    init_nom ();
    nb = parent->nb_objet_sur_lieu;
    adr = parent->liste_point;
    pl.x = x;
    pl.y = y;
    l = adr[--nb] - pl;
    xm = nb--;
    d = l * l;
    for (; nb >= 0; nb--)
      {
	l = adr[nb] - pl;
	d1 = l * l;
	if (d1 < d)
	  {
	    d = d1;
	    xm = nb;
	  }
      }
    p = adr[xm];
  }
  void move (int mx, int my);
  void actualise ();
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// Class point_milieu_segment, middle of a segment
class point_milieu_segment:public point_c
{
  public:
  droite_c * parent;
  point_milieu_segment ():point_c ()
  {
    classe = POINT_MI_SE;
  }
  point_milieu_segment (liste_elem & lp):point_c ()
  {
    parent = (droite_c *) lp.lire (1);
    classe = POINT_MI_SE;
    init_nom ();
    actualise ();
  }
  point_milieu_segment (liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    parent = (droite_c *) lp.lire (1);;
    init_nom ();
    classe = POINT_MI_SE;
    actualise ();
  }
  virtual void init_nom (void);
  void move (int mx, int my)
  {
  };
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// Class point_milieu_2points, middle of two points
class point_milieu_2points:public point_c
{
  public:
  point_c * pt1, *pt2;
  point_milieu_2points ():point_c ()
  {
    strcpy (nom_type, _ ("%1 the middle of %2"));
    classe = POINT_MI_2PT;
  }
  point_milieu_2points (liste_elem & lp):point_c ()
  {
    pt1 = (point_c *) lp.lire (1);;
    pt2 = (point_c *) lp.lire (2);;
    init_nom ();
    classe = POINT_MI_2PT;
    actualise ();
  }
  point_milieu_2points (liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    pt1 = (point_c *) lp.lire (1);;
    pt2 = (point_c *) lp.lire (2);;
    init_nom ();
    classe = POINT_MI_2PT;
    actualise ();
  }
  virtual void init_nom (void);
  void move (int mx, int my)
  {
  };
  void actualise ();
  void sauve_disk (FILE * f);
  liste_elem *parents (liste_elem * liste_parent);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// Class point_sur_segment, free point on a segment
class point_sur_segment:public point_c
{
  public:
  double xm;			// point's abscissa on the segment

  droite_c *parent;
    point_sur_segment ():point_c ()
  {
    classe = POINT_SE;
  }
  point_sur_segment (double x, double y, liste_elem & lp):point_c ()
  {
    droite_s co;
    vecteur_s u, am;
    double pente;
    classe = POINT_SE;
    parent = (droite_c *) lp.lire (1);;
    co = parent->coordonnees ();
    u.x = co.b.x - co.a.x;
    u.y = co.b.y - co.a.y;
    if (u.x == 0 && u.y == 0)
      {
	p.x = co.a.x;
	p.y = co.a.y;
	xm = 0;
	couleur = ROUGE;
	return;
      }
    pente = parent->pente ();
    if (fabs (pente) > 1)
      {
	p.y = y;
	if (pente != 1E100)
	  p.x = (y - co.a.y) / pente + co.a.x;
	else
	  p.x = co.a.x;
      }
    else
      {
	p.x = x;
	p.y = pente * (x - co.a.x) + co.a.y;
      }
    am.x = co.a.x - p.x;
    am.y = co.a.y - p.y;
    xm = sqrt (am.x * am.x + am.y * am.y) / sqrt (u.x * u.x + u.y * u.y);
    if (-am.x * u.x - am.y * u.y < 0)
      xm = 0;
    if (xm > 1)
      xm = 1;
    p.x = xm * u.x + co.a.x;
    p.y = xm * u.y + co.a.y;
    couleur = ROUGE;
    init_nom ();
  }
  point_sur_segment (double x, double y, liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    droite_s co;
    vecteur_s u, am;
    double pente;
    classe = POINT_SE;
    parent = (droite_c *) lp.lire (1);;
    co = parent->coordonnees ();
    u.x = co.b.x - co.a.x;
    u.y = co.b.y - co.a.y;
    if (u.x == 0 && u.y == 0)
      {
	p.x = co.a.x;
	p.y = co.a.y;
	xm = 0;
	return;
      }
    pente = parent->pente ();
    if (fabs (pente) > 1)
      {
	p.y = y;
	if (pente != 1E100)
	  p.x = (y - co.a.y) / pente + co.a.x;
	else
	  p.x = co.a.x;
      }
    else
      {
	p.x = x;
	p.y = pente * (x - co.a.x) + co.a.y;
      }
    am.x = co.a.x - p.x;
    am.y = co.a.y - p.y;
    xm = sqrt (am.x * am.x + am.y * am.y) / sqrt (u.x * u.x + u.y * u.y);
    if (-am.x * u.x - am.y * u.y < 0)
      xm = 0;
    if (xm > 1)
      xm = 1;
    p.x = xm * u.x + co.a.x;
    p.y = xm * u.y + co.a.y;
    init_nom ();
  }
  void move (int mx, int my);
  void actualise ();
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// Class point_sur_cercle, free point a circle
class point_sur_cercle:public point_c
{
  public:
  double xm;			// curv abscissa on circle

  cercle_c *parent;		// parent circle

    point_sur_cercle (void):point_c ()
  {
    classe = POINT_CE;
  }
  point_sur_cercle (double x, double y, liste_elem & lp):point_c ()
  {
    double k, den, r;
    point_s centre, move;

    classe = POINT_CE;
    parent = (cercle_c *) lp.lire (1);;
    centre = parent->centre ();
    move.x = x;
    move.y = y;
    move = move - centre;
    r = parent->rayon ();
    den = sqrt (move * move);
    if (den == 0)
      k = 0;
    else
      k = r / den;
    p = k * move + centre;
    // compute the curv abscissa
    xm = atan2 (p.y - centre.y, p.x - centre.x);
    init_nom ();
  }
  point_sur_cercle (double x, double y, liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    double k, den, r;
    point_s centre, move;
    classe = POINT_CE;
    parent = (cercle_c *) lp.lire (1);
    centre = parent->centre ();
    move.x = x;
    move.y = y;
    move = move - centre;
    r = parent->rayon ();
    den = sqrt (move * move);
    if (den == 0)
      k = 0;
    else
      k = r / den;
    p = k * move + centre;
    // compute the curv abscissa
    xm = atan2 (p.y - centre.y, p.x - centre.x);
    couleur = ROUGE;
    init_nom ();
  }
  void move (int mx, int my);
  void actualise ();
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// Class point_sur_arc_cercle, free point a arc circle
class point_sur_arc_cercle:public point_c
{
  public:
  double xm;			// abscissa on arc circle on  [ 0 ; 1 ]

  arc_cercle_c *parent;		// parent arc circle

    point_sur_arc_cercle (void):point_c ()
  {
    classe = POINT_AR;
  }
  point_sur_arc_cercle (double x, double y, liste_elem & lp):point_c ()
  {
    classe = POINT_AR;
    parent = (arc_cercle_c *) lp.lire (1);;
    p.x = x;
    p.y = y;
    move (0, 0);
    init_nom ();
  }
  void move (int mx, int my);
  void actualise ();
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// Intersection point between XXX and YYY objects : point_sur_XXX_et_YYY
// point_sur_droite_et_droite
class point_sur_droite_et_droite:public point_c
{
  public:
  droite_c * d1, *d2;
  point_sur_droite_et_droite (void):point_c ()
  {
    classe = POINT_DR_DR;
  }
  point_sur_droite_et_droite (liste_elem & lp):point_c ()
  {
    d1 = (droite_c *) lp.lire (1);
    d2 = (droite_c *) lp.lire (2);
    actualise ();
    couleur = ROUGE;
    classe = POINT_DR_DR;
    init_nom ();
  }
  point_sur_droite_et_droite (liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    d1 = (droite_c *) lp.lire (1);
    d2 = (droite_c *) lp.lire (2);
    actualise ();
    classe = POINT_DR_DR;
    init_nom ();
  }
  void move (int mx, int my)
  {
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// point_sur_droite_et_demi_droite
class point_sur_droite_et_demi_droite:public point_c
{
  public:
  droite_c * droite;
  demi_droite_c *demi_droite;
    point_sur_droite_et_demi_droite (void):point_c ()
  {
    classe = POINT_DR_DD;
  }
  point_sur_droite_et_demi_droite (liste_elem & lp):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DROITE)
      {
	droite = (droite_c *) parent1;
	demi_droite = (demi_droite_c *) parent2;
      }
    else
      {
	droite = (droite_c *) parent2;
	demi_droite = (demi_droite_c *) parent1;
      }
    actualise ();
    couleur = ROUGE;
    init_nom ();
    classe = POINT_DR_DD;
  }
  point_sur_droite_et_demi_droite (liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DROITE)
      {
	droite = (droite_c *) parent1;
	demi_droite = (demi_droite_c *) parent2;
      }
    else
      {
	droite = (droite_c *) parent2;
	demi_droite = (demi_droite_c *) parent1;
      }
    actualise ();
    classe = POINT_DR_DD;
    init_nom ();
  }
  void move (int mx, int my)
  {
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// point_sur_droite_et_segment
class point_sur_droite_et_segment:public point_c
{
  public:
  droite_c * droite;
  segment_c *segment;
    point_sur_droite_et_segment (void):point_c ()
  {
    classe = POINT_DR_SE;
  }
  point_sur_droite_et_segment (liste_elem & lp):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DROITE)
      {
	droite = (droite_c *) parent1;
	segment = (segment_c *) parent2;
      }
    else
      {
	droite = (droite_c *) parent2;
	segment = (segment_c *) parent1;
      }
    actualise ();
    couleur = ROUGE;
    classe = POINT_DR_SE;
    init_nom ();
  }
  point_sur_droite_et_segment (liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DROITE)
      {
	droite = (droite_c *) parent1;
	segment = (segment_c *) parent2;
      }
    else
      {
	droite = (droite_c *) parent2;
	segment = (segment_c *) parent1;
      }
    actualise ();
    classe = POINT_DR_SE;
    init_nom ();
  }
  void move (int mx, int my)
  {
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// point_sur_demi_droite_et_demi_droite
class point_sur_demi_droite_et_demi_droite:public point_c
{
  public:
  demi_droite_c * dd1, *dd2;
  point_sur_demi_droite_et_demi_droite (void):point_c ()
  {
    classe = POINT_DD_DD;
  }
  point_sur_demi_droite_et_demi_droite (liste_elem & lp):point_c ()
  {
    dd1 = (demi_droite_c *) lp.lire (1);
    dd2 = (demi_droite_c *) lp.lire (2);
    actualise ();
    couleur = ROUGE;
    classe = POINT_DD_DD;
    init_nom ();
  }
  point_sur_demi_droite_et_demi_droite (liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    dd1 = (demi_droite_c *) lp.lire (1);
    dd2 = (demi_droite_c *) lp.lire (2);
    actualise ();
    classe = POINT_DD_DD;
    init_nom ();
  }
  void move (int mx, int my)
  {
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// point_sur_demi_droite_et_segment
class point_sur_demi_droite_et_segment:public point_c
{
  public:
  demi_droite_c * demi_droite;
  segment_c *segment;
    point_sur_demi_droite_et_segment (void):point_c ()
  {
    classe = POINT_DD_SE;
  }
  point_sur_demi_droite_et_segment (liste_elem & lp):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DEMI_DROITE)
      {
	demi_droite = (demi_droite_c *) parent1;
	segment = (segment_c *) parent2;
      }
    else
      {
	demi_droite = (demi_droite_c *) parent2;
	segment = (segment_c *) parent1;
      }
    actualise ();
    couleur = ROUGE;
    classe = POINT_DD_SE;
    init_nom ();
  }
  point_sur_demi_droite_et_segment (liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DEMI_DROITE)
      {
	demi_droite = (demi_droite_c *) parent1;
	segment = (segment_c *) parent2;
      }
    else
      {
	demi_droite = (demi_droite_c *) parent2;
	segment = (segment_c *) parent1;
      }
    actualise ();
    classe = POINT_DD_SE;
    init_nom ();
  }
  void move (int mx, int my)
  {
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// point_sur_segment_et_segment
class point_sur_segment_et_segment:public point_c
{
  public:
  segment_c * s1, *s2;
  point_sur_segment_et_segment (void):point_c ()
  {
    classe = POINT_SE_SE;
  }
  point_sur_segment_et_segment (liste_elem & lp):point_c ()
  {
    s1 = (segment_c *) lp.lire (1);;
    s2 = (segment_c *) lp.lire (2);
    actualise ();
    couleur = ROUGE;
    classe = POINT_SE_SE;
    init_nom ();
  }
  point_sur_segment_et_segment (liste_elem & lp, char a, char b, char c, char d):point_c (a, b, c, d)
  {
    s1 = (segment_c *) lp.lire (1);
    s2 = (segment_c *) lp.lire (2);
    actualise ();
    classe = POINT_SE_SE;
    init_nom ();
  }
  void move (int mx, int my)
  {
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// classe de base point d'intersection de droite et cercle
class point_sur_cercle_et_droite:public point_c
{
  public:
  droite_c * droite;
  cercle_c *cercle;
    point_sur_cercle_et_droite (void):point_c ()
  {
    classe = POINT_DR_CE;
  }
  point_sur_cercle_et_droite (liste_elem & lp, char s):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DROITE)
      {
	droite = (droite_c *) parent1;
	cercle = (cercle_c *) parent2;
      }
    else
      {
	droite = (droite_c *) parent2;
	cercle = (cercle_c *) parent1;
      }
    couleur = ROUGE;
    signe = s;
    classe = POINT_DR_CE;
    actualise ();
    init_nom ();
  }
  point_sur_cercle_et_droite (liste_elem & lp, char a, char b, char c, char d, char s):point_c (a, b, c, d)
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DROITE)
      {
	droite = (droite_c *) parent1;
	cercle = (cercle_c *) parent2;
      }
    else
      {
	droite = (droite_c *) parent2;
	cercle = (cercle_c *) parent1;
      }
    signe = s;
    classe = POINT_DR_CE;
    actualise ();
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
//
class point_sur_cercle_et_demi_droite:public point_c
{
  public:
  demi_droite_c * demi_droite;
  cercle_c *cercle;
    point_sur_cercle_et_demi_droite (void):point_c ()
  {
    classe = POINT_DD_CE;
  }
  point_sur_cercle_et_demi_droite (liste_elem & lp, char s):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DEMI_DROITE)
      {
	demi_droite = (demi_droite_c *) parent1;
	cercle = (cercle_c *) parent2;
      }
    else
      {
	demi_droite = (demi_droite_c *) parent2;
	cercle = (cercle_c *) parent1;
      }
    couleur = ROUGE;
    signe = s;
    actualise ();
    classe = POINT_DD_CE;
    init_nom ();
  }
  point_sur_cercle_et_demi_droite (liste_elem & lp, char a, char b, char c, char d, char s):point_c (a, b, c, d)
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DEMI_DROITE)
      {
	demi_droite = (demi_droite_c *) parent1;
	cercle = (cercle_c *) parent2;
      }
    else
      {
	demi_droite = (demi_droite_c *) parent2;
	cercle = (cercle_c *) parent1;
      }
    signe = s;
    actualise ();
    classe = POINT_DD_CE;
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
//
class point_sur_cercle_et_segment:public point_c
{
  public:
  segment_c * segment;
  cercle_c *cercle;
    point_sur_cercle_et_segment ():point_c ()
  {
    classe = POINT_SE_CE;
  }
  point_sur_cercle_et_segment (liste_elem & lp, char s):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_SEGMENT)
      {
	segment = (segment_c *) parent1;
	cercle = (cercle_c *) parent2;
      }
    else
      {
	segment = (segment_c *) parent2;
	cercle = (cercle_c *) parent1;
      }
    couleur = ROUGE;
    signe = s;
    actualise ();
    classe = POINT_SE_CE;
    init_nom ();
  }
  point_sur_cercle_et_segment (liste_elem & lp, char a, char b, char c, char d, char s):point_c (a, b, c, d)
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_SEGMENT)
      {
	segment = (segment_c *) parent1;
	cercle = (cercle_c *) parent2;
      }
    else
      {
	segment = (segment_c *) parent2;
	cercle = (cercle_c *) parent1;
      }
    signe = s;
    actualise ();
    classe = POINT_SE_CE;
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
//
class point_sur_cercle_et_cercle:public point_c
{
  public:
  cercle_c * cercle1, *cercle2;
  point_sur_cercle_et_cercle (void):point_c ()
  {
    classe = POINT_CE_CE;
  }
  point_sur_cercle_et_cercle (liste_elem & lp, char s):point_c ()
  {
    cercle1 = (cercle_c *) lp.lire (1);
    cercle2 = (cercle_c *) lp.lire (2);
    couleur = ROUGE;
    signe = s;
    actualise ();
    classe = POINT_CE_CE;
    init_nom ();
  }
  point_sur_cercle_et_cercle (liste_elem & lp, char a, char b, char c, char d, char s):point_c (a, b, c, d)
  {
    cercle1 = (cercle_c *) lp.lire (1);
    cercle2 = (cercle_c *) lp.lire (2);
    signe = s;
    actualise ();
    classe = POINT_CE_CE;
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};

// Intersection point with arc circle object
class point_sur_arc_cercle_et_droite:public point_c
{
  public:
  droite_c * droite;
  arc_cercle_c *arc_cercle;
    point_sur_arc_cercle_et_droite (void):point_c ()
  {
    classe = POINT_DR_AR;
  }
  point_sur_arc_cercle_et_droite (liste_elem & lp, char s):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DROITE)
      {
	droite = (droite_c *) parent1;
	arc_cercle = (arc_cercle_c *) parent2;
      }
    else
      {
	droite = (droite_c *) parent2;
	arc_cercle = (arc_cercle_c *) parent1;
      }
    couleur = ROUGE;
    signe = s;
    classe = POINT_DR_AR;
    actualise ();
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
class point_sur_arc_cercle_et_demi_droite:public point_c
{
  public:
  demi_droite_c * demi_droite;
  arc_cercle_c *arc_cercle;
    point_sur_arc_cercle_et_demi_droite (void):point_c ()
  {
    classe = POINT_DD_AR;
  }
  point_sur_arc_cercle_et_demi_droite (liste_elem & lp, char s):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_DEMI_DROITE)
      {
	demi_droite = (demi_droite_c *) parent1;
	arc_cercle = (arc_cercle_c *) parent2;
      }
    else
      {
	demi_droite = (demi_droite_c *) parent2;
	arc_cercle = (arc_cercle_c *) parent1;
      }
    couleur = ROUGE;
    signe = s;
    classe = POINT_DD_AR;
    actualise ();
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
class point_sur_arc_cercle_et_segment:public point_c
{
  public:
  segment_c * segment;
  arc_cercle_c *arc_cercle;
    point_sur_arc_cercle_et_segment (void):point_c ()
  {
    classe = POINT_SE_AR;
  }
  point_sur_arc_cercle_et_segment (liste_elem & lp, char s):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_SEGMENT)
      {
	segment = (segment_c *) parent1;
	arc_cercle = (arc_cercle_c *) parent2;
      }
    else
      {
	segment = (segment_c *) parent2;
	arc_cercle = (arc_cercle_c *) parent1;
      }
    couleur = ROUGE;
    signe = s;
    classe = POINT_SE_AR;
    actualise ();
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
class point_sur_arc_cercle_et_cercle:public point_c
{
  public:
  cercle_c * cercle;
  arc_cercle_c *arc_cercle;
    point_sur_arc_cercle_et_cercle (void):point_c ()
  {
    classe = POINT_CE_AR;
  }
  point_sur_arc_cercle_et_cercle (liste_elem & lp, char s):point_c ()
  {
    figure_c *parent1, *parent2;
    parent1 = (figure_c *) lp.lire (1);
    parent2 = (figure_c *) lp.lire (2);
    if (parent1->type == FIG_CERCLE)
      {
	cercle = (cercle_c *) parent1;
	arc_cercle = (arc_cercle_c *) parent2;
      }
    else
      {
	cercle = (cercle_c *) parent2;
	arc_cercle = (arc_cercle_c *) parent1;
      }
    couleur = ROUGE;
    signe = s;
    classe = POINT_CE_AR;
    actualise ();
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
class point_sur_arc_cercle_et_arc_cercle:public point_c
{
  public:
  arc_cercle_c * arc_cercle1;
  arc_cercle_c *arc_cercle2;
    point_sur_arc_cercle_et_arc_cercle (void):point_c ()
  {
    classe = POINT_AR_AR;
  }
  point_sur_arc_cercle_et_arc_cercle (liste_elem & lp, char s):point_c ()
  {
    arc_cercle1 = (arc_cercle_c *) lp.lire (1);
    arc_cercle2 = (arc_cercle_c *) lp.lire (2);

    couleur = ROUGE;
    signe = s;
    classe = POINT_AR_AR;
    actualise ();
    init_nom ();
  }
  void actualise ();
  liste_elem *parents (liste_elem * liste_parent);
  void move (int mx, int my)
  {
  }
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};


// Transformed Point
// By reflexion
class reflexion_point:public point_c
{
  public:
  droite_c * axe;
  point_c *pt;
    reflexion_point (void):point_c ()
  {
    classe = POINT_RE;
  }
  reflexion_point (liste_elem & lp):point_c ()
  {
    figure_c *f1, *f2;
    f1 = (figure_c *) lp.lire (1);
    f2 = (figure_c *) lp.lire (2);
    if (f1->type == FIG_POINT)
      {
	pt = (point_c *) f1;
	axe = (droite_c *) f2;
      }
    else
      {
	pt = (point_c *) f2;
	axe = (droite_c *) f1;
      }
    actualise ();
    classe = POINT_RE;
    init_nom ();
  }
  point_s coordonnees (void);
  void move (int mx, int my);
  void actualise (void);
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};

// By symetry
class symetrie_point:public point_c
{
  public:
  point_c * pt, *symetrie;
  symetrie_point ():point_c ()
  {
    classe = POINT_SY;
  }
  symetrie_point (liste_elem & lp):point_c ()
  {
    pt = (point_c *) lp.lire (2);
    symetrie = (point_c *) lp.lire (1);
    actualise ();
    classe = POINT_SY;
    init_nom ();
  }
  point_s coordonnees (void);
  void move (int mx, int my);
  void actualise (void);
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};

// By translation
class translation_point:public point_c
{
  public:
  vecteur_c * v;
  point_c *pt;
    translation_point ():point_c ()
  {
    classe = POINT_TR;
  }
  translation_point (liste_elem & lp):point_c ()
  {
    figure_c *f1, *f2;
    f1 = (figure_c *) lp.lire (1);
    f2 = (figure_c *) lp.lire (2);
    if (f1->type == FIG_POINT)
      {
	pt = (point_c *) f1;
	v = (vecteur_c *) f2;
      }
    else
      {
	pt = (point_c *) f2;
	v = (vecteur_c *) f1;
      }
    actualise ();
    classe = POINT_TR;
    init_nom ();
  }
  point_s coordonnees (void);
  void move (int mx, int my);
  void actualise (void);
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// By rotation
class rotation_point:public point_c
{
  public:
  valeur_c * v;
  point_c *c, *pt;
    rotation_point ():point_c ()
  {
    classe = POINT_RO;
  }
  rotation_point (liste_elem & lp):point_c ()
  {
    figure_c *f1, *f2, *f3;
    f1 = (figure_c *) lp.lire (1);
    f2 = (figure_c *) lp.lire (2);
    f3 = (figure_c *) lp.lire (3);
    if (f1->type == FIG_POINT)
      {
	c = (point_c *) f1;
	if (f2->type == FIG_POINT)
	  {
	    v = (valeur_c *) f3;
	    pt = (point_c *) f2;
	  }
	else
	  {
	    v = (valeur_c *) f2;
	    pt = (point_c *) f3;
	  }
      }
    else
      {
	c = (point_c *) f2;
	pt = (point_c *) f3;
	v = (valeur_c *) f1;
      }
    actualise ();
    classe = POINT_RO;
    init_nom ();
  }
  point_s coordonnees (void);
  void move (int mx, int my);
  void actualise (void);
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};
// By homothetie
class homothetie_point:public point_c
{
  public:
  valeur_c * v;
  point_c *c, *pt;
    homothetie_point ():point_c ()
  {
    classe = POINT_HO;
  }
  homothetie_point (liste_elem & lp):point_c ()
  {
    figure_c *f1, *f2, *f3;
    f1 = (figure_c *) lp.lire (1);
    f2 = (figure_c *) lp.lire (2);
    f3 = (figure_c *) lp.lire (3);
    if (f1->type == FIG_POINT)
      {
	c = (point_c *) f1;
	if (f2->type == FIG_POINT)
	  {
	    v = (valeur_c *) f3;
	    pt = (point_c *) f2;
	  }
	else
	  {
	    v = (valeur_c *) f2;
	    pt = (point_c *) f3;
	  }
      }
    else
      {
	c = (point_c *) f2;
	pt = (point_c *) f3;
	v = (valeur_c *) f1;
      }
    actualise ();
    classe = POINT_HO;
    init_nom ();
  }
  point_s coordonnees (void);
  void move (int mx, int my);
  void actualise (void);
  liste_elem *parents (liste_elem * liste_parent);
  void sauve_disk (FILE * f);
  void lire_disk (FILE * f);
  char dependance (figure_c * fig);
};

#endif
