/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/*
 * Copyright (c) 1995   The Geometry Center
 */

/*
 * geomview.c 
 * 
 * This file implements open_geomview(), which opens a connection
 * to a Geomview process, starting up Geomview if necessary.
 * 
 * Communication is actually done via a named pipe in the /tmp/geomview
 * directory.  This pipe and dir are created if they don't yet exist.
 * 
 * This code was originally written by Stuart Levy for togeomview.c; I
 * (mbp) modified it for Pisces.
 * (worfolk) modified it for dstool.
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/file.h>
#ifdef AIX
#define _BSD  1		/* Get FNDELAY from fcntl.h */
#endif
#include <fcntl.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <sys/un.h>

#ifndef NeXT
#include <unistd.h>
#endif

#ifndef O_NONBLOCK
# define O_NONBLOCK O_NDELAY
#endif
#ifndef FNDELAY
# define FNDELAY O_NDELAY
#endif
#ifndef FNONBLK
# define FNONBLK FNDELAY
#endif

#include <utilities.h>
#include <defaults.h>

/* Max number of times to try opening the pipe to Geomview; these
 * tries are done a 1 second intervals. We might need to try more than
 * once because we can't actually open the pipe until Geomview is
 * alive and listening on the other end, and it might take a few
 * seconds for Geomview to come up. */
#define MAX_TRIES 60

static char *pipedir = "/tmp/geomview";	/* dir for named pipe */
static char *toname = "dstool.togv";	/* name of pipe to Geomview */
static char *frname = "dstool.frgv";	/* name of pipe from Geomview */

/* gvpath gives the pathname for invoking geomview; specifying just
 * "geomview" causes the shell to search for it in the $PATH.
 * If it's NULL, we assume that Geomview isn't available;
 * see tcl/mainloop.c for details. */
char *gvpath = GEOMVIEW_EXECUTABLE;

static int interrupted = 0;

static void start_gv(void);
static void gv_interrupt_sig(int sig);
extern FILE *gv_to;
extern FILE *gv_from;

/*-----------------------------------------------------------------------
 * Function:	open_geomview
 * Description:	Open a stream to Geomview, starting a new Geomview
 *		  process if one is not alread listening
 * Returns:	the opened stream
 * Author:	mbp
 * Date:	Thu Sep 15 22:25:58 1994
 * Notes:	The returned stream is open for writing only; you can
 *		  write to it as though it were a file.  Returns NULL
 *		  if unable to comply.
 *
 */
void open_geomview(FILE **to, FILE **from)
{
  int try, tofd = -1;
  char topipename[BUFSIZ], frpipename[BUFSIZ];
  static char *geomview[] = { "geomview", NULL };
  struct stat st;
  FILE *fp = NULL;
  extern int errno;

  if (gvpath == NULL) return;

/*  printf ("%s\n", gvpath); */
  
  if(access(pipedir, W_OK) < 0) {
    mkdir(pipedir, 0777);
    chmod(pipedir, 0777);
  }

  sprintf(topipename, "%s/%s", pipedir, toname);
  sprintf(frpipename, "%s/%s", pipedir, frname);
  
  if(stat(topipename, &st) == 0 && (st.st_mode&S_IFMT) != S_IFIFO) {
    unlink(topipename);
  }
  if(stat(frpipename, &st) == 0 && (st.st_mode&S_IFMT) != S_IFIFO) {
    unlink(frpipename);
  }
  
  if(access(topipename, 0) < 0) {
    mknod(topipename, S_IFIFO, 0);
    chmod(topipename, 0666);
  }
  if(access(frpipename, 0) < 0) {
    mknod(frpipename, S_IFIFO, 0);
    chmod(frpipename, 0666);
  }
  tofd = open(topipename, O_WRONLY|O_NONBLOCK);
  if(tofd >= 0) {
    fcntl(tofd, F_SETFL, fcntl(tofd, F_GETFL, 0) & ~(FNDELAY|FNONBLK|O_NONBLOCK));
  } else if(errno == ENXIO) {
    start_gv();
    try = 0;
    while (!interrupted && tofd < 0 && try < MAX_TRIES) {
      tofd = open(topipename, O_WRONLY|O_NONBLOCK);
      if (tofd >= 0) {
	fcntl(tofd, F_SETFL, fcntl(tofd, F_GETFL, 0) & ~(FNDELAY|FNONBLK|O_NONBLOCK));
      } else {
	++try;
	sleep(1);
      }
    }
  }

  if (tofd >= 0) {
    fp = fdopen(tofd, "w");
    if (from != NULL) {
      *from = fopen(frpipename, "r");
    }
  }
  *to = fp;

}

/*
 * interrupt() is a signal handler used to get a message from the
 * child process in the event that it's unable to fire up Geomview.
 */
static void
gv_interrupt_sig(int sig) { 
  interrupted = 1;
}

/*
 * start_gv() starts a copy of Geomview
 */
static void
start_gv()
{
  static char cmd[BUFSIZ];
  char *args[] = {
    GEOMVIEW_EXECUTABLE,
    "-c",
    cmd,
    NULL
    };

  sprintf(cmd, "(command %s/%s %s/%s)", pipedir, toname, pipedir, frname);
  signal(SIGALRM, gv_interrupt_sig);
  if(fork() == 0) {
    close(0);
#if defined(NeXT) || defined(BSD)
    setpgrp(0,getpid());
#else /* SysV style */
    setpgrp();
#endif
    execvp(gvpath, &args[0]);
    execvp("geomview", &args[0]);
    execvp("gv", &args[0]);
/*    perror("Couldn't exec geomview nor gv"); */
    kill(getppid(), SIGALRM);
    _exit(1);
  }
}


void
geomview_send(char *str)
{
  if (gv_to == NULL) return;
  fprintf(gv_to, str);
  fflush(gv_to);
}

void
geomview_wait(void)
{
  char buf[256];

  if ((gv_to == NULL) || (gv_from == NULL)) return;
  geomview_send("(echo \"ok\\n\")\n");
  fgets(buf,255,gv_from);
}


void
geomview_transform(int n, double f[])
{
  int i, j;

  if (gv_to == NULL) return;

  fprintf(gv_to, "(progn\n");
  for (i=0; i<n; i++)
    {
      fprintf(gv_to, "(read transform { transform define ttt%d\n",i);
      for (j=0; j<16; j++)
	{
	  fprintf(gv_to, "%f ",f[16*i+j]);
	}
      fprintf(gv_to, "} )\n");

    }
  fprintf(gv_to, ")\n");
  fflush(gv_to);
}


FILE *
geomview_get_fp()
{
  return gv_to;
}
