#import "GVApp.h"

#import "EventView.h"
#import "MainPanelDel.h"
#import "InspectorDel.h"
#import "PreferencesDel.h"
#import "ToolPanelDel.h"

#import "PrintCam.h"

#include "main.h"

//---------------
#import <stdlib.h>
#import <sys/param.h>
#import <objc/NXBundle.h>

#import <defaults/defaults.h>

#include "mg.h"
#include "mgrib.h"
#include "main.h"
#include "drawer.h"
#include "ui.h"
#include "comm.h"
#include "lang.h"
#include "rman.h"
#include "libc.h"

#include <sys/stat.h>
#include <sys/file.h>
#include <string.h>
#include <streams/streams.h>

snap_entry snapshot_table[] =
{
    {NULL, NULL}
};

static void parse_args(int argc, char *argv[]);

extern HandleOps CommandOps, GeomOps;

void env_init()
{
#define MAXDIRS 100
  extern char *getenv();
  char *cp;
  char str[MAXPATHLEN + 1];

  cp = strrchr(NXArgv[0], '/');
  if(cp) {
    sprintf(str, "GEOMVIEW_DIR=%.*s", (int)(cp-NXArgv[0]), NXArgv[0]);
    putenv(strdup(str));
  }
/* example of CONFIG.gv file:
   #
   # By default, $GEOMVIEW_DIR refers to the geomview.app directory;
   # if the data files and geomview external modules are moved elsewhere
   # this could be overridden by the line:
   #
   #  ( setenv GEOMVIEW_DIR /u/gcg/ngrap)
   #
   # In any case, CONFIG.gv should include something like the following,
   # to set the search paths for the above.
   #
   ( set-load-path  ( . $GEOMVIEW_DIR/data/geom $GEOMVIEW_DIR/data ) )
   ( set-emodule-path ( $GEOMVIEW_DIR/modules/next ) )
   #
   # Normally the main init file (read before the user's own ~/.geomview)
   # lives in "geomview.app/geomview.gv"; this could be overridden by e.g.:
   #
   # ( setenv GEOMVIEW_SYSTEM_INITFILE   $GEOMVIEW_DIR/data/.geomview )
   #
   # CONFIG.gv is intended for configuring pathnames only; other initialization
   # should be done in the file specified by GEOMVIEW_SYSTEM_INITFILE.
*/

  if (![[NXBundle mainBundle] getPath:str forResource:"CONFIG" ofType:"gv"] ) 
    return;

  loadfile(str, &CommandOps, 0);

  if((geomview_system_initfile = getenv("GEOMVIEW_SYSTEM_INITFILE"))==NULL) {
    [[NXBundle mainBundle] getPath:str forResource:"geomview" ofType:"gv"];
    geomview_system_initfile = strdup(str);
  }
}

static void
parse_args(int argc, char *argv[])
{
  char **av;
  int ac;

  for(ac = argc, av = argv+1; --ac > 0; av++) {
    if(!parse_common_arg(&ac, &av)) {
	usage(argv[0], "");
    }
  }
}

LDEFINE(cull_backface, LVOID,	
"(cull-backface [on|off])\n\
	Select whether back-facing polygons should be displayed.\n\
	Initially on: all polygons are displayed.  When off, polygons whose\n\
	vertices are arranged clockwise on the screen are hidden.  Useful for\n\
	simulating two-sided surface coloring.")
{
  int on;
  LDECLARE(("cull-backface", LBEGIN,
    LKEYWORD, &on,
    LEND));
//  backface(on == YES_KEYWORD || on == ON_KEYWORD);
  gv_redraw(ALLCAMS);
  return Lt;
}

//----------------
struct panel panels[] = {
    { NULL, 	0,       	0,			0, 0,-1,-1 },
    { NULL,	"geomview", 	"geomview",  		1, 0,-1,-1 },
    { NULL,	"Tools", 	"[Pt] Tools",		1, 1,-1,-1 },
    { NULL,	"Appearance", 	"[Pa] Appearance", 	1, 1,-1,-1 },
    { NULL,	"Commands", 	"[PC] Commands",	0, 1,-1,-1 },
    { NULL,	"Save",		"[>] Save",		0, 1,-1,-1 },
    { NULL,	"Load",		"[<] Load",		0, 1,-1,-1 },
};

struct saveops save[] = {
    { SAVE_WIO,	&CommandOps,	0,	 	"Commands"		},
    { SAVE_WIO, &GeomOps,	SELF,		"Geometry alone"	},
    { SAVE_WIO, &GeomOps,	WORLDGEOM,	"Geometry [in world]"	},
    { SAVE_WIO, &GeomOps,	UNIVERSE,	"Geometry [in universe]" },
    { SAVE_RMan, NULL,		TIFF_KEYWORD,	"RMan [->tiff]"		},
    { SAVE_RMan, NULL,		FRAME_KEYWORD,	"RMan [->frame]"	},
    { SAVE_SNAP, NULL,		0,	 	"PPM snapshot"		},
    { SAVE_WIO, &CamOps,	UNIVERSE, 	"Camera"		},
    { SAVE_WIO, &TransOps,	WORLDGEOM,	"Transform [to world]"	},
    { SAVE_WIO, &TransOps,	UNIVERSE,	"Transform [to universe]"},
    { SAVE_WIO, &WindowOps,	UNIVERSE,	"Window"		},
    { SAVE_PANELS, NULL,	0,		"Panels"		},
};

#include <sys/types.h>
#include <sys/time.h>

/* tmp context debugging include */
#include <mgri.h>

#define streq(s1,s2)  (!strcmp(s1,s2))

UIState uistate;

extern char builddate[], buildinfo[];

static struct perf {		/* Performance metering */
    int interval;		/* Interval between auto-reports */
    int mindt, maxdt, meandt;	/* Integer milliseconds */
    int cycles;			/* # cycles where we actually did something */
    struct timeval then;
} perf;

void
timing(int interval)
{
    if(perf.cycles > 0) {
	printf("%d..%d ms/cycle, mean %d ms over %d cycles\n",
		perf.mindt, perf.maxdt,
		perf.cycles ? perf.meandt/perf.cycles : 0,
		perf.cycles);
	fflush(stdout);
    }
    perf.mindt = 9999999, perf.maxdt = -1, perf.meandt = perf.cycles = 0;
    perf.interval = interval;
}

static void
perftick()
{
    int dt;
    struct timeval now;
    gettimeofday(&now, NULL);
    dt = (now.tv_sec - perf.then.tv_sec)*1000 +
	 (now.tv_usec - perf.then.tv_usec)/1000;
    if(dt > 0) {
	if(dt < perf.mindt) perf.mindt = dt;
	if(dt > perf.maxdt) perf.maxdt = dt;
	perf.meandt += dt;
	if(++perf.cycles == perf.interval)
	    timing(perf.interval);
    }
}

void ui_init()
{
  cui_init();
  clight_init();
//  panel_init(); do not think that I need one of these.

  gv_event_mode( OBJROTATE );
  set_light( light_count() > 0 ? 1 : 0 ); // should this really be needed??
//  ui_select(WORLDGEOM);  
  [[NXApp inspectorDel] appearance];
  uistate.cursor_on = 1;
  return;
}

/*
 * ui_update() is called once each time around the main loop.
 * Anything needing synchronous updates can go here.
 * Right now we just update the emodule browser; changes in it can
 * arise asynchronously, as processes die.
 */
void ui_update()
{
  int i;
  emodule *em;
  if(uistate.emod_check) {
    for(i=0, em=VVEC(uistate.emod,emodule); i<VVCOUNT(uistate.emod); ) {
	if(VVEC(uistate.emod,emodule)[i].pid < 0 &&
	    (em->link == NULL || PoolInputFile(em->link) == NULL)) {
		emodule_reap(em);
	} else {
	    i++, em++;
	}
    }
    uistate.emod_check = 0;
  }
  if(uistate.emod_changed) {
    [[NXApp mainPanelDel] buildExternalBrowser];
    uistate.emod_changed = 0;
  }
  return;
}


void RedrawSelf(DPSTimedEntry num, double now, void *userdata)
{
  int max, nseen;
  struct timeval notime;
  fd_set fds;

  if((button.left||button.middle||button.right)&&(mousedragage++==1)) {
    gv_rawevent(event.dev,-1, event.x, event.y, ++event.t );
  }
  PoolInputFDs(&fds, &max);
  notime.tv_sec = notime.tv_usec = 0;
  nseen = select(max+1, &fds, NULL, NULL, &notime);
  gettimeofday(&perf.then, NULL);
  PoolInAll(&fds, &nseen);
  ui_update();
  gv_update_draw( ALLCAMS, 0. );
  if(perf.interval > 0)
    perftick();
}

static int suitable_id(int savetype, char **namep)
{
   struct saveops *sp = &save[savetype];
   int id = drawer_idbyname(*namep);

   if(savetype >= 0 && savetype < COUNT(save)) {
	if(sp->ops == &CamOps || sp->ops == &WindowOps
			    || sp->special == SAVE_RMan
			    || sp->special == SAVE_SNAP) {
	    if(!ISCAM(id)) id = FOCUSID;
	} else if(sp->ops == &GeomOps && !ISGEOM(id))
	    id = GEOMID(uistate.targetgeom);
   }
   *namep = drawer_id2name(id);
   return (drawer_get_object(id) == NULL) ? NOID : id;
}

void ui_pickcolor( int drawer_code )
{
}

void ui_curson(int on)
{
/*
  if( (uistate.cursor_on = (on<0) ? !uistate.cursor_on : on) )
    curson();
  else cursoff();
*/
}

int ui_name2panel(char *name)
{
  register int i;
  if(strcasecmp(name, "main") == 0) return P_MAIN;
  if(strcasecmp(name, "camera") == 0) return P_INSPECTOR;
  if(strcasecmp(name, "lights") == 0) return P_INSPECTOR;
  for(i=P_MAX; --i > 0 && strncasecmp(name, panels[i].name, 4) != 0; )   ;
  return i;
}

void ui_replace_mode(char *name, PFI proc, int type, int index)
{
  fprintf(stderr,"The function ui_replace_mode is not being supported\n");
}

void ui_highlight(int newid)
{
  int newtype = TYPEOF(newid);
  DGeom *dg;
  DView *v;
  static Color black = {0.0, 0.0, 0.0};
  Color c;
  
  if ( newtype == T_GEOM &&
      (dg = (DGeom *) drawer_get_object(newid)) && dg->bboxap ) {
    c = uistate.pickedbboxcolor;
    drawer_color( GEOMID(uistate.targetgeom), DRAWER_BBOXCOLOR, &black );
    drawer_color( newid, DRAWER_BBOXCOLOR, &c );
  } else if (newtype == T_CAM) { // pop cam window in case it's obscured 
    v = (DView *) drawer_get_object(newid);
    if (v) {
//      mggl_ctxselect(v->mgctx); // winset happens in here 
//      winpop();
    }
  }
}

/* ui_action is separate from ui_event_mode code because they're 
   keeping different old values around 
 */
void ui_action(int val)
{ 
  [[NXApp toolPanelDel] action:val];
}

void ui_center(int oid)
{
  if ( (oid == NOID) || (oid == CENTERID) ) {
    oid = uistate.centerid;
  } else {
    set_ui_center(oid);
  }
  [[NXApp toolPanelDel] setCenterField];
}

int ui_panelshown(int index) 
{ return (unsigned)index >= P_MAX ? 0 : panels[index].shown; }


void ui_popup_message(char *s)
{
  NXRunAlertPanel("Geomview Message", s, "OK", NULL, NULL);
  return;
}

/*
 * Show what's happening on the keyboard.
 * Commands are enclosed in [brackets].  In case of error, a '?' appears
 * following the close-bracket.
 */
void ui_keyboard(int ch)
{
  static char kybd[9];
  static int nextc = 0;
  if(ch <= 0) {
    ui_keyboard(']');
    if(ch < 0) ui_keyboard('?');
    nextc = 0;
  } else {
    if(nextc >= sizeof(kybd)-1) {
	bcopy(kybd+1, kybd, sizeof(kybd)-2);
	nextc = sizeof(kybd)-2;
    } else if(nextc == 0) {
	kybd[nextc++] = '[';
    }
    kybd[nextc++] = ch;
    kybd[nextc] = '\0';
//    if([NXApp mainPanelDel]) [[NXApp mainPanelDel] setKeyboardString:kybd];
  }
  return;
}

void ui_objectchange()
{
  [[NXApp mainPanelDel] objectChange];
  [[NXApp inspectorDel] objectChange];
  [[NXApp menuCloseCamera] setEnabled:((drawer_cam_count()>0) ? YES : NO)];
  return;
}

void ui_add_mode(char *name, PFI proc, int type)
{
  ui_install_mode(name, proc, type);
  return;
}

void ui_remove_mode(char *name)
{
  int mode = ui_mode_index(name);
  ui_uninstall_mode(name);
  if(mode==uistate.mode_current) {
    gv_event_mode( uistate.modenames[0] );
  }
  return;
}

void ui_maybe_refresh(int oid)
{
  DObject *obj;

  obj = drawer_get_object(oid);
  
  if(obj == NULL) return;
  if (ISGEOM(obj->id)) [[NXApp inspectorDel] appearance];
  else if(ISCAM(obj->id)) [[NXApp inspectorDel] camera];
      
  [[NXApp toolPanelDel] setTargetField];
  [[NXApp inspectorDel] setSpace];
  [[NXApp toolPanelDel] setSpace];
  [[NXApp toolPanelDel] setModel];
  return;
}

void ui_event_mode(char *mode)
{
  uistate.mode_current = ui_mode_index(mode);

  [[NXApp toolPanelDel] event_mode];
  return;
}

void ui_mousefocus(int index)
{
  uistate.mousefocus = index;
  [[NXApp inspectorDel] mouseFocus];
  [[NXApp toolPanelDel] setModel];
  return;
}


static int final_initted = 0;
/*
 * This is called after all other initialization (command-line, &c) is done.
 * It makes the main control window visible, if appropriate.
 */
void ui_final_init()
{
  int i;
  final_initted = 1;
//  ui_select(CAMID(uistate.targetcam));
  [[NXApp inspectorDel] camera];
//  ui_select(GEOMID(uistate.targetgeom));
  [[NXApp inspectorDel] appearance];
  lights_changed_check();
  for(i=1; i<P_MAX; i++)
    if(panels[i].shown) ui_showpanel(i, 1);
  ui_objectchange();
}

void ui_showpanel( int index, int show )
{
  struct panel *p = &panels[index];
  if((unsigned) index >= P_MAX) return;
  if(show < 0) show = !p->shown;
  p->shown = show;
  if(!final_initted)
    return;

  switch(index) {
    case P_INPUT: if(show) [NXApp open:NXApp]; break;
    case P_SAVE: if(show) [NXApp save:NXApp]; break;
    case P_MAIN: [[NXApp panelMain:NXApp] display]; break;
    case P_TOOL: [[NXApp toolPanel:NXApp] display]; break;
    case P_INSPECTOR: [[NXApp inspector:NXApp] display]; break;
    case P_COMMANDS: 
      if(show) [[[NXApp commandText] window] makeKeyAndOrderFront:NXApp];
      else [[[NXApp commandText] window] close];
      break;
  }
  return;
}
/* handles callbacks from mgri */
static void callbackfunc( mgcontext *thectx, int type)
{
  int viewid = drawer_idbyctx(thectx);
  if(viewid) {
    switch(type) {
      case MG_RIWINDOWICONIFIED:
          gv_freeze(drawer_idbyctx(thectx), 1); // freeze: don't draw here now.
	  break;
      case MG_RIWINDOWDEICONIFIED:
	  gv_freeze(drawer_idbyctx(thectx), 0); // permit thawing now.
          gv_redraw(drawer_idbyctx(thectx)); // drawer_redraw(viewid);
	  break;
      case MG_RIWINDOWRESIZED:
          gv_redraw(drawer_idbyctx(thectx)); // drawer_redraw(viewid);
	  break;
      case MG_RIWINDOWCLOSED:            // WINDOWWILLCLOSE
          gv_delete(drawer_idbyctx(thectx)); // drawer_delete(viewid);
	  break;
      case MG_RIWINDOWBECAMEMAIN:
      case MG_RIWINDOWBECAMEKEY:
          gv_winenter(drawer_idbyctx(thectx)); // drawer_winenter(viewid);
	  break;
    }
  }
}

/* We don't need to know about this. */
void ui_windowWillOpen(DView *dv) { }

void ui_windowDidOpen(DView *dv)
{
  char *winId;
  char *viewId;
  NXRect r;
      
  mgctxset(MG_RICALLBACK, &callbackfunc, MG_END);
  mgctxget(MG_NXWINDOW, &winId);
  mgctxget(MG_NXVIEW, &viewId);
      
  [(id)winId addToEventMask:(NX_KEYDOWNMASK | NX_KEYDOWNMASK)];
  [NXApp makeKeyResponderFor:(id)winId];
  [(id)viewId getBounds:&r];
  [(id)viewId addSubview:[[EventView alloc] initFrame:&r]];
  [(id)viewId setAutoresizeSubviews:YES];
  [(id)winId orderFrontRegardless];
  [(id)winId makeKeyWindow];

  return;
}

void set_light_display(int lightno)
{
  [[NXApp inspectorDel] lights];
  return;
}

void ui_light_button()
{
  ui_objectchange();
  [[NXApp toolPanelDel] lightEdit];
  [[NXApp mainPanelDel] lightEdit];
//  [[NXApp inspectorDel] setShowLights];
  return;
}

void ui_lights_changed()
{
  [[NXApp mainPanelDel] lightsChanged];
  set_light( uistate.current_light );
  return;
}


LDEFINE(ui_panel, LVOID,
	    "(ui-panel       PANELNAME  {on|off} [ WINDOW ] )\n\
	Do or don't display the given user-interface panel.\n\
	Case is ignored in panel names.  Current PANELNAMEs are:\n\
		geomview	main panel\n\
		tools		motion controls\n\
		appearance	appearance controls\n\
		cameras		camera controls\n\
		lighting	lighting controls\n\
		command		command entry box\n\
		credits		geomview credits\n\
	By default, the \"geomview\" and \"tools\" panels appear when\n\
	geomview starts.  If the optional Window is supplied, a\n\
	\"position\" clause (e.g. (ui-panel obscure on { position xmin\n\
	xmax ymin ymax }) sets the panel's default position.  (Only\n\
	xmin and ymin values are actually used.)  A present but empty\n\
	Window, e.g.  \"(ui-panel obscure on {})\" causes interactive\n\
	positioning.")
{
  char *panelname;
  int index, on;
  struct panel *p;
  WindowStruct *ws=NULL;
  WnPosition wp;
  
  LDECLARE(("ui-panel", LBEGIN,
	    LSTRING, &panelname,
	    LKEYWORD, &on,
	    LOPTIONAL,
	    LWINDOW, &ws,
	    LEND));
  
  if((index = ui_name2panel(panelname)) == 0) {
    fprintf(stderr, "ui-panel: expected name of a panel, got \"%s\"\n",
	    panelname);
    return Lnil;
  }
  if (ws) {
    p = &panels[index];
    if(WnGet(ws->wn, WN_PREFPOS, &wp) > 0) {
      p->x0 = wp.xmin;
      p->y0 = wp.ymin;
    }
  }
  ui_showpanel( index, boolval("ui-panel", on) );
  return Lt;
}

LDEFINE(ui_center, LVOID,
       "(ui-center      ID)\n\
	Set the center for user interface (i.e. mouse) controlled\n\
	motions to object ID.")
{
  int oid;
  LDECLARE(("ui-center", LBEGIN,
	    LID, &oid,
	    LEND));

  if ( (oid == NOID) || (oid == CENTERID) ) {
    oid = uistate.centerid;
  } else {
    set_ui_center(oid);
  }
//  may_set_sinput(CenterIdInput, drawer_id2name(oid));
  return Lt;
}

LDEFINE(ui_freeze, LVOID,
        "(ui-freeze [on|off])\n\
        Toggle updating user interface panels. Ignored on the NeXT."
)
{
  int kw = -1;
  LDECLARE(("ui-freeze", LBEGIN,
        LOPTIONAL,
        LKEYWORD, &kw,
        LEND));
  return Lnil;
}


LDEFINE(ui_emodule_define, LVOID,
"(emodule-define  NAME  SHELL-COMMAND ...)\n\
	Define an external module called NAME, which then appears in the\n\
	external-module browser.  The SHELL-COMMAND string\n\
	is a UNIX shell command which invokes the module.\n\
	See emodule-run for discussion of external modules.")
{
  emodule *em;
  char *title, *cmd;
  LDECLARE(("emodule-define", LBEGIN,
	    LSTRING, &title,
	    LSTRING, &cmd,
	    LEND));

  em = ui_emodule_install(0, strdup(title), (PFI)emodule_run);
  em->text = strdup(cmd);
  [[NXApp mainPanelDel] buildExternalBrowser];
  return Lt;
}


LDEFINE(ui_emodule_start, LVOID,
"(emodule-start  NAME)\n\
	Starts the external module NAME, defined by emodule-define.\n\
	Equivalent to clicking on the corresponding module-browser entry.")
{
  emodule *em;
  int i;
  char *name;
  LDECLARE(("emodule-start", LBEGIN,
	    LSTRING, &name,
	    LEND));
  i = ui_emodule_index(name, &em);
  if(i < 0) return Lnil;
  (*em->func)(em);
  [[NXApp mainPanelDel] adjustExternalBrowser:i];
  return Lt;
}

/*-----------------------------------------------------------------------
 * Function:	ename_fix
 * Description:	fix up an emodule name for sorting
 * Args:	c: write fixed name here
 *		e: orig name
 *		n: length
 * Returns:	nothing
 * Author:	mbp
 * Date:	Thu Jan 28 11:06:44 1993
 * Notes:	converts names of the form  [D]XXX to XX0[D]
 *		where D is any string and XXX is any string and
 *		XX0 is that string with its last char replaced by '0'.
 *
 *		When sorting modules in the module  browser the
 *		"fixed" names are compared.  The fixed names are not
 *		stored anywhere or used otherwise.
 */
static void ename_fix(c,e,n)
char *c,*e;
int n;
{
  char num[80];
  int ni=0, ei=0;
  if (e[0] != '[') goto copy;
  ei = 1;
  num[0] = '\0';
  while (ei<n && e[ei++] != ']') num[ni++] = e[ei-1];
  if (ei == n) goto copy;
  num[ni] = '\0';
  strncpy(c,e+ei,n);
  c[strlen(c)-1] = '0';
  strcat(c,"[");
  strcat(c,num);
  strcat(c,"]");
  return;
 copy:
  strncpy(c,e,n);
  return;
}

static int emod_compare(emodule *e1,emodule *e2)
{
  char c1[80], c2[80];

  ename_fix(c1,e1->name,80);
  ename_fix(c2,e2->name,80);

  return streq(c1,c2);
}

LDEFINE(emodule_sort, LVOID,
"(emodule-sort)\n\
	Sorts the modules in the application browser alphabetically.")
{
  emodule *em;
  size_t nel, size;
  LDECLARE(("emodule-sort", LBEGIN,
	    LEND));

  em = VVEC(uistate.emod, emodule);
  nel = VVCOUNT(uistate.emod);
  size = sizeof(emodule);

  qsort(em, nel, size, 
       (int (*)(const void *, const void *))emod_compare);
  [[NXApp mainPanelDel] buildExternalBrowser];
  return Lt;
}


LDEFINE(ui_target, LVOID,
       "(ui-target      ID [yes|no])\n\
	Set the target of user actions (the selected line of the\n\
	target object browser) to ID.  The second argument specifies\n\
	whether to make ID the current object regardless of its type.\n\
	If \"no\", then ID becomes the current object of its type\n\
	(geom or camera).  The default is \"yes\".  This command may\n\
	result in a change of motion modes based on target choice.")
{
  DObject *obj;
  int oid, newtype;
  int immediate=YES_KEYWORD;

  LDECLARE(("ui-target", LBEGIN,
	    LID, &oid,
	    LOPTIONAL,
	    LKEYWORD, &immediate,
	    LEND));
  immediate = boolval("ui-target", immediate);
  newtype = TYPEOF(oid);

  if (oid == uistate.targetid ) return Lt;
  if ( !(obj = drawer_get_object(oid))
      || (oid == TARGETID) || (oid == CENTERID)
      || (oid == SELF) || (oid == UNIVERSE) || (oid == PRIMITIVE)) {
    [[NXApp toolPanelDel] setTargetField];
    return Lt;
  }
  if (immediate) {
    /* ui_highlight must be called before set_ui_target_id */
    ui_highlight(oid); 
    set_ui_target_id(oid); 
    if(uistate.lights_shown) light_edit_mode(0);
    [[NXApp mainPanelDel] adjustPickBrowser:oid]; 
    [[NXApp toolPanelDel] setTargetField];
  } else {
    /* immediate == NOIMMEDIATE: for cases like deleting geometry
     * with the command language when current target is a camera.
     * update targettype but don't change browser.
     */
    switch (newtype) {
    case T_GEOM: uistate.targetgeom = INDEXOF(oid); break;
    case T_CAM: break;
    }
  }
  switch (newtype) {
    /* If our target object is not a camera we want to make sure
     * that camera actions (i.e. some keyboard events, etc) are done
     * on the focus camera.
     */
    case T_GEOM: uistate.targetcam = INDEXOF(FOCUSID); break;
    case T_CAM: uistate.targetcam = INDEXOF(oid); break;
  }
  [[NXApp inspectorDel] targetChange];
  return Lt;
}

@implementation GVApp

- appWillInit:sender
{
  int i;
  CameraStruct cs;

  mgdevice_RI();
  init_geomview(NXArgc, NXArgv);
  parse_args(NXArgc, NXArgv);	/* Load command & data file(s) */

  cs.h = NULL;
  cs.cam = NULL;

  for(i = 0; i < nwins; i++)
    if (!dview[i]) gv_new_camera(NULL,&cs);

  ui_final_init();

  return self;
}

// add this as a categorie ??
- (int)selectedRow:sender
{
  int index=0, notused;
  id matrix=[sender matrixInColumn:0];
  [matrix getRow:&index andCol:&notused ofCell:[matrix selectedCell]];
  return index;	
}

- info:sender
{
  char versioninfo[256];
  if(!infoPanel) 
    [NXApp loadNibSection:"InfoPanel.nib" owner:self withNames:NO];
  [infoPanel makeKeyAndOrderFront:self];
  sprintf(versioninfo, "Version %5s Copyright 1994, The Geometry Center.",GEOMVIEW_VERSION);
  [compileString setStringValue:buildinfo];
  [versionString setStringValue:versioninfo];
  return self;
}

- panelMain:sender
{
  if (!mainPanelDel) 
    mainPanelDel = [[MainPanelDel alloc] init];
  if(sender!=NXApp) ui_showpanel(P_MAIN,1);
  return mainPanelDel;
}

- inspector:sender
{
  if (!inspectorDel) 
    inspectorDel = [[InspectorDel alloc] init];
  if(sender!=NXApp) ui_showpanel(P_INSPECTOR,1);
  return inspectorDel;
}

- preferences:sender
{
  if (!preferencesDel)
    preferencesDel = [[PreferencesDel alloc] init];
  else [preferencesDel display];
  return preferencesDel;
}

- toolPanel:sender
{
  if (!toolPanelDel)
    toolPanelDel = [[ToolPanelDel alloc] init];
  if(sender!=NXApp) ui_showpanel(P_TOOL,1);
  return toolPanelDel;
}

- appDidInit:sender
{
  DPSAddTimedEntry(0.1, RedrawSelf, (void *)self, 1);
  return self;
}

- deleteObj:sender
{
  gv_delete( uistate.targetid );
  return self;
}

- deleteCamera:sender
{
  if(CAMID(uistate.mousefocus)) gv_delete(CAMID(uistate.mousefocus));
  return self;
}

- halt:sender
{
  drawer_stop(NOID);
  return self;
}

- addCamara:sender
{
  CameraStruct cs;
  cs.h = NULL;
  cs.cam = NULL;
  gv_new_camera(NULL, &cs);
  return self;
}

- center:sender
{
  drawer_center(uistate.targetid);
  return self;
}

int isstdin(char *pathname)
{
  return (pathname!=NULL && strcmp(pathname+strlen(pathname)-2,"/-")==0);
}


- save:sender
{
  id savepanel = [SavePanel new];
  int success;
  int count = 0;
    
  int id = uistate.savewhat;
  DObject *dobj = drawer_get_object(id);
  int savetype;
  
  [savepanel setDelegate:self];
  [savepanel setTitle:"Save"];
  [savepanel setPrompt:"File:"];
  [saveTypeBrowser loadColumnZero];
  [savepanel setAccessoryView:[saveAccessory contentView]];

  if(dobj == NULL || dobj->name[0] == NULL)
    id = WORLDGEOM, dobj = (DObject*)dgeom[0];
//  fl_set_object_focus(SavePanel, ui_saveFileInput);
  [saveObject setStringValue: dobj->name[0]];
  if ([self selectedRow:saveTypeBrowser]==-1) 
    [[saveTypeBrowser matrixInColumn:0] selectCellAt:1 :0];
  savetype=(&save[[self selectedRow:saveTypeBrowser]])->flag;
 
  success = [savepanel runModalForDirectory:NULL file:NULL];
  
  return self;
} 

- print:sender
{
  DView *view;
  Camera *cam = NULL;
  WnWindow *win;
  Appearance *ap;
  mgcontext *ctx;
  char *rib_buffer;
  NXStream *tokens;
  int max;    /* max buffer size */
  FILE *f;
  char fname[64];
  
  sprintf(fname,"/tmp/renderXXXXXX");
  mktemp((char *)fname);
  strcat(fname,".rib");
  f = fopen(fname, "w");
  
  view = (DView *)drawer_get_object(FOCUSID);
  if(!ISCAM(FOCUSID) || view == NULL) {
    OOGLError(1, "rib print routine: bad view!");
    return self;
  }
  mgctxselect(view->mgctx);
  mgctxget(MG_CAMERA, &cam);
  /* Copy so that changed flags are set */
  ap = ApCopy(mggetappearance(), NULL);
  mgctxget(MG_WINDOW, &win);
  if(cam == NULL || ap == NULL || win == NULL) {
    OOGLError(1, "rib print routine: trouble, %x %x %x", cam,ap,win);
    return self;
  }
  
  mgdevice_RIB();
  ctx = mgctxcreate(MG_CAMERA, cam,
		    MG_APPEAR, ap,
		    MG_WINDOW, win,
		    MG_BACKGROUND, &view->backcolor,
		    MG_SPACE, spaceof(WORLDGEOM),
		    MG_RIBFORMAT, MG_RIBASCII,
		    MG_RIBLINEMODE, rman.line,
		    MG_RIBFILE, f,
		    MG_RIBDISPLAY, rman.display,
		    MG_RIBDISPLAYNAME, "geom.tiff",
		    MG_RIBBACKING, rman.background,
		    MG_END);
  
  mgworldbegin();
  GeomDraw(view->Item);
  mgworldend();
  mgrib_terminatebuffer();

  //printCam = [[PrintCam alloc] init];
  
  /* following for stream support which doesn't seem to work
  mgrib_tokenbuffer(&rib_buffer, &max);
  tokens = NXOpenMemory((const char *)rib_buffer, max, NX_READONLY);
  [printCam setTokenStream:tokens];
  */
  
  mgrib_flushbuffer();
  fclose(f);
  [printCam setTokenFile:fname];
  [printCam printPSCode:self];
  
  
  mgctxdelete(ctx);
  ApDelete(ap);
  mgctxselect(view->mgctx);	/* Revert to previous device */
  
  /* NXCloseMemory(tokens,NX_SAVEBUFFER); */
  unlink(fname);
  
  return self;
}
  
void panelplace(id window, int index, FILE *outf)
{
   NXRect r;
   if(window == NULL) return;
   [window getFrame:&r];
   fprintf(outf, "  (ui-panel %s %s { position %d %d %d %d })\n",
   	panels[index].name,
	panels[index].shown ? "on" : "off",
	(int)r.origin.x, (int)(r.origin.x+r.size.width)-1,
	(int)r.origin.y, (int)(r.origin.y+r.size.height)-1);
}

int ui_savepanels(char *fname)
{
  FILE *f = fopen(fname, "w");
  if(f == NULL) return 1;
  fprintf(f, " (progn \n");
  panelplace([[NXApp mainPanelDel] window], P_MAIN, f);
  panelplace([[NXApp toolPanelDel] window], P_TOOL, f);
  panelplace([[NXApp inspectorDel] window], P_INSPECTOR, f);
  panelplace([[NXApp commandText] window], P_COMMANDS, f);
  fprintf(f, " )\n");
  return fclose(f);
}

- (BOOL)panelValidateFilenames:sender
{
  char *fname = (char *)[sender filename];
  char *object = (char *)[saveObject stringValue];
  int savetype = [self selectedRow:saveTypeBrowser];

  int oid; // err;
  Pool *p;
  struct saveops *sp;

  if(object == NULL || *object == '\0' || savetype < 0)
    return NO;
  if((oid = suitable_id(savetype, &object)) == NOID) {
    [saveObject setStringValue:"?"];
    return NO;
  }

  sp = &save[savetype];
  switch(sp->special) {
  case SAVE_WIO:
    p = PoolStreamTemp(fname, isstdin(fname)?stdout:NULL, 1, sp->ops);
    if(p == NULL) {
	fprintf(stderr, "Can't open output: %s: %s\n", fname, sperror());
	return NO;
    }
    worldio(sp->ops, p, sp->flag, oid);
    if(PoolOutputFile(p) != stdout) PoolClose(p);
    PoolDelete(p);
    break;
  case SAVE_RMan:
    gv_rib_display(sp->flag, fname);
    gv_rib_snapshot(oid, fname);
    break;
  case SAVE_SNAP:
      gv_snapshot(oid, fname, "ppm", 0, 0);
//    err = gv_snapshot(oid, fname);
//    fprintf(stderr, "\7");	/* Ring bell when done */
//    if(err)
//	return NO;
    break;
  case SAVE_PANELS:
    if(ui_savepanels(fname)) return NO;
    break; 
  }
  return YES;
}

- open:sender
{
  id openpanel = [OpenPanel new];
  int success;
  char **dirp = getfiledirs();
  static int first = 1;
  int count = 0;
  
  if(first) {
      [pathBrowser setDelegate:self];
      [pathBrowser loadColumnZero];
      [pathBrowser setTarget:self];
      [pathBrowser setAction:@selector(pathSelected:)];
      [pathBrowser setTitle:"Path List" ofColumn:0];
  } else [pathBrowser loadColumnZero];
      
  [openpanel setTitle:"Open an Object"];
  [openpanel setPrompt:"File:"];
  [openpanel setAccessoryView:pathBrowser];
  
  if(first) {
      first = 0;
      success = [openpanel runModalForDirectory:dirp[0] file:NULL];
  } else {
      success = [openpanel runModal];
  }
  if (success) loadfile((char *)[openpanel filename],&CommandOps, 1);;

  return self;
} 

- (BOOL)doesTreatFilePackagesAsDirectories
{
  return NO;
}

- openBundle:sender
{
  NXBundle *test;
  id openpanel = [OpenPanel new];
  double tmp=0;
/*  strspn("boo","bla");
  erf(tmp);
  tanh(tmp);
  asinh(tmp);
  atanh(tmp);
*/
  [openpanel runModal];
  test = [[NXBundle alloc] initForDirectory:(char *)[openpanel filename]];
  
  [[[test classNamed:"TestObject"] alloc] init];
  
  return self;
}

- pathSelected:sender
{
  char **dirp = getfiledirs();
  id openpanel = [OpenPanel new];
  
  [openpanel setDirectory:dirp[[[sender matrixInColumn:0] selectedRow]]];
  
  return self;
}

- (BOOL)textWillEnd:sender
{
  const char *str=[commandText stringValue];
  if (str && *str)
    comm_object((char *)str, &CommandOps, NULL, NULL, COMM_NOW);
  return NO; 
}

- textDidEnd:(id)sender :endChar:(unsigned short)whyEnd
{
  return self;
}

- (BOOL)appAcceptsAnotherFile:sender
{
     return YES;
}

-(int)app:sender openFile:(const char *)fname type:(const char *)type
{
  loadfile((char *)fname,&CommandOps, 1);
  return 0;
}

- makeKeyResponderFor:window
{    
    id next;
    
    [window addToEventMask:NX_KEYDOWNMASK];

    next = [window firstResponder];
    if(next) {
	while([next nextResponder]) next = [next nextResponder];
	[next setNextResponder:self];
    } else [window makeFirstResponder:self];
    
    return self;
}

- keyDown:(NXEvent *)ev
{
  if(ev->data.key.charCode==27) {
    NXBeep();
    return self; // Trap the esc!!
  }
  if(ev->data.key.charCode==22 /* control-v */)
  {
      mgri_debugcontext();
      return self;
  }
  sendevent(ev->data.key.charCode, 1,ev);
  return self;
}

- keyUp:(NXEvent *)theEvent
{
  return self;
}

- mainPanelDel {return mainPanelDel;}
- inspectorDel {return inspectorDel;}
- preferencesDel {return preferencesDel;}
- toolPanelDel {return toolPanelDel;}
- commandText {return commandText;}
- menuCloseCamera {return menuCloseCamera;}

- saveTypeBrowser:sender
{
   char *object = (char *)[saveObject stringValue];
   if(suitable_id([self selectedRow:sender], &object) != NOID)
       [saveObject setStringValue:object];
  return self;
}

- (int)browser:sender fillMatrix:theMatrix inColumn:(int)column
{
  char **dirp = getfiledirs();
  int i;
  int count = 0;
  NXBrowserCell *bCell;

  if(sender==pathBrowser) {
    /* remove old fields */
    /* [theMatrix removeColAt:0 andFree:YES]; */
    
    /* reconstruct new fields */
    for(i=0;dirp[i]!=NULL;i++) {
    	char *start;

	/* cut off anything prior to 'Geomview.app' if it's there */
	/* but ...DO WE REALLY WANT THIS? smw */
	start = strstr(dirp[i], "Geomview.app");
	if(start == NULL) start = dirp[i];
	[theMatrix addRow];
	bCell=[theMatrix cellAt:count :0];
	[bCell setStringValue:start];
	[bCell setLoaded:YES];
	[bCell setLeaf:YES];
	count++;
    }
    [theMatrix sizeToCells];
    
    return count;
  }
  else if(sender==saveTypeBrowser) {
    for(i=0;i<COUNT(save);i++) {
      [theMatrix addRow];
      bCell=[theMatrix cellAt:i :0];
      [bCell setStringValue:save[i].name];
      [bCell setLoaded:YES];
      [bCell setLeaf:YES];
    }
    return COUNT(save);    
  }
  return 0;
}

@end
