/*
    Copyright (C) 1996-1999  Ulric Eriksson <ulric@edu.stockholm.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston,
    MA 02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <signal.h>
#include <unistd.h>
#include <locale.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include "../config.h"
#include "cmalloc.h"
#include "common.h"

/*#define DEBUG
*/
#include "traceme.h"

/* ---
common.c

Device independent functions for all three programs.
--- */

/* ---
Figure out the names of directories
*/

char *libexecdir; /* where plugins are; default /usr/local/libexec/siag */
char *datadir;	  /* where Scheme files are; default /usr/local/share/siag */
char *docdir;	  /* where online docs are; default /usr/local/doc/siag */
char *siaghelp;	  /* what displays html; default $docdir/common/siaghelp */
char *siag_basedir;	/* $HOME/.siag */
char *version;	  /* version string */

void common_init(char *fmt)
{
	char *home = getenv("SIAGHOME");
	char *docs = getenv("SIAGDOCS");
	char *help = getenv("SIAGHELP");
	char *lang;
	char p[1024];

	if (home) {
		libexecdir = datadir = cstrdup(home);
	} else {
		libexecdir = LIBEXECDIR;
		datadir = DATADIR;
	}
	if (docs) {
		docdir = cstrdup(docs);
	} else {
		docdir = DOCDIR;
	}
	if (help) {
		siaghelp = cstrdup(help);
	} else {
		sprintf(p, "%s/common/siaghelp", docdir);
		siaghelp = cstrdup(p);
	}
	if (fmt) {
		sprintf(p, fmt, VERSION_NO);
	} else {
		sprintf(p, "Siag Office %s", VERSION_NO);
	}
	version = cstrdup(p);
	home = getenv("HOME");
	if (home == NULL) home = "";
	sprintf(p, "%s/.siag", home);
	siag_basedir = cstrdup(p);
	mkdir(siag_basedir, 0700);

	/* i18n */
	lang = getenv("LC_ALL");
	if (lang == NULL) lang = getenv("LC_MESSAGES");
	if (lang == NULL) lang = getenv("LANG");
	if (lang) {
		sprintf(p, "%s/common/dictionary.%s", datadir, lang);
		load_dictionary(p);
	}
}

/* ---
Strip trailing newline, if there is one
*/

void chomp(char *p)
{
	if ((p = strchr(p, '\n'))) *p = '\0';
}

/* ---
Check if a string should be interpreted as a positive answer.
*/

int itsayes(char *p)
{
	return !cstrcasecmp(p, "yes")
		|| !cstrcasecmp(p, "1")
		|| !cstrcasecmp(p, "on")
		|| !cstrcasecmp(p, translate("yes"))
		|| !cstrcasecmp(p, translate("on"));
}

/* ---
Case insensitive compare
*/

int cstrcasecmp(const char *p, const char *q)
{
	int c;

	while (!(c = toupper(*p)-toupper(*q)) && *p) {
		p++;
		q++;
	}
	return c;
}

/* ---
Case insensitive compare, length limited
*/

int cstrncasecmp(const char *p, const char *q, size_t n)
{
	size_t i = 0;
	int c = 0;

	while ((i < n) && !(c = toupper(*p)-toupper(*q)) && *p) {
		p++;
		q++;
		i++;
	}
	return c;
}

/* ---
Delete all temp directories whose processes are no longer around.
NB: Doesn't do anything at all.
*/

void stale_dirs(char *dn)
{
}

/* a little bookkeeping to keep track of temp files and make sure they
   are deleted when child processes terminate */

static int ndeletia;
static struct {
	long p;
	char *fn;
} *deletia;

/* ---
associate a temp file with a process
*/

void deletia_add(long p, char *fn)
{
	int i;

	for (i = 0; i < ndeletia; i++)
		if (deletia[i].fn == NULL) break;
	if (i == ndeletia) {
		ndeletia++;
		deletia = crealloc(deletia, ndeletia*sizeof *deletia);
	}
	deletia[i].p = p;
	deletia[i].fn = cstrdup(fn);
}

/* ---
mark all files related to a process to be deleted later
p == 0 marks all files
*/

void deletia_mark(long p)
{
	int i;

	for (i = 0; i < ndeletia; i++) {
		if (p == 0 || p == deletia[i].p)
			deletia[i].p = 0;
	}
}

/* ---
remove files that have previously been marked
*/

void deletia_reap(void)
{
	int i;

	for (i = 0; i < ndeletia; i++) {
		if (deletia[i].p == 0 && deletia[i].fn) {
			remove(deletia[i].fn);
			cfree(deletia[i].fn);
			deletia[i].fn = NULL;
		}
	}
}

/* ---
Signal handler for dead children.
*/

void waitforchild(int i)
{
	pid_t p;
	long lp;

	p = waitpid(-1, NULL, WNOHANG);
	lp = (long)p;
	if (p > 0) deletia_mark(lp);
	signal(SIGCHLD, waitforchild);
}

#define TAR_MAGIC "ustar"

/* ---
Check if a file is a tar archive and if an index file is present.
Returns 1 for success.
*/

int tryuntar(char *fn, char *index)
{
	FILE *fp;
	char b[1024];
	int result, n;
	struct stat statbuf;

	fp = fopen(fn, "r");
	if (!fp) {
		return 0;
	}
	n = fread(b, 1, 300, fp);
	if (n != 300) {
		fclose(fp);
		return 0;
	}
	if (strncmp(b+257, TAR_MAGIC, strlen(TAR_MAGIC))) {
		fclose(fp);
		return 0;
	}
	fclose(fp);
	sprintf(b,
		"mkdir -p %s/untar;"
		"cat %s 2> /dev/null |(cd %s/untar;tar xf - %s) 2> /dev/null",
		siag_basedir, fn, siag_basedir, index);
	system(b);
	sprintf(b, "%s/untar/%s", siag_basedir, index);
	result = !stat(b, &statbuf);
	sprintf(b, "rm -rf %s/untar", siag_basedir);
	system(b);
	return result;
}

/* ---
A function that doesn't do anything. Can be used to set break points
during debugging.
*/

void dummy_fun(void)
{
	;
}

static struct {
	char *key, *xl;
} *dict = NULL;

static long nw = 0;

/* ---
Load the message string dictionary from $(SIAGHOME)/common/dictionary.$(LANG)
*/

void load_dictionary(char *fn)
{
        FILE *fp;
        char b[1024], *p, *q;

	/* convert everything after language code to upper case */
	p = strchr(fn, '_');
	if (p) {
		q = p;
		while (*q) {
			*q = toupper(*q);
			q++;
		}
	}

	if (p) {
		q = strchr(p, '.');
	} else {
		q = NULL;
	}

	fp = fopen(fn, "r");
	
	if (!fp && q) {
		/* try abridged file name */
		*q = '\0';
		fp = fopen(fn, "r");
	}
	if (!fp && p) {
		/* try abridged file name */
		*p = '\0';
		fp = fopen(fn, "r");
	}
        if (!fp) {
#ifdef DEBUG
                fprintf(stderr, "Can't open dictionary\n");
#endif
                return;
        }
        while (fgets(b, sizeof b, fp)) {
                chomp(b);
                p = strchr(b, '\t');
                if (!p) continue;
                *p++ = 0;
                p += strspn(p, "\t");
                dict = crealloc(dict, (nw+1)*(sizeof dict[0]));
                dict[nw].key = cstrdup(b);
                dict[nw++].xl = cstrdup(p);
        }
}

/* ---
Translate a message string. If no match is found, the key is returned.
By using English keys, an application can keep working even when the
dictionary is incomplete.
990113: tree search through sorted list
*/

char *translate(char *key)
{
	long lower = 0, upper = nw-1;
        long i;
	int d;

	while (lower <= upper) {
		i = (lower+upper)/2;
		d = strcmp(key, dict[i].key);
		if (d == 0) return dict[i].xl;
		else if (d > 0) lower = i+1;
		else upper = i-1;
	}
        return key;
}


