/* type exit to exit */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "lisp.h"


static NODE *heap;
static NODE **free_ptr;
static NODE **stack;
static int *a;
static int *b;


static NODE _nil;
NODE *nil, *t, *arglist;
static NODE *eof, *lparen, *rparen, *period;
static NODE *quote, *quote_char;
static NODE *sum, *product, *power, *tensor, *im;
static NODE *symbol_table[128];
static NODE *nullstr;
static int free_count;
static int tos;
static char buf[BUFLEN + 1];

static int quit=0;

static FILE *infile;
static FILE *outfile;

static unsigned int _stklen = 10000; /* stack size */


int hash(unsigned char *s);
void gc(void);
NODE *append(NODE *p1, NODE *p2);
NODE *eval_append(NODE *p);
NODE *lisp_read();
NODE *ratom();
NODE *read_list();
NODE *cons(NODE *p1, NODE *p2);
NODE *new_symbol(char* p);
NODE *new_string();
NODE *new_number(int a, int b);
NODE *eval_user_function(NODE *p);
NODE *eval_return(NODE *p);
NODE *eval_symbol();
NODE *subst(NODE *p1, NODE *p2, NODE *p3);
NODE *scan_number(char*p);
NODE *read_symbol();
NODE *read_string(char* buf);
NODE *ksum(int n);
NODE *expand(int);
NODE *kproduct(int n);
void prin_list(NODE *p);


//forward declarations
void push_vars(NODE *p);
void pop_vars(NODE *p);
void untag(NODE *p);
void print(NODE *p);
void prin(NODE *p);
void terpri();
int eos(int c);
int equal(NODE *p1, NODE *p2);
int lessp(NODE *p1, NODE *p2);
int isnum(NODE *p);
int iszero(NODE *p);
int isone(NODE *p);
int gcd(int a, int b);
void multiply(int *pa, int *pb, int a, int b);
void add(int *pa, int *pb, int a, int b);



void read_eval_print(void)
{
    NODE *p;
    printf("\n> ");
    p = lisp_read();
    push(p);
    p = eval(p);
    pop();
    print(p);
}

/* the temp stack keeps intermediate results from garbage collection */

void push(NODE *p)
{
    if (tos == MAX)
    {
        printf("temp stack overflow\n");
        exit(1);
    }
    stack[tos++] = p;
}

NODE *pop()
{
    if (tos == 0)
    {
        printf("temp stack underflow\n");
        exit(1);
    }
    return stack[--tos];
}

NODE *lookup(char *s)
{
    NODE *p;
    int x;
    if (strcmp(s, "nil") == 0)
        return nil;
    x = hash((unsigned char*)s);
    p = symbol_table[x];
    while (p != nil)
    {
        if (strcmp(p->head->s, s) == 0)
            return p->head;
        p = p->tail;
    }
    p = new_symbol(s);
    p = cons(p, symbol_table[x]);
    symbol_table[x] = p;
    return p->head;
}

int hash(unsigned char *s)
{
	int x = 0;
	while (*s)
		x ^= *s++;
	return x & 0x7f;
}

NODE *eval_nil(NODE *p)
{
    return nil;
}

NODE *eval_number(NODE *p)
{
    return p;
}

NODE *eval_string(NODE *p)
{
    return p;
}

/* this is the main entry point for evaluating functions */

/* in eval_pair, we call f of the head of the list p */

/* example: in the list (cons a b), f of the head of the list is eval_cons */

NODE *eval_pair(NODE *p)
{
    return (p->head->f)(p);
}

int cmp(const void*pp,const void*qq)
{
    NODE **p=(NODE**)pp, **q=(NODE**)qq;
    if (equal(*p, *q))
        return 0;
    else if (lessp(*p, *q))
        return -1;
    else
        return 1;
}

//TODO what is the difference between plus, sum and sum1 ?
NODE *eval_sum(NODE *p)
{
    int h;
    NODE *q;

    h = tos;

    p = p->tail;

    while (p != nil)
    {
        q = eval(p->head);
        if (q->head == sum)
        {
            q = q->tail;
            while (q != nil)
            {
                push(q->head);
                q = q->tail;
            }
        } else
            push(q);
        p = p->tail;
    }
    return ksum(tos - h);
}

NODE *eval_sum1(NODE *p)
{
    int h;
    NODE *q;

    h = tos;

    p = eval(arg1);

    while (p != nil)
    {
        q = p->head;
        if (q->head == sum)
        {
            q = q->tail;
            while (q != nil)
            {
                push(q->head);
                q = q->tail;
            }
        }
        else
            push(q);
        p = p->tail;
    }
    return ksum(tos - h);
}

/* in eval_symbol below, p can point to a list or symbol */

/* how do we get to eval_symbol when p points to a list? */

/* via eval_pair above */

/* example: suppose x is defined by (setq x '(plus arg1 arg2)) */

/* now we consider the various ways x can be evaluated */

/* to evaluate the symbol x, eval_symbol is called */

/* the binding (plus arg1 arg2) is returned unevaluated */

/* to evaluate the list (x 1 2), eval_pair is called */

/* eval_pair calls eval_symbol via (p->head->f) */

/* eval_symbol calls eval_user_function because p is the list (x 1 2) */

/* eval_user_function evaluates (plus arg1 arg2) and returns 3 */

/* so, at the user prompt, if we type x we get (plus arg1 arg2) and if we */

/* type (x 1 2) we get 3 */

/* why is the function check done here and not in eval_pair? */

/* because eval_pair is called for every function, i.e., cons, etc. */

/* we only need to check user defined symbols so we do the check here */

NODE *eval_symbol(NODE *p)
{
    if (p->f == eval_pair)
        return eval_user_function(p);
    else
        return p->binding;
}


NODE *eval_user_function(NODE *p)
{
    push(arglist);
    arglist = eval_args(p->tail);
    if (p->head->binding->f == eval_pair)
        p = eval(p->head->binding);
    else
        /* this is the case where the function name is undefined */
        /* return the same function with args evaluated */
        p = cons(p->head, arglist);
    arglist = pop();
    return p;
}

NODE *eval_args(NODE *p)
{
    NODE *p1, *p2;
    if (p != nil)
    {
        p1 = eval(p->head);
        push(p1);
        p2 = eval_args(p->tail);
        pop();
        p = cons(p1, p2);
    }
    return p;
}

NODE *eval_and(NODE *p)
{
    while (p->tail != nil)
    {
        p = p->tail;
        if (eval(p->head) == nil)
            return nil;
    }
    return t;
}

NODE *eval_append(NODE *p)
{
    NODE *p1, *p2;
    p1 = eval(arg1);
    push(p1);
    p2 = eval(arg2);
    push(p2);
    p = append(p1, p2);
    pop();
    pop();
    return p;
}

NODE *append(NODE *p1, NODE *p2)
{
    if (p1 == nil)
        return p2;
    else
        return cons(p1->head, append(p1->tail, p2));
}

/*TODO
NODE *eval_arg(NODE *p)
{
    return arglist->head;
}
*/

NODE *eval_arg1(NODE *p)
{
    return arglist->head;
}

NODE *eval_arg2(NODE *p)
{
    return arglist->tail->head;
}

NODE *eval_arg3(NODE *p)
{
    return arglist->tail->tail->head;
}

NODE *eval_arg4(NODE *p)
{
    return arglist->tail->tail->tail->head;
}

NODE *eval_arg5(NODE *p)
{
    return arglist->tail->tail->tail->tail->head;
}

NODE *eval_arg6(NODE *p)
{
    return arglist->tail->tail->tail->tail->tail->head;
}

NODE *eval_arg7(NODE *p)
{
    return arglist->tail->tail->tail->tail->tail->tail->head;
}

NODE *eval_arg8(NODE *p)
{
    return arglist->tail->tail->tail->tail->tail->tail->tail->head;
}

NODE *eval_arglist(NODE *p)
{
    return arglist;
}

NODE *eval_atom(NODE *p)
{
    if (eval(arg1)->f == eval_pair)
        return nil;
    else
        return t;
}

NODE *eval_head(NODE *p)
{
    return eval(arg1)->head;
}

NODE *eval_tail(NODE *p)
{
    return eval(arg1)->tail;
}

/*TODO
 NODE *eval_caar(NODE *p)
{
    return eval(arg1)->head->head;
}

NODE *eval_cadr(NODE *p)
{
    return eval(arg1)->tail->head;
}

NODE *eval_cdar(NODE *p)
{
    return eval(arg1)->head->tail;
}

NODE *eval_cddr(NODE *p)
{
    return eval(arg1)->tail->tail;
}

NODE *eval_caaar(NODE *p)
{
    return eval(arg1)->head->head->head;
}

NODE *eval_caadr(NODE *p)
{
    return eval(arg1)->tail->head->head;
}

NODE *eval_cadar(NODE *p)
{
    return eval(arg1)->head->tail->head;
}

NODE *eval_caddr(NODE *p)
{
    return eval(arg1)->tail->tail->head;
}

NODE *eval_cdaar(NODE *p)
{
    return eval(arg1)->head->head->tail;
}

NODE *eval_cdadr(NODE *p)
{
	return eval(arg1)->tail->head->tail;
}

NODE *eval_cddar(NODE *p)
{
	return eval(arg1)->head->tail->tail;
}

NODE *eval_cdddr(NODE *p)
{
	return eval(arg1)->tail->tail->tail;
}
*/

NODE *eval_while(NODE *p)
{
    p = p->tail;
    while (eval(p->head) != nil)
        eval(p->tail->head);
    return nil;
}

NODE *eval_cond(NODE *p)
{
    while (p->tail != nil)
    {
        p = p->tail;
        if (eval(p->head->head) != nil)
            return eval(p->head->tail->head);
    }
    return nil;
}

NODE *eval_cons(NODE *p)
{
    NODE *p1, *p2;
    p1 = eval(arg1);
    push(p1);
    p2 = eval(arg2);
    pop();
    return cons(p1, p2);
}

/* just like setq except arg2 is not evaluated */

NODE *eval_define(NODE *p)
{
    arg1->binding = arg2;
    return arg1;
}

NODE *eval_eq(NODE *p)
{
    NODE *p1, *p2;
    p1 = eval(arg1);
    push(p1);
    p2 = eval(arg2);
    pop();
    if (p1 == p2)
        return t;
    else
        return nil;
}

NODE *eval_equal(NODE *p)
{
    NODE *p1, *p2;
    p1 = eval(arg1);
    push(p1);
    p2 = eval(arg2);
    pop();
    if (equal(p1, p2))
        return t;
    else
        return nil;
}

NODE *eval_eval(NODE *p)
{
    p = eval(arg1);
    push(p);
    p = eval(p);
    pop();
    return p;
}

NODE *eval_exit(NODE *p)
{
    quit=1;
    return nil;
}

NODE *eval_expt(NODE *p)
{
    int a, b, i, j;
    NODE *p1, *p2;

    p1 = eval(arg1);
    push(p1);
    p2 = eval(arg2);
    pop();

    if (p1->f != eval_number || p2->f != eval_number)
        return nil;

    /* anything to zero power is one */

    if (p2->numer == 0)
        return new_number(1, 1);

    /* zero to any power (other than zero) is zero */

    if (p1->numer == 0)
        return p1;

    /* one to any power (other than zero) is one */

    if (p1->numer == 1 && p1->denom == 1)
        return p1;

    /* error if power is not integer */

    if (p2->denom != 1)
        return nil;

    j = p2->numer;

    if (j < 0)
        j = -j;

    a = 1;
    b = 1;

    for (i = 0; i < j; i++)
    {
        a *= p1->numer;
        b *= p1->denom;
    }

    if (p2->numer > 0)
        return new_number(a, b);
    else
        return new_number(b, a);
}

NODE *eval_fixp(NODE *p)
{
    p = eval(arg1);
    if (p->f == eval_number && p->denom == 1)
        return t;
    else
        return nil;
}

NODE *eval_gc(NODE *p)
{
    gc();
    return new_number(free_count, 1);
}

NODE *eval_greaterp(NODE *p)
{
    NODE *p1, *p2;
    p1 = eval(arg1);
    push(p1);
    p2 = eval(arg2);
    pop();
    if (lessp(p2, p1))
        return t;
    else
        return nil;
}

NODE *eval_goto(NODE *p)
{
    return p;
}

NODE *eval_lessp(NODE *p)
{
    NODE *p1, *p2;
    p1 = eval(arg1);
    push(p1);
    p2 = eval(arg2);
    pop();
    if (lessp(p1, p2))
        return t;
    else
        return nil;
}

NODE *eval_list(NODE *p)
{
    NODE *p1, *p2;
    p = p->tail;
    if (p != nil)
    {
        p1 = eval(p->head);
        push(p1);
        p2 = eval_list(p);
        pop();
        p = cons(p1, p2);
    }
    return p;
}

NODE *eval_load(NODE *p)
{
    FILE *f = infile;
    p = eval(arg1);
    if (p->f != eval_string)
        return nil;
    infile = fopen(p->s, "r");
    if (infile == NULL) {
        infile = f;
        printf("fopen %s returns NULL\n", p->s);
        return nil;
    }
    push(p);
    for (;;)
    {
        p = lisp_read();
        if (p == eof)
            break;
        push(p);
        eval(p);
        pop();
    }
    infile = f;
    return pop();
}

NODE *eval_nhead(NODE *p)
{
    int n = 0;
    p = eval(arg1);
    while (p != nil)
    {
        n++;
        p = p->tail;
    }
    return new_number(n, 1);
}

NODE *eval_not(NODE *p)
{
    if (eval(arg1) == nil)
        return t;
    else
        return nil;
}

NODE *eval_null(NODE *p)
{
    if (eval(arg1) == nil)
        return t;
    else
        return nil;
}

NODE *eval_numberp(NODE *p)
{
    if (eval(arg1)->f == eval_number)
        return t;
    else
        return nil;
}

NODE *eval_onep(NODE *p)
{
    p = eval(arg1);
    if (p->f == eval_number && p->numer == 1 && p->denom == 1)
        return t;
    else
        return nil;
}

NODE *eval_or(NODE *p)
{
    while (p->tail != nil)
    {
        p = p->tail;
        if (eval(p->head) != nil)
            return t;
    }
    return nil;
}

NODE *eval_pheads(NODE *p)
{
    p = eval(arg1);
    while (p != nil)
    {
        print(p->head);
        p = p->tail;
    }
    return nullstr;
}

NODE *eval_plus(NODE *p)
{
    int a, b;
    NODE *q;

    p = p->tail;
    q = eval(p->head);
    if (q->f != eval_number)
        return nil;
    a = q->numer;
    b = q->denom;

    while (p->tail != nil)
    {
        p = p->tail;
        q = eval(p->head);
        if (q->f != eval_number)
            return nil;
        a = a * q->denom + b * q->numer;
        b *= q->denom;
    }
    return new_number(a, b);
}


void add(int *pa, int *pb, int a, int b)
{
    int sgn, t, x, y;

    a = *pa * b + *pb * a;
    b = *pb * b;

    if (a == 0)
	b = 1;
    else
    {
        sgn = 1;
        if (a < 0)
        {
            a = -a;
	    sgn *= -1;
        }
        x = a;
        y = b;
        while (y)
        {
	    t = x % y;
	    x = y;
	    y = t;
	}
	a = sgn * a / x;
	b /= x;
    }

    *pa = a;
    *pb = b;
}

NODE *add3(NODE *p, NODE *q)
{
    int a, b, sgn, t, x, y;

    a = p->numer * q->denom + p->denom * q->numer;
    b = p->denom * q->denom;

    if (a == 0)
	b = 1;
    else
    {
        sgn = 1;
        if (a < 0)
        {
	    a = -a;
	    sgn *= -1;
	}
	x = a;
	y = b;
        while (y)
        {
	    t = x % y;
	    x = y;
	    y = t;
	}
	a = sgn * a / x;
        b /= x;
    }
    return new_number(a, b);
}

NODE *eval_prin(NODE *p)
{
    NODE *x = nil;
    while (p->tail != nil)
    {
        p = p->tail;
        x = eval(p->head);
        prin(x);
    }
    return x;
}

NODE *eval_print(NODE *p)
{
    NODE *q = nil;
    while (p->tail != nil)
    {
        p = p->tail;
        q = eval(p->head);
        print(q);
    }
    return q;
}

/* (product a b c) */

NODE *eval_product(NODE *p)
{
    int flag = 0, n = 0;
    NODE *q;
    p = p->tail;
    while (p != nil)
    {
        q = eval(p->head);
        if (q->head == product)
        {
            q = q->tail;
            while (q != nil)
            {
                if (q->head->head == sum)
                    flag = 1;
                push(q->head);
                n++;
                q = q->tail;
            }
        }
        else
        {
            if (q->head == sum)
                flag = 1;
            push(q);
            n++;
        }
        p = p->tail;
    }
    if (flag)
        return expand(n);
    else
        return kproduct(n);
}

/* (product '(a b c)) */

NODE *eval_product1(NODE *p)
{
    int flag = 0, n = 0;
    NODE *q;
    p = eval(p->tail->head);
    while (p != nil)
    {
        q = p->head;
        if (q->head == product)
        {
            q = q->tail;
            while (q != nil)
            {
                if (q->head->head == sum)
                    flag = 1;
                push(q->head);
                n++;
                q = q->tail;
            }
        }
        else
        {
            if (q->head == sum)
                flag = 1;
            push(q);
            n++;
        }
        p = p->tail;
    }
    if (flag)
        return expand(n);
    else
        return kproduct(n);
}

NODE *eval_prog(NODE *p)
{
    NODE *q, *u, *x;
    push_vars(p->tail->head);
    x = nil;
    q = p->tail->tail;
    while (q != nil)
    {
        u = eval(q->head);

        if (u->head->f == eval_return)
        {
            x = eval(u->tail->head);
            break;
        }
        else if (u->head->f == eval_goto)
        {
            q = p->tail->tail;
            while (q != nil && q->head != u->tail->head)
                q = q->tail;
        } else
            q = q->tail;
    }
    pop_vars(p->tail->head);
    return x;
}

void push_vars(NODE *p)
{
    while (p != nil)
    {
        push(p->head->binding);
        p->head->binding = nil;
        p = p->tail;
    }
}

/* recursively descend the var list first because pop order is the reverse of push */

void pop_vars(NODE *p)
{
    if (p == nil)
        return;
    pop_vars(p->tail);
    p->head->binding = pop();
}

NODE *eval_quote(NODE *p)
{
    return arg1;
}

NODE *eval_quotient(NODE *p)
{
    int a, b;
    NODE *q;

    p = p->tail;
    q = eval(p->head);
    if (q->f != eval_number)
        return nil;
    a = q->numer;
    b = q->denom;

    while (p->tail != nil)
    {
        p = p->tail;
        q = eval(p->head);
        if (q->f != eval_number)
            return nil;
        a *= q->denom;
        b *= q->numer;
    }
    return new_number(a, b);
}

NODE *eval_return(NODE *p)
{
    return p;
}

NODE *eval_reverse(NODE *p)
{
    NODE *t = nil;
    p = eval(arg1);
    push(p);
    while (p != nil)
    {
        t = cons(p->head, t);
        p = p->tail;
    }
    pop();
    return t;
}

NODE *eval_setq(NODE *p)
{
    arg1->binding = eval(arg2);
    return arg1;
}

NODE *eval_subst(NODE *p)
{
    NODE *p1, *p2, *p3, *temp;

    p1 = eval(arg1);
    push(p1);

    p2 = eval(arg2);
    push(p2);

    p3 = eval(arg3);
    push(p3);

    temp = subst(p1, p2, p3);

    pop();
    pop();
    pop();

    return temp;
}

/* substitute p1 for p2 in p3 */

NODE *subst(NODE *p1, NODE *p2, NODE *p3)
{
    NODE *p4, *p5;

    if (equal(p2, p3))
        return p1;
    else if (p3->f == eval_pair)
    {
        p4 = subst(p1, p2, p3->head);
        push(p4);
        p5 = subst(p1, p2, p3->tail);
        pop();
        return cons(p4, p5);
    }
    else
        return p3;
}

NODE *eval_terpri(NODE *p)
{
    terpri();
    return nil;
}

NODE *eval_times(NODE *p)
{
    int a, b;
    NODE *q;

    p = p->tail;
    q = eval(p->head);
    if (q->f != eval_number)
        return nil;
    a = q->numer;
    b = q->denom;

    while (p->tail != nil)
    {
        p = p->tail;
        q = eval(p->head);
        if (q->f != eval_number)
            return nil;
        a *= q->numer;
        b *= q->denom;
    }
    return new_number(a, b);
}

NODE *eval_zerop(NODE *p)
{
    p = eval(arg1);
    if (p->f == eval_number && p->numer == 0)
        return t;
    else
        return nil;
}

int Quit(void)
{
    return quit;
}

void init_io(void)
{
    infile = stdin;
    outfile = stdout;
}

void cleanup(void)
{
    free(heap);
    free(free_ptr);
    free(stack);
    free(a);
    free(b);
}

void init(void)
{
    int i;

    heap = (NODE*)malloc(NNODE*sizeof(NODE));
    if (heap == NULL)
    {
        exit(1);
    }
    free_ptr = (NODE**)malloc(NNODE*sizeof(NODE *));
    if (free_ptr == NULL)
    {
        exit(1);
    }
    stack = (NODE**)malloc(MAX*sizeof(NODE *));
    if (stack == NULL)
    {
        exit(1);
    }
    a = (int*)malloc(MAX*sizeof(int));
    if (a == NULL)
    {
        exit(1);
    }
    b = (int*)malloc(MAX*sizeof(int));
    if (b == NULL)
    {
        exit(1);
    }


    
    init_io();
    quit=0;
    nil = &_nil;
    nil->head = nil;
    nil->tail = nil;
    nil->binding = nil;
    nil->f = eval_nil;
    nil->s = "nil";

    arglist = nil;

    for (i = 0; i < NNODE; i++)
        free_ptr[i] = heap + i;

    free_count = NNODE;

    for (i = 0; i < 128; i++)
        symbol_table[i] = nil;

    lookup("and")->f = eval_and;
    lookup("append")->f = eval_append;
    /*TODO
     lookup("arg")->f = eval_arg;
     */
    lookup("arg1")->f = eval_arg1;
    lookup("arg2")->f = eval_arg2;
    lookup("arg3")->f = eval_arg3;
    lookup("arg4")->f = eval_arg4;
    lookup("arg5")->f = eval_arg5;
    lookup("arg6")->f = eval_arg6;
    lookup("arg7")->f = eval_arg7;
    lookup("arg8")->f = eval_arg8;
    lookup("arglist")->f = eval_arglist;
    lookup("atom")->f = eval_atom;
    lookup("head")->f = eval_head;
    lookup("tail")->f = eval_tail;
    /*TODO
     lookup("caar")->f = eval_caar;
    lookup("cadr")->f = eval_cadr;
    lookup("cdar")->f = eval_cdar;
    lookup("cddr")->f = eval_cddr;
    lookup("caaar")->f = eval_caaar;
    lookup("caadr")->f = eval_caadr;
    lookup("cadar")->f = eval_cadar;
    lookup("caddr")->f = eval_caddr;
    lookup("cdaar")->f = eval_cdaar;
    lookup("cdadr")->f = eval_cdadr;
    lookup("cddar")->f = eval_cddar;
    lookup("cdddr")->f = eval_cdddr;
*/
    lookup("cond")->f = eval_cond;
    lookup("while")->f = eval_while;
    lookup("cons")->f = eval_cons;
    lookup("define")->f = eval_define;
    lookup("eq")->f = eval_eq;
    lookup("equal")->f = eval_equal;
    lookup("eval")->f = eval_eval;
    lookup("exit")->f = eval_exit;
    lookup("expt")->f = eval_expt;
    lookup("fixp")->f = eval_fixp;
    lookup("gc")->f = eval_gc;
    lookup("goto")->f = eval_goto;
    lookup("greaterp")->f = eval_greaterp;
    lookup("lessp")->f = eval_lessp;
    lookup("list")->f = eval_list;
    lookup("load")->f = eval_load;
    lookup("nhead")->f = eval_nhead;
    lookup("not")->f = eval_not;
    lookup("null")->f = eval_null;
    lookup("numberp")->f = eval_numberp;
    lookup("onep")->f = eval_onep;
    lookup("or")->f = eval_or;
    lookup("pheads")->f = eval_pheads;
    lookup("plus")->f = eval_plus;
    lookup("prin")->f = eval_prin;
    lookup("print")->f = eval_print;
    lookup("product")->f = eval_product;
    lookup("product1")->f = eval_product1;
    lookup("prog")->f = eval_prog;
    lookup("quote")->f = eval_quote;
    lookup("quotient")->f = eval_quotient;
    lookup("return")->f = eval_return;
    lookup("reverse")->f = eval_reverse;
    lookup("setq")->f = eval_setq;
    lookup("sum")->f = eval_sum;
    lookup("sum1")->f = eval_sum1;
    lookup("subst")->f = eval_subst;
    lookup("terpri")->f = eval_terpri;
    lookup("times")->f = eval_times;
    lookup("zerop")->f = eval_zerop;

    t = lookup("t");
    eof = lookup("eof");
    quote = lookup("quote");
    im = lookup("i");
    lparen = lookup("lparen");
    rparen = lookup("rparen");
    period = lookup("period");
    quote_char = lookup("quote_char");
    power = lookup("power");
    product = lookup("product");
    sum = lookup("sum");
    tensor = lookup("tensor");
    nullstr = lookup("");
}

NODE *cons(NODE *p1, NODE *p2)
{
    NODE *p;
    if (free_count == 0)
    {
        push(p1);
        push(p2);
        gc();
        pop();
        pop();
    }
    p = free_ptr[--free_count];
    p->f = eval_pair;
    p->head = p1;
    p->tail = p2;
    p->binding = nil;
    return p;
}

NODE *new_number(int a, int b)
{
    int sgn, t, x, y;
    NODE *p;

    if (b == 0)
    {
        printf("divide by zero\n");
        return nil;
    }

    if (a == 0)
        b = 1;
    else
    {
        sgn = 1;
        if (a < 0)
        {
            a = -a;
            sgn *= -1;
        }
        if (b < 0)
        {
            b = -b;
            sgn *= -1;
        }
        x = a;
        y = b;
        while (y)
        {
            t = x % y;
            x = y;
            y = t;
        }
        a = sgn * a / x;
        b /= x;
    }

    if (free_count == 0)
        gc();
    p = free_ptr[--free_count];
    p->f = eval_number;
    p->head = nil;
    p->tail = nil;
    p->binding = nil;
    p->numer = a;
    p->denom = b;
    return p;
}

NODE *new_string(char *s)
{
    NODE *p;
    if (free_count == 0)
        gc();
    p = free_ptr[--free_count];
    p->f = eval_string;
    p->head = nil;
    p->tail = nil;
    p->binding = nil;
    p->s = strdup(s);
    return p;
}

NODE *new_symbol(char *s)
{
    NODE *p;
    if (free_count == 0)
        gc();
    p = free_ptr[--free_count];
    p->f = eval_symbol;
    p->head = nil;
    p->tail = nil;
    p->binding = p;
    p->s = strdup(s);
    return p;
}

void gc(void)
{
    int i;

    /* tag everything */

    for (i = 0; i < NNODE; i++)
        heap[i].tag = 1;

    /* untag what's used */

    untag(arglist);

    for (i = 0; i < 128; i++)
        untag(symbol_table[i]);

    for (i = 0; i < tos; i++)
        untag(stack[i]);

    /* collect everything that's still tagged */

    free_count = 0;

    for (i = 0; i < NNODE; i++)
        if (heap[i].tag)
            free_ptr[free_count++] = heap + i;

    if (free_count == 0) {
        printf("out of memory\n");
        exit(1);
    }
}
void untag(NODE *p)
{
    if (p->tag)
    {
        p->tag = 0;
        untag(p->head);
        untag(p->tail);
        untag(p->binding);
    }
}

void print(NODE *p)
{
    prin(p);
    terpri();
}

void prin(NODE *p)
{
    if (p->f == eval_pair)
        prin_list(p);
    else if (p->f == eval_number)
        if (p->denom == 1)
            fprintf(outfile, "%d", p->numer);
        else
            fprintf(outfile, "%d/%d", p->numer, p->denom);
    else
        fprintf(outfile, "%s", p->s);
}

void prin_list(NODE *p)
{
    fprintf(outfile, "(");
    prin(p->head);
    while (p->tail->f == eval_pair)
    {
        fprintf(outfile, " ");
        p = p->tail;
        prin(p->head);
    }
    if (p->tail == nil)
        fprintf(outfile, ")");
    else
    {
        fprintf(outfile, " . ");
        prin(p->tail);
        fprintf(outfile, ")");
    }
}

void terpri()
{
    fprintf(outfile, "\n");
}

NODE *lisp_read()
{
    NODE *p = ratom();
    if (p == lparen)
        p = read_list();
    else if (p == quote_char)
    {
        p = lisp_read();
        p = cons(p, nil);
        p = cons(quote, p);
    }
    return p;
}

NODE *read_list()
{
    NODE *p1, *p2;

    p1 = lisp_read();

    if (p1 == eof)
    {
        printf("unexpected eof\n");
        return nil;
    }

    if (p1 == rparen)
        return nil;

    if (p1 == period)
    {
        p1 = lisp_read();
        push(p1);
        p2 = ratom();
        pop();
        if (p2 != rparen)
            printf("missing ) after dot tail\n");
        return p1;
    }

    push(p1);
    p2 = read_list();
    pop();
    return cons(p1, p2);
}

/* read atom */

NODE *ratom()
{
    int c, k;
    NODE *p;

    /* skip spaces and comments */

    for (;;)
    {

        c = fgetc(infile);

        if (c == ';')
            while (c != EOF && c != '\n')
                c = fgetc(infile);

        if (c == EOF || c > ' ')
            break;
    }

    switch (c)
    {
    case EOF:
        return eof;
    case '.':
        return period;
    case '(':
        return lparen;
    case ')':
        return rparen;
    case '\'':
        return quote_char;
    case '\"':
        return read_string(buf);
    default:
        ungetc(c, infile);
        return read_symbol();
    }
}

NODE *read_string(char* buf)
{
    int c, i;

    for (i = 0; i < BUFLEN; i++)
    {
        c = fgetc(infile);
        if (c == EOF || c == '\"')
            break;
        buf[i] = c;
    }

    buf[i] = 0;

    if (i == BUFLEN)
        printf("input buffer overflow\n");

    return new_string(buf);
}

NODE *read_symbol()
{
    int c, i;
    NODE *p;

    for (i = 0; i < BUFLEN; i++)
    {
        c = fgetc(infile);
        if (c == EOF)
            break;
        if (eos(c))
        {
            ungetc(c, infile);
            break;
        }
        buf[i] = c;
    }

    buf[i] = 0;

    if (i == BUFLEN)
        printf("input buffer overflow\n");

    if (*buf == '+' || *buf == '-' || isdigit(*buf))
    {
        p = scan_number(buf);
        if (p == nil)
            printf("syntax error in number: %s\n", buf);
        return p;
    }
    else
        return lookup(buf);
}

/* end of symbol? */
int eos(int c)
{
    if (isspace(c))
        return 1;
    switch (c)
    {
    case '.':
    case ';':
    case '(':
    case ')':
    case '\'':
        return 1;
    default:
        return 0;
    }
}

NODE *scan_number(char*p)
{
    int a, b, sgn;

    a = 0;
    b = 1;
    sgn = 1;

    if (*p == '+')
        p++;
    else if (*p == '-')
    {
        p++;
        sgn = -1;
    }

    if (!isdigit(*p))
        return nil;

    while (isdigit(*p))
    {
        a = 10 * a + *p - '0';
        p++;
    }

    if (*p == '/')
    {
        p++;
        b = 0;
        while (isdigit(*p))
        {
            b = 10 * b + *p - '0';
            p++;
        }
        if (b == 0)
            return nil;
    }
    if (*p)
        return nil;
    return new_number(sgn * a, b);
}


void load(char *s)
{
    NODE *p;
    infile = fopen(s, "r");
    if (infile == NULL)
    {
        printf("fopen %s returns NULL\n", s);
        return;
    }
    for (;;)
    {
        p = lisp_read();
        if (p == eof)
            break;
        push(p);
        eval(p);
        pop();
    }
}

int equal(NODE *p1, NODE *p2)
{
    if (p1 == p2)
        return 1;
    else if (p1->f != p2->f)
        return 0;
    else if (p1->f == eval_number)
        if (p1->numer == p2->numer && p1->denom == p2->denom)
            return 1;
        else
            return 0;
    else if (p1->f == eval_string)
        if (strcmp(p1->s, p2->s) == 0)
            return 1;
        else
            return 0;
    else if (p1->f == eval_pair)
        if (equal(p1->head, p2->head))
            return equal(p1->tail, p2->tail);
        else
            return 0;
    else
        return 0;
}

/* lessp compares different objects too */

/* nil is less than anything else */

/* a number is greater than nil and less than anything else */

/* a list is greater than anything else */

/* objects between numbers and lists (symbols, strings and functions) */

/* are compared string-wise using print names */

/* lists are compared recursively using above rules */

/* why all this effort? */

/* so commutative symbolic expressions can have a canonical form */

int lessp(NODE *p1, NODE *p2)
{
    if (p1 == p2)
        return 0;

    if (p1 == nil)
        return 1;

    if (p2 == nil)
        return 0;

    if (p1->f == eval_number && p2->f == eval_number)
        if (p1->numer * p2->denom < p2->numer * p1->denom)
            return 1;
        else
            return 0;

    if (p1->f == eval_number)
        return 1;

    if (p2->f == eval_number)
        return 0;

    if (p1->f != eval_pair && p2->f != eval_pair)
        if (strcmp(p1->s, p2->s) < 0)
            return 1;
        else
            return 0;

    if (p1->f != eval_pair)
        return 1;

    if (p2->f != eval_pair)
        return 0;

    if (lessp(p1->head, p2->head))
        return 1;

    if (lessp(p2->head, p1->head))
        return 0;

    return lessp(p1->tail, p2->tail);
}

int isnum(NODE *p)
{
    if (p->f == eval_number)
        return 1;
    else
        return 0;
}

int iszero(NODE *p)
{
    if (p->f == eval_number && p->numer == 0)
        return 1;
    else
        return 0;
}

int isone(NODE *p)
{
    if (p->f == eval_number && p->numer == 1 && p->denom == 1)
        return 1;
    else
        return 0;
}

int gcd(int a, int b)
{
    int t;
    if (a < 0)
        a = -a;
    if (b < 0)
        b = -b;
    while (b)
    {
        t = a % b;
        a = b;
        b = t;
    }
    if (b < 0)
        a *= -1;
    return a;
}

NODE *ksum(int n)
{
    int a0, b0, h, i, j;
    NODE *p, *q;

    h = tos - n;

    /* add numbers */

    a0 = 0;
    b0 = 1;

    for (i = 0; i < n; i++)
    {
        p = stack[h + i];
        if (isnum(p))
        {
            add(&a0, &b0, p->numer, p->denom);
            stack[h + i] = nil;
        }
    }

    /* remove numeric coefficients */

    for (i = 0; i < n; i++)
    {
        p = stack[h + i];
        if (p->head == product && isnum(p->tail->head))
        {
            p = p->tail;
            a[h + i] = p->head->numer;
            b[h + i] = p->head->denom;
            p = p->tail;
            if (p->tail == nil)
                p = p->head;
            else
                p = cons(product, p);
            stack[h + i] = p;
        }
        else
        {
            a[h + i] = 1;
            b[h + i] = 1;
        }
    }

    /* combine terms */

    for (i = 0; i < n - 1; i++)
    {
        p = stack[h + i];
        if (p == nil)
            continue;
        for (j = i + 1; j < n; j++)
        {
            q = stack[h + j];
            if (q == nil)
                continue;
            if (equal(p, q))
            {
                add(a + h + i, b + h + i, a[h + j], b[h + j]);
                stack[h + j] = nil;
                if (a[h + i] == 0) {
                    stack[h + i] = nil;
                    break;
                }
            }
        }
    }

    /* put the coefficients back */

    for (i = 0; i < n; i++)
    {
        p = stack[h + i];
        if (p == nil)
            continue;
        if (a[h + i] == 1 && b[h + i] == 1)
            continue;
        q = new_number(a[h + i], b[h + i]);
        if (p->head == product)
            p = p->tail;
        else
        {
            push(q); /* save q from garbage collection */
            p = cons(p, nil);
            pop();
        }
        p = cons(q, p);
        p = cons(product, p);
        stack[h + i] = p;
    }

    /* add number */

    if (a0 != 0)
    {
        for (i = 0; i < n; i++)
            if (stack[h + i] == nil)
                break;
        if (i == n) {
            printf("bug in ksum()\n");
            exit(1);
        }
        stack[h + i] = new_number(a0, b0);
    }

    /* remove nils */

    j = 0;
    for (i = 0; i < n; i++)
        if (stack[h + i] != nil)
            stack[h + j++] = stack[h + i];

    /* sort terms */

    qsort(stack + h, j, sizeof (NODE *), cmp);

    /* result */

    switch (j)
    {
    case 0:
        p = new_number(0, 1);
        break;
    case 1:
        p = stack[h];
        break;
    default:
        p = nil;
        for (i = j - 1; i >= 0; i--)
            p = cons(stack[h + i], p);
        p = cons(sum, p);
        break;
    }

    tos = h;

    return p;
}


NODE *kproduct(int n)
{
    int a0, b0, flag, h, i, j;
    NODE *p, *q;

    h = tos - n;

    /* multiply numbers */

    a0 = 1;
    b0 = 1;

    for (i = 0; i < n; i++)
    {
        p = stack[h + i];
        if (isnum(p))
        {
            if (p->numer == 0)
            {
                tos = h;
                return new_number(0, 1);
            }
            multiply(&a0, &b0, p->numer, p->denom);
            stack[h + i] = nil;
        }
    }

    /* remove numeric exponents */

    /* note: removing exponent may yield sum */

    for (i = 0; i < n; i++)
    {
        a[h + i] = 1;
        b[h + i] = 1;
        p = stack[h + i];
        if (p->head != power)
            continue;
        if (isnum(arg2))
        {
            stack[h + i] = arg1;
            a[h + i] = arg2->numer;
            b[h + i] = arg2->denom;
        }
        else if (arg2->head == product && isnum(arg2->tail->head)) {
            if (arg2->tail->tail->tail == nil)
                q = arg2->tail->tail->head; // a of (product 2 a)
            else
                q = cons(product, arg2->tail->tail);
            q = cons(q, nil);
            q = cons(arg1, q);
            q = cons(power, q);
            stack[h + i] = q;
            a[h + i] = arg2->tail->head->numer;
            b[h + i] = arg2->tail->head->denom;
        }
    }

    /* combine factors */

    for (i = 0; i < n - 1; i++)
    {
        p = stack[h + i];
        if (p == nil)
            continue;
        for (j = i + 1; j < n; j++)
        {
            q = stack[h + j];
            if (q == nil)
                continue;
            if (p == im && q == im)
            {
                a0 *= -1;
                stack[h + i] = nil;
                stack[h + j] = nil;
                break;
            }
            else if (p->head == tensor && q->head == tensor)
            {
                p = append(p->tail, q->tail);
                p = cons(tensor, p);
                stack[h + i] = p;
                stack[h + j] = nil;
            }
            else if (equal(p, q))
            {
                add(a + h + i, b + h + i, a[h + j], b[h + j]);
                stack[h + j] = nil;
                if (a[h + i] == 0) {
                    stack[h + i] = nil;
                    break;
                }
            }
        }
    }

    /* restore exponents */

    for (i = 0; i < n; i++)
    {
        p = stack[h + i];
        if (p == nil)
            continue;
        if (a[h + i] == 1 && b[h + i] == 1)
            continue;
        q = new_number(a[h + i], b[h + i]);
        if (p->head == power)
        {
            if (arg2->head == product)
                q = cons(q, arg2->tail);
            else
            {
                push(q); /* save from garbage collection */
                q = cons(q, cons(arg2, nil));
                pop();
            }
            q = cons(product, q);
            q = cons(q, nil);
            q = cons(arg1, q);
        } else
            q = cons(p, cons(q, nil));
        q = cons(power, q);
        if (q->tail->head == sum && isnum(q->tail->tail->head))
            q = eval(q); /* possibly expand sum to power */
        stack[h + i] = q;
    }

    /* restore coefficient */

    if (a0 != 1 || b0 != 1)
    {
        for (i = 0; i < n; i++)
            if (stack[h + i] == nil)
                break;
        if (i == n)
        {
            printf("bug in kproduct()\n");
            exit(1);
        }
        stack[h + i] = new_number(a0, b0);
    }

    /* remove nils */

    j = 0;
    flag = 0;
    for (i = 0; i < n; i++)
    {
        p = stack[h + i];
        if (p != nil) {
            stack[h + j++] = p;
            if (p->head == sum)
                flag = 1;
        }
    }

    if (flag)
    {
        tos = h + j;
        return expand(j);
    }

    /* sort factors */

    qsort(stack + h, j, sizeof (NODE *), cmp);

    /* result */

    switch (j)
    {
    case 0:
        p = new_number(1, 1);
        break;
    case 1:
        p = stack[h];
        break;
    default:
        p = nil;
        for (i = j - 1; i >= 0; i--)
            p = cons(stack[h + i], p);
        p = cons(product, p);
        break;
    }

    tos = h;

    return p;
}

void multiply(int *pa, int *pb, int a, int b)
{
    int x;
    a *= *pa;
    b *= *pb;
    if (a == 0)
    {
        *pa = 0;
        *pb = 1;
    }
    else
    {
        x = gcd(a, b);
        *pa = a / x;
        *pb = b / x;
    }
}

NODE *expand(int n)
{
    NODE *p;
    int h, h1, h2, i, j, k, m;

    h = tos - n;

    /* a[i] = stack index of factor i */

    /* b[i] = number of terms in factor i */

    m = 1;
    for (i = 0; i < n; i++)
    {
        p = stack[h + i];
        if (p->head == sum)
        {
            a[h + i] = tos;
            j = 0;
            p = p->tail;
            while (p != nil)
            {
                push(p->head);
                p = p->tail;
                j++;
            }
        }
        else
        {
            a[h + i] = h + i;
            j = 1;
        }
        b[h + i] = j;
        m = j * m;
    }

    h1 = tos;

    for (i = 0; i < m; i++)
    {
        k = i;
        h2 = tos;
        for (j = 0; j < n; j++)
        {
            p = stack[a[h + j] + k % b[h + j]];
            k = k / b[h + j];
            if (p->head == product)
            {
                p = p->tail;
                while (p != nil)
                {
                    push(p->head);
                    p = p->tail;
                }
            }
            else
                push(p);
        }
        p = kproduct(tos - h2);
        if (p->head == sum)
        {
            p = p->tail;
            while (p != nil)
            {
                push(p->head);
                p = p->tail;
            }
        }
        else
            push(p);
    }

    p = ksum(tos - h1);

    tos = h;

    return p;
}

int get_int(NODE* p)
{
    if (p->f != eval_number)
    {
        printf("value not a number\n");
        exit(1);
    }
    return p->numer/p->denom;
}

float get_float(NODE* p)
{
    if (p->f != eval_number)
    {
        printf("value not a number\n");
        exit(1);
    }
    return ((float)p->numer)/p->denom;
}

NODE* get_arg(int index) // Get zero-base arg from command line
{
    int i;
    NODE* res=arglist;
    for (i=0;i<index;i++)
    {
        res=res->tail;
    }
    return res->head;
}


